/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer_flat.h"


#define DATA   (&flat_buffer->data)
#define VALID  (flat_buffer->valid)




static void
gimp_flat_buffer_delete (GimpBuffer * buffer)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  /* free any image memory we hold */
  d_uninit (DATA);

  /* clean up the base class */
  gimp_buffer_uninit (GIMP_BUFFER (flat_buffer));
  
  /* delete the object */
  g_free (flat_buffer);
}


static gboolean
gimp_flat_buffer_alloc (GimpBuffer   *buffer,
                        GimpPortion  *portion,
                        Alloc         how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  switch (how)
    {
    case ALLOC_ALLOC:
      if (d_alloc (DATA, (buffer->width *
                          buffer->height *
                          buffer->depth)) != TRUE)
        {
          g_warning ("alloc of flatbuffer failed");
          return FALSE;
        }
      return TRUE;

    case ALLOC_UNALLOC:
      if (d_unalloc (DATA) != TRUE)
        {
          g_warning ("unalloc of flatbuffer failed");
          return FALSE;
        }
      VALID = FALSE;
      return TRUE;

    case ALLOC_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}


static gboolean
gimp_flat_buffer_validate (GimpBuffer  *buffer,
                           GimpPortion *portion, 
                           Validate     how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  switch (how)
    {
    case VALIDATE_VALIDATE:
      if (VALID == TRUE)
        return TRUE;

      if (! d_is_alloced (DATA))
        {
          if (buffer->autoalloc == FALSE)
            return FALSE;
          
          if (gimp_flat_buffer_alloc (buffer, portion,
                                      ALLOC_ALLOC) != TRUE)
            {
              g_warning ("flatbuffer autoalloc failed");
              return FALSE;
            }
        }

      if (d_write (DATA) != TRUE)
        {
          g_warning ("flatbuffer validate use failed");
          return FALSE;
        }

      /* HOOK: PERFORM VALIDATION HERE */
      memset (DATA->data, 0, (buffer->width *
                              buffer->height *
                              buffer->depth));
      
      if (d_release (DATA) != TRUE)
        {
          g_warning ("flatbuffer validate unuse failed");
          return FALSE;
        }

      VALID = TRUE;
      return TRUE;


    case VALIDATE_INVALIDATE:
      if (d_usecount (DATA) != 0)
        {
          g_warning ("tried to invalidate an in-use flatbuffer");
          return FALSE;
        }

      VALID = FALSE;
      return TRUE;


    case VALIDATE_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}



static gboolean
gimp_flat_buffer_use (GimpBuffer  *buffer,
                      GimpPortion *portion, 
                      Use          how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  if ((VALID != TRUE) && ((how == USE_READ) ||
                          (how == USE_UPDATE) ||
                          (how == USE_WRITE)))
    {
      if (buffer->autovalidate == FALSE)
        return FALSE;
          
      if (gimp_flat_buffer_validate (buffer, portion,
                                     VALIDATE_VALIDATE) != TRUE)
        {
          g_warning ("flatbuffer autovalidate failed");
          return FALSE;
        }
    }

  switch (how)
    {
    case USE_READ:
    case USE_UPDATE:
      if (d_read (DATA) != TRUE)
        {
          g_warning ("flatbuffer clean use failed");
          return FALSE;
        }
      return TRUE;

    case USE_WRITE:
      if (d_write (DATA) != TRUE)
        {
          g_warning ("flatbuffer dirty use failed");
          return FALSE;
        }
      return TRUE;

    case USE_RELEASE:
      if (d_release (DATA) != TRUE)
        {
          g_warning ("flatbuffer unuse failed");
          return FALSE;
        }
      return TRUE;

    case USE_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}


static gboolean
gimp_flat_buffer_query (GimpBuffer    *buffer,
                        GimpPortion   *portion,
                        GimpMemStatus *status)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  status->alloced   = (d_is_alloced (DATA) ? TRUE : FALSE);
  status->valid     = VALID;
  status->usecount  = (d_usecount (DATA));

  return TRUE;
}


static gboolean
gimp_flat_buffer_data (GimpBuffer      *buffer,
                       GimpPortion     *portion,
                       GimpPixelArray  *array)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  /* describe the memory layout */
  array->tag       = buffer->tag;
  array->width     = MIN (buffer->width,
                          portion->focus.b.x) - portion->focus.a.x;
  array->height    = MIN (buffer->height,
                          portion->focus.b.y) - portion->focus.a.y;
  array->pixstride = buffer->depth;
  array->rowstride = buffer->depth * buffer->width;

  /* get a pointer to the memory */
  array->data = DATA;

  /* and save an offset to the first pixel */
  array->offset = (portion->focus.a.y * array->rowstride) +
                  (portion->focus.a.x * array->pixstride);

  return TRUE;
}


static GimpFlatBufferClass my_class =
{
  {
    STORAGE_FLAT,
    gimp_flat_buffer_delete,
    NULL, /* focus */
    gimp_flat_buffer_alloc,
    gimp_flat_buffer_validate,
    gimp_flat_buffer_use,
    gimp_flat_buffer_query,
    gimp_flat_buffer_data
  }
};






GimpFlatBuffer *
gimp_flat_buffer_new (Tag   tag,
                      gint  width,
                      gint  height)
{
  GimpFlatBuffer * flat_buffer;

  /* alloc a new object */
  flat_buffer = g_new (GimpFlatBuffer, 1);

  /* init the base class */
  gimp_buffer_init (GIMP_BUFFER (flat_buffer),
                    tag, width, height,
                    width, 0, height, 0);

  /* init the memory pointer */
  d_init (DATA);

  /* it has no valid data yet */
  VALID = FALSE;

  /* remember what sort of object we are */
  GIMP_BUFFER (flat_buffer)->klass = (void*) &my_class;

  /* return the new object */
  return flat_buffer;
}


