(define (qq) (load "memtest.scm"))


;; --------------------------------------------------------
;;
;;             direct access to memory state
;;

;; find field in fields and return corresponding data
(define (read-field field fields data)
  (cond
   ((null? fields) #f)
   ((null? data) #f)
   ((eq? field (car fields)) (car data))
   (#t (read-field field (cdr fields) (cdr data)))))

;; fetch a field from mem header
(define (mem-head mem field)
  (read-field field
              '(id usecount joined data)
              (car (gimpmem-info mem))))

;; fetch a field from mem body
(define (mem-body mem field)
  (read-field field
              '(data size ptrcount usecount sharable)
              (cadr (gimpmem-info mem))))




;; --------------------------------------------------------
;;
;;            high level access to memory state
;;

;; verify internal consistency of memptr
(define (mem-valid? mem)

  (define (valid-ptrs?)
    (let ((rc #f))

      ;; if joined, must be sharable
      (if (mem-head mem 'joined)
          (if (not (mem-body mem 'sharable))
              (set! rc "ptr-1")))

      ;; must have at least one ptr
      (if (< (mem-body mem 'ptrcount) 1)
          (set! rc "ptr-2"))

      ;; if many ptrs, must be joined and sharable
      (if (> (mem-body mem 'ptrcount) 1)
          (if (not (and (mem-body mem 'sharable)
                        (mem-head mem 'joined)))
              (set! rc "ptr-3")))

      (if rc
          (begin (display rc) #f)
          #t)))

  (define (valid-uses?)
    (let ((rc #f))

      ;; uses in head needs 1 use in body
      (if (> 0 (mem-head mem 'usecount))
          (if (< (mem-body mem 'usecount) 1)
              (set! rc "use-1")))
      
      ;; body uses <= ptrcount
      (if (> (mem-body mem 'usecount)
             (mem-body mem 'ptrcount))
          (set! rc "use-2"))

      (if rc
          (begin (display rc) #f)
          #t)))

  (define (valid-data?)
    (let ((rc #f))

      ;; body data non-null
      (if (equal? (mem-body mem 'data) "")
          (set! rc "data-1"))

      ;; size > 0
      (if (< (mem-body mem 'size) 1)
          (set! rc "data-2"))

      ;; head usecount 0 -> head data null
      (if (equal? (mem-head mem 'usecount) 0)
          (if (not (equal? (mem-head mem 'data) "(nil)"))
              (set! rc "data-3")))
                
      ;; head usecount>0 -> head data == body data
      (if (> (mem-head mem 'usecount) 0)
          (if (not (equal? (mem-head mem 'data)
                           (mem-body mem 'data)))
              (set! rc "data-4")))

      (if rc
          (begin (display rc) #f)
          #t)))

  ;; verify a mem
  (or (equal? (mem-head mem 'id) "")
      (and (valid-ptrs?)
           (valid-uses?)
           (valid-data?))))

;; get the major state of the memptr
(define (mem-state mem)
  (cond
   ((not (gimpmem? mem))          'UNINIT)
   ((equal? (mem-head mem 'id) "") 'UNALLOC)
   ((> (mem-body mem 'ptrcount) 1) 'SHARED)
   ((mem-body mem 'sharable)       'SHARABLE)
   (#t                             'UNSHARABLE)))





;; --------------------------------------------------------
;;
;;                  testing framework
;;

;; perform an action on a mem
(define (mem-do mem c)
  (let* ((cmd (car c)))
    (display c)
    (newline)
    (and (cond
          ((eq? cmd 'alloc)   (gimpmem-alloc mem (cadr c)))
          ((eq? cmd 'unalloc) (gimpmem-unalloc mem))
          ((eq? cmd 'read)    (gimpmem-use mem #f))
          ((eq? cmd 'write)   (gimpmem-use mem #t))
          ((eq? cmd 'release) (gimpmem-unuse mem))
          ((eq? cmd 'join)    (gimpmem-join mem (cadr c)))
          ((eq? cmd 'split)   (gimpmem-split mem))
          (#t #f))
         (mem-valid? mem))))

;; perform a series of actions on a mem
(define (run-test mem test)
  (or (null? test)
      (and (mem-do mem (car test))
           (run-test mem (cdr test)))))

(define the-test '((alloc 1000)
                   (write)
                   (read)
                   (release)
                   (release)))

(define (z) (run-test (make-gimpmem) the-test))
