/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gr_simple_correlator.h>
#include <gr_simple_framer_sync.h>
#include <gr_io_signature.h>
#include <assert.h>
#include <stdexcept>
#include <gr_count_bits.h>


static const int BITS_PER_BYTE = 8;
static const int OVERHEAD = 6;	    	   // 4 bytes sync, 1 byte seqno, 1 byte cmd
static const int THRESHOLD = 3;

gr_simple_correlator_sptr
gr_make_simple_correlator (int payload_bytesize)
{
  return gr_simple_correlator_sptr (new gr_simple_correlator (payload_bytesize));
}

gr_simple_correlator::gr_simple_correlator (int payload_bytesize)
  : gr_block ("simple_correlator",
	      gr_make_io_signature (1, 1, sizeof (float)),
	      gr_make_io_signature (1, 1, sizeof (unsigned char))),
    d_payload_bytesize (payload_bytesize),
    d_state (ST_LOOKING), d_seqno (0), d_bi (0), d_best_fit (0)
{
  d_bufsize = payload_bytesize + OVERHEAD;
  d_buf = new unsigned char [d_bufsize * BITS_PER_BYTE];
  d_pktbuf = new unsigned char [d_bufsize];

  set_output_multiple (payload_bytesize);

  enter_looking ();
}

gr_simple_correlator::~gr_simple_correlator ()
{
  delete d_buf;
  delete d_pktbuf;
}
   

void
gr_simple_correlator::enter_looking ()
{
  fprintf (stderr, ">>> enter_looking\n");
  d_state = ST_LOOKING;
  for (int i = 0; i < OVERSAMPLE; i++)
    d_shift_reg[i] = 0;
}

void
gr_simple_correlator::enter_locked ()
{
  fprintf (stderr, ">>> enter_locked\n");
  d_state = ST_LOCKED;
  d_bi = 0;
}

static void
packit (unsigned char *pktbuf, const unsigned char *bitbuf, int bitcount)
{
  for (int i = 0; i < bitcount; i += 8){
    int t = bitbuf[i+0] & 0x1;
    t = (t << 1) | (bitbuf[i+1] & 0x1);
    t = (t << 1) | (bitbuf[i+2] & 0x1);
    t = (t << 1) | (bitbuf[i+3] & 0x1);
    t = (t << 1) | (bitbuf[i+4] & 0x1);
    t = (t << 1) | (bitbuf[i+5] & 0x1);
    t = (t << 1) | (bitbuf[i+6] & 0x1);
    t = (t << 1) | (bitbuf[i+7] & 0x1);
    *pktbuf++ = t;
  }
}

inline static int slice (float x)
{
  return x >= 0 ? 0 : 1;	// FIXME sign?
}

int
gr_simple_correlator::general_work (int noutput_items,
				    gr_vector_int &ninput_items,
				    gr_vector_const_void_star &input_items,
				    gr_vector_void_star &output_items)
{
  const float *in = (const float *) input_items[0];
  unsigned char *out = (unsigned char *) output_items[0];

  
  int n = 0;
  int nin = ninput_items[0];
  nin = ((nin + OVERSAMPLE - 1) / OVERSAMPLE) * OVERSAMPLE;	// round down

  int decision;

  while (n < nin - OVERSAMPLE - 1){
    switch (d_state){
    case ST_LOOKING:
      for (int i = 0; i < OVERSAMPLE; i++){
	decision = slice (in[n]);
	d_shift_reg[i] = (d_shift_reg[i] << 1) | decision;
      }

      for (int i = 0; i < OVERSAMPLE; i++){
	int hamming_dist = gr_count_bits32 (d_shift_reg[i] ^ GRSF_SYNC);
	printf ("%2d\n", hamming_dist);
	if (hamming_dist < THRESHOLD){
	  d_best_fit = i;		// FIXME kludge
	  enter_locked ();
	  break;
	}
      }
      n += OVERSAMPLE;
      break;
      
    case ST_LOCKED:
      decision = slice (in[n + d_best_fit]);
      d_buf[d_bi++] = decision;
      if (d_bi >= d_bufsize){
	// got complete packet...  write it to output...
	// packet needs to be packed from bits to bytes.
	// FIXME do it.
	// print some stuff

	// packit (d_pktbuf, d_buf, d_bufsize);
	// printf ("seqno = %02x\n", d_pktbuf[4]);
	enter_looking ();
      }
      n += OVERSAMPLE;
      break;
      
    default:
      assert (0);
    }
  }

  consume_each (n);
  return 0;
}
