// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

namespace gcdriver {
using namespace ost;
using namespace std;

Session::Session(BayonneSpan *s, timeslot_t ts, device_t *devif) :
BayonneSession(&Driver::dialogic, ts, s),
BayonneMsgport(&Driver::dialogic),
TimerPort()
{
	devnode = devif;
	short wparm;
	char bparm;
	const char *cp;

	disconnecting = false;
	blocked = true;
	accepted = false;
	routing = true;
	waiting = false;
	callstate = GCST_NULL;
	crn = -1;

	logevents = &cerr;

	if(devif->chdev > -1)
		iomap[devif->chdev] = this;

	if(devif->tsdev > -1)
		iomap[devif->tsdev] = this;

	if(devif->linedev > -1)
		iomap[devif->linedev] = this;

	strcpy(msgname, logname);
	msgport = dynamic_cast<BayonneMsgport *>(this);
	crn = -1;

        intts.sc_numts = voxts.sc_numts = faxts.sc_numts = 1;
        voxts.sc_tsarrayp = &tsinfo[0];
        intts.sc_tsarrayp = &tsinfo[1];
        faxts.sc_tsarrayp = &tsinfo[2];
	callstate = GCST_NULL;

        digtpt[0].tp_type = IO_CONT;
        digtpt[0].tp_termno = DX_MAXDTMF;
        digtpt[0].tp_length = 1;
        digtpt[0].tp_flags = TF_MAXDTMF;
        digtpt[1].tp_type = IO_CONT;
        digtpt[1].tp_termno = DX_DIGMASK;
        digtpt[1].tp_flags = TF_DIGMASK;
        digtpt[2].tp_type = IO_CONT;
        digtpt[2].tp_termno = DX_MAXSIL;
        digtpt[2].tp_length = 0;
        digtpt[2].tp_flags = TF_10MS;
        digtpt[3].tp_type = IO_CONT;
        digtpt[3].tp_termno = DX_LCOFF;
        digtpt[3].tp_length = 3;
        digtpt[3].tp_flags = TF_LCOFF | TF_10MS;
        digtpt[4].tp_type = IO_EOT;
        digtpt[4].tp_termno = DX_MAXTIME;
        digtpt[4].tp_flags = TF_MAXTIME | TF_10MS;

        iofile.io_type = IO_DEV | IO_UIO | IO_EOT;
        iofile.io_bufp = 0;
        iofile.io_offset = 0;
        iofile.io_length = -1;
        iofile.io_fhandle = timeslot;
        iofile.io_nextp = NULL;

        iotone.io_type = IO_MEM | IO_EOT;
        iotone.io_offset = 0;
        iotone.io_fhandle = 0;
        iotone.io_nextp = NULL;

	cp = getKeyString("talkoff");
	if(cp)
	{
	        wparm = atoi(cp);
                dx_setparm(devnode->chdev, DXCH_DTMFTLK, (void *)&wparm);
        }

	cp = getKeyString("debounce");
	if(cp)
	{
		wparm = atoi(cp);
		dx_setparm(devnode->chdev, DXCH_DTMFDEB, (void *)&wparm);
	}

	bridge = BR_TDM;
	iface = devnode->iface;

	if(devnode->iface != IF_PSTN)
		return;

	cp = getKeyString("winktimer");
	if(cp)
	{
                bparm = atoi(cp) / 10;
		dx_setparm(devnode->chdev, DXCH_WINKLEN, (void *)&bparm);
	}

	cp = getKeyString("winkdelay");
	if(cp)
	{
                bparm = atoi(cp) / 10;
                dx_setparm(devnode->chdev, DXCH_WINKDLY, (void *)&bparm);
	}

	cp = getKeyString("minwink");
	if(cp)
	{
                bparm = atoi(cp) / 10;
		dx_setparm(devnode->chdev, DXCH_MINRWINK, (void *)&bparm);
	}

	cp = getKeyString("maxwink");
	if(cp)
	{
                bparm = atoi(cp) / 10;
		dx_setparm(devnode->chdev, DXCH_MAXRWINK, (void *)&bparm);
	}
}

void Session::initial(void)
{
}

void Session::run(void)
{
	Event event;
	timeout_t timeout;

	slog.debug("%s: msgport starting; chdev=%d, tsdev=%d, linedev=%d", 
		logname, devnode->chdev, devnode->tsdev, devnode->linedev);

	for(;;)
	{
		timeout = getRemaining();

		event.id = TIMER_EXPIRED;
		event.timeslot = timeslot;

		if(timeout)
		{
			if(timeout == TIMEOUT_INF)
				timeout = 0;

			wait(&event, timeout);
		}

		switch(event.id)
		{
		case MSGPORT_SHUTDOWN:
			Thread::sync();
		case MSGPORT_WAKEUP:
			continue;
		}
		putEvent(&event);
	}
}

Session::~Session()
{
	shutdown();
	stopChannel(EV_ASYNC);
	tsRelease();

        if(crn != -1)
        {
                gc_DropCall(crn, GC_NORMAL_CLEARING, EV_SYNC);
                gc_ReleaseCallEx(crn, EV_SYNC);
        }

	if(devnode->btype == SPRINGWARE)
		dx_close(devnode->chdev);
}

timeout_t Session::getRemaining(void)
{
	return TimerPort::getTimer();
}

void Session::startTimer(timeout_t timer)
{
	TimerPort::setTimer(timer);
	msgport->update();
}

void Session::stopTimer(void)
{
	TimerPort::endTimer();
	msgport->update();
}

void Session::stopChannel(int mode)
{
	if(devnode->chdev > -1)
	        if(dx_stopch(devnode->chdev, mode) < 0)
			slog.error("%s: dx_stopch failed; reason=%s",
				logname, ATDV_ERRMSGP(devnode->chdev));
}

void Session::tsRelease(void)
{
	if(devnode->tsdev > -1)
		nr_scunroute(devnode->tsdev, SC_DTI, devnode->chdev, SC_VOX, SC_FULLDUP);
	else
		nr_scunroute(devnode->chdev, SC_LSI, devnode->chdev, SC_VOX, SC_FULLDUP);
}

void Session::tsConnect(void)
{
	tsRelease();
	if(devnode->tsdev > -1)
		nr_scroute(devnode->tsdev, SC_DTI, devnode->chdev, SC_VOX, SC_FULLDUP);
	else
		nr_scroute(devnode->chdev, SC_LSI, devnode->chdev, SC_VOX, SC_FULLDUP);
}

bool Session::enterInitial(Event *event)
{
	if(devnode->iface == IF_PSTN)
		setOffhook(false);
	if(event->id == DEVICE_UNBLOCKED)
		setState(STATE_IDLE);
	return true;
}

bool Session::enterReset(Event *event)
{
	if(event->id == ENTER_STATE)
		resetChannel();
	else
		crn = -1;

	return false;
}

bool Session::enterPlay(Event *event)
{
        unsigned flags = MD_PCM | PM_SR8;

	switch(event->id)
	{
	case ENTER_STATE:
		if(state.audio.mode == Audio::modeReadAny)
                        update_pos = false;
                else
                        update_pos = true;
                audio.play(state.audio.list, state.audio.mode);
                if(!audio.isOpen())
                {
                        slog.error("%s: audio file access error", logname);
                        error("no-files");
                        setRunning();
                        return true;
                }
        	switch(audio.getEncoding())
        	{
		case Audio::pcm16Mono:
        	case Audio::mulawAudio:
                	flags = MD_PCM | PM_SR8;
                	break;
        	case Audio::alawAudio:
                	flags = MD_PCM | PM_ALAW | PM_SR8;
                	break;
        	case Audio::voxADPCM:
                	flags = MD_ADPCM | PM_SR6;
                	break;
	        default:
			update_pos = false;
                	slog.error("%s: unsupported codec", logname);
			error("no-codec");
			setRunning();
			return true;
		}
	        dx_play(devnode->chdev, &iofile, 
			Driver::dialogic.getPlayTPT(), EV_ASYNC | flags);

		return false;
	}
	return false;
}

bool Session::enterRecord(Event *event)
{
        unsigned flags = MD_PCM | PM_SR8;

	switch(event->id)
	{
	case ENTER_STATE:
		audio.record(state.audio.list[0], state.audio.mode, state.audio.note);
                if(!audio.isOpen())
                {
                        slog.error("%s: audio file access error", logname);
                        error("no-files");
                        setRunning();
                        return true;
                }
        	switch(audio.getEncoding())
        	{
        	case Audio::mulawAudio:
                	flags = MD_PCM | PM_SR8;
                	break;
        	case Audio::alawAudio:
                	flags = MD_PCM | PM_ALAW | PM_SR8;
                	break;
        	case Audio::voxADPCM:
                	flags = MD_ADPCM | PM_SR6;
                	break;
	        default:
			update_pos = false;
                	slog.error("%s: unsupported codec", logname);
			error("no-codec");
			setRunning();
			return true;
		}
		update_pos = true;
        	DV_TPT *rectpt = Driver::dialogic.getRecordTPT();

        	if(state.audio.silence)
        	{
                	if(Driver::dialogic.getBoardType() == DM3)
                        	rectpt[0].tp_length = state.audio.silence / 100;
                	else
                        	rectpt[1].tp_length = state.audio.silence / 100;
        	}
        	else
        	{
                	if(Driver::dialogic.getBoardType() == DM3)
                       		rectpt = NULL;
                	else
                        	rectpt[1].tp_length = 0;
        	}
		dx_rec(devnode->chdev, &iofile, rectpt, EV_ASYNC | flags);
		return false;
	}
	return false;
}


bool Session::enterPickup(Event *event)
{
	switch(event->id)
	{
	case ENTER_STATE:
		if(offhook)
		{
			setRunning();
			return true;
		}

		if(devnode->iface == IF_PSTN)
			return false;

		if(!callProgress(IN_BAND_INFO) || !acceptCall())
		{
			setState(STATE_HANGUP);
			return true;
		}
		return false;
	case STOP_DISCONNECT:
	case TIMER_EXPIRED:
		if(devnode->iface == IF_PSTN)
			return false;

		setState(STATE_HANGUP);
		return true;
	case CALL_ACCEPT:
		setRunning();
		return true;
	}
	return false;
}
					
bool Session::enterHangup(Event *event)
{
	switch(event->id)
	{
	case TIMER_EXPIRED:
		if(devnode->iface == IF_PSTN)
			setState(STATE_IDLE);
		setState(STATE_RESET);
		return true;
	case CALL_DISCONNECT:
		setState(STATE_RELEASE);
		return true;
	case ENTER_STATE:
		stopChannel(EV_ASYNC);
		setOffhook(false);

		exiting = true;
		newTid();
		
		if(thread)
			delete thread;

		if(image)
			BayonneSession::detach();

		if(!hangupCall())
		{
			setState(STATE_IDLE);
			return true;
		}
		startTimer(driver->getHangupTimer());
		return true;		
	}
	return false;
}

bool Session::enterIdle(Event *event)
{
	switch(event->id)
	{
	case ENTER_STATE:
		if(crn != -1)
			releaseCall();
		callstate = GCST_NULL;
		waitCall();
		update_pos = disconnecting = accepted = false;
		return false;
	case CALL_OFFERED:
		// ???
		return true;
	case DEVICE_BLOCKED:
		return true;
	case DEVICE_UNBLOCKED:
		return waitCall();
	}
	return false;
}

bool Session::enterRelease(Event *event)
{
	if(event->id == ENTER_STATE)
		releaseCall();
	else
		crn = -1;

	return false;
}

bool Session::enableDTMF(void)
{
	if(Driver::dialogic.getBoardType() == DM3)
	{
		if(dx_setevtmsk(devnode->chdev, DM_DIGITS) < 0)
			return false;
		return true;
	}
	if(dx_setevtmsk(devnode->chdev, DM_LCOFF|DM_LCON|DM_DIGITS) < 0)
		return false;
	return true;
}

void Session::disableDTMF(void)
{
	if(Driver::dialogic.getBoardType() == DM3)
		dx_setevtmsk(devnode->chdev, DM_DIGOFF);
	else if(devnode->iface == IF_PSTN)
		dx_setevtmsk(devnode->chdev, DM_LCOFF|DM_LCON|DM_DIGOFF|DM_RINGS);
	else
		dx_setevtmsk(devnode->chdev, DM_LCOFF|DM_LCON|DM_DIGOFF);
}

bool Session::waitCall(void)
{
        if(gc_WaitCall(devnode->linedev, NULL, NULL, -1, EV_ASYNC) == 0)
                return true;
        else
                return false;
}

bool Session::callProgress(int indicator)
{
        int state = -1;
        gc_GetCallState(crn, &state);
        slog(Slog::levelDebug) << logname << "State:: " << state << endl;

        if(gc_CallProgress(crn, indicator) == GC_SUCCESS)
                return true;
        else
                return false;
}

bool Session::acceptCall(void)
{
        if(offhook || accepted)
                return false;

        if(gc_AcceptCall(crn, 0, EV_ASYNC) < 0)
                return false;

        accepted = true;

        return true;
}

bool Session::answerCall(void)
{
        if(offhook)
                return false;

        if(gc_AnswerCall(crn, 0, EV_ASYNC) < 0)
                return false;

        offhook = true;
        return true;
}

bool Session::hangupCall(void)
{
        int cause;

        if(offhook && accepted)
                return false;

        disconnecting = true;

        switch(callstate)
        {
        case GCST_ACCEPTED:
        case GCST_OFFERED:
                cause = GC_CALL_REJECTED;
                break;
        default:
                cause = GC_NORMAL_CLEARING;
        }

        if(crn != -1)
        {
                if(gc_DropCall(crn, cause, EV_ASYNC) < 0)
                {
                        resetChannel();
                        return false;
                }
        }
        else resetChannel();

        offhook = false;
        return true;
}

void Session::releaseCall(void)
{
        disconnecting = false;

        if(gc_ReleaseCallEx(crn, EV_ASYNC) < 0)
		return;

        crn = -1;
}

void Session::resetChannel(void)
{
        gc_ResetLineDev(devnode->linedev, EV_ASYNC);
}

bool Session::postEvent(Event *event)
{
	bool rtn = putEvent(event);
	queEvent(NULL);			// reset timers...
	return rtn;
}

void Session::setOffhook(bool flag)
{
	offhook = flag;
	if(devnode->iface != IF_PSTN)
		return;

	if(offhook)
		dx_sethook(devnode->chdev, DX_OFFHOOK, EV_ASYNC);
	else
		dx_sethook(devnode->chdev, DX_ONHOOK, EV_ASYNC);
}

void Session::clrAudio(void)
{
	if(!audio.isOpen())
		goto skip;

	stopChannel(EV_ASYNC);	
	if(!update_pos)
		goto skip;

	audio.getPosition(audio.var_position, 12);
        update_pos = false;
skip:
	BayonneSession::clrAudio();
}	
				
} // namespace
