#!/usr/bin/env python

import sys
import math
from optparse import OptionParser

from gnuradio import gr, eng_notation
from gnuradio import audio
from gnuradio import usrp
from gnuradio import blks
from gnuradio.eng_option import eng_option

from gnuradio.wxgui import stdgui, fftsink
import wx

class nbfm_rx_graph (stdgui.gui_flow_graph):
    def __init__(self,frame,panel,vbox,argv):
        stdgui.gui_flow_graph.__init__ (self,frame,panel,vbox,argv)

        ########################################################
        # Parse command line arguments

        usage = "usage: %prog [options] freqency"
        parser = OptionParser(option_class=eng_option, usage=usage)
        parser.add_option("-N", "--no-gui", action="store_true", default=False)
        (options, args) = parser.parse_args()
        if len(args) != 1:
            parser.print_help()
            sys.exit(1)

        RF_freq = eng_notation.str_to_num (args[0])
        if RF_freq < 1e6:
            RF_freq *= 1e6

        no_gui = options.no_gui
        
        ########################################################
        # Set up basic constants and parameters

        adc_rate = 64e6                              # 64 MS/s
        IF_freq = 40e3
        
        while RF_freq > adc_rate:
            RF_freq = RF_freq - adc_rate
            
        if RF_freq > adc_rate/2:
            RF_freq = adc_rate - RF_freq
        else:
            RF_freq = RF_freq

        #usrp_decim = 250
        usrp_decim = 125
        IF_rate = adc_rate // usrp_decim              # 256 kS/s
        # IF_decim = 4
        IF_decim = 8
        quad_rate = IF_rate // IF_decim               # 64 kS/s
        audio_decimation = 4
        audio_rate = quad_rate // audio_decimation    # 16 kS/s


        max_dev = 5e3       # max deviation
        tau = 75e-6         # time constant for fm deemph filter
        
        ########################################################
        # Instantiate Blocks
        
        # usrp is data source
        src = usrp.source_c (0, usrp_decim)      # Set up USRP, set decim rate
        src.set_rx_freq (0, IF_freq - RF_freq)   # Tune it to correct freq
        src.set_pga(0,20)                        # Turn on internal amp, 20 dB on chan 0

        IF_freq = RF_freq + src.rx_freq(0)       # compute actual IF freq

        # Create filter to get actual channel we want
        chan_coeffs = gr.firdes.low_pass (1.0,                # gain
                                          IF_rate,            # sampling rate
                                          11e3,               # low pass cutoff freq
                                          4e3,                # width of trans. band
                                          gr.firdes.WIN_HANN) # filter type 

        print "len(chan_coeffs) =", len(chan_coeffs)
        # Decimating Channel filter with frequency translation
        # complex in and out, float taps
        ddc = gr.freq_xlating_fir_filter_ccf (IF_decim,       # decimation rate
                                              chan_coeffs,    # taps
                                              -IF_freq,       # frequency translation amount
                                              IF_rate)        # input sample rate

        # instantiate the guts of the single channel receiver
        fmrx = blks.nbfm_rx (self, audio_rate, quad_rate, max_dev=max_dev, tau=tau)

        # sound card as final sink
        audio_sink = audio.sink (int (audio_rate))
        
        # now wire it all together
        self.connect (src, ddc, fmrx, (audio_sink, 0))
        
        # FFT Displays
        if 1 and not(no_gui):
            usrp_fft, fft_win1 = \
                      fftsink.make_fft_sink_c (self, panel, "USRP Output", 512,
                                               IF_rate, 0, 100)
            self.connect (src, usrp_fft)
            vbox.Add (fft_win1, 1, wx.EXPAND)

        if 1 and not(no_gui):
            pre_demod, fft_win2 = \
                       fftsink.make_fft_sink_c (self, panel, "Pre - Demodulation", 512,
                                                quad_rate, 0, 140)
            self.connect (ddc, pre_demod)
            vbox.Add (fft_win2, 1, wx.EXPAND)

        if 0 and not(no_gui):
            post_deemph, fft_win3 = \
                         fftsink.make_fft_sink_f (self, panel, "With Deemph", 512,
                                                  IF_rate, -60, 40)
            self.connect (fmrx.deemph,post_deemph)
            vbox.Add (fft_win3, 1, wx.EXPAND)

        if 0 and not(no_gui):
            post_filt, fft_win4 = \
                       fftsink.make_fft_sink_f (self, panel, "Post Filter", 512,
                                                audio_rate, -60, 40)
            self.connect (fmrx.audio_filter,post_filt)
            vbox.Add (fft_win4, 1, wx.EXPAND)
        

def parseargs (args):
    nargs = len (args)
    if nargs == 1:
        return( float (args[0]))
    else:
        sys.stderr.write ('usage: nbfm_rcv freq\n')
        sys.exit (1)

def main():
    app = stdgui.stdapp (nbfm_rx_graph, "NBFM RX")
    app.MainLoop ()
    
if __name__ == '__main__':
    main()
