#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

__all__ = ['tv_rx']

import math
def int_seq_to_str (seq):
    """convert a sequence of integers into a string"""
    return ''.join (map (chr, seq))

def str_to_int_seq (str):
    """convert a string to a list of integers"""
    return map (ord, str)

def control_byte_1(fast_tuning_p, reference_divisor):
    c = 0x88
    if fast_tuning_p:
        c |= 0x40
    if reference_divisor == 512:
        c |= 0x3 << 1
    elif reference_divisor == 640:
        c |= 0x0 << 1
    elif reference_divisor == 1024:
        c |= 0x1 << 1
    else:
        assert 0
    return c

def control_byte_2(target_freq, shutdown_tx_PGA):
    if target_freq < 158e6:        # VHF low
        c = 0xa0
    elif target_freq < 464e6:      # VHF high
        c = 0x90
    else:                          # UHF
        c = 0x30
    if shutdown_tx_PGA:
        c |= 0x08
    return c

class tv_rx (object):
    def __init__ (self, usrp, which):
        """
        Control microtune 4937 based USRP daughterboard.
        
        @param usrp: instance of usrp.source_c
        @param which: which slot: 0 or 1 corresponding to slot RX_A or RX_B respectively
        @type which: int
        """
        self._u = usrp
        self._which = which
        assert (0 <= which and which <= 1)
        self._i2c_addr = (0x60, 0x61)[which]

        self._first_IF = 43.75e6
        self._reference_divisor = 640
        self._fast_tuning = False
        
        # set default values
        self._set_pga(0)
        self._set_rfagc(0)
        self._set_ifagc(0)
        
    # Gain setting
    def _set_rfagc(self,gain):
        assert gain <= 60 and gain >= 0
        if gain == 60:
            voltage = 4
        else:
            voltage = gain/60.0 * 2.25 + 1.25
        dacword = int(4096*voltage/1.22/3.3)    # 1.22 = opamp gain

        assert dacword>=0 and dacword<4096
        self.rfagc = dacword
        self._u.write_aux_dac(self._which, 1, dacword)

    def _set_ifagc(self,gain):
        assert gain <= 35 and gain >= 0
        if gain == 35:
            voltage = 3.5
        else:
            voltage = gain/35.0 * 2.1 + 1.4
        dacword = int(4096*voltage/1.22/3.3)    # 1.22 = opamp gain

        assert dacword>=0 and dacword<4096
        self.rfagc = dacword
        self._u.write_aux_dac(self._which, 0, dacword)

    def _set_pga(self,pga_gain):
        assert pga_gain >=0 and pga_gain <=20
        if(self._which == 0):
            self._u.set_pga (0, pga_gain)
        else:
            self._u.set_pga (2, pga_gain)
            
    def set_gain(self,gain):
        assert gain>=0 and gain<=115
        if gain>60:
            rfgain = 60
            gain = gain - 60
        else:
            rfgain = gain
            gain = 0
        if gain > 35:
            ifgain = 35
            gain = gain - 35
        else:
            ifgain = gain
            gain = 0
        pgagain = gain
        self._set_rfagc(rfgain)
        self._set_ifagc(ifgain)
        self._set_pga(pgagain)
        
    def set_freq(self, target_freq):
        """
        Set RF frequency.

        @returns (ok, actual_freq) where ok is True or False and actual_freq
        is the actual frequency tuned to.
        """

        target_f_osc = target_freq + self._first_IF;
        f_ref = 4e6 / self._reference_divisor

        divisor = int((target_f_osc + (f_ref * 4)) / (f_ref * 8))
        actual_freq = (f_ref * 8 * divisor) - self._first_IF

        if (divisor & ~0x7fff) != 0:		# must be 15-bits or less
            return False, 0
        
        # build i2c command string
        buf = [0] * 4
        buf[0] = (divisor >> 8) & 0xff          # DB1
        buf[1] = divisor & 0xff                 # DB2
        buf[2] = control_byte_1(self._fast_tuning, self._reference_divisor)
        buf[3] = control_byte_2(actual_freq, True)

        ok = self._u.write_i2c(self._i2c_addr, int_seq_to_str (buf))

        return (ok, actual_freq)
