/* filefuncs.c: Implementation of file-based functions in Meta-HTML. */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Fri Mar 22 13:54:13 1996.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#include "language.h"

static void pf_get_file_properties (PFunArgs);
static void pf_set_file_properties (PFunArgs);
static void pf_directory_contents (PFunArgs);
static void pf_file_exists (PFunArgs);

static char *unix_username_from_uid (uid_t uid);
static void unix_add_date (Symbol *sym, long ticks);

static PFunDesc func_table[] =
{
  { "GET-FILE-PROPERTIES",	0, 0, pf_get_file_properties },
  { "SET-FILE-PROPERTIES",	0, 0, pf_set_file_properties },
  { "DIRECTORY-CONTENTS",	0, 0, pf_directory_contents },
  { "FILE-EXISTS",		0, 0, pf_file_exists },
  { (char *)NULL,		0, 0, (PFunHandler *)NULL }
};

PACKAGE_INITIALIZER (initialize_file_functions)
DOC_SECTION (FILE-OPERATORS)

/* <get-file-properties /www/nirvana/docs/welcome.mhtml>

     returns:
       ((name . "welcome.mhtml")
        (full-name . "/www/nirvana/docs/welcome.mhtml")
	(size . 2188)
	(type . FILE)
	(created 1 21 95 01 01 00 298398348)
	(written 1 21 95 01 01 00 298349349)
	(read    3 21 96 01 01 00 928948938)
	(creator . bfox))

     Values for TYPE: FILE, DIRECTORY, EXECUTABLE. */
static char *
get_file_properties (char *filename)
{
  char *result = (char *)NULL;

  if (filename != (char *)NULL)
    {
      struct stat finfo;
      int error;

      error = stat (filename, &finfo);

      if (error != -1)
	{
	  Package *package = symbol_get_package ((char *)NULL);
	  Symbol *sym;
	  char buffer[128];

	  /* FULL-NAME... */
	  sym = symbol_intern_in_package (package, "FULL-NAME");
	  symbol_add_value (sym, filename);

	  /* NAME... */
	  {
	    char *temp = strrchr (filename, '/');

	    if (temp != (char *)NULL)
	      temp++;
	    else
	      temp = filename;

	    sym = symbol_intern_in_package (package, "NAME");
	    symbol_add_value (sym, temp);
	  }

	  /* SIZE... */
	  sprintf (buffer, "%ld", (long)finfo.st_size);
	  sym = symbol_intern_in_package (package, "SIZE");
	  symbol_add_value (sym, buffer);

	  /* CREATED... */
	  sym = symbol_intern_in_package (package, "CREATED");
	  unix_add_date (sym, finfo.st_ctime);

	  /* WRITTEN... */
	  sym = symbol_intern_in_package (package, "WRITTEN");
	  unix_add_date (sym, finfo.st_mtime);

	  /* READ... */
	  sym = symbol_intern_in_package (package, "READ");
	  unix_add_date (sym, finfo.st_atime);

	  /* TYPE... */
	  sym = symbol_intern_in_package (package, "TYPE");
	  if (S_ISDIR (finfo.st_mode))
	    symbol_add_value (sym, "DIRECTORY");
	  else if ((S_IXOTH & finfo.st_mode))
	    symbol_add_value (sym, "EXECUTABLE");
	  else if (S_ISREG (finfo.st_mode))
	    symbol_add_value (sym, "FILE");

	  /* CREATOR... */
	  sym = symbol_intern_in_package (package, "CREATOR");
	  symbol_add_value (sym, unix_username_from_uid (finfo.st_uid));

	  result = package_to_alist (package, 0);
	  symbol_destroy_package (package);
	}
    }

  return (result);
}

static void
unix_add_date (Symbol *sym, long ticks)
{
  struct tm *ti = localtime ((const time_t *)&ticks);
  char buffer[40];

  sprintf (buffer, "%02d", ti->tm_mon + 1);
  symbol_add_value (sym, buffer);

  sprintf (buffer, "%02d", ti->tm_mday);
  symbol_add_value (sym, buffer);

  sprintf (buffer, "%02d", ti->tm_year);
  symbol_add_value (sym, buffer);

  sprintf (buffer, "%02d", ti->tm_hour);
  symbol_add_value (sym, buffer);

  sprintf (buffer, "%02d", ti->tm_min);
  symbol_add_value (sym, buffer);

  sprintf (buffer, "%02d", ti->tm_sec);
  symbol_add_value (sym, buffer);

  sprintf (buffer, "%ld", (unsigned long) ticks);
  symbol_add_value (sym, buffer);
}

static char *
unix_username_from_uid (uid_t uid)
{
  static uid_t last_uid = -1;
  static char last_name[50];

  if (uid != last_uid)
    {
      struct passwd *entry = getpwuid (uid);

      last_name[0] = '\0';
      last_uid = uid;

      if (entry != (struct passwd *)NULL)
	strcpy (last_name, entry->pw_name);

    }
  return (last_name);
}

/* File properties are returned in an alist representing operating system
   information about the named path.  Pathnames must be given fully; they
   are *not* relative to web space in any way.  This is probably a huge
   security hole, but it can't be any worse than CGI-EXEC.  Yeeesh. */
DEFUN (pf_get_file_properties, pathname,
"Return an association-list containing operating system information
about the file or directory named by <var path>. <var pathname> must be
given fully; it is <i>not</i> relative to Web space in any way.

If the file exists and is accessible, the members of the returned
association-list which are guaranteed to be present are:

<table border>
  <tr>
    <td align=left><b>Member</b></td>
    <td align=left><b>Value</b></td>
    <td align=left><b>Description</b></td>
  </tr>
  <tr>
     <td align=left> <b>NAME</b></td>
     <td align=left><i>welcome.mhtml</i></td>
     <td align=left>
        The name of the file or directory, without any of the path
        information.
     </td>
  </tr>
  <tr>
     <td align=left><b>FULL-NAME</b></td>
     <td align=left>/www/site/docs/welcome.mhtml</td>
     <td align=left>
       The name of the file or directory, with full path
       information.  This should be identical to <var PATH> as
       received by <b><code>get-file-properties</code></b>.
     </td>
   </tr>
   <tr>
     <td align=left> <b>SIZE</b> </td>
     <td align=left>2188</td>
     <td align=left>
       The size of the file in bytes.
     </td>
   </tr>
   <tr>
     <td align=left><b>TYPE</b></td>
     <td align=left>FILE</td>
     <td align=left>
        The <i>type</i> of the file.  This will either be
       <b><code>FILE</code></b> or <b><code>DIRECTORY</code></b>.
     </td>
   </tr>
   <tr>
     <td align=center colspan=3>
       <b>In addition to the above fields, the following fields appear
       on Unix based systems.</b>
     </td>
   </tr>
   <tr>
     <td align=left><b>CREATED</b></td>
     <td align=left>6 29 96 10 3 24</td>
     <td align=left>
       The date on which this file was created.  The value is an
       array, with ordered values being: month, day, year, hours,
       minutes, and seconds.
     </td>
   </tr>
   <tr>
     <td align=left><b>WRITTEN</b></td>
     <td align=left>6 29 96 10 3 24</td>
     <td align=left>
       The date on which this file was last written.  The value is an
       array, with ordered values being: month, day, year, hours,
       minutes, and seconds.
     </td>
    </tr>
    <tr>
      <td align=left><b>READ</b></td>
      <td align=left>6 30 96 19 27 51</td>
      <td align=left>
        The date on which this file was last read.  The value is an
        array, with ordered values being: month, day, year, hours,
        minutes, and seconds.
      </td>
    </tr>
    <tr>
       <td align=left><b>CREATOR</b></td>
       <td align=left>bfox</td>
       <td align=left>
         The system identifier of the user who created this file.
       </td>
     </tr>
</table>

Example:
<complete-example>
<get-file-properties />
<complete-example>")
{
  char *filename = mhtml_evaluate_string (get_positional_arg (vars, 0));
  char *result = get_file_properties (filename);

  if (result != (char *)NULL)
    {
      bprintf_insert (page, start, "%s", result);
      *newstart += strlen (result);
      free (result);
    }

  if (filename != (char *)NULL)
    free (filename);
}

DEFUN (pf_file_exists, pathname,
"Returns \"true\" if <var pathname> is an existing file or directory.

<var pathname> is absolute -- it is looked up in absolute file system
space, not in Web space.  To find out if a file <code>foo</code> in
the current directory exists, use:
<example>
<file-exists <thisdir>/foo>
</example>")
{
  char *filename = mhtml_evaluate_string (get_positional_arg (vars, 0));

  if (!empty_string_p (filename))
    {
      struct stat finfo;

      if ((stat (filename, &finfo)) != -1)
	{
	  bprintf_insert (page, start, "true");
	  *newstart += 4;
	  if (debug_level > 5)
	    page_debug ("<file-exists %s> --> true", filename);
	}
      else if (debug_level > 5)
	page_debug ("<file-exists %s> -->", filename);
    }
}

/* Not yet implemented. */
static void
pf_set_file_properties (PFunArgs)
{
}

DEFUN (pf_directory_contents,
       pathname &optional package-name &key matching=pattern,
"Returns a newline separated list of association lists for
the files matching <var pattern>.

When <var package-name> is supplied, each variable in
<var package-name> is the name of a file in <var pathname>, and
the value of each variable is the association list for that
file.")
{
  char *dirname = mhtml_evaluate_string (get_positional_arg (vars, 0));
  char *matching = mhtml_evaluate_string (get_value (vars, "matching"));

  if (!empty_string_p (dirname))
    {
      DIR *dir = opendir (dirname);
  
      if (dir != (DIR *)NULL)
	{
	  char *packname;
	  Package *package;
	  Symbol *sym;
	  struct dirent *entry;
	  char buffer[1024];
	  int offset = 0;
	  regex_t re;
	  regmatch_t where[2];
	  int check_re = 0;

	  if (!empty_string_p (matching))
	    {
	      regcomp (&re, matching, REG_EXTENDED);
	      check_re = 1;
	    }

	  packname = mhtml_evaluate_string (get_positional_arg (vars, 1));
	  if ((packname != (char *)NULL) && empty_string_p (packname))
	    package = CurrentPackage;
	  else
	    package = symbol_get_package (packname);

	  sprintf (buffer, "%s", dirname);
	  offset = strlen (buffer);
	  if (buffer[offset - 1] != '/')
	    buffer[offset++] = '/';

	  while ((entry = readdir (dir)) != (struct dirent *)NULL)
	    {
	      char *info;

	      strncpy (&buffer[offset], entry->d_name, D_NAMELEN (entry));
	      buffer[offset + D_NAMELEN (entry)] = '\0';

	      if ((check_re == 0) ||
		  (regexec (&re, buffer + offset, 1, where, 0) == 0))
		{
		  info = get_file_properties (buffer);
		  sym = symbol_intern_in_package (package, buffer + offset);
		  symbol_add_value (sym, info);
		  if (info != (char *)NULL)
		    free (info);
		}
	    }

	  if (packname != (char *)NULL)
	    {
	      free (packname);
	    }
	  else
	    {
	      char *result = package_to_alist (package, 0);
	      if (!empty_string_p (result))
		{
		  bprintf_insert (page, start, "%s\n", result);
		  *newstart += 1 + strlen (result);
		}

	      if (result != (char *)NULL)
		free (result);
	    }

	  if (check_re)
	    regfree (&re);
	}
    }

  if (matching != (char *)NULL) free (matching);
  if (dirname != (char *)NULL) free (dirname);
}
