/* PSPP - a program for statistical analysis.
   Copyright (C) 2015 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#ifndef BARCHART_H
#define BARCHART_H

#include "libpspp/str.h"
#include "libpspp/hmap.h"
#include "data/value.h"
#include "output/chart.h"


struct category
{
  struct hmap_node node;
  int idx;                     /* Unique zero based index */
  struct string label;         /* The label to be displayed for this category */
  union value val;             /* The value of this category */
  int width;                   /* The width of VAL */
};


struct barchart
  {
    struct chart chart;

    /* Should the chart be displayed as percentages */
    bool percent;

    /* The categories */
    struct freq **cats;

    /* The total number of categories (regardless of level) */
    int n_nzcats;

    /* The number of primary categories */
    int n_pcats;

    /* The largest count of all the categories */
    double largest;

    /* The sum of all the counts */
    double total_count;

    /* The label for the ordinate (vertical axis) */
    char *ylabel;

    /* The variables holding the categorical values */
    const struct variable **var;
    int n_vars;

    int widths[2];

    /* A hash table of struct category indexed by VAL */
    struct hmap primaries;

    /* A hash table of struct category indexed by VAL */
    struct hmap secondaries;

    /* A array of pointers to the members of the above hmap,
       (the secondaries) sorted by VAL */
    struct category **ss;
  };


struct variable;
struct freq;

struct chart *barchart_create (const struct variable **, int n_vars,
				    const char *ylabel, bool percent,
                                    struct freq *const *, int n_cats);

/* This boilerplate for barchart, a subclass of chart, was
   autogenerated by mk-class-boilerplate. */

#include <assert.h>
#include "libpspp/cast.h"

extern const struct chart_class barchart_class;

/* Returns true if SUPER is a barchart, otherwise false. */
static inline bool
is_barchart (const struct chart *super)
{
  return super->class == &barchart_class;
}

/* Returns SUPER converted to barchart.  SUPER must be a barchart, as
   reported by is_barchart. */
static inline struct barchart *
to_barchart (const struct chart *super)
{
  assert (is_barchart (super));
  return UP_CAST (super, struct barchart, chart);
}

/* Returns INSTANCE converted to chart. */
static inline struct chart *
barchart_super (const struct barchart *instance)
{
  return CONST_CAST (struct chart *, &instance->chart);
}

/* Increments INSTANCE's reference count and returns INSTANCE. */
static inline struct barchart *
barchart_ref (const struct barchart *instance)
{
  return to_barchart (chart_ref (&instance->chart));
}

/* Decrements INSTANCE's reference count, then destroys INSTANCE if
   the reference count is now zero. */
static inline void
barchart_unref (struct barchart *instance)
{
  chart_unref (&instance->chart);
}

/* Returns true if INSTANCE's reference count is greater than 1,
   false otherwise. */
static inline bool
barchart_is_shared (const struct barchart *instance)
{
  return chart_is_shared (&instance->chart);
}

static inline void
barchart_submit (struct barchart *instance)
{
  chart_submit (&instance->chart);
}

#endif /* output/charts/barchart.h */
