/*
 * Copyright (c) 2021, Oracle and/or its affiliates.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2.0, as
 * published by the Free Software Foundation.
 *
 * This program is also distributed with certain software (including
 * but not limited to OpenSSL) that is licensed under separate terms,
 * as designated in a particular file or component or in included license
 * documentation.  The authors of MySQL hereby grant you an
 * additional permission to link the program and your derivative works
 * with the separately licensed software that they have included with
 * MySQL.
 *
 * Without limiting anything contained in the foregoing, this file,
 * which is part of MySQL Connector/Node.js, is also subject to the
 * Universal FOSS Exception, version 1.0, a copy of which can be found at
 * http://oss.oracle.com/licenses/universal-foss-exception.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License, version 2.0, for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
 */

'use strict';

/* eslint-env node, mocha */

const config = require('../../../config');
const expect = require('chai').expect;
const fixtures = require('../../../fixtures');
const mysqlx = require('../../../..');

describe('regressions introduced by server patches related to prepared statements', () => {
    const baseConfig = { schema: config.schema || 'mysql-connector-nodejs_test' };

    let collection, schema, session;

    beforeEach('create default schema', () => {
        return fixtures.createSchema(baseConfig.schema);
    });

    beforeEach('create session using default schema', () => {
        const defaultConfig = Object.assign({}, config, baseConfig);

        return mysqlx.getSession(defaultConfig)
            .then(s => {
                session = s;
            });
    });

    beforeEach('load default schema', () => {
        schema = session.getDefaultSchema();
    });

    beforeEach('create collection', () => {
        return schema.createCollection('test')
            .then(c => {
                collection = c;
            });
    });

    afterEach('drop default schema', () => {
        return session.dropSchema(schema.getName());
    });

    afterEach('close session', () => {
        return session.close();
    });

    context('BUG#33331322', () => {
        let autoGeneratedIds;

        beforeEach('add fixtures', () => {
            return collection
                .add({ name: 'foo' })
                .add({ name: 'bar' })
                .execute()
                .then(res => {
                    autoGeneratedIds = res.getGeneratedIds();
                });
        });

        it('hits a match when performing the index lookup for auto generated ids', () => {
            const docs = [{ _id: autoGeneratedIds[0], name: 'foo' }, { _id: autoGeneratedIds[1], name: 'bar' }];
            const stmt = collection.find('_id = :id');

            return stmt.bind('id', autoGeneratedIds[0])
                .execute()
                .then(res => {
                    return expect(res.fetchOne()).to.deep.equal(docs[0]);
                })
                .then(() => {
                    return stmt.bind('id', autoGeneratedIds[1])
                        .execute();
                })
                .then(res => {
                    return expect(res.fetchOne()).to.deep.equal(docs[1]);
                })
                .then(() => {
                    // check if the index was used
                    return session.sql('select EVENT_NAME, NO_INDEX_USED, NO_GOOD_INDEX_USED from performance_schema.events_statements_history ORDER BY EVENT_ID desc limit 1')
                        .execute();
                })
                .then(res => {
                    // the last prepared statement should have both
                    // NO_INDEX_USED and NO_GOOD_INDEX_USED set to 0
                    return expect(res.fetchOne()).to.deep.equal(['statement/com/Execute', 0, 0]);
                });
        });
    });
});
