/*
  +----------------------------------------------------------------------+
  | PHP Version 6                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Georg Richter <georg@mysql.com>                             |
  |          Andrey Hristov <andrey@mysql.com>                           |
  |          Ulf Wendel <uwendel@mysql.com>                              |
  +----------------------------------------------------------------------+
*/

/* $Id: header,v 1.17 2006/01/01 13:09:48 sniper Exp $ */
#include "php.h"
#include "ext/standard/url.h"
#include "mysqlnd.h"
#include "mysqlnd_priv.h"

#define MYSQLND_SILENT




#define mysqlnd_array_init(arg, field_count) \
{ \
	ALLOC_HASHTABLE_REL(Z_ARRVAL_P(arg));\
	zend_u_hash_init(Z_ARRVAL_P(arg), (field_count), NULL, ZVAL_PTR_DTOR, 0, 0);\
	Z_TYPE_P(arg) = IS_ARRAY;\
}


extern MYSQLND_CHARSET *mysqlnd_charsets;

/* {{{ mysqlnd_command_to_text 
 */
static
const char * const mysqlnd_command_to_text[COM_END] =
{
  "SLEEP", "QUIT", "INIT_DB", "QUERY", "FIELD_LIST",
  "CREATE_DB", "DROP_DB", "REFRESH", "SHUTDOWN", "STATISTICS",
  "PROCESS_INFO", "CONNECT", "PROCESS_KILL", "DEBUG", "PING",
  "TIME", "DELAYED_INSERT", "CHANGE_USER", "BINLOG_DUMP",
  "TABLE_DUMP", "CONNECT_OUT", "REGISTER_SLAVE",
  "STMT_PREPARE", "STMT_EXECUTE", "STMT_SEND_LONG_DATA", "STMT_CLOSE",
  "STMT_RESET", "SET_OPTION", "STMT_FETCH", "DAEMON"
};
/* }}} */


static const char * mysqlnd_server_gone = "MySQL server has gone away";
static const char * mysqlnd_out_of_sync = "Commands out of sync; you can't run this command now";


/* {{{ php_mysqlnd_free_field_metadata */
void php_mysqlnd_free_field_metadata(MYSQLND_FIELD *meta)
{
	if (meta) {
		if (meta->root) {
			efree(meta->root);
			meta->root = NULL;
		}
		if (meta->def) {
			efree(meta->def);
			meta->def = NULL;
		}
	}
}
/* }}} */


/* {{{ mysqlnd_unbuffered_free_last_data */
void mysqlnd_unbuffered_free_last_data(MYSQLND_RES *result)
{
	if (result->last_row_data) {
		int i;
		for (i = 0; i < result->field_count; i++) {
			if (result->type == MYSQLND_RES_PS) {
				/* For now PS always does create copies when fetching data */
				zval_ptr_dtor(&result->last_row_data[i]);			
			} else {
				mysqlnd_palloc_zval_ptr_dtor(&(result->last_row_data[i]), result->zval_cache, FALSE);
			}
		}
		/* Free last row's zvals */
		efree(result->last_row_data);
		result->last_row_data = NULL;
	}
	if (result->last_row_buffer) {
		/* Nothing points to this buffer now, free it */
		efree(result->last_row_buffer);
		result->last_row_buffer = NULL;
	}
}
/* }}} */


/* {{{ mysqlnd_internal_free_result_contents */
void mysqlnd_internal_free_result_contents(MYSQLND_RES *result)
{
	int i;
	MYSQLND_FIELD *meta = result->fields;

	/* The user has used the old API - mysqlnd_fetch_row() */
	if (result->last_row) {
		efree(result->last_row);
		result->last_row = NULL;
	}

	if (!result->data) {
		mysqlnd_unbuffered_free_last_data(result);
	} else {
		zval **current_row;
		zend_uchar *current_buffer;
		int j;

		for (i = 0; i < result->row_count; i++) {
			current_row = result->data[i];
			current_buffer = result->row_buffers[i];
			for (j = 0; j < result->field_count; j++) {
				if (result->type == MYSQLND_RES_PS) {
					/* For now PS always does create copies when fetching data */
					zval_ptr_dtor(&current_row[j]);
				} else {
					/* Free only if we haven't referenced it */
					mysqlnd_palloc_zval_ptr_dtor(&(current_row[j]), result->zval_cache, FALSE);
				}
			}
			efree(current_row);
			efree(current_buffer);
		}
		efree(result->data);
		efree(result->row_buffers);
		result->data		= NULL;
		result->row_buffers	= NULL;
		result->data_cursor = NULL;
		result->row_count = 0;
	}
	if (result->lengths) {
		efree(result->lengths);
		result->lengths = NULL;
	}

	result->conn = NULL;

	if (meta) {
		i = result->field_count;
		while (i--) {
			php_mysqlnd_free_field_metadata(meta++);
		}
		efree(result->fields);
		result->fields = NULL;
	}
	if (result->row_packet) {
		if (result->type == MYSQLND_RES_NORMAL) {
			PACKET_FREE((php_mysql_packet_row*) result->row_packet);
		} else {
			PACKET_FREE((php_mysql_packet_row*) result->row_packet);
		}
		result->row_packet = NULL;
	}

	if (result->zend_hash_keys) {
		efree(result->zend_hash_keys);
		result->zend_hash_keys = NULL;
	}

	if (result->row_buffer) {
		efree(result->row_buffer);
		result->row_buffer = NULL;
		result->row_buffer_len = 0;
	}
	if (result->zval_cache) {
		mysqlnd_palloc_free_cache_reference(&result->zval_cache);
	}
}
/* }}} */


/* {{{ mysqlnd_internal_free_result */
void mysqlnd_internal_free_result(MYSQLND_RES *result TSRMLS_DC)
{
	/*
	  result->conn is an address if this is an unbuffered query.
	  In this case, decrement the reference counter in the connection
	  object and if needed free the latter. If quit_sent is no
	*/
	if (result->conn && (!(--result->conn->references)) && result->conn->state == CONN_QUIT_SENT) {
		/*
		  No multithreading issues as we don't share the connection :)
		  This will free the object too, of course because references has
		  reached zero.
		*/
		mysqlnd_conn_free(result->conn TSRMLS_CC);
		result->conn = NULL;
	}
	mysqlnd_internal_free_result_contents(result);
	efree(result);
}
/* }}} */


/* {{{ mysqlnd_conn_free_contents */
static
void mysqlnd_conn_free_contents(MYSQLND *conn TSRMLS_DC)
{
	mysqlnd_local_infile_default(conn);
	if (conn->current_result) {
		mysqlnd_internal_free_result_contents(conn->current_result);
		efree(conn->current_result);
		conn->current_result = NULL;
	}

	if (conn->stream) {
		php_stream_free(conn->stream, (conn->persistent) ? PHP_STREAM_FREE_RSRC_DTOR : PHP_STREAM_FREE_CLOSE);
		conn->stream = NULL;
	}
	if (conn->host) {
		pefree(conn->host, conn->persistent);
		conn->host = NULL;
	}
	if (conn->user) {
		pefree(conn->user, conn->persistent);
		conn->user = NULL;
	}
	if (conn->passwd) {
		pefree(conn->passwd, conn->persistent);
		conn->passwd = NULL;
	}
	if (conn->unix_socket) {
		pefree(conn->unix_socket, conn->persistent);
		conn->unix_socket = NULL;
	}
	if (conn->scheme) {
		pefree(conn->scheme, conn->persistent);
		conn->scheme = NULL;
	}
	if (conn->server_version) {
		pefree(conn->server_version, conn->persistent);
		conn->server_version = NULL;
	}
	if (conn->host_info) {
		pefree(conn->host_info, conn->persistent);
		conn->host_info = NULL;
	}
	if (conn->scramble) {
		pefree(conn->scramble, conn->persistent);
		conn->scramble = NULL;
	}
	if (conn->last_message) {
		pefree(conn->last_message, conn->persistent);
		conn->last_message = NULL;
	}
	if (conn->options.num_commands) {
		uint i;
		for (i=0; i < conn->options.num_commands; i++) {
			pefree(conn->options.init_commands[i], conn->persistent);
		}
		pefree(conn->options.init_commands, conn->persistent);
		conn->options.init_commands = NULL;
	}
	if (conn->options.cfg_file) {
		pefree(conn->options.cfg_file, conn->persistent);
		conn->options.cfg_file = NULL;
	}
	if (conn->options.cfg_section) {
		pefree(conn->options.cfg_section, conn->persistent);
		conn->options.cfg_section = NULL;
	}
	if (conn->options.ssl_key) {
		pefree(conn->options.ssl_key, conn->persistent);
		conn->options.ssl_key = NULL;
	}
	if (conn->options.ssl_cert) {
		pefree(conn->options.ssl_cert, conn->persistent);
		conn->options.ssl_cert = NULL;
	}
	if (conn->options.ssl_ca) {
		pefree(conn->options.ssl_ca, conn->persistent);
		conn->options.ssl_ca = NULL;
	}
	if (conn->options.ssl_capath) {
		pefree(conn->options.ssl_capath, conn->persistent);
		conn->options.ssl_capath = NULL;
	}
	if (conn->options.ssl_cipher) {
		pefree(conn->options.ssl_cipher, conn->persistent);
		conn->options.ssl_cipher = NULL;
	}
	if (conn->zval_cache) {
		mysqlnd_palloc_free_cache_reference(&conn->zval_cache);
		conn->zval_cache = NULL;
	}
	if (conn->cmd_buffer.buffer) {
		pefree(conn->cmd_buffer.buffer, conn->persistent);
		conn->cmd_buffer.buffer = NULL;
	}
}
/* }}} */


/* {{{ mysqlnd_conn_free */
void mysqlnd_conn_free(MYSQLND *conn TSRMLS_DC)
{
	mysqlnd_conn_free_contents(conn TSRMLS_CC);

	/* If there are result sets attached to us, don't clean. They will do */
	if (!conn->references) {
		pefree(conn, conn->persistent);
	}
}
/* }}} */


/* {{{ mysqlnd_conn__dtor */
void mysqlnd_conn_rsrc_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC)
{
	MYSQLND *conn = (MYSQLND *) rsrc->ptr;

#ifndef MYSQLND_SILENT
	php_printf("CLOSING CONNECTION\n");
#endif
	/* be silent */
	conn->tmp_int = 1;
	mysqlnd_close(conn TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_result_rsrc_dtor */
void mysqlnd_result_rsrc_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC)
{
#ifndef MYSQLND_SILENT
	php_printf("CLOSING RESULT\n");
#endif
	mysqlnd_free_result((MYSQLND_RES *) rsrc->ptr TSRMLS_CC);
	rsrc->ptr = NULL;
}
/* }}} */


/* {{{ mysqlnd_stmt_rsrc_dtor */
void mysqlnd_stmt_rsrc_dtor(zend_rsrc_list_entry *rsrc TSRMLS_DC)
{
#ifndef MYSQLND_SILENT
	php_printf("CLOSING STMT\n");
#endif
	mysqlnd_stmt_close((MYSQLND_STMT *) rsrc->ptr TSRMLS_CC);
	efree(rsrc->ptr);
}
/* }}} */


/* {{{ mysqlnd_simple_command_handle_response */
enum_func_status
mysqlnd_simple_command_handle_response(MYSQLND *conn, enum php_mysql_packet_type ok_packet,
									   zend_bool silent, enum php_mysqlnd_server_command command
									   TSRMLS_DC)
{
	enum_func_status ret;
	switch (ok_packet) {
		case PROT_OK_PACKET:{
			php_mysql_packet_ok ok_response;
			PACKET_INIT_ALLOCA(ok_response, PROT_OK_PACKET);
			if (FAIL == (ret = PACKET_READ_ALLOCA(ok_response, conn))) {
				if (!silent) {
					php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error while reading %s's OK packet",
									 mysqlnd_command_to_text[command]);
				}
			} else {
#ifndef MYSQLND_SILENT
				php_printf("\tOK from server\n");
#endif
				if (0xFF == ok_response.field_count) {
					/* The server signalled error. Set the error */
					SET_CLIENT_ERROR(conn->error_info, ok_response.error_no,
									 ok_response.sqlstate, ok_response.error); 
					ret = FAIL;
					/*
					  Cover a protocol design error: error packet does not
					  contain the server status. Therefore, the client has no way
					  to find out whether there are more result sets of
					  a multiple-result-set statement pending. Luckily, in 5.0 an
					  error always aborts execution of a statement, wherever it is
					  a multi-statement or a stored procedure, so it should be
					  safe to unconditionally turn off the flag here.
					*/
					conn->upsert_status.server_status &= ~SERVER_MORE_RESULTS_EXISTS;
					
				} else {
					SET_NEW_MESSAGE(conn->last_message, conn->last_message_len,
									ok_response.message, ok_response.message_len);

					conn->upsert_status.warning_count = ok_response.warning_count;
					conn->upsert_status.server_status = ok_response.server_status;
					conn->upsert_status.affected_rows = ok_response.affected_rows;
					conn->upsert_status.last_insert_id = ok_response.last_insert_id;
				}
			}
			PACKET_FREE_ALLOCA(ok_response);
			break;
		}
		case PROT_EOF_PACKET:{
			php_mysql_packet_eof ok_response;
			PACKET_INIT_ALLOCA(ok_response, PROT_EOF_PACKET);
			if (FAIL == (ret = PACKET_READ_ALLOCA(ok_response, conn))) {
				SET_CLIENT_ERROR(conn->error_info, CR_MALFORMED_PACKET, UNKNOWN_SQLSTATE,
								 "Malformed packet");
				if (!silent) {
					php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error while reading %s's EOF packet",
								 	mysqlnd_command_to_text[command]);
				}
			} else if (0xFE != ok_response.field_count) {
				SET_CLIENT_ERROR(conn->error_info, CR_MALFORMED_PACKET, UNKNOWN_SQLSTATE,
								 "Malformed packet");
				if (!silent) {
					php_error_docref(NULL TSRMLS_CC, E_WARNING,
									"EOF packet expected, field count wasn't 0xFE but 0x%2X",
									ok_response.field_count);
				}
			} else {
#ifndef MYSQLND_SILENT
				php_printf("\tOK from server\n");
#endif
			}
			PACKET_FREE_ALLOCA(ok_response);
			break;
		}
		default:
			ret = FAIL;
			SET_CLIENT_ERROR(conn->error_info, CR_MALFORMED_PACKET, UNKNOWN_SQLSTATE,
							 "Malformed packet");
			php_error_docref(NULL TSRMLS_CC, E_ERROR, "Wrong response packet %d passed to the function",
							 ok_packet);
			break;
	}
	return ret;
}
/* }}} */


/* {{{ mysqlnd_simple_command */
enum_func_status
mysqlnd_simple_command(MYSQLND *conn, enum php_mysqlnd_server_command command,
					   const char * const arg, size_t arg_len,
					   enum php_mysql_packet_type ok_packet, zend_bool silent TSRMLS_DC)
{
	enum_func_status ret = PASS;
	php_mysql_packet_command cmd_packet;

	switch (conn->state) {
		case CONN_READY:
			break;
		case CONN_QUIT_SENT:
			SET_CLIENT_ERROR(conn->error_info, CR_SERVER_GONE_ERROR, UNKNOWN_SQLSTATE, mysqlnd_server_gone); 
			return FAIL;	
		default:
			SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
							 mysqlnd_out_of_sync); 
			return FAIL;
	}

	/* clean UPSERT info */
	memset(&conn->upsert_status, 0, sizeof(conn->upsert_status));
	conn->upsert_status.affected_rows = (mynd_ulonglong) ~0;
	SET_EMPTY_ERROR(conn->error_info);

	PACKET_INIT_ALLOCA(cmd_packet, PROT_CMD_PACKET);
	cmd_packet.command = command;
	if (arg && arg_len) {
		cmd_packet.argument = arg;
		cmd_packet.arg_len  = arg_len;
	}

	/* Reset packet_no, or we will get bad handshake! */
	conn->packet_no = 0;

	if (! PACKET_WRITE_ALLOCA(cmd_packet, conn)) {
		if (!silent) {
			php_error(E_WARNING, "Error while sending %s packet", mysqlnd_command_to_text[command]);
		}
		SET_CLIENT_ERROR(conn->error_info, CR_SERVER_GONE_ERROR, UNKNOWN_SQLSTATE, mysqlnd_server_gone); 
		ret = FAIL;
	} else if (ok_packet != PROT_LAST) {
		ret = mysqlnd_simple_command_handle_response(conn, ok_packet, silent, command TSRMLS_CC);
	}

	/*
	  There is no need to call FREE_ALLOCA on cmd_packet as the
	  only allocated string is cmd_packet.argument and it was passed
	  to us. We should not free it.
	*/

	return ret;
}
/* }}} */


/* {{{ mysqlnd_set_server_option */
enum_func_status
mysqlnd_set_server_option(MYSQLND *conn, enum_mysqlnd_server_option option TSRMLS_DC)
{
	char buffer[2];
	int2store(buffer, (uint) option);
	return mysqlnd_simple_command(conn, COM_SET_OPTION, buffer, sizeof(buffer), PROT_EOF_PACKET, FALSE TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_autocommit */
enum_func_status
mysqlnd_autocommit(MYSQLND *conn, zend_bool mode TSRMLS_DC)
{
	return mysqlnd_query(conn, mode ? "set autocommit=1":"set autocommit=0", 16 TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_commit */
enum_func_status
mysqlnd_commit(MYSQLND *conn TSRMLS_DC)
{
	return mysqlnd_query(conn, "COMMIT", 6 TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_rollback */
enum_func_status
mysqlnd_rollback(MYSQLND *conn TSRMLS_DC)
{
	return mysqlnd_query(conn, "ROLLBACK", 8 TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_start_psession */
void mysqlnd_start_psession(MYSQLND *conn) 
{

}
/* }}} */


/* {{{ mysqlnd_end_psession */
void mysqlnd_end_psession(MYSQLND *conn)
{

}
/* }}} */


/* {{{ mysqlnd_init */
MYSQLND *mysqlnd_init(zend_bool persistent)
{
	MYSQLND *ret = pecalloc(1, sizeof(MYSQLND), persistent);

	ret->upsert_status.affected_rows = (mynd_ulonglong) ~0;
	ret->persistent = persistent;

	return ret;
}
/* }}} */


/* {{{ mysqlnd_connect */
MYSQLND *mysqlnd_connect(MYSQLND *conn,
						 char *host, char *user,
						 char *passwd, unsigned int passwd_len,
						 char *db, unsigned int db_len,
						 unsigned int port,
						 char *socket,
						 unsigned int mysql_flags,
						 MYSQLND_ZVAL_PCACHE *zval_cache TSRMLS_DC)
{
	char *transport = NULL, *errstr = NULL;
	int transport_len, errcode = 0;
	unsigned int streams_options = ENFORCE_SAFE_MODE;
	unsigned int streams_flags = STREAM_XPORT_CLIENT | STREAM_XPORT_CONNECT;
	zend_bool self_alloced = FALSE;
	struct timeval tv;
	zend_bool unix_socket = FALSE;

	php_mysql_packet_greet greet_packet;
	php_mysql_packet_auth *auth_packet;
	php_mysql_packet_ok ok_packet;

	if (conn && conn->state != CONN_ALLOCED) {
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 mysqlnd_out_of_sync);
		return NULL;
	}

	if (!host) {
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Host is not valid");
		return NULL;
	}
	if (!port && !socket) {
		port = 3306;
	}
	if (!strncasecmp(host, "localhost", sizeof("localhost") - 1)) {
		if (!socket) {
			socket = "/tmp/mysql.sock";
		}
		transport_len = spprintf(&transport, 0, "unix://%s", socket);
		unix_socket = TRUE;
	} else {
		transport_len = spprintf(&transport, 0, "tcp://%s:%d", host, port);
	}

	PACKET_INIT_ALLOCA(greet_packet, PROT_GREET_PACKET);
	PACKET_INIT(auth_packet, PROT_AUTH_PACKET, php_mysql_packet_auth *);
	PACKET_INIT_ALLOCA(ok_packet, PROT_OK_PACKET);

	if (!conn) {
		conn = ecalloc(1, sizeof(MYSQLND));
		self_alloced = TRUE;
	}

	conn->state	= CONN_ALLOCED;
	conn->packet_no = 0;

	if (conn->options.timeout_connect) {
		tv.tv_sec = conn->options.timeout_connect;
		tv.tv_usec = 0;
	}
	if (conn->persistent) {
		conn->scheme = pestrndup(transport, transport_len, 1);
		efree(transport);
	} else {
		conn->scheme = transport;
	}
	conn->stream = php_stream_xport_create(conn->scheme, transport_len, streams_options, streams_flags,
										   (conn->persistent) ? "mysqli_plink" : NULL, 
										   (conn->options.timeout_connect) ? &tv : NULL,
										   NULL /*ctx*/,
										   &errstr, &errcode);
	
	if (errstr || !conn->stream) {
		goto err;
	}

	if (conn->options.timeout_read)
	{
		tv.tv_sec = conn->options.timeout_read;
		tv.tv_usec = 0;
		php_stream_set_option(conn->stream, PHP_STREAM_OPTION_READ_TIMEOUT, 0, &tv);
	}

#if A0
	if (!unix_socket) {
		/* Set TCP_NODELAY */
		conn->stream->ops->set_option(conn->stream, PHP_STREAM_OPTION_NODELAY, 0, NULL TSRMLS_CC);
	}
#endif

	if (FAIL == PACKET_READ_ALLOCA(greet_packet, conn)) {
#ifndef MYSQLND_SILENT
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error while reading greeting packet");
#endif
		goto err;
	}

	if (greet_packet.pre41) {
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Connecting to MySQL Server before 4.1"
						" is not supported. Server is %-.32s", greet_packet.server_version);
		SET_CLIENT_ERROR(conn->error_info, CR_NOT_IMPLEMENTED, UNKNOWN_SQLSTATE,
						 "Connecting to 3.22, 3.23 & 4.0 servers is not supported"); 
		goto err;	
	}

	conn->thread_id			= greet_packet.thread_id;
	conn->protocol_version	= greet_packet.protocol_version;
	conn->server_version	= greet_packet.server_version;
	greet_packet.server_version = NULL; /* The string will be freed otherwise */

	/* we allow load data local infile by default */
	mysql_flags  |= CLIENT_LOCAL_FILES;

	auth_packet->user		= user? user:"";
	auth_packet->password	= passwd? passwd:"";
	auth_packet->charset_no	= greet_packet.charset_no;
	auth_packet->db			= db;
	auth_packet->db_len		= db_len;
	auth_packet->max_packet_size= 3UL*1024UL*1024UL*1024UL;
	auth_packet->client_flags= mysql_flags;

	auth_packet->server_scramble_buf = pemalloc(SCRAMBLE_LENGTH, conn->persistent);
	memcpy(auth_packet->server_scramble_buf, greet_packet.scramble_buf, SCRAMBLE_LENGTH);
	PACKET_WRITE(auth_packet, conn);

	if (FAIL == PACKET_READ_ALLOCA(ok_packet, conn) || ok_packet.field_count >= 0xFE) {
		pefree(auth_packet->server_scramble_buf, conn->persistent);
		if (ok_packet.field_count == 0xFE) {
			/* pre41 server !*/
			php_error_docref(NULL TSRMLS_CC, E_WARNING, "mysqlnd cannot connect to MySQL < 4.1 servers");
		} else if (ok_packet.field_count == 0xFF) {
			if (ok_packet.sqlstate[0]) {
				if (!self_alloced) {
					strncpy(conn->error_info.sqlstate, ok_packet.sqlstate, sizeof(conn->error_info.sqlstate));
				}
#ifndef MYSQLND_SILENT
				php_error_docref(NULL TSRMLS_CC, E_WARNING, "ERROR:%d [SQLSTATE:%s] %s",
								 ok_packet.error_no, ok_packet.sqlstate, ok_packet.error);
#endif
			} else {
#ifndef MYSQLND_SILENT
				php_error_docref(NULL TSRMLS_CC, E_WARNING, "ERROR:%d %s",
								 ok_packet.error_no, ok_packet.error);
#endif
			}
			if (!self_alloced) {
				conn->error_info.error_no = ok_packet.error_no;
				strncpy(conn->error_info.error, ok_packet.error, sizeof(conn->error_info.error));
			}
		}
	} else {
		conn->state				= CONN_READY;
		/* disable multi_query */
		if (FAIL ==	mysqlnd_set_server_option(conn, MYSQL_OPTION_MULTI_STATEMENTS_OFF TSRMLS_CC)) {
			conn->state			= CONN_ALLOCED;
			goto err;
		}
		conn->multistatements = FALSE; 

		conn->user				= pestrdup(user, conn->persistent);
		conn->passwd			= pestrndup(passwd, passwd_len, conn->persistent);
		conn->port				= port;
		if (host && !socket) {
			char *p;

			conn->host = pestrdup(host, conn->persistent);
			spprintf(&p, 0, "MySQL host info: %s via TCP/IP", conn->host);
			if (conn->persistent) {
				conn->host_info = pestrdup(p, 1);
				efree(p);
			} else {
				conn->host_info = p;
			}
		} else {
			conn->unix_socket	= pestrdup(socket, conn->persistent);
			conn->host_info		= pestrdup("MySQL host info: Localhost via UNIX socket", conn->persistent);
		}
		conn->client_flag		= auth_packet->client_flags;
		conn->max_packet_size	= auth_packet->max_packet_size;
		conn->scramble			= auth_packet->server_scramble_buf;
		/* todo: check if charset is available */
		conn->charset			= mysqlnd_find_charset_nr(greet_packet.charset_no);
		conn->server_capabilities = greet_packet.server_capabilities;
		conn->upsert_status.warning_count = 0;
		conn->upsert_status.server_status = greet_packet.server_status;
		SET_NEW_MESSAGE(conn->last_message, conn->last_message_len,
						ok_packet.message, ok_packet.message_len);

		SET_EMPTY_ERROR(conn->error_info);

		PACKET_FREE_ALLOCA(greet_packet);
		PACKET_FREE(auth_packet);
		PACKET_FREE_ALLOCA(ok_packet);

		conn->zval_cache = mysqlnd_palloc_get_cache_reference(zval_cache);
		conn->cmd_buffer.length = 128L*1024L;
		conn->cmd_buffer.buffer = pemalloc(conn->cmd_buffer.length, conn->persistent);

		return conn;
	}
err:
	PACKET_FREE_ALLOCA(greet_packet);
	PACKET_FREE(auth_packet);
	PACKET_FREE_ALLOCA(ok_packet);

	if (conn->scheme) {
		pefree(transport, conn->persistent);
		conn->scheme = NULL;
	}
	if (errstr) {
		conn->error_info.error_no = errcode;
		strncpy(conn->error_info.error, errstr, sizeof(conn->error_info.error));
		conn->error_info.sqlstate[0] = '\0';

		php_error_docref(NULL TSRMLS_CC, E_WARNING, "[%d] %.64s", errcode, errstr);
		efree(errstr);
	}
	/* This will also close conn->stream if it has been opened */
	mysqlnd_conn_free_contents(conn TSRMLS_CC);

	if (self_alloced) {
		/*
		  We have alloced, thus there are no references to this
		  object - we are free to kill it!
		*/
		pefree(conn, conn->persistent);
	}
	return NULL;
}
/* }}} */


/* {{{ mysqlnd_read_result_metadata */
enum_func_status
mysqlnd_read_result_metadata(MYSQLND *conn, MYSQLND_RES *result TSRMLS_DC)
{
	int i = 0;	
	php_mysql_packet_res_field field_packet;

	/* Make it safe to call it repeatedly for PS */
	if (!result->fields) {
		result->fields = ecalloc(result->field_count, sizeof(MYSQLND_FIELD));
	}
	if (!result->zend_hash_keys) {
		result->zend_hash_keys = emalloc(result->field_count * sizeof(unsigned long));
	}
	/* 1. Read all fields metadata */

	/* It's safe to reread without freeing */
	PACKET_INIT_ALLOCA(field_packet, PROT_RSET_FLD_PACKET);
	for (;i < result->field_count; i++) {

		if (result->fields[i].root) {
			/* We re-read metadata for PS */
			efree(result->fields[i].root);
			result->fields[i].root = NULL;
		}

		field_packet.metadata = &(result->fields[i]);
		if (FAIL == PACKET_READ_ALLOCA(field_packet, conn)) {
			PACKET_FREE_ALLOCA(field_packet);
			goto error;
		}
		result->zend_hash_keys[i] = zend_get_hash_value(field_packet.metadata->name,
														field_packet.metadata->name_length + 1);
	}
	PACKET_FREE_ALLOCA(field_packet);

	/*
	  2. Follows an EOF packet, which the client of mysqlnd_read_result_metadata()
	     should consume.
	  3. If there is a result set, it follows. The last packet will have 'eof' set
	  	 If PS, then no result set follows.
	*/

	return PASS;
error:
	mysqlnd_internal_free_result_contents(result);
	return FAIL;
}
/* }}} */


static enum_func_status
mysqlnd_query_read_result_set_header(MYSQLND *conn TSRMLS_DC)
{
	enum_func_status ret;
	php_mysql_packet_rset_header rset_header;

	ret = FAIL;
	PACKET_INIT_ALLOCA(rset_header, PROT_RSET_HEADER_PACKET);
	do {
		if (FAIL == (ret = PACKET_READ_ALLOCA(rset_header, conn))) {
			php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error reading result set's header");
			break;
		}

		if (rset_header.error_info.error_no) {
#ifndef MYSQLND_SILENT
			if (rset_header.error_info.sqlstate[0]) {
				php_error_docref(NULL TSRMLS_CC, E_WARNING, "ERROR:%d [SQLSTATE:%s] %s",
								 rset_header.error_info.error_no,
								 rset_header.error_info.sqlstate,
								 rset_header.error_info.error);
			} else {
				php_error_docref(NULL TSRMLS_CC, E_WARNING, "ERROR:%d %s",
								 rset_header.error_info.error_no,
								 rset_header.error_info.error);
			}
#endif
			/*
			  Cover a protocol design error: error packet does not
			  contain the server status. Therefore, the client has no way
			  to find out whether there are more result sets of
			  a multiple-result-set statement pending. Luckily, in 5.0 an
			  error always aborts execution of a statement, wherever it is
			  a multi-statement or a stored procedure, so it should be
			  safe to unconditionally turn off the flag here.
			*/
			conn->upsert_status.server_status &= ~SERVER_MORE_RESULTS_EXISTS;

			/*
			  This will copy the error code and the messages, as they
			  are buffers in the struct
			*/
			conn->error_info = rset_header.error_info;
			ret = FAIL;
			break;
		}
		conn->error_info.error_no = 0;

		switch (rset_header.field_count) {
			case MYSQLND_NULL_LENGTH:	/* LOAD DATA LOCAL INFILE */
				conn->last_query_type = QUERY_LOAD_LOCAL;
				ret = mysqlnd_handle_local_infile(conn, rset_header.info_or_local_file TSRMLS_CC);

				conn->state = (ret == PASS)? CONN_READY:CONN_QUIT_SENT;
				break;
			case 0:				/* UPSERT		*/
				conn->last_query_type = QUERY_UPSERT;
				conn->field_count = rset_header.field_count;
				conn->upsert_status.warning_count = rset_header.warning_count;
				conn->upsert_status.server_status = rset_header.server_status;
				conn->upsert_status.affected_rows = rset_header.affected_rows;
				conn->upsert_status.last_insert_id = rset_header.last_insert_id;
				SET_NEW_MESSAGE(conn->last_message, conn->last_message_len,
								rset_header.info_or_local_file, rset_header.info_or_local_file_len);
				/* Result set can follow UPSERT statement, check server_status */
				if (conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS) {
					conn->state = CONN_NEXT_RESULT_PENDING;
				} else {
					conn->state = CONN_READY;
				}
				ret = PASS;
				break;
			default:{			/* Result set	*/
				php_mysql_packet_eof fields_eof;

				memset(&conn->upsert_status, 0, sizeof(conn->upsert_status));
				conn->last_query_type = QUERY_SELECT;
				conn->state = CONN_FETCHING_DATA;
				conn->current_result = ecalloc(1, sizeof(MYSQLND_RES));
				conn->current_result->field_count = conn->field_count = rset_header.field_count;
				conn->current_result->type = MYSQLND_RES_NORMAL;
				conn->current_result->zval_cache = mysqlnd_palloc_get_cache_reference(conn->zval_cache);

				if (FAIL == (ret = mysqlnd_read_result_metadata(conn, conn->current_result TSRMLS_CC))) {
					efree(conn->current_result);
					conn->current_result = NULL;
					break;
				}

				/* Check for SERVER_STATUS_MORE_RESULTS if needed */
				PACKET_INIT_ALLOCA(fields_eof, PROT_EOF_PACKET);
				if (FAIL == (ret = PACKET_READ_ALLOCA(fields_eof, conn))) {
					mysqlnd_internal_free_result_contents(conn->current_result);
					efree(conn->current_result);
					PACKET_FREE_ALLOCA(fields_eof);
					conn->current_result = NULL;
					break;
				}
				conn->upsert_status.warning_count = fields_eof.warning_count;
				conn->upsert_status.server_status = fields_eof.server_status;
				PACKET_FREE_ALLOCA(fields_eof);

				ret = PASS;
				break;
			}
		}
	} while (0);
	PACKET_FREE_ALLOCA(rset_header);
	return ret;
}


/* {{{ mysqlnd_query */
/* 
  If conn->error_info.error_no is not zero, then we had an error.
  Still the result from the query is PASS
*/
enum_func_status
mysqlnd_query(MYSQLND *conn, const char *query, unsigned int query_len TSRMLS_DC)
{

	if (PASS != mysqlnd_simple_command(conn, COM_QUERY, query, query_len,
									   PROT_LAST /* we will handle the OK packet*/,
									   FALSE TSRMLS_CC)) {
		return FAIL;		
	}

	/*
	  Here read the result set. We don't do it in simple_command because it need
	  information from the ok packet. We will fetch it ourselves.
	*/
	return mysqlnd_query_read_result_set_header(conn TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_fetch_lengths_unbuffered */
static
unsigned long * mysqlnd_fetch_lengths_unbuffered(MYSQLND_RES * result)
{
	return result->lengths;
}
/* }}} */


/* {{{ mysqlnd_fetch_lengths_buffered */
/*
  Do lazy initialization for buffered results. As PHP strings have
  length inside, this function makes not much sense in the context
  of PHP, to be called as separate function. But let's have it for
  completeness.
*/
static 
unsigned long * mysqlnd_fetch_lengths_buffered(MYSQLND_RES *result)
{
	int i;
	zval **previous_row;

	/*
	  If:
	  - unbuffered result
	  - first row hs not been read
	  - last_row has been read
	*/
	if ((result->data_cursor == NULL) || result->data_cursor == result->data ||
		((result->data_cursor - result->data) > result->row_count)) {
		return NULL;/* No rows or no more rows */
	}

	previous_row = *(result->data_cursor - 1);
	for (i = 0; i < result->field_count; i++) {
		result->lengths[i] = (Z_TYPE_P(previous_row[i]) == IS_NULL)? 0:Z_STRLEN_P(previous_row[i]);
	}

	return result->lengths;
}
/* }}} */


/* {{{ mysqlnd_fetch_lengths */
unsigned long * mysqlnd_fetch_lengths(MYSQLND_RES *result TSRMLS_DC)
{
	return result->fetch_lengths(result);
}
/* }}} */


/* {{{ mysqlnd_fetch_row_unbuffered */
enum_func_status
mysqlnd_fetch_row_unbuffered(MYSQLND_RES *result, void *param, unsigned int flags,
							zend_bool *fetched_anything TSRMLS_DC)
{
	enum_func_status ret;
	zval *row = (zval *) param;
	unsigned int i, field_count = result->field_count;
	php_mysql_packet_row *row_packet = (php_mysql_packet_row *) result->row_packet;

	if (result->eof_reached) {
		/* No more rows obviously */
		return PASS;
	}
	if (result->type == MYSQLND_RES_NORMAL && result->conn->state != CONN_FETCHING_DATA) {
		SET_CLIENT_ERROR(result->conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 mysqlnd_out_of_sync); 
		return FAIL;
	}
	/* Let the row packet fill our buffer and skip additional malloc + memcpy */
	row_packet->skip_extraction = row? FALSE:TRUE;

	/*
	   If we skip rows (row == NULL) we have to
	   mysqlnd_unbuffered_free_last_data() before it. The function returns always true.
	*/
	if (PASS == (ret = PACKET_READ(row_packet, result->conn)) && !row_packet->eof) {
		mysqlnd_unbuffered_free_last_data(result);

		result->last_row_data = row_packet->fields;
		result->last_row_buffer = row_packet->row_buffer;
		row_packet->fields = NULL;
		row_packet->row_buffer = NULL;

		if (row) {
			HashTable *row_ht = Z_ARRVAL_P(row);
			for (i = 0; i < field_count; i++) {
				zval *data = result->last_row_data[i];
				int len = (Z_TYPE_P(data) == IS_NULL)? 0:Z_STRLEN_P(data);

				result->lengths[i] = len;

				/* Forbid ZE to free it, we will clean it */
				ZVAL_ADDREF(data);

				if ((flags & MYSQLND_FETCH_BOTH) == MYSQLND_FETCH_BOTH) {
					ZVAL_ADDREF(data);
				}
				if (flags & MYSQLND_FETCH_NUM) {
					zend_hash_next_index_insert(row_ht, &data, sizeof(zval *), NULL);
				}
				if (flags & MYSQLND_FETCH_ASSOC) {
					/* zend_hash_quick_update needs length + trailing zero */
					/* QQ: Error handling ? */
					zend_hash_quick_update(row_ht,
									   		result->fields[i].name, result->fields[i].name_length + 1,
									   		result->zend_hash_keys[i],
									   		(void *) &data, sizeof(zval *), NULL);
				}
				if (result->fields[i].max_length < len) {
					result->fields[i].max_length = len;
				}
			}
		}

		result->row_count++;
		*fetched_anything = TRUE;
	}

	/* Mark the connection as usable again */
	if (row_packet->eof) {
		result->eof_reached = TRUE;
		result->conn->upsert_status.warning_count = row_packet->warning_count;
		result->conn->upsert_status.server_status = row_packet->server_status;
		/*
		  result->row_packet will be cleaned when
		  destroying the result object
		*/
		if (result->conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS) {
			result->conn->state = CONN_NEXT_RESULT_PENDING;
		} else {
			result->conn->state = CONN_READY;
		}
		mysqlnd_unbuffered_free_last_data(result);
		*fetched_anything = FALSE;
	}

	return PASS;
}
/* }}} */


/* {{{ mysqlnd_use_result */
MYSQLND_RES *mysqlnd_use_result(MYSQLND *conn TSRMLS_DC)
{
	MYSQLND_RES *result;

	/* Nothing to store for UPSERT/LOAD DATA */
	if (conn->last_query_type != QUERY_SELECT || conn->state != CONN_FETCHING_DATA || !conn->current_result) {
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 mysqlnd_out_of_sync); 
		return NULL;
	}

	result = conn->current_result;
	conn->current_result = NULL;

	result->fetch_row		= mysqlnd_fetch_row_unbuffered;
	result->fetch_lengths	= mysqlnd_fetch_lengths_unbuffered;

	result->conn = conn;
	conn->references++;
	/*
	  Will be freed in the mysqlnd_internal_free_result_contents() called
	  by the resource destructor. mysqlnd_fetch_row_unbuffered() expects
	  this to be not NULL.
	*/
	PACKET_INIT(result->row_packet, PROT_ROW_PACKET, php_mysql_packet_row *);
	((php_mysql_packet_row *)result->row_packet)->field_count = result->field_count;
	((php_mysql_packet_row *)result->row_packet)->binary_protocol = FALSE;
	result->lengths = emalloc(result->field_count * sizeof(unsigned long));

	/* No multithreading issues as we don't share the connection :) */

	return result;
}
/* }}} */


/* {{{ mysqlnd_fetch_row_buffered */
enum_func_status
mysqlnd_fetch_row_buffered(MYSQLND_RES *result, void *param, unsigned int flags,
						   zend_bool *fetched_anything TSRMLS_DC)
{
	unsigned int i;
	zval *row = (zval *) param;

	/* If we haven't read everything */
	if (result->data_cursor && (result->data_cursor - result->data) < result->row_count) {
		zval **current_row = *result->data_cursor;
		for (i = 0; i < result->field_count; i++) {
			zval *data = current_row[i];

			/*
			  Let us later know what to do with this zval. If ref_count > 1, we will just
			  decrease it, otherwise free it. zval_ptr_dtor() make this very easy job.
			*/
			ZVAL_ADDREF(data);
			
			if ((flags & MYSQLND_FETCH_BOTH) == MYSQLND_FETCH_BOTH) {
				ZVAL_ADDREF(data);
			}
			if (flags & MYSQLND_FETCH_NUM) {
				zend_hash_next_index_insert(Z_ARRVAL_P(row), &data, sizeof(zval *), NULL);
			}
			if (flags & MYSQLND_FETCH_ASSOC) {
				/* zend_hash_quick_update needs length + trailing zero */
				/* QQ: Error handling ? */
				zend_hash_quick_update(Z_ARRVAL_P(row),
									   result->fields[i].name, result->fields[i].name_length + 1,
									   result->zend_hash_keys[i],
									   (void *) &data, sizeof(zval *), NULL);
			}
		}
		result->data_cursor++;
		*fetched_anything = TRUE;
	} else {
		result->data_cursor = NULL;
		*fetched_anything = FALSE;
#ifndef MYSQLND_SILENT
		php_printf("NO MORE DATA\n ");
#endif
	}
	return PASS;
}
/* }}} */


/* {{{ mysqlnd_store_result */
MYSQLND_RES *mysqlnd_store_result(MYSQLND *conn TSRMLS_DC)
{
	enum_func_status ret;
	int next_extend = 32, free_rows;
	php_mysql_packet_row row_packet;
	MYSQLND_RES *result;
	
	/* Nothing to store for UPSERT/LOAD DATA*/
	if (conn->last_query_type != QUERY_SELECT || conn->state != CONN_FETCHING_DATA || !conn->current_result) {
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 mysqlnd_out_of_sync); 
		return NULL;
	}

	result = conn->current_result;
	conn->current_result = NULL;

	result->fetch_row		= mysqlnd_fetch_row_buffered;
	result->fetch_lengths	= mysqlnd_fetch_lengths_buffered;

	conn->state = CONN_FETCHING_DATA;

	result->lengths = emalloc(result->field_count * sizeof(unsigned long));

	/* Create room for 'next_extend' rows */

	result->conn = NULL;	/* store result does not reference  the connection */
	result->data 	= emalloc(next_extend * sizeof(zval **));
	result->row_buffers = emalloc(next_extend * sizeof(zend_uchar *));

	free_rows = next_extend;

	PACKET_INIT_ALLOCA(row_packet, PROT_ROW_PACKET);
	row_packet.field_count = result->field_count;
	row_packet.binary_protocol = FALSE;
	/* Let the row packet fill our buffer and skip additional malloc + memcpy */
	while (FAIL != (ret = PACKET_READ_ALLOCA(row_packet, conn)) && !row_packet.eof) {
		int i;
		zval **current_row;

		if (!free_rows) {
			unsigned long total_rows = free_rows = next_extend = next_extend * 5 / 3; /* extend with 33% */
			total_rows += result->row_count;
			result->data = erealloc(result->data, total_rows * sizeof(zval **));
			result->row_buffers = erealloc(result->row_buffers, total_rows * sizeof(zend_uchar *));
		}
		free_rows--;
		current_row = result->data[result->row_count] = row_packet.fields;
		result->row_buffers[result->row_count] = row_packet.row_buffer;
		result->row_count++;
		
		/* So row_packet's destructor function won't efree() it */
		row_packet.fields = NULL;
		row_packet.row_buffer = NULL;

		for (i = 0; i < row_packet.field_count; i++) {
			unsigned long len =  (Z_TYPE_P(current_row[i]) == IS_NULL)? 0:Z_STRLEN_P(current_row[i]);

			if (result->fields[i].max_length < len) {
				result->fields[i].max_length = len;
			}
		}
		/*
		  No need to FREE_ALLOCA as we can reuse the
		  'lengths' and 'fields' arrays. For lengths its absolutely safe.
		  'fields' is reused because the ownership of the strings has been
		  transfered above. 
		*/
	}
	/* Finally clean */
	if (row_packet.eof) { 
		conn->upsert_status.warning_count = row_packet.warning_count;
		conn->upsert_status.server_status = row_packet.server_status;
	}
	PACKET_FREE_ALLOCA(row_packet);
	/* We can realloc here to save some memory, if free_rows > 0 ?*/

	if (conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS) {
		conn->state = CONN_NEXT_RESULT_PENDING;
	} else {
		conn->state = CONN_READY;
	}

	if (PASS == ret) {
		/* Position at the first row */
		result->data_cursor = result->data;
		/* No multithreading issues as we don't share the connection :) */
	} else {
		mysqlnd_internal_free_result_contents(result);
		efree(result);
		result = NULL;
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Pretty serious error");
	}
	return result;
}
/* }}} */


/* {{{ mysqlnd_free_result */
enum_func_status
mysqlnd_free_result(MYSQLND_RES *result TSRMLS_DC)
{
	zend_bool fetched_anything;
	if (result->conn && !result->eof_reached) {
		/* We have to fetch all data to clean the line */
		while ((PASS == mysqlnd_fetch_row_unbuffered(result, NULL, 0, &fetched_anything TSRMLS_CC)) &&
			   fetched_anything == TRUE) {
		}		
	}
	mysqlnd_internal_free_result(result TSRMLS_CC);
	return PASS;
}
/* }}} */


/* {{{ mysqlnd_fetch_row_ex */
enum_func_status
mysqlnd_fetch_row_ex(MYSQLND_RES *result, zval *row, unsigned int flags,
					 zend_bool *fetched_anything TSRMLS_DC)
{
	result->fetch_row(result, row, flags, fetched_anything TSRMLS_CC);
	return PASS;
}
/* }}} */


/* {{{ _mysqlnd_fetch_row */
MYSQLND_ROW _mysqlnd_fetch_row(MYSQLND_RES *result TSRMLS_DC ZEND_FILE_LINE_DC)
{
	unsigned int flags = MYSQLND_FETCH_NUM; 
	zval* row;
	MYSQLND_ROW ret = NULL;
	zend_bool fetched_anything;

	MAKE_STD_ZVAL(row);
	mysqlnd_array_init(row, result->field_count);

	if (PASS == result->fetch_row(result, row, flags, &fetched_anything TSRMLS_CC) &&
		fetched_anything == TRUE)
	{
		zval **entry;
		uint i = 0;
		if (!result->last_row) {
			result->last_row = emalloc(result->field_count * sizeof(char *));
		}
		ret = result->last_row;

		zend_hash_internal_pointer_reset(Z_ARRVAL_P(row));
		while (zend_hash_get_current_data(Z_ARRVAL_P(row), (void **)&entry) == SUCCESS) {
			/* Everything is a string */
			ret[i++] = Z_STRVAL_PP(entry);
			zend_hash_move_forward(Z_ARRVAL_P(row));
		}
	}
	zval_ptr_dtor(&row);
	return ret;
}
/* }}} */


/* {{{ mysqlnd_data_seek */
enum_func_status
mysqlnd_data_seek(MYSQLND_RES *result, mynd_ulonglong row)
{
	if (!result->data || row >= result->row_count) {
		return FAIL;
	}
	result->data_cursor = result->data + row;
	
	return PASS;
}
/* }}} */


/* {{{ mysqlnd_errno */
unsigned int mysqlnd_errno(const MYSQLND * const conn)
{
	return conn->error_info.error_no;
}
/* }}} */


/* {{{ mysqlnd_error */
const char *mysqlnd_error(const MYSQLND * const conn)
{
	return conn->error_info.error;
}
/* }}} */


/* {{{ mysqlnd_sqlstate */
const char *mysqlnd_sqlstate(const MYSQLND * const conn)
{
	return conn->error_info.sqlstate[0] ? conn->error_info.sqlstate:MYSQLND_SQLSTATE_NULL;
}
/* }}} */


/* {{{ mysqlnd_real_escape_string */
ulong mysqlnd_real_escape_string(MYSQLND *conn, char *newstr, char *escapestr, int escapestr_len TSRMLS_DC)
{
	const char 	*newstr_s = newstr;
	const char 	*newstr_e = newstr + 2 * escapestr_len;
	const char 	*end = escapestr + escapestr_len;
	zend_bool	escape_overflow = FALSE;
	
	for (;escapestr < end; escapestr++) {
		char	esc = '\0';

		/* check unicode characters */
		if (conn->charset->char_maxlen > 1) {
			uint len = conn->charset->mb_valid(escapestr, end);

			/* check possible overflow */
			if (newstr + len > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			/* copy mb char without escaping it */
			while (len--)
				*newstr++ = *escapestr++;
			escapestr--;
			continue;
		}
		if (conn->charset->char_maxlen > 1  &&  conn->charset->mb_charlen(*escapestr) > 1) {
			esc = *escapestr;
		} else {
			switch (*escapestr) {
				case 0:
					esc = '0';
					break;
				case '\n':
					esc = 'n';
					break;
				case '\r':
					esc = 'r';
					break;
				case '\\':
				case '\'':
				case '"':
					esc = *escapestr;
					break;
				case '\032':
					esc = 'Z';
					break;
			}
		}
		if (esc) {
			if (newstr + 2 > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			/* copy escaped character */
			*newstr++ = '\\';
			*newstr++ = esc;
		} else {
			if (newstr + 1 > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			/* copy non escaped character */
			*newstr++ = *escapestr;
		}
	}
	*newstr = '\0';

	if (escape_overflow)
		return (ulong)~0;
	return (ulong)(newstr - newstr_s);
}
/* }}} */


/* {{{ mysqlnd_dump_debug_info */
enum_func_status
mysqlnd_dump_debug_info(MYSQLND *conn TSRMLS_DC)
{
	return mysqlnd_simple_command(conn, COM_DEBUG, NULL, 0, PROT_EOF_PACKET, FALSE TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_select_db */
enum_func_status
mysqlnd_select_db(MYSQLND *conn, const char * const db, unsigned int db_len TSRMLS_DC)
{
	enum_func_status ret;
	ret = mysqlnd_simple_command(conn, COM_INIT_DB, db, db_len, PROT_OK_PACKET, FALSE TSRMLS_CC);
	/*
	  The server sends 0 but libmysql doesn't read it and has established
	  a protocol of giving back -1. Thus we have to follow it :(
	*/
	conn->upsert_status.affected_rows = -1;

	return ret;
}
/* }}} */


/* {{{ mysqlnd_ping */
enum_func_status
mysqlnd_ping(MYSQLND *conn TSRMLS_DC)
{
	enum_func_status ret;
	ret = mysqlnd_simple_command(conn, COM_PING, NULL, 0, PROT_OK_PACKET, FALSE TSRMLS_CC);
	/*
	  The server sends 0 but libmysql doesn't read it and has established
	  a protocol of giving back -1. Thus we have to follow it :(
	*/
	conn->upsert_status.affected_rows = -1;

	return ret;
}
/* }}} */


/* {{{ mysqlnd_stat */
enum_func_status
mysqlnd_stat(MYSQLND *conn, char **message, unsigned int * message_len TSRMLS_DC)
{
	enum_func_status ret;
	php_mysql_packet_stats stats_header;

	ret = mysqlnd_simple_command(conn, COM_STATISTICS, NULL, 0, PROT_LAST, FALSE TSRMLS_CC);
	if (FAIL == ret) {
		return FAIL;
	}
	PACKET_INIT_ALLOCA(stats_header, PROT_STATS_PACKET);
	if (FAIL == (ret = PACKET_READ_ALLOCA(stats_header, conn))) {
		return FAIL;
	}
	*message = stats_header.message;
	*message_len = stats_header.message_len;
	/* Ownership transfer */
	stats_header.message = NULL;
	PACKET_FREE_ALLOCA(stats_header);

	return PASS;
}
/* }}} */


/* {{{ mysqlnd_kill */
enum_func_status
mysqlnd_kill(MYSQLND *conn, unsigned long pid TSRMLS_DC)
{
	enum_func_status ret;
	char buff[4];

	int4store(buff, pid);

	/* If we kill ourselves don't expect OK packet, PROT_LAST will skip it */
	if (pid != conn->thread_id) {
		ret = mysqlnd_simple_command(conn, COM_PROCESS_KILL, buff, 4, PROT_OK_PACKET, FALSE TSRMLS_CC);
		/*
		  The server sends 0 but libmysql doesn't read it and has established
		  a protocol of giving back -1. Thus we have to follow it :(
		*/
		conn->upsert_status.affected_rows = -1;	
	} else {
		ret = mysqlnd_simple_command(conn, COM_PROCESS_KILL, buff, 4, PROT_LAST, FALSE TSRMLS_CC);
		conn->state = CONN_QUIT_SENT;
	}
	return ret;
}
/* }}} */


/* {{{ mysqlnd_refresh */
enum_func_status
mysqlnd_refresh(MYSQLND *conn, unsigned long options TSRMLS_DC)
{
	mysqlnd_1b bits[1];
	int1store(bits, options);

	return mysqlnd_simple_command(conn, COM_REFRESH, bits, 1, PROT_OK_PACKET, FALSE TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_set_character_set */
enum_func_status
mysqlnd_set_charset(MYSQLND *conn, char *csname TSRMLS_DC)
{
	MYSQLND_CHARSET	*charset = mysqlnd_find_charset_name(csname);
	char query[MAX_CHARSET_LEN + 12];

	if (!charset) {
		SET_CLIENT_ERROR(conn->error_info, CR_CANT_FIND_CHARSET, UNKNOWN_SQLSTATE, "Invalid characterset or character set not supported");
		return FAIL;
	}

	strcpy(query, "SET NAMES ");
	strcat(query, csname);

	if (FAIL == mysqlnd_query(conn, query, strlen(query) TSRMLS_CC)) {
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error executing query");
	} else if (conn->error_info.error_no) {
		return FAIL;
	}
	conn->charset = charset;
	return PASS;
}
/* }}} */


/* {{{ mysqlnd_shutdown */
enum_func_status
mysqlnd_shutdown(MYSQLND *conn, unsigned long level TSRMLS_DC)
{
	mysqlnd_1b bits[1];
	int1store(bits, level);

	return mysqlnd_simple_command(conn, COM_SHUTDOWN, bits, 1, PROT_OK_PACKET, FALSE TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_close */
enum_func_status
mysqlnd_close(MYSQLND *conn TSRMLS_DC)
{
	enum_func_status ret;
	switch (conn->state) {
		case CONN_READY:
			ret =  mysqlnd_simple_command(conn, COM_QUIT, NULL, 0, PROT_LAST,
										 conn->tmp_int? TRUE : FALSE TSRMLS_CC);
			/* Do nothing */
			break;
		case CONN_NEXT_RESULT_PENDING:
		case CONN_QUERY_SENT:
		case CONN_FETCHING_DATA:
#ifndef MYSQLND_SILENT
			php_printf("Brutally closing connection [%p][%s]\n", conn, conn->scheme);
#endif
			/*
			  Do nothing, the connection will be brutally closed
			  and the server will catch it and free close from its side.
			*/
			ret = PASS;
			break;
		case CONN_ALLOCED:
		case CONN_QUIT_SENT:
			/*
			  We get here if the user has called mysqlnd_close(), and then later the
			  resource destructor has kicked in.
			*/
			break;
	}
	conn->state = CONN_QUIT_SENT;
	mysqlnd_conn_free_contents(conn TSRMLS_CC);
	mysqlnd_conn_free(conn TSRMLS_CC);
	return ret;
}
/* }}} */


/* {{{ mysqlnd_num_fields */
unsigned int mysqlnd_num_fields(const MYSQLND_RES * const res)
{
	return res->field_count;
}
/* }}} */


/* {{{ mysqlnd_num_fields */
mynd_ulonglong mysqlnd_num_rows(const MYSQLND_RES * const res)
{
	return res->row_count;
}
/* }}} */

/* {{{ mysqlnd_field_count */
unsigned int mysqlnd_field_count(const MYSQLND * const conn)
{
	return conn->field_count;
}
/* }}} */


/* {{{ mysqlnd_fetch_field */
MYSQLND_FIELD *
mysqlnd_fetch_field(MYSQLND_RES *result)
{
	if (result->current_field >= result->field_count)
		return NULL;
	return &result->fields[result->current_field++];
}
/* }}} */


/* {{{ mysqlnd_field_seek */
MYSQLND_FIELD_OFFSET
mysqlnd_field_seek(MYSQLND_RES *result, MYSQLND_FIELD_OFFSET field_offset)
{
	MYSQLND_FIELD_OFFSET return_value = result->current_field;
	result->current_field = field_offset;
	return return_value;
}
/* }}} */


/* {{{ mysqlnd_field_tell */
MYSQLND_FIELD_OFFSET
mysqlnd_field_tell(MYSQLND_RES *result)
{
	return result->current_field;
}
/* }}} */


/* {{{ mysqlnd_fetch_field_direct */
MYSQLND_FIELD *
mysqlnd_fetch_field_direct(MYSQLND_RES *result, MYSQLND_FIELD_OFFSET fieldnr)
{
	return &result->fields[fieldnr];
}
/* }}} */


/* {{{ mysqlnd_insert_id */
mynd_ulonglong mysqlnd_insert_id(const MYSQLND * const conn)
{
	return conn->upsert_status.last_insert_id;
}
/* }}} */


/* {{{ mysqlnd_affected_rows */
mynd_ulonglong mysqlnd_affected_rows(const MYSQLND * const conn)
{
	return conn->upsert_status.affected_rows;
}
/* }}} */


/* {{{ mysqlnd_warning_count */
unsigned int mysqlnd_warning_count(const MYSQLND * const conn)
{
	return conn->upsert_status.warning_count;
}
/* }}} */


/* {{{ mysqlnd_info */
const char *mysqlnd_info(const MYSQLND * const conn)
{
	return conn->last_message;
}
/* }}} */



/* {{{ mysqlnd_client_info */
const char *mysqlnd_get_client_info()
{
	return MYSQLND_VERSION;
}
/* }}} */


/* {{{ mysqlnd_client_version */
uint mysqlnd_get_client_version()
{
	return MYSQLND_VERSION_ID;
}
/* }}} */


/* {{{ mysqlnd_get_server_info */
const char *mysqlnd_get_server_info(const MYSQLND * const conn)
{
	return(conn->server_version);
}
/* }}} */



/* {{{ mysqlnd_get_host_info */
const char *mysqlnd_get_host_info(const MYSQLND * const conn)
{
	return(conn->host_info);
}
/* }}} */


/* {{{ mysqlnd_get_proto_info */
unsigned int mysqlnd_get_proto_info(const MYSQLND *const conn)
{
	return(conn->protocol_version);
}
/* }}} */

/* {{{ mysqlnd_thread_id */
mynd_ulonglong mysqlnd_thread_id(const MYSQLND *const conn)
{
	return(conn->thread_id);
}
/* }}} */


/* {{{ mysqlnd_get_server_version */
unsigned long mysqlnd_get_server_version(const MYSQLND * const conn)
{
	long major, minor, patch;
	char *p = conn->server_version;

	if (!conn->server_version) {
		return 0;
	}

	major = strtol(p, &p, 10);
	p += 1; /* consume the dot */
	minor = strtol(p, &p, 10);
	p += 1; /* consume the dot */
	patch = strtol(p, &p, 10);
	
	return (unsigned long)(major * 10000L + (unsigned long)(minor * 100L + patch));
}
/* }}} */


zend_bool mysqlnd_more_results(const MYSQLND * const conn)
{
	/* (conn->state == CONN_NEXT_RESULT_PENDING) too */
	return conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS? TRUE:FALSE;
}


enum_func_status
mysqlnd_next_result(MYSQLND *conn TSRMLS_DC)
{
	enum_func_status ret;
	/* For UPSERT statement we don't set conn->state */
	if (conn->state != CONN_NEXT_RESULT_PENDING) {
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						mysqlnd_out_of_sync);
		return FAIL;
	}

	SET_EMPTY_ERROR(conn->error_info);
	conn->upsert_status.affected_rows= ~(mynd_ulonglong) 0;
	/*
	  We are sure that there is a result set, since conn->state is set accordingly
	  in mysqlnd_store_result() or mysqlnd_fetch_row_unbuffered()
	*/
	if (FAIL == (ret = mysqlnd_query_read_result_set_header(conn TSRMLS_CC))) {
#ifndef MYSQLND_SILENT
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Serious error");
#endif
		conn->state = CONN_QUIT_SENT;
	}

	return ret;
}


/* {{{ mysqlnd_field_type_name */
const char *mysqlnd_field_type_name(enum mysqlnd_field_types field_type)
{
	switch(field_type) {
		case FIELD_TYPE_STRING:
		case FIELD_TYPE_VAR_STRING:
			return "string";
			break;
		case FIELD_TYPE_TINY:
		case FIELD_TYPE_SHORT:
		case FIELD_TYPE_LONG:
		case FIELD_TYPE_LONGLONG:
		case FIELD_TYPE_INT24:
			return "int";
			break;
		case FIELD_TYPE_FLOAT:
		case FIELD_TYPE_DOUBLE:
		case FIELD_TYPE_DECIMAL:
		case FIELD_TYPE_NEWDECIMAL:
			return "real";
			break;
		case FIELD_TYPE_TIMESTAMP:
			return "timestamp";
			break;
		case FIELD_TYPE_YEAR:
			return "year";
			break;
		case FIELD_TYPE_DATE:
		case FIELD_TYPE_NEWDATE:
			return "date";
			break;
		case FIELD_TYPE_TIME:
			return "time";
			break;
		case FIELD_TYPE_SET:
			return "set";
			break;
		case FIELD_TYPE_ENUM:
			return "enum";
			break;
		case FIELD_TYPE_GEOMETRY:
			return "geometry";
			break;
		case FIELD_TYPE_DATETIME:
			return "datetime";
			break;
		case FIELD_TYPE_TINY_BLOB:
		case FIELD_TYPE_MEDIUM_BLOB:
		case FIELD_TYPE_LONG_BLOB:
		case FIELD_TYPE_BLOB:
			return "blob";
			break;
		case FIELD_TYPE_NULL:
			return "null";
			break;
		default:
			return "unknown";
			break;
	}
}
/* }}} */


/* {{{ mysqlnd_change_user */
enum_func_status
mysqlnd_change_user(MYSQLND *conn, const char *user, const char *passwd, const char *db TSRMLS_DC)
{
	/*
	  User could be max 16 * 3 (utf8), pass is 20 usually, db is up to 64*3
	  Stack space is not that expensive, so use a bit more to be protected against
	  stack overrungs.
	*/
	int user_len;
	enum_func_status ret;
	php_mysql_packet_chg_user_resp chg_user_resp;
	char buffer[768], *p = buffer;

	if (!user) {
		user = "";
	}
	if (!passwd) {
		passwd = "";
	}
	if (!db) {
		db = "";
	}

	/* 1. user ASCIIZ */
	user_len = strlen(user);
	memcpy(p, user, user_len);
	p += user_len;
	*p++ = '\0';

	/* 2. password SCRAMBLE_LENGTH followed by the scramble or \0 */
	if (passwd[0]) {
		*p++ = SCRAMBLE_LENGTH;
		php_mysqlnd_scramble(p, conn->scramble, passwd);
		p += SCRAMBLE_LENGTH;
	} else {
		*p++ = '\0';
	}

	/* 3. db ASCIIZ */
	if (db[0]) {
		int db_len = strlen(db);
		memcpy(p, db, db_len);
		p += db_len;
	}
	*p++ = '\0';

	if (PASS != mysqlnd_simple_command(conn, COM_CHANGE_USER, buffer, p - buffer,
									   PROT_LAST /* we will handle the OK packet*/,
									   FALSE TSRMLS_CC)) {
		return FAIL;
	}

	PACKET_INIT_ALLOCA(chg_user_resp, PROT_CHG_USER_PACKET);
	ret = PACKET_READ_ALLOCA(chg_user_resp, conn);
	conn->error_info = chg_user_resp.error_info;
	PACKET_FREE_ALLOCA(chg_user_resp);

	if (conn->error_info.error_no) {
		ret = FAIL;
		/*
		  COM_CHANGE_USER is broken in 5.1. At least in 5.1.15 and 5.1.14, 5.1.11 is immune.
		  bug#25371 mysql_change_user() triggers "packets out of sync"
		  When it gets fixed, there should be one more check here
		*/
	 	if (mysqlnd_get_server_version(conn) > 50113L) {
			php_mysql_packet_ok redundant_error_packet;
			PACKET_INIT_ALLOCA(redundant_error_packet, PROT_OK_PACKET);
			PACKET_READ_ALLOCA(redundant_error_packet, conn);
			PACKET_FREE_ALLOCA(redundant_error_packet);
		}
	}

	/*
	  Here we should close all statements. Unbuffered queries should not be a
	  problem as we won't allow sending COM_CHANGE_USER.
	*/

	return ret;
}
/* }}} */


/* {{{ mysqlnd_options */
enum_func_status
mysqlnd_options(MYSQLND *conn, enum mysqlnd_option option, const char *value)
{
	switch (option) {
		case MYSQL_OPT_CONNECT_TIMEOUT:
			conn->options.timeout_connect = *(uint*) value;
			break;
		case MYSQL_OPT_READ_TIMEOUT:
			conn->options.timeout_read = *(uint*) value;
			break;
		case MYSQL_OPT_WRITE_TIMEOUT:
			conn->options.timeout_write = *(uint*) value;
			break;
  		case MYSQL_OPT_LOCAL_INFILE:
			if (!value || (*(uint*) value) ? 1 : 0) {
				conn->options.flags |= CLIENT_LOCAL_FILES;
			} else {
				conn->options.flags &= ~CLIENT_LOCAL_FILES;
			}
		    break;
		case MYSQL_OPT_COMPRESS:
		case MYSQL_INIT_COMMAND:
  		case MYSQL_READ_DEFAULT_FILE:
		case MYSQL_READ_DEFAULT_GROUP:
		case MYSQL_SET_CLIENT_IP:
		case MYSQL_REPORT_DATA_TRUNCATION:
		case MYSQL_OPT_SSL_VERIFY_SERVER_CERT:
			/* currently not supported. Todo!! */
			break;	
		case MYSQL_SET_CHARSET_DIR:
		case MYSQL_SET_CHARSET_NAME:
		case MYSQL_OPT_RECONNECT:
  		case MYSQL_OPT_PROTOCOL:
			/* we don't need external character sets, all character sets are
			   compiled in. For compatibility we just ignore this setting.
			   Same for protocol, we don't support old protocol */
		case MYSQL_OPT_USE_REMOTE_CONNECTION:
		case MYSQL_OPT_USE_EMBEDDED_CONNECTION:
		case MYSQL_OPT_GUESS_CONNECTION:
			/* todo: throw an error, we don't support embedded */
			break;

		case MYSQL_OPT_NAMED_PIPE:
		case MYSQL_SHARED_MEMORY_BASE_NAME:
		case MYSQL_OPT_USE_RESULT:
		case MYSQL_SECURE_AUTH:
			/* not sure, todo ? */
		default:
			return FAIL;
	}
	return PASS;
}
/* }}} */


/* {{{ _mysqlnd_fetch_into */
void _mysqlnd_fetch_into(MYSQLND_RES *result, unsigned int flags, zval *return_value TSRMLS_DC ZEND_FILE_LINE_DC)
{
	zend_bool fetched_anything;
	/*
	  Hint Zend how many elements we will have in the hash. Thus it won't
	  extend and rehash the hash constantly.
	*/
	mysqlnd_array_init(return_value, mysqlnd_num_fields(result) * 2);
	if (FAIL == result->fetch_row(result, (void *)return_value, flags, &fetched_anything TSRMLS_CC)) {
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error while reading a row");
		RETURN_FALSE;
	} else if (fetched_anything == FALSE) {
		zval_dtor(return_value);
		RETURN_NULL();
	}
	/*
	  return_value is IS_NULL for no more data and an array for data. Thus it's ok
	  to return here.
	*/
}
/* }}} */


/* {{{ _mysqlnd_fetch_all */
void _mysqlnd_fetch_all(MYSQLND_RES *result, unsigned int flags, zval *return_value TSRMLS_DC ZEND_FILE_LINE_DC)
{
	zval			*row;
	ulong			i=0;


	/* mysqlnd_fetch_all works with buffered resultsets only */
	if (result->conn || !result->row_count || !result->data_cursor || 
		result->data_cursor >= result->data + result->row_count) {
		RETURN_NULL();
	}	

	mysqlnd_array_init(return_value, result->row_count);

	while (result->data_cursor && (result->data_cursor - result->data) < result->row_count) {
		MAKE_STD_ZVAL(row);
		mysqlnd_fetch_into(result, flags, row);
		add_index_zval(return_value, i++, row);	
	}
}
/* }}} */

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
