/*
  +----------------------------------------------------------------------+
  | PHP Version 6                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Georg Richter <georg@mysql.com>                             |
  |          Andrey Hristov <andrey@mysql.com>                           |
  |          Ulf Wendel <uwendel@mysql.com>                              |
  +----------------------------------------------------------------------+
*/
#include "php.h"
#include "php_globals.h"
#include "mysqlnd.h"
#include "mysqlnd_priv.h"
#include "ext/standard/sha1.h"


#define MYSQLND_SILENT
#define MYSQLND_DUMP_HEADER_N_BODY2
#define MYSQLND_DUMP_HEADER_N_BODY_FULL2

#define MYSQLND_MAX_PACKET_SIZE (256L*256L*256L-1)

#define	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, buf_size, packet_type) \
	{ \
		if (FAIL == mysqlnd_read_header((conn), &((packet)->header) TSRMLS_CC)) {\
			return FAIL;\
		}\
		if (!mysqlnd_read_body((conn)->stream, (buf), \
										MIN(sizeof(buf), (packet)->header.size) TSRMLS_CC)) { \
			php_error(E_WARNING, "Empty %s packet body", (packet_type));\
			return FAIL; \
		} \
	}


extern mysqlnd_packet_methods packet_methods[];

static const char	*unknown_sqlstate= "HY000";


/* {{{ php_mysqlnd_net_field_length 
   Get next field's length */
unsigned long php_mysqlnd_net_field_length(zend_uchar **packet)
{
	register zend_uchar *p= (zend_uchar *)*packet;

	if (*p < 251) {
		(*packet)++;
		return (unsigned long) *p;
	}

	switch (*p) {
		case 251:
			(*packet)++;
			return MYSQLND_NULL_LENGTH;
		case 252:
			(*packet) += 3;
			return (unsigned long) uint2korr(p+1);
		case 253:
			(*packet) += 4;
			return (unsigned long) uint3korr(p+1);
		default:
			(*packet) += 9;
			return (unsigned long) uint4korr(p+1);
	}
}
/* }}} */


/* {{{ php_mysqlnd_net_field_length_ll 
   Get next field's length */
mynd_ulonglong php_mysqlnd_net_field_length_ll(zend_uchar **packet)
{
	register zend_uchar *p= (zend_uchar *)*packet;

	if (*p < 251) {
		(*packet)++;
		return (mynd_ulonglong) *p;
	}

	switch (*p) {
		case 251:
			(*packet)++;
			return (mynd_ulonglong) MYSQLND_NULL_LENGTH;
		case 252:
			(*packet) += 3;
			return (mynd_ulonglong) uint2korr(p+1);
		case 253:
			(*packet) += 4;
			return (mynd_ulonglong) uint3korr(p+1);
		default:
			(*packet) += 9;
			return (mynd_ulonglong) uint8korr(p+1);
	}
}
/* }}} */


/* {{{ php_mysqlnd_net_store_length */
zend_uchar *php_mysqlnd_net_store_length(zend_uchar *packet, mynd_ulonglong length)
{
	if (length < (mynd_ulonglong) L64(251)) {
		*packet=(zend_uchar) length;
		return packet+1;
	}

	if (length < (mynd_ulonglong) L64(65536)) {
		*packet++= 252;
		int2store(packet,(uint) length);
		return packet+2;
	}

	if (length < (mynd_ulonglong) L64(16777216)) {
		*packet++= 253;
		int3store(packet,(ulong) length);
		return packet+3;
	}
	*packet++= 254;
	int8store(packet,length);
	return  packet+8;
}
/* }}} */


/* {{{ php_mysqlnd_read_error_from_line */
static
enum_func_status php_mysqlnd_read_error_from_line(zend_uchar *buf, size_t buf_len,
												  char *error, int error_buf_len,
												  unsigned int *error_no, char *sqlstate)
{
	zend_uchar *p = buf;
	int error_msg_len= 0;
	if (buf_len > 2) {
		*error_no = uint2korr(p);
		p+= 2;
		/* sqlstate is following */
		if (*p == '#') {
			memcpy(sqlstate, ++p, MYSQLND_SQLSTATE_LENGTH);
			p+= MYSQLND_SQLSTATE_LENGTH;
		}
		error_msg_len = buf_len - (p - buf);
		error_msg_len = MIN(error_msg_len, error_buf_len - 1);
		memcpy(error, p, error_msg_len);
	} else {
		*error_no = CR_UNKNOWN_ERROR;
		memcpy(sqlstate, unknown_sqlstate, MYSQLND_SQLSTATE_LENGTH);
	}
	sqlstate[MYSQLND_SQLSTATE_LENGTH] = '\0';
	error[error_msg_len]= '\0';
	return FAIL;
} 
/* }}} */


/* We assume that MYSQLND_HEADER_SIZE is 4 bytes !! */
#define STORE_HEADER_SIZE(safe_storage, buffer)  int4store((safe_storage), (buffer))
#define RESTORE_HEADER_SIZE(buffer, safe_storage) STORE_HEADER_SIZE((safe_storage), (buffer))

/* {{{ mysqlnd_stream_write */
/*
  IMPORTANT : It's expected that buf has place in the beginning for MYSQLND_HEADER_SIZE !!!!
  			  This is done for performance reasons in the caller of this function.
			  Otherwise we will have to do send two TCP packets, or do new alloc and memcpy.
			  Neither are quick, thus the clients of this function are obligated to do
			  what they are asked for.

  `count` is actually the length of the payload data. Thus :
  count + MYSQLND_HEADER_SIZE = sizeof(buf) (not the pointer but the actual buffer)
*/
size_t mysqlnd_stream_write(MYSQLND *conn, char * const buf, size_t count TSRMLS_DC)
{
	zend_uchar safe_storage[MYSQLND_HEADER_SIZE];
	size_t old_chunk_size = conn->stream->chunk_size;
	size_t ret, left = count;
	zend_uchar *p = buf;

	while (left > MYSQLND_MAX_PACKET_SIZE) {
		STORE_HEADER_SIZE(safe_storage, p);
		int3store(p, MYSQLND_MAX_PACKET_SIZE);
		int1store(p + 3, conn->packet_no);		
		conn->packet_no++;
		conn->stream->chunk_size = MYSQLND_MAX_PACKET_SIZE;
		ret = php_stream_write(conn->stream, p, MYSQLND_MAX_PACKET_SIZE + MYSQLND_HEADER_SIZE);
		RESTORE_HEADER_SIZE(p, safe_storage);

		p += MYSQLND_MAX_PACKET_SIZE;
		left -= MYSQLND_MAX_PACKET_SIZE;
	}
	/* Even for zero size payload we have to send a packet */
	STORE_HEADER_SIZE(safe_storage, p);
	int3store(p, left);
	int1store(p + 3, conn->packet_no);		
	conn->packet_no++;
	conn->stream->chunk_size = left + MYSQLND_HEADER_SIZE;
	ret = php_stream_write(conn->stream, p, left + MYSQLND_HEADER_SIZE);
	RESTORE_HEADER_SIZE(p, safe_storage);

	conn->stream->chunk_size = old_chunk_size;

	return ret;
}
/* }}} */


/* {{{ mysqlnd_read_body */
static
size_t mysqlnd_read_body(php_stream *stream, zend_uchar *buf, size_t size TSRMLS_DC)
{
	size_t ret;
	char *p = (char *)buf;
	int i, iter=0;

	do {
		size -= (ret = php_stream_read(stream, p, size));
#ifdef MYSQLND_DUMP_HEADER_N_BODY
		if (size || iter++) {
			php_printf("read=%d buf=%p p=%p chunk_size=%d left=%d\n",
						ret, buf, p ,stream->chunk_size, size);
		}
#endif
		p += ret;
	} while (size > 0);

#ifdef MYSQLND_DUMP_HEADER_N_BODY_FULL
	php_printf("\tBODY: requested=%d last_read=%3d\n\t", p - (char*)buf, ret);
	for (i = 0 ; i < p - (char*)buf; i++) printf("%c-", buf[i]); 	php_printf("\n\t");
	for (i = 0 ; i < p - (char*)buf; i++) printf("%.2X ", *((char*)buf+i));	php_printf("\n");
#endif

	return p - (char*)buf;
}
/* }}} */


/* {{{ mysqlnd_read_header */
static enum_func_status
mysqlnd_read_header(MYSQLND *conn, mysqlnd_packet_header *header TSRMLS_DC)
{
	char buffer[4];
	char *p = buffer;
	int to_read = 4, ret;

	do {
		if (!(ret= php_stream_read(conn->stream, p, to_read))) {
			php_error(E_WARNING, "Error while reading header from socket");
			return FAIL;
		}
		p += ret;
		to_read -= ret;
	} while (to_read);

	header->size = uint3korr(buffer);
	header->packet_no = uint1korr(buffer + 3);

	if (conn->packet_no == header->packet_no) {
		/*
		  Have to increase the number, so we can send correct number back. It will
		  round at 255 as this is unsigned char. The server needs this for simple
		  flow control checking.
		*/
		conn->packet_no++;
#ifdef MYSQLND_DUMP_HEADER_N_BODY
		php_printf("HEADER: packet_no=%d size=%3d\n", header->packet_no, header->size);
#endif
		return PASS;
	}

	php_error(E_WARNING, "Packets out of order. Expected %d received %d. Packet size=%d",
			  conn->packet_no, header->packet_no, header->size);
	return FAIL;
}
/* }}} */


/* {{{ php_mysqlnd_greet_read */
static enum_func_status
php_mysqlnd_greet_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	zend_uchar buf[256];
	register zend_uchar *p= buf;
	register php_mysql_packet_greet *packet= (php_mysql_packet_greet *) _packet;

	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, sizeof(buf), "greeting");

	packet->protocol_version = uint1korr(p);
	p++;

	packet->server_version = pestrdup((char *)p, conn->persistent);
	p+= strlen(packet->server_version) + 1; /* eat the '\0' */

	packet->thread_id = uint4korr(p);
	p+=4;

	memcpy(packet->scramble_buf, p, SCRAMBLE_LENGTH_323);
	p+= 8;

	/* pad1 */
	p++;

	packet->server_capabilities = uint2korr(p);
	p+= 2;

	packet->charset_no = uint1korr(p);
	p++;

	packet->server_status = uint2korr(p);
	p+= 2;

	/* pad2 */
	p+= 13;

	if (p - buf < packet->header.size) {
		/* scramble_buf is split into two parts */
		memcpy(packet->scramble_buf + SCRAMBLE_LENGTH_323,
		   		p, SCRAMBLE_LENGTH - SCRAMBLE_LENGTH_323);
	} else {
		packet->pre41 = TRUE;
	}

	return PASS;
}
/* }}} */


/* {{{ php_mysqlnd_greet_free_mem */
static
void php_mysqlnd_greet_free_mem(void *_packet, zend_bool alloca)
{
	php_mysql_packet_greet *p= (php_mysql_packet_greet *) _packet;
	if (p->server_version) {
		efree(p->server_version);
		p->server_version = NULL;
	}
	if (!alloca) {
		efree(p);
	}
}
/* }}} */


#define MYSQLND_CAPABILITIES (CLIENT_LONG_PASSWORD | CLIENT_LONG_FLAG | CLIENT_TRANSACTIONS | \
				CLIENT_PROTOCOL_41 | CLIENT_SECURE_CONNECTION | \
				CLIENT_MULTI_RESULTS)


/* {{{ php_mysqlnd_crypt */
static void php_mysqlnd_crypt(unsigned char *buffer, const unsigned char *s1, const unsigned char *s2, unsigned int len)
{
	const unsigned char *s1_end= s1 + len;
	while (s1 < s1_end) {
		*buffer++= *s1++ ^ *s2++;
	}
}
/* }}} */


/* {{{ php_mysqlnd_scramble */
void php_mysqlnd_scramble(unsigned char *buffer, const unsigned char *scramble,
						  const unsigned char *password)
{
	PHP_SHA1_CTX context;
	unsigned char sha1[SHA1_MAX_LENGTH];
	unsigned char sha2[SHA1_MAX_LENGTH];
	

	/* Phase 1: hash password */
	PHP_SHA1Init(&context);
	PHP_SHA1Update(&context, password, strlen((char *)password));
	PHP_SHA1Final(sha1, &context);

	/* Phase 2: hash sha1 */
	PHP_SHA1Init(&context);
	PHP_SHA1Update(&context, (unsigned char*)sha1, SHA1_MAX_LENGTH);
	PHP_SHA1Final(sha2, &context);

	/* Phase 3: hash scramble + sha2 */
	PHP_SHA1Init(&context);
	PHP_SHA1Update(&context, scramble, SCRAMBLE_LENGTH);
	PHP_SHA1Update(&context, (unsigned char*)sha2, SHA1_MAX_LENGTH);
	PHP_SHA1Final(buffer, &context);

	/* let's crypt buffer now */
	php_mysqlnd_crypt(buffer, (const unsigned char *)buffer, (const unsigned  char *)sha1, SHA1_MAX_LENGTH);
}
/* }}} */


/* {{{ php_mysqlnd_auth_write */
static
size_t php_mysqlnd_auth_write(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	char buffer[1024];
	register char *p= buffer + MYSQLND_HEADER_SIZE; /* start after the header */
	int len;
	register php_mysql_packet_auth *packet= (php_mysql_packet_auth *) _packet;

	packet->client_flags |= MYSQLND_CAPABILITIES;

	if (packet->db) {
		packet->client_flags |= CLIENT_CONNECT_WITH_DB;
	}

	if (PG(open_basedir) && strlen(PG(open_basedir))) {
		packet->client_flags ^= CLIENT_LOCAL_FILES;
	}
	
	/* don't allow multi_queries via connect parameter */
	packet->client_flags ^= CLIENT_MULTI_STATEMENTS;
	int4store(p, packet->client_flags);
	p+= 4;

	int4store(p, packet->max_packet_size);
	p+= 4;

	int1store(p, packet->charset_no);
	p++;

	memset(p, 0, 23); /* filler */
	p+= 23;	

	len= strlen(packet->user);
	strncpy(p, packet->user, len);
	p+= len;
	*p++ = '\0';

	/* copy scrambled pass*/
	if (packet->password && packet->password[0]) {
		/* In 4.1 we use CLIENT_SECURE_CONNECTION and thus the len of the buf should be passed */
		int1store(p, 20);
		p++;
		php_mysqlnd_scramble((unsigned char*)p, packet->server_scramble_buf,
							 (unsigned char *)packet->password);
		p+= 20;
	} else {
		/* Zero length */
		int1store(p, 0);
		p++;
	}

	if (packet->db) {
		memcpy(p, packet->db, packet->db_len);
		p+= packet->db_len;
		*p++= '\0';
	}
	/* Handle CLIENT_CONNECT_WITH_DB */
	/* no \0 for no DB */

	return mysqlnd_stream_write(conn, buffer, p - buffer TSRMLS_CC);
}
/* }}} */

/* {{{ php_mysqlnd_auth_free_mem */
static
void php_mysqlnd_auth_free_mem(void *_packet, zend_bool alloca)
{
	if (!alloca) {
		efree((php_mysql_packet_auth *) _packet);
	}
}
/* }}} */


/* {{{ php_mysqlnd_ok_read */
static enum_func_status
php_mysqlnd_ok_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	zend_uchar buf[1024];
	zend_uchar *p= buf;
	int i;
	register php_mysql_packet_ok *packet= (php_mysql_packet_ok *) _packet;

	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, sizeof(buf), "OK");

	/* Should be always 0x0 or 0xFF for error */
	packet->field_count= uint1korr(p);
	p++;

	if (0xFF == packet->field_count) {
		php_mysqlnd_read_error_from_line(p, packet->header.size - 1,
										 packet->error, sizeof(packet->error),
										 &packet->error_no, packet->sqlstate);
		return PASS;
	}
	/* Everything was fine! */
	packet->affected_rows  = php_mysqlnd_net_field_length_ll(&p);
	packet->last_insert_id = php_mysqlnd_net_field_length_ll(&p);

	packet->server_status = uint2korr(p);
	p+= 2; 

	packet->warning_count = uint2korr(p);
	p+= 2;

	/* There is a message */
	if (packet->header.size > p - buf && (i = php_mysqlnd_net_field_length(&p))) {
		packet->message = estrndup((char *)p, MIN(i, sizeof(buf) - (p - buf)));
		packet->message_len = i;
	} else {
		packet->message = NULL;
	}

	return PASS;
}
/* }}} */


/* {{{ php_mysqlnd_ok_free_mem */
static
void php_mysqlnd_ok_free_mem(void *_packet, zend_bool alloca)
{
	php_mysql_packet_ok *p= (php_mysql_packet_ok *) _packet;
	if (p->message) {
		efree(p->message);
		p->message = NULL;
	}
	if (!alloca) {
		efree(p);
	}
}
/* }}} */



/* {{{ php_mysqlnd_eof_read */
static enum_func_status
php_mysqlnd_eof_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	zend_uchar buf[5]; /* EOF packet is since 4.1 five bytes long*/
	zend_uchar *p= buf;
	php_mysql_packet_eof *packet= (php_mysql_packet_eof *) _packet;

	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, sizeof(buf), "EOF");

	/* Should be always 0xFE */
	packet->field_count= uint1korr(p);
	p++;

	packet->warning_count = uint2korr(p);
	p+= 2;
	packet->server_status = uint2korr(p);

	return PASS;
}
/* }}} */


/* {{{ php_mysqlnd_eof_free_mem */
static
void php_mysqlnd_eof_free_mem(void *_packet, zend_bool alloca)
{
	if (!alloca) {
		efree(_packet);
	}
}
/* }}} */


/* {{{ php_mysqlnd_cmd_write */
size_t php_mysqlnd_cmd_write(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	/* Let's have some space, which we can use, if not enough, we will allocate new buffer */
	php_mysql_packet_command *packet= (php_mysql_packet_command *) _packet;

	if (!packet->argument || !packet->arg_len) {
		char buffer[MYSQLND_HEADER_SIZE + 1];

		int1store(buffer + MYSQLND_HEADER_SIZE, packet->command);
		return mysqlnd_stream_write(conn, buffer, 1 TSRMLS_CC);
	} else {
		size_t tmp_len = packet->arg_len + 1 + MYSQLND_HEADER_SIZE, ret;
		zend_uchar *tmp, *p;

		tmp = (tmp_len > conn->cmd_buffer.length)? emalloc(tmp_len):conn->cmd_buffer.buffer;
		p = tmp + MYSQLND_HEADER_SIZE; /* skip the header */

		int1store(p, packet->command);
		p++;

		memcpy(p, packet->argument, packet->arg_len);

		ret = mysqlnd_stream_write(conn, tmp, tmp_len - MYSQLND_HEADER_SIZE TSRMLS_CC);
		if (tmp != conn->cmd_buffer.buffer) {
			efree(tmp);
		}
		return ret;
	}
}
/* }}} */


/* {{{ php_mysqlnd_cmd_free_mem */
static
void php_mysqlnd_cmd_free_mem(void *_packet, zend_bool alloca)
{
	if (!alloca) {
		efree((php_mysql_packet_command *) _packet);
	}
}
/* }}} */


/* {{{ php_mysqlnd_rset_header_read */
static enum_func_status
php_mysqlnd_rset_header_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	zend_uchar buf[1024];
	zend_uchar *p= buf;
	size_t len;
	php_mysql_packet_rset_header *packet= (php_mysql_packet_rset_header *) _packet;

	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, sizeof(buf), "resultset header ");

	/*
	  Don't increment. First byte is 0xFF on error, but otherwise is starting byte
	  of encoded sequence for length.
	*/
	if (*p == 0xFF) {
		/* Error */
		p++;
		php_mysqlnd_read_error_from_line(p, packet->header.size - 1,
										 packet->error_info.error, sizeof(packet->error_info.error),
										 &packet->error_info.error_no, packet->error_info.sqlstate);
		return PASS;
	}

	packet->field_count= php_mysqlnd_net_field_length(&p);
	switch (packet->field_count) {
		case MYSQLND_NULL_LENGTH:
			/*
			  First byte in the packet is the field count.
			  Thus, the name is size - 1. And we add 1 for a trailing \0.
			*/
			len = packet->header.size - 1;
			packet->info_or_local_file = emalloc(len + 1);
			memcpy(packet->info_or_local_file, p, len);
			packet->info_or_local_file[len] = '\0';
			packet->info_or_local_file_len = len;
			break;
		case 0x00:
			packet->affected_rows = php_mysqlnd_net_field_length_ll(&p);
			packet->last_insert_id= php_mysqlnd_net_field_length_ll(&p);
			packet->server_status = uint2korr(p);
			p+=2;
			packet->warning_count = uint2korr(p);
			p+=2;
			/* Check for additional textual data */
			if (packet->header.size  > (p - buf) && (len = php_mysqlnd_net_field_length(&p))) {
				packet->info_or_local_file = emalloc(len + 1);
				memcpy(packet->info_or_local_file, p, len);
				packet->info_or_local_file[len] = '\0';
				packet->info_or_local_file_len = len;
			}
			break;
		default:
			/* Result set */
			break;
	}

	return PASS;
}
/* }}} */


/* {{{ php_mysqlnd_rset_header_free_mem */
static
void php_mysqlnd_rset_header_free_mem(void *_packet, zend_bool alloca)
{
	php_mysql_packet_rset_header *p= (php_mysql_packet_rset_header *) _packet;
	if (p->info_or_local_file) {
		efree(p->info_or_local_file);
		p->info_or_local_file = NULL;
	}
	if (!alloca) {
		efree(p);
	}
}
/* }}} */


/* {{{ php_mysqlnd_rset_field_read */
static enum_func_status
php_mysqlnd_rset_field_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	zend_uchar buf[4096]; /* Should be enough for the metadata of a single row */
	php_mysql_packet_res_field *packet= (php_mysql_packet_res_field *) _packet;
	zend_uchar *p = buf;
	char *root_ptr;
	size_t len, total_len = 0;
	MYSQLND_FIELD *meta;

	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, sizeof(buf), "field");

	if (packet->skip_parsing) {
		return PASS;
	}

	meta = packet->metadata;
	if ((len = php_mysqlnd_net_field_length(&p))) {
		meta->catalog = (char *)p;
		p += (meta->catalog_length = len);
		total_len += len + 1;
	}

	if ((len = php_mysqlnd_net_field_length(&p))) {
		meta->db = (char *)p;
		p += (meta->db_length = len);
		total_len += len + 1;
	}

	if ((len = php_mysqlnd_net_field_length(&p))) {
		meta->table = (char *)p;
		p += (meta->table_length = len);
		total_len += len + 1;
	}

	if ((len = php_mysqlnd_net_field_length(&p))) {
		meta->org_table = (char *)p;
		p += (meta->org_table_length = len);
		total_len += len + 1;
	}

	if ((len = php_mysqlnd_net_field_length(&p))) {
		meta->name = (char *)p;
		p += (meta->name_length = len);
		total_len += len + 1;
	}

	if ((len = php_mysqlnd_net_field_length(&p))) {
		meta->org_name = (char *)p;
		p += (meta->org_name_length = len);
		total_len += len + 1;
	}


	/* 1 byte filler */
	p++;

	meta->charsetnr = uint2korr(p);
	p += 2;

	meta->length = uint4korr(p);
	p += 4;

	meta->type = uint1korr(p);
	p += 1;

	meta->flags = uint2korr(p);
	p += 2;

	meta->decimals = uint2korr(p);
	p += 1;

	/* 2 byte filler */
	p +=2;

	if (packet->header.size > (p - buf) && (len = php_mysqlnd_net_field_length(&p))) {
		meta->def = emalloc(len + 1);
		memcpy(meta->def, p, len);
		meta->def[len] = '\0';
		meta->def_length = len;
		p += len;
	}

	root_ptr = meta->root = emalloc(total_len);
	/* Now do allocs */
	if (meta->catalog) {
		len = meta->catalog_length;
		meta->catalog = memcpy(root_ptr, meta->catalog, len);
		*(root_ptr +=len) = '\0';
		root_ptr++;
	}

	if (meta->db) {
		len = meta->db_length;
		meta->db = memcpy(root_ptr, meta->db, len);
		*(root_ptr + len) = '\0';
	}

	if (meta->table) {
		len = meta->table_length;
		meta->table = memcpy(root_ptr, meta->table, len);
		*(root_ptr +=len) = '\0';
		root_ptr++;
	}

	if (meta->org_table) {
		len = meta->org_table_length;
		meta->org_table = memcpy(root_ptr, meta->org_table, len);
		*(root_ptr +=len) = '\0';
		root_ptr++;
	}

	if (meta->name) {
		len = meta->name_length;
		meta->name = memcpy(root_ptr, meta->name, len);
		*(root_ptr +=len) = '\0';
		root_ptr++;
	}

	if (meta->org_name) {
		len = meta->org_name_length;
		meta->org_name = memcpy(root_ptr, meta->org_name, len);
		*(root_ptr +=len) = '\0';
		root_ptr++;
	}

#ifndef MYSQLND_SILENT
		php_printf("\tFIELD=[%s.%s.%s]\n",  meta->db? meta->db:"*NA*",
											meta->table? meta->table:"*NA*",
											meta->name? meta->name:"*NA*");
#endif

	return PASS;
}
/* }}} */


/* {{{ php_mysqlnd_rset_field_free_mem */
static
void php_mysqlnd_rset_field_free_mem(void *_packet, zend_bool alloca)
{
	php_mysql_packet_res_field *p= (php_mysql_packet_res_field *) _packet;

	/* p->metadata was passed to us as temporal buffer */
	if (!alloca) {
		efree(p);
	}
}
/* }}} */


static
enum_func_status
php_mysqlnd_read_row_ex(MYSQLND *conn, zend_uchar **buf, int buf_size,
						unsigned int *data_size TSRMLS_DC)
{
	enum_func_status ret = PASS;
	mysqlnd_packet_header header;
	zend_uchar *new_buf = NULL, *p = *buf;
	zend_bool first_iteration = TRUE;

	/*
	  To ease the process the server splits everything in packets up to 2^24 - 1.
	  Even in the case the payload is evenly divisible by this value, the last
	  packet will be empty, namely 0 bytes. Thus, we can read every packet and ask
	  for next one if they have 2^24 - 1 sizes. But just read the header of a
	  zero-length byte, don't read the body, there is no such.
	*/

	*data_size = 0;
	while (1) {
		if (FAIL == mysqlnd_read_header(conn, &header TSRMLS_CC)) {
			ret = FAIL;
			break;
		}

		*data_size += header.size;

		if (first_iteration && header.size > buf_size) {
			/*
			  We need a trailing \0 for the last string, in case of text-mode,
			  to be able to implement read-only variables.
			*/
			p = new_buf = emalloc(header.size + 1);
			conn->stream->chunk_size = header.size;
		} else if (!first_iteration) {
			/* Empty packet after MYSQLND_MAX_PACKET_SIZE packet. That's ok, break */
			if (!header.size) {
				break;
			}

			/*
			  We are here after first iteration, then our buffer is 
			  definitely not the local buffer, so we won't test it.
			  We have to realloc the buffer.

			  We need a trailing \0 for the last string, in case of text-mode,
			  to be able to implement read-only variables.
			*/
			new_buf = realloc(new_buf, *data_size + 1);
			/* The position could have changed, recalculate */
			p = new_buf + (*data_size - header.size);
		}

		if (!mysqlnd_read_body(conn->stream, p, header.size TSRMLS_CC)) {
			php_error(E_WARNING, "Empty row packet body");
			ret = FAIL;
			break;
		}

		if (header.size < MYSQLND_MAX_PACKET_SIZE) {
			break;
		}
		first_iteration = FALSE;
	}
	if (ret == PASS && new_buf) {
		*buf = new_buf;
	}
	return ret;
}






/* {{{ php_mysqlnd_rowp_read */
/*
  packet->fields is created by this function, if normal statements
  packet->fields is passed from outside, if PS
*/
static enum_func_status
php_mysqlnd_rowp_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	zend_uchar *p;
	zend_uchar *null_ptr, bit;
	enum_func_status ret = PASS;
	size_t data_size = 0;
	size_t old_chunk_size = conn->stream->chunk_size;
	php_mysql_packet_row *packet= (php_mysql_packet_row *) _packet;


	if (FAIL == (ret = php_mysqlnd_read_row_ex(conn, &packet->row_buffer, 0, &data_size TSRMLS_CC))) {
		goto end;
	}

	/* packet->row_buffer is of size 'data_size + 1' */
	packet->header.size = data_size;

	if ((*(p = packet->row_buffer)) == 0xFE && data_size < 8) { /* EOF */
		packet->eof = TRUE;
		p++;
		if (data_size > 1) {
			packet->warning_count = uint2korr(p);
			p += 2;
			packet->server_status = uint2korr(p);
			/* Seems we have 3 bytes reserved for future use */
		}
		goto end;
	}

	packet->eof = FALSE;
	/* packet->field_count is set by the user of the packet */

	if (!packet->skip_extraction) {
		int i;
		zval **current_field, **end_field, **start_field;
		zend_bool last_field_was_string;

		if (!packet->fields) {
			/*
			  old-API will probably set packet->fields to NULL every time, though for
			  unbuffered sets it makes not much sense as the zvals in this buffer matter,
			  not the buffer. Constantly allocating and deallocating brings nothing.

			  For PS - if stmt_store() is performed, thus we don't have a cursor, it will
			  behave just like old-API buffered. Cursors will behave like a bit different,
			  but mostly like old-API unbuffered and thus will populate this array with
			  value.
			*/
			packet->fields = (zval **) emalloc(packet->field_count * sizeof(zval *));
		}

		end_field = (current_field = start_field = packet->fields) + packet->field_count;

		if (packet->binary_protocol) {
			goto binary_protocol;
		}
		for (; current_field < end_field; current_field++) {
			/* Don't reverse the order. It is significant!*/
			zend_uchar *this_field_len_pos = p;
			void *obj;
			zend_bool allocated;
			unsigned long len = php_mysqlnd_net_field_length(&p);

			obj = mysqlnd_palloc_get_zval(conn->zval_cache, &allocated);
			if (allocated) {
				*current_field = (zval *) obj;
			} else {
				*current_field = &((mysqlnd_zval *) obj)->zv;			
				((mysqlnd_zval *) obj)->ze_alloced = FALSE;
			}

			if (current_field > start_field && last_field_was_string) {
				/*
				  Normal queries: 
				    We have to put \0 now to the end of the previous field,
				    if it was a string. IS_NULL doesn't matter. Because we
				    have already read our length, then we can overwrite it
				    in the row buffer.
				    This statement terminates the previous field, not the current one.
 
				    NULL_LENGTH is encoded in one byte, so we can stick a \0 there.
				    Any string length is encoded in at least one byte, so we can stick
				    a \0 there.
				*/

				*this_field_len_pos = '\0';
			}

			/* NULL or NOT NULL, this is the question! */
			if (len == MYSQLND_NULL_LENGTH) {
				ZVAL_NULL(*current_field);
				last_field_was_string = FALSE;
			} else {
				ZVAL_STRINGL(*current_field, p, len, 0);
				p += len;
				last_field_was_string = TRUE;
			}
		}
		if (last_field_was_string) {
			/* Normal queries: The buffer has one more byte at the end, because we need it */
			packet->row_buffer[data_size] = '\0';
		}
		goto end;
binary_protocol:
		/* skip the first byte, not 0xFE -> 0x0, status */
		p++;
		null_ptr= p;
		p += (packet->field_count + 9)/8;		/* skip null bits */
 		bit	= 4;								/* first 2 bits are reserved */

		for (i = 0; current_field < end_field; current_field++, i++) {
			MAKE_STD_ZVAL(*current_field);
			if (*null_ptr & bit) {
				ZVAL_NULL(*current_field);
			} else {
				fetch_functions[packet->fields_metadata[i].type].func(*current_field,
																	  &packet->fields_metadata[i],
																	  0, &p TSRMLS_CC);
			}
		    if (!((bit<<=1) & 255)) {
				bit= 1;					/* To next byte */
				null_ptr++;
			}
		}
		/* Normal queries: The buffer has one more byte at the end, because we need it */
		packet->row_buffer[data_size] = '\0';
	}
end:
	conn->stream->chunk_size = old_chunk_size;
	return ret;
}
/* }}} */


/* {{{ php_mysqlnd_rowp_free_mem */
static
void php_mysqlnd_rowp_free_mem(void *_packet, zend_bool alloca)
{
	php_mysql_packet_row *p= (php_mysql_packet_row *) _packet;
	if (p->row_buffer) {
		efree(p->row_buffer);
		p->row_buffer = NULL;
	}
	/*
	  Don't free packet->fields :
	  - normal queries -> store_result() | fetch_row_unbuffered() will transfer
	    the ownership and NULL it.
	  - PS will pass in it the bound variables, we have to use them! and of course
	    not free the array. As it is passed to us, we should not clean it ourselves.
	
	*/
	if (!alloca) {
		efree(p);
	}
}
/* }}} */



/* {{{ php_mysqlnd_stats_read */
static enum_func_status
php_mysqlnd_stats_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	zend_uchar buf[1024];
	php_mysql_packet_stats *packet= (php_mysql_packet_stats *) _packet;

	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, sizeof(buf), "statistics");

	packet->message = emalloc(packet->header.size + 1);
	memcpy(packet->message, buf, packet->header.size);
	packet->message[packet->header.size] = '\0';
	packet->message_len = packet->header.size;

	return PASS;
}
/* }}} */


/* {{{ php_mysqlnd_stats_free_mem */
static
void php_mysqlnd_stats_free_mem(void *_packet, zend_bool alloca)
{
	php_mysql_packet_stats *p= (php_mysql_packet_stats *) _packet;
	if (p->message) {
		efree(p->message);
		p->message = NULL;
	}
	if (!alloca) {
		efree(p);
	}
}
/* }}} */


/* 1 + 4 (id) + 2 (field_c) + 2 (param_c) + 1 (filler) + 2 (warnings ) */
#define PREPARE_RESPONSE_SIZE_41 9
#define PREPARE_RESPONSE_SIZE_50 12

/* {{{ php_mysqlnd_prepare_read */
static enum_func_status
php_mysqlnd_prepare_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	/* In case of an error, we should have place to put it */
	zend_uchar buf[1024];
	zend_uchar *p= buf;
	unsigned int data_size;
	php_mysql_packet_prepare_response *packet= (php_mysql_packet_prepare_response *) _packet;

	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, sizeof(buf), "prepare");
	
	data_size = packet->header.size;
	packet->error_code = uint1korr(p);
	p++;

	if (0xFF == packet->error_code) {
		php_mysqlnd_read_error_from_line(p, data_size - 1,
										 packet->error_info.error, sizeof(packet->error_info.error),
										 &packet->error_info.error_no,
										 packet->error_info.sqlstate);
		return PASS;
	}

	if (data_size != PREPARE_RESPONSE_SIZE_41 &&
		data_size != PREPARE_RESPONSE_SIZE_50 &&
		!(data_size > PREPARE_RESPONSE_SIZE_50)) {
		php_error(E_WARNING, "Wrong COM_STMT_PREPARE response size. Received %d", data_size);
		return FAIL;
	}

	packet->stmt_id = uint4korr(p);
	p += 4;

	/* Number of columns in result set */
	packet->field_count = uint2korr(p);
	p += 2;

	packet->param_count = uint2korr(p);
	p += 2;

	/* 0x0 filler sent by the server for 4.1 clients */
	p++;

	if (data_size >= 12) {
		packet->warning_count= uint2korr(p);
	}

#ifndef MYSQLND_SILENT
	php_printf("\tPrepare packet read: stmt_id=%d fields=%d params=%d\n",
				packet->stmt_id, packet->field_count, packet->param_count);
#endif

	return PASS;
}
/* }}} */


/* {{{ php_mysqlnd_prepare_free_mem */
static
void php_mysqlnd_prepare_free_mem(void *_packet, zend_bool alloca)
{
	php_mysql_packet_prepare_response *p= (php_mysql_packet_prepare_response *) _packet;
	if (!alloca) {
		efree(p);
	}
}
/* }}} */


/* {{{ php_mysqlnd_chg_user_read */
static enum_func_status
php_mysqlnd_chg_user_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	/* There could be an error message */
	zend_uchar buf[1024];
	zend_uchar *p= buf;
	php_mysql_packet_chg_user_resp *packet= (php_mysql_packet_chg_user_resp *) _packet;

	PACKET_READ_HEADER_AND_BODY(packet, conn, buf, sizeof(buf), "change user response ");

	/*
	  Don't increment. First byte is 0xFF on error, but otherwise is starting byte
	  of encoded sequence for length.
	*/

	/* Should be always 0x0 or 0xFF for error */
	packet->field_count= uint1korr(p);
	p++;

	if (packet->header.size == 1 && buf[0] == 0xFE &&
		packet->server_capabilities & CLIENT_SECURE_CONNECTION) {
		/* We don't handle 3.23 authentication */
		return FAIL;
	}

	if (0xFF == packet->field_count) {
		php_mysqlnd_read_error_from_line(p, packet->header.size - 1,
										 packet->error_info.error,
										 sizeof(packet->error_info.error),
										 &packet->error_info.error_no,
										 packet->error_info.sqlstate);
	}

	return PASS;
}
/* }}} */


/* {{{ php_mysqlnd_chg_user_free_mem */
static
void php_mysqlnd_chg_user_free_mem(void *_packet, zend_bool alloca)
{
	if (!alloca) {
		efree(_packet);
	}
}
/* }}} */


/* {{{ php_mysqlnd_binrowp_read */
static enum_func_status
php_mysqlnd_binrowp_read(void *_packet, MYSQLND *conn TSRMLS_DC)
{
	/* Try with 1024byte buffer on the stack, if too small, allocate bigger */
	int buf_size = 512*1024L; 
	zend_uchar *buf, *p;
	enum_func_status ret = PASS;
	unsigned int data_size = 0;
	unsigned int old_chunk_size = conn->stream->chunk_size;

	php_mysql_packet_binrow *packet= (php_mysql_packet_binrow *) _packet;

	if (!packet->row_buffer) {
		packet->row_buffer = malloc(buf_size);
	}
	buf = packet->row_buffer;
	if (FAIL == (ret = php_mysqlnd_read_row_ex(conn, &buf, buf_size, &data_size TSRMLS_CC))) {
		goto end;
	}

	packet->header.size = data_size;

	if ((*(p = buf)) == 0xFE && data_size < 8) { /* EOF */
		packet->eof = TRUE;
		p++;

		if (data_size > 1) {
			packet->warning_count = uint2korr(p);
			p += 2;
			packet->server_status = uint2korr(p);
			// Seems we have 3 bytes reserved for future use
		}

		goto end;
	}

	packet->eof = FALSE;
	/* packet->field_count is set by the user */

	/*
	  Allocate this buffer only when this function is used for first
	  time with specified packet
	*/
	if (!packet->fields) {
		int i;
		packet->fields = (zval **) emalloc(packet->field_count * sizeof(zval *));
		for (i = 0; i < packet->field_count; i++) {
			INIT_PZVAL(packet->fields[i]);
		}
	}

	{
		zval **current_field, **end_field;
		end_field = (current_field = packet->fields) + packet->field_count;

		for (;current_field < end_field; current_field++) {
			ulong len;
			/* NULL or NOT NULL, this is the question! */
			if ((len=(ulong) php_mysqlnd_net_field_length(&p)) == MYSQLND_NULL_LENGTH) { 
				ZVAL_NULL(*current_field);
			} else {
				ZVAL_STRINGL(*current_field, p, len, 1);
				p += len;
			}
		}
	}
end:
	if (buf != packet->row_buffer) {
		efree(buf);
	}
	conn->stream->chunk_size = old_chunk_size;
	return ret;
}
/* }}} */


/* {{{ php_mysqlnd_binrowp_free_mem */
static
void php_mysqlnd_binrowp_free_mem(void *_packet, zend_bool alloca)
{
	php_mysql_packet_binrow *p= (php_mysql_packet_binrow *) _packet;
	if (p->fields) {
		int i;
		for (i = 0; i < p->field_count; i++) {
			zval_dtor(p->fields[i]);
		}
		/*
		  Don't free every single instance, we have transfered the
		  ownership.
		*/
		efree(p->fields);
		p->fields = NULL;
	}
	if (p->row_buffer) {
		free(p->row_buffer);
		p->row_buffer = NULL;
	}
	/* We get p->lengths passed to us, so we should not free it */
	if (!alloca) {
		efree(p);
	}
}
/* }}} */


/* {{{ packet_methods 
 */
mysqlnd_packet_methods packet_methods[PROT_LAST] =
{
	{
		sizeof(php_mysql_packet_greet),
		php_mysqlnd_greet_read,
		NULL, /* write */
		php_mysqlnd_greet_free_mem,
	}, /* PROT_GREET_PACKET */
	{
		sizeof(php_mysql_packet_auth),
		NULL, /* read */
		php_mysqlnd_auth_write,
		php_mysqlnd_auth_free_mem,
	}, /* PROT_AUTH_PACKET */
	{
		sizeof(php_mysql_packet_ok),
		php_mysqlnd_ok_read, /* read */
		NULL, /* write */
		php_mysqlnd_ok_free_mem,
	}, /* PROT_OK_PACKET */
	{
		sizeof(php_mysql_packet_eof),
		php_mysqlnd_eof_read, /* read */
		NULL, /* write */
		php_mysqlnd_eof_free_mem,
	}, /* PROT_EOF_PACKET */
	{
		sizeof(php_mysql_packet_command),
		NULL, /* read */
		php_mysqlnd_cmd_write, /* write */
		php_mysqlnd_cmd_free_mem,
	}, /* PROT_CMD_PACKET */
	{
		sizeof(php_mysql_packet_rset_header),
		php_mysqlnd_rset_header_read, /* read */
		NULL, /* write */
		php_mysqlnd_rset_header_free_mem,
	}, /* PROT_RSET_HEADER_PACKET */
	{
		sizeof(php_mysql_packet_res_field),
		php_mysqlnd_rset_field_read, /* read */
		NULL, /* write */
		php_mysqlnd_rset_field_free_mem,
	}, /* PROT_RSET_FLD_PACKET */
	{
		sizeof(php_mysql_packet_row),
		php_mysqlnd_rowp_read, /* read */
		NULL, /* write */
		php_mysqlnd_rowp_free_mem,
	}, /* PROT_ROW_PACKET */
	{
		sizeof(php_mysql_packet_stats),
		php_mysqlnd_stats_read, /* read */
		NULL, /* write */
		php_mysqlnd_stats_free_mem,
	}, /* PROT_STATS_PACKET */
	{
		sizeof(php_mysql_packet_prepare_response),
		php_mysqlnd_prepare_read, /* read */
		NULL, /* write */
		php_mysqlnd_prepare_free_mem,
	}, /* PROT_PREPARE_RESP_PACKET */
	{
		sizeof(php_mysql_packet_chg_user_resp),
		php_mysqlnd_chg_user_read, /* read */
		NULL, /* write */
		php_mysqlnd_chg_user_free_mem,
	}, /* PROT_CHG_USER_PACKET */
	{
		sizeof(php_mysql_packet_binrow),
		php_mysqlnd_binrowp_read, /* read */
		NULL, /* write */
		php_mysqlnd_binrowp_free_mem,
	}, /* PROT_BINROW_PACKET */
};
/* }}} */


/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
