/*
  +----------------------------------------------------------------------+
  | PHP Version 6                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 2006-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Georg Richter <georg@mysql.com>                             |
  |          Andrey Hristov <andrey@mysql.com>                           |
  |          Ulf Wendel <uwendel@mysql.com>                              |
  +----------------------------------------------------------------------+
*/

/* $Id: header,v 1.17 2006/01/01 13:09:48 sniper Exp $ */
#include "php.h"
#include "mysqlnd.h"
#include "mysqlnd_wireprotocol.h"
#include "mysqlnd_priv.h"
#include "mysqlnd_statistics.h"
#include "ext/standard/basic_functions.h"

#define MYSQLND_SILENT

/* the server doesn't support 4byte utf8, but let's make it forward compatible */
#define MYSQLND_MAX_ALLOWED_USER_LEN	256  /* 64 char * 4byte */
#define MYSQLND_MAX_ALLOWED_DB_LEN		256  /* 64 char * 4byte */
/*
  TODO :
  - Don't bind so tightly the metadata with the result set. This means
    that the metadata reading should not expect a MYSQLND_RES pointer, it
	does not need it, but return a pointer to the metadata (MYSQLND_FIELD *).
	For normal statements we will then just assign it to a member of
	MYSQLND_RES. For PS statements, it will stay as part of the statement
	(MYSQLND_STMT) between prepare and execute. At execute the new metadata
	will be sent by the server, so we will discard the old one and then
	finally attach it to the result set. This will make the code more clean,
	as a prepared statement won't have anymore stmt->result != NULL, as it
	is now, just to have where to store the metadata.

 - Change mysqlnd_simple_command to accept a heap dynamic array of MYSQLND_STRING
   terminated by a string with ptr being NULL. Thus, multi-part messages can be
   sent to the network like writev() and this can save at least for
   mysqlnd_stmt_send_long_data() new malloc. This change will probably make the
   code in few other places cleaner.
*/

extern MYSQLND_CHARSET *mysqlnd_charsets;




static const char * mysqlnd_server_gone = "MySQL server has gone away";
const char * mysqlnd_out_of_sync = "Commands out of sync; you can't run this command now";

MYSQLND_STATS *mysqlnd_global_stats = NULL;
static zend_bool mysqlnd_library_initted = FALSE;


/* {{{ mysqlnd_library_init */
PHPAPI void mysqlnd_library_init()
{
	if (mysqlnd_library_initted == FALSE) {
		mysqlnd_library_initted = TRUE;
 		_mysqlnd_init_ps_subsystem();
		mysqlnd_global_stats = calloc(1, sizeof(MYSQLND_STATS));
#ifdef ZTS
		mysqlnd_global_stats->LOCK_access = tsrm_mutex_alloc();
#endif
	}
}
/* }}} */


/* {{{ mysqlnd_library_end */
PHPAPI void mysqlnd_library_end()
{
	if (mysqlnd_library_initted == TRUE) {
#ifdef ZTS
		tsrm_mutex_free(mysqlnd_global_stats->LOCK_access);
#endif
		free(mysqlnd_global_stats);
		mysqlnd_global_stats = NULL;
		mysqlnd_library_initted = FALSE;
	}
}
/* }}} */


/* {{{ php_mysqlnd_free_field_metadata */
static
void php_mysqlnd_free_field_metadata(MYSQLND_FIELD *meta, zend_bool persistent)
{
	if (meta) {
		if (meta->root) {
			pefree(meta->root, persistent);
			meta->root = NULL;
		}
		if (meta->def) {
			pefree(meta->def, persistent);
			meta->def = NULL;
		}
	}
}
/* }}} */


/* {{{ mysqlnd_unbuffered_free_last_data */
void mysqlnd_unbuffered_free_last_data(MYSQLND_RES *result TSRMLS_DC)
{
	MYSQLND_RES_UNBUFFERED *unbuf = result->unbuf;
	if (!unbuf) {
		return;
	}

	if (unbuf->last_row_data) {
		int i;
		zend_bool copy_ctor_called;
		for (i = 0; i < result->field_count; i++) {
			if (result->type == MYSQLND_RES_PS) {
				/* Do nothing, before assigning we will clean */
				zval_ptr_dtor(&unbuf->last_row_data[i]);
			} else {
				mysqlnd_palloc_zval_ptr_dtor(&(unbuf->last_row_data[i]),
											 result->zval_cache, FALSE,
											 &copy_ctor_called TSRMLS_CC);
				if (copy_ctor_called) {
					MYSQLND_INC_CONN_STATISTIC(&result->conn->stats,
											   STAT_COPY_ON_WRITE_PERFORMED);
					/* Increase global stats */
				} else {
					MYSQLND_INC_CONN_STATISTIC(&result->conn->stats,
											   STAT_COPY_ON_WRITE_SAVED);
				}
			}
		}
		/* Free last row's zvals */
		efree(unbuf->last_row_data);
		unbuf->last_row_data = NULL;
	}
	if (unbuf->last_row_buffer) {
		/* Nothing points to this buffer now, free it */
		efree(unbuf->last_row_buffer);
		unbuf->last_row_buffer = NULL;
	}
}
/* }}} */


/* {{{ mysqlnd_free_buffered_data */
void mysqlnd_free_buffered_data(MYSQLND_RES *result TSRMLS_DC)
{
	MYSQLND_ZVAL_PCACHE  *zval_cache = result->zval_cache;
	MYSQLND_RES_BUFFERED *set = result->data;
	enum_mysqlnd_res_type result_type = result->type;
	unsigned int field_count = result->field_count;
	unsigned int row;

	for (row = 0; row < result->data->row_count; row++) {
		unsigned int col;
		zval **current_row = current_row = set->data[row];
		zend_uchar *current_buffer = set->row_buffers[row];

		for (col = 0; col < field_count; col++) {
			if (result_type == MYSQLND_RES_PS) {
				/* For now PS always does create copies when fetching data */
				zval_ptr_dtor(&current_row[col]);

				MYSQLND_INC_CONN_STATISTIC(NULL, STAT_COPY_ON_WRITE_PERFORMED);
			} else {
				zend_bool copy_ctor_called;
				/* Free only if we haven't referenced it */
				mysqlnd_palloc_zval_ptr_dtor(&(current_row[col]), zval_cache,
											 FALSE, &copy_ctor_called TSRMLS_CC);

				MYSQLND_INC_CONN_STATISTIC(NULL, copy_ctor_called? STAT_COPY_ON_WRITE_PERFORMED:
																   STAT_COPY_ON_WRITE_SAVED);
			}
		}
		pefree(current_row, set->persistent);
		pefree(current_buffer, set->persistent);
	}
	pefree(set->data, set->persistent);
	pefree(set->row_buffers, set->persistent);
	set->data			= NULL;
	set->row_buffers	= NULL;
	set->data_cursor	= NULL;
	set->row_count	= 0;
	if (set->persistent) {
		mysqlnd_qcache_free_cache_reference(&set->qcache);
	}
	pefree(set, set->persistent);
	result->data		= NULL;
}
/* }}} */


/* {{{ mysqlnd_internal_free_result_buffers */
void mysqlnd_internal_free_result_buffers(MYSQLND_RES *result TSRMLS_DC)
{

	if (!result->data) {
		mysqlnd_unbuffered_free_last_data(result TSRMLS_CC);
		if (result->unbuf) {
			efree(result->unbuf);
			result->unbuf = NULL;
		}
	} else {
		mysqlnd_free_buffered_data(result TSRMLS_CC);
	}

	if (result->lengths) {
		efree(result->lengths);
		result->lengths = NULL;
	}
}
/* }}} */


/* {{{ mysqlnd_internal_free_result_metadata */
static
void mysqlnd_internal_free_result_metadata(MYSQLND_RES_METADATA *meta, zend_bool persistent TSRMLS_DC)
{
	int i;
	MYSQLND_FIELD *fields;

	if ((fields = meta->fields)) {
		i = meta->field_count;
		while (i--) {
			php_mysqlnd_free_field_metadata(fields++, persistent);
		}
		pefree(meta->fields, persistent);
		meta->fields = NULL;
	}

	if (meta->zend_hash_keys) {
#if PHP_MAJOR_VERSION >= 6
		if (UG(unicode)) {
			for (i = 0; i < meta->field_count; i++) {
				if (meta->zend_hash_keys[i].ustr.v) {
					pefree(meta->zend_hash_keys[i].ustr.v, persistent);
				}
			}
		}
#endif
		pefree(meta->zend_hash_keys, persistent);
		meta->zend_hash_keys = NULL;
	}
	pefree(meta, persistent);
}
/* }}} */


/* {{{ mysqlnd_internal_free_result_contents */
void mysqlnd_internal_free_result_contents(MYSQLND_RES *result TSRMLS_DC)
{

	result->m.free_result_buffers(result TSRMLS_CC);

	if (result->row_packet) {
		if (result->type == MYSQLND_RES_NORMAL) {
			PACKET_FREE(result->row_packet);
		} else {
			PACKET_FREE(result->row_packet);
		}
		result->row_packet = NULL;
	}

	result->conn = NULL;

	if (result->meta) {
		mysqlnd_internal_free_result_metadata(result->meta, FALSE TSRMLS_CC);
		result->meta = NULL;
	}

	if (result->zval_cache) {
		mysqlnd_palloc_free_cache_reference(&result->zval_cache);
	}
}
/* }}} */


/* {{{ mysqlnd_internal_free_result */
void mysqlnd_internal_free_result(MYSQLND_RES *result TSRMLS_DC)
{
	/*
	  result->conn is an address if this is an unbuffered query.
	  In this case, decrement the reference counter in the connection
	  object and if needed free the latter. If quit_sent is no
	*/
	if (result->conn) {
		result->conn->m->free_reference(result->conn TSRMLS_CC);
		result->conn = NULL;
	}

	mysqlnd_internal_free_result_contents(result TSRMLS_CC);
	efree(result);
}
/* }}} */


/* {{{ mysqlnd_conn_free_contents */
static
void mysqlnd_conn_free_contents(MYSQLND *conn TSRMLS_DC)
{
	mysqlnd_local_infile_default(conn);
	if (conn->current_result) {
		mysqlnd_internal_free_result_contents(conn->current_result TSRMLS_CC);
		efree(conn->current_result);
		conn->current_result = NULL;
	}

	if (conn->net.stream) {
		php_stream_free(conn->net.stream, (conn->persistent) ? PHP_STREAM_FREE_RSRC_DTOR : PHP_STREAM_FREE_CLOSE);
		conn->net.stream = NULL;
	}
	if (conn->host) {
		pefree(conn->host, conn->persistent);
		conn->host = NULL;
	}
	if (conn->user) {
		pefree(conn->user, conn->persistent);
		conn->user = NULL;
	}
	if (conn->passwd) {
		pefree(conn->passwd, conn->persistent);
		conn->passwd = NULL;
	}
	if (conn->unix_socket) {
		pefree(conn->unix_socket, conn->persistent);
		conn->unix_socket = NULL;
	}
	if (conn->scheme) {
		pefree(conn->scheme, conn->persistent);
		conn->scheme = NULL;
	}
	if (conn->server_version) {
		pefree(conn->server_version, conn->persistent);
		conn->server_version = NULL;
	}
	if (conn->host_info) {
		pefree(conn->host_info, conn->persistent);
		conn->host_info = NULL;
	}
	if (conn->scramble) {
		pefree(conn->scramble, conn->persistent);
		conn->scramble = NULL;
	}
	if (conn->last_message) {
		pefree(conn->last_message, conn->persistent);
		conn->last_message = NULL;
	}
	if (conn->options.num_commands) {
		unsigned int i;
		for (i=0; i < conn->options.num_commands; i++) {
			pefree(conn->options.init_commands[i], conn->persistent);
		}
		pefree(conn->options.init_commands, conn->persistent);
		conn->options.init_commands = NULL;
	}
	if (conn->options.cfg_file) {
		pefree(conn->options.cfg_file, conn->persistent);
		conn->options.cfg_file = NULL;
	}
	if (conn->options.cfg_section) {
		pefree(conn->options.cfg_section, conn->persistent);
		conn->options.cfg_section = NULL;
	}
	if (conn->options.ssl_key) {
		pefree(conn->options.ssl_key, conn->persistent);
		conn->options.ssl_key = NULL;
	}
	if (conn->options.ssl_cert) {
		pefree(conn->options.ssl_cert, conn->persistent);
		conn->options.ssl_cert = NULL;
	}
	if (conn->options.ssl_ca) {
		pefree(conn->options.ssl_ca, conn->persistent);
		conn->options.ssl_ca = NULL;
	}
	if (conn->options.ssl_capath) {
		pefree(conn->options.ssl_capath, conn->persistent);
		conn->options.ssl_capath = NULL;
	}
	if (conn->options.ssl_cipher) {
		pefree(conn->options.ssl_cipher, conn->persistent);
		conn->options.ssl_cipher = NULL;
	}
	if (conn->zval_cache) {
		mysqlnd_palloc_free_cache_reference(&conn->zval_cache);
		conn->zval_cache = NULL;
	}
	if (conn->qcache) {
		mysqlnd_qcache_free_cache_reference(&conn->qcache);
		conn->qcache = NULL;
	}
	if (conn->net.cmd_buffer.buffer) {
		pefree(conn->net.cmd_buffer.buffer, conn->persistent);
		conn->net.cmd_buffer.buffer = NULL;
	}
}
/* }}} */


/* {{{ _mysqlnd_conn_dtor */
static
void _mysqlnd_conn_dtor(MYSQLND *conn TSRMLS_DC)
{
	conn->m->free_contents(conn TSRMLS_CC);

	pefree(conn, conn->persistent);
}
/* }}} */


/* {{{ mysqlnd_simple_command_handle_response */
enum_func_status
mysqlnd_simple_command_handle_response(MYSQLND *conn, enum php_mysql_packet_type ok_packet,
									   zend_bool silent, enum php_mysqlnd_server_command command
									   TSRMLS_DC)
{
	enum_func_status ret;
	switch (ok_packet) {
		case PROT_OK_PACKET:{
			php_mysql_packet_ok ok_response;
			PACKET_INIT_ALLOCA(ok_response, PROT_OK_PACKET);
			if (FAIL == (ret = PACKET_READ_ALLOCA(ok_response, conn))) {
				if (!silent) {
					php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error while reading %s's OK packet",
									 mysqlnd_command_to_text[command]);
				}
			} else {
#ifndef MYSQLND_SILENT
				php_printf("\tOK from server\n");
#endif
				if (0xFF == ok_response.field_count) {
					/* The server signalled error. Set the error */
					SET_CLIENT_ERROR(conn->error_info, ok_response.error_no,
									 ok_response.sqlstate, ok_response.error); 
					ret = FAIL;
					/*
					  Cover a protocol design error: error packet does not
					  contain the server status. Therefore, the client has no way
					  to find out whether there are more result sets of
					  a multiple-result-set statement pending. Luckily, in 5.0 an
					  error always aborts execution of a statement, wherever it is
					  a multi-statement or a stored procedure, so it should be
					  safe to unconditionally turn off the flag here.
					*/
					conn->upsert_status.server_status &= ~SERVER_MORE_RESULTS_EXISTS;
					conn->upsert_status.affected_rows = (mynd_ulonglong) ~0;
				} else {
					SET_NEW_MESSAGE(conn->last_message, conn->last_message_len,
									ok_response.message, ok_response.message_len);

					conn->upsert_status.warning_count = ok_response.warning_count;
					conn->upsert_status.server_status = ok_response.server_status;
					conn->upsert_status.affected_rows = ok_response.affected_rows;
					conn->upsert_status.last_insert_id = ok_response.last_insert_id;
				}
			}
			PACKET_FREE_ALLOCA(ok_response);
			break;
		}
		case PROT_EOF_PACKET:{
			php_mysql_packet_eof ok_response;
			PACKET_INIT_ALLOCA(ok_response, PROT_EOF_PACKET);
			if (FAIL == (ret = PACKET_READ_ALLOCA(ok_response, conn))) {
				SET_CLIENT_ERROR(conn->error_info, CR_MALFORMED_PACKET, UNKNOWN_SQLSTATE,
								 "Malformed packet");
				if (!silent) {
					php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error while reading %s's EOF packet",
								 	mysqlnd_command_to_text[command]);
				}
			} else if (0xFF == ok_response.field_count) {
				/* The server signalled error. Set the error */
				SET_CLIENT_ERROR(conn->error_info, ok_response.error_no,
								 ok_response.sqlstate, ok_response.error); 
				conn->upsert_status.affected_rows = (mynd_ulonglong) ~0;
			} else if (0xFE != ok_response.field_count) {
				SET_CLIENT_ERROR(conn->error_info, CR_MALFORMED_PACKET, UNKNOWN_SQLSTATE,
								 "Malformed packet");
				if (!silent) {
					php_error_docref(NULL TSRMLS_CC, E_WARNING,
									"EOF packet expected, field count wasn't 0xFE but 0x%2X",
									ok_response.field_count);
				}
			} else {
#ifndef MYSQLND_SILENT
				php_printf("\tOK from server\n");
#endif
			}
			PACKET_FREE_ALLOCA(ok_response);
			break;
		}
		default:
			ret = FAIL;
			SET_CLIENT_ERROR(conn->error_info, CR_MALFORMED_PACKET, UNKNOWN_SQLSTATE,
							 "Malformed packet");
			php_error_docref(NULL TSRMLS_CC, E_ERROR, "Wrong response packet %d passed to the function",
							 ok_packet);
			break;
	}
	return ret;
}
/* }}} */


/* {{{ mysqlnd_simple_command */
enum_func_status
mysqlnd_simple_command(MYSQLND *conn, enum php_mysqlnd_server_command command,
					   const char * const arg, size_t arg_len,
					   enum php_mysql_packet_type ok_packet, zend_bool silent TSRMLS_DC)
{
	enum_func_status ret = PASS;
	php_mysql_packet_command cmd_packet;

	switch (conn->state) {
		case CONN_READY:
			break;
		case CONN_QUIT_SENT:
			SET_CLIENT_ERROR(conn->error_info, CR_SERVER_GONE_ERROR, UNKNOWN_SQLSTATE, mysqlnd_server_gone); 
			return FAIL;	
		default:
			SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
							 mysqlnd_out_of_sync); 
			return FAIL;
	}

	/* clean UPSERT info */
	memset(&conn->upsert_status, 0, sizeof(conn->upsert_status));
	conn->upsert_status.affected_rows = (mynd_ulonglong) ~0;
	SET_EMPTY_ERROR(conn->error_info);

	PACKET_INIT_ALLOCA(cmd_packet, PROT_CMD_PACKET);
	cmd_packet.command = command;
	if (arg && arg_len) {
		cmd_packet.argument = arg;
		cmd_packet.arg_len  = arg_len;
	}

	if (! PACKET_WRITE_ALLOCA(cmd_packet, conn)) {
		if (!silent) {
			php_error(E_WARNING, "Error while sending %s packet", mysqlnd_command_to_text[command]);
		}
		SET_CLIENT_ERROR(conn->error_info, CR_SERVER_GONE_ERROR, UNKNOWN_SQLSTATE, mysqlnd_server_gone); 
		ret = FAIL;
	} else if (ok_packet != PROT_LAST) {
		ret = mysqlnd_simple_command_handle_response(conn, ok_packet, silent, command TSRMLS_CC);
	}

	/*
	  There is no need to call FREE_ALLOCA on cmd_packet as the
	  only allocated string is cmd_packet.argument and it was passed
	  to us. We should not free it.
	*/

	return ret;
}
/* }}} */


/* {{{ _mysqlnd_set_server_option */
static enum_func_status
_mysqlnd_set_server_option(MYSQLND * const conn, enum_mysqlnd_server_option option TSRMLS_DC)
{
	char buffer[2];
	int2store(buffer, (uint) option);
	return mysqlnd_simple_command(conn, COM_SET_OPTION, buffer, sizeof(buffer), PROT_EOF_PACKET, FALSE TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_start_psession */
PHPAPI void mysqlnd_restart_psession(MYSQLND *conn) 
{
	MYSQLND_INC_CONN_STATISTIC(&conn->stats, STAT_CONNECT_REUSED);
}
/* }}} */


/* {{{ mysqlnd_end_psession */
PHPAPI void mysqlnd_end_psession(MYSQLND *conn)
{

}
/* }}} */


/* {{{ mysqlnd_connect */
PHPAPI MYSQLND *mysqlnd_connect(MYSQLND *conn,
						 char *host, char *user,
						 char *passwd, unsigned int passwd_len,
						 char *db, unsigned int db_len,
						 unsigned int port,
						 char *socket,
						 unsigned int mysql_flags,
						 MYSQLND_ZVAL_PCACHE *zval_cache TSRMLS_DC)
{
	char *transport = NULL, *errstr = NULL;
	int transport_len, errcode = 0;
	unsigned int streams_options = ENFORCE_SAFE_MODE;
	unsigned int streams_flags = STREAM_XPORT_CLIENT | STREAM_XPORT_CONNECT;
	zend_bool self_alloced = FALSE;
	struct timeval tv;
	zend_bool unix_socket = FALSE;

	php_mysql_packet_greet greet_packet;
	php_mysql_packet_auth *auth_packet;
	php_mysql_packet_ok ok_packet;

	if (conn && conn->state != CONN_ALLOCED) {
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 mysqlnd_out_of_sync);
		return NULL;
	}

	if (!host || !host[0]) {
		host = "localhost";
	}
	if (!user || !user[0]) {
		user = php_get_current_user();
	}
	if (!passwd) {
		passwd = "";
		passwd_len = 0;
	}
	if (!db) {
		db = "";
		db_len = 0;
	}
	if (!port && !socket) {
		port = 3306;
	}
#ifndef PHP_WIN32
	if (!strncasecmp(host, "localhost", sizeof("localhost") - 1)) {
		if (!socket) {
			socket = "/tmp/mysql.sock";
		}
		transport_len = spprintf(&transport, 0, "unix://%s", socket);
		unix_socket = TRUE;
	} else 
#endif
	{
		transport_len = spprintf(&transport, 0, "tcp://%s:%d", host, port);
	}

	PACKET_INIT_ALLOCA(greet_packet, PROT_GREET_PACKET);
	PACKET_INIT(auth_packet, PROT_AUTH_PACKET, php_mysql_packet_auth *);
	PACKET_INIT_ALLOCA(ok_packet, PROT_OK_PACKET);

	if (!conn) {
		conn = mysqlnd_init(FALSE);
		self_alloced = TRUE;
	}

	conn->state	= CONN_ALLOCED;
	conn->net.packet_no = 0;

	if (conn->options.timeout_connect) {
		tv.tv_sec = conn->options.timeout_connect;
		tv.tv_usec = 0;
	}
	if (conn->persistent) {
		conn->scheme = pestrndup(transport, transport_len, 1);
		efree(transport);
	} else {
		conn->scheme = transport;
	}
	conn->net.stream = php_stream_xport_create(conn->scheme, transport_len, streams_options, streams_flags,
												(conn->persistent) ? "mysqli_plink" : NULL, 
												(conn->options.timeout_connect) ? &tv : NULL,
												NULL /*ctx*/, &errstr, &errcode);
	
	if (errstr || !conn->net.stream) {
		goto err;
	}

	if (conn->options.timeout_read)
	{
		tv.tv_sec = conn->options.timeout_read;
		tv.tv_usec = 0;
		php_stream_set_option(conn->net.stream, PHP_STREAM_OPTION_READ_TIMEOUT, 0, &tv);
	}

	if (!unix_socket) {
		/* Set TCP_NODELAY */
		mysqlnd_set_sock_no_delay(conn->net.stream);
	}

	if (FAIL == PACKET_READ_ALLOCA(greet_packet, conn)) {
#ifndef MYSQLND_SILENT
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error while reading greeting packet");
#endif
		goto err;
	} else if (greet_packet.error_no) {
		SET_CLIENT_ERROR(conn->error_info, greet_packet.error_no,
						 greet_packet.sqlstate, greet_packet.error);
		goto err;	
	} else if (greet_packet.pre41) {
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Connecting to 3.22, 3.23 & 4.0 "
						" is not supported. Server is %-.32s", greet_packet.server_version);
		SET_CLIENT_ERROR(conn->error_info, CR_NOT_IMPLEMENTED, UNKNOWN_SQLSTATE,
						 "Connecting to 3.22, 3.23 & 4.0 servers is not supported");
		goto err;
	}

	conn->thread_id			= greet_packet.thread_id;
	conn->protocol_version	= greet_packet.protocol_version;
	conn->server_version	= greet_packet.server_version;
	greet_packet.server_version = NULL; /* The string will be freed otherwise */

	/* we allow load data local infile by default */
	mysql_flags  |= CLIENT_LOCAL_FILES;

	auth_packet->user		= user;
	auth_packet->password	= passwd;
	auth_packet->charset_no	= greet_packet.charset_no;
	auth_packet->db			= db;
	auth_packet->db_len		= db_len;
	auth_packet->max_packet_size= 3UL*1024UL*1024UL*1024UL;
	auth_packet->client_flags= mysql_flags;

	conn->scramble = auth_packet->server_scramble_buf = pemalloc(SCRAMBLE_LENGTH, conn->persistent);
	memcpy(auth_packet->server_scramble_buf, greet_packet.scramble_buf, SCRAMBLE_LENGTH);
	PACKET_WRITE(auth_packet, conn);

	if (FAIL == PACKET_READ_ALLOCA(ok_packet, conn) || ok_packet.field_count >= 0xFE) {
		if (ok_packet.field_count == 0xFE) {
			/* pre41 server !*/
			php_error_docref(NULL TSRMLS_CC, E_WARNING, "mysqlnd cannot connect to MySQL < 4.1 servers");
		} else if (ok_packet.field_count == 0xFF) {
			if (ok_packet.sqlstate[0]) {
				if (!self_alloced) {
					strncpy(conn->error_info.sqlstate, ok_packet.sqlstate, sizeof(conn->error_info.sqlstate));
				}
#ifndef MYSQLND_SILENT
				php_error_docref(NULL TSRMLS_CC, E_WARNING, "ERROR:%d [SQLSTATE:%s] %s",
								 ok_packet.error_no, ok_packet.sqlstate, ok_packet.error);
#endif
			}
			if (!self_alloced) {
				conn->error_info.error_no = ok_packet.error_no;
				strncpy(conn->error_info.error, ok_packet.error, sizeof(conn->error_info.error));
			}
		}
	} else {
		conn->state				= CONN_READY;

		conn->user				= pestrdup(user, conn->persistent);
		conn->passwd			= pestrndup(passwd, passwd_len, conn->persistent);
		conn->port				= port;
		if (host && !socket) {
			char *p;

			conn->host = pestrdup(host, conn->persistent);
			spprintf(&p, 0, "MySQL host info: %s via TCP/IP", conn->host);
			if (conn->persistent) {
				conn->host_info = pestrdup(p, 1);
				efree(p);
			} else {
				conn->host_info = p;
			}
		} else {
			conn->unix_socket	= pestrdup(socket, conn->persistent);
			conn->host_info		= pestrdup("MySQL host info: Localhost via UNIX socket", conn->persistent);
		}
		conn->client_flag		= auth_packet->client_flags;
		conn->max_packet_size	= auth_packet->max_packet_size;
		/* todo: check if charset is available */
		conn->charset			= mysqlnd_find_charset_nr(greet_packet.charset_no);
		conn->server_capabilities = greet_packet.server_capabilities;
		conn->upsert_status.warning_count = 0;
		conn->upsert_status.server_status = greet_packet.server_status;
		SET_NEW_MESSAGE(conn->last_message, conn->last_message_len,
						ok_packet.message, ok_packet.message_len);

		SET_EMPTY_ERROR(conn->error_info);

		PACKET_FREE_ALLOCA(greet_packet);
		PACKET_FREE(auth_packet);
		PACKET_FREE_ALLOCA(ok_packet);

		conn->zval_cache = mysqlnd_palloc_get_cache_reference(zval_cache);
		conn->net.cmd_buffer.length = 128L*1024L;
		conn->net.cmd_buffer.buffer = pemalloc(conn->net.cmd_buffer.length, conn->persistent);

		MYSQLND_INC_CONN_STATISTIC(&conn->stats, STAT_CONNECT_SUCCESS);

		{
			uint as_unicode = 1;
			conn->m->set_client_option(conn, MYSQLND_OPT_NUMERIC_AND_DATETIME_AS_UNICODE,
									   (char *)&as_unicode);
		}

		return conn;
	}
err:
	PACKET_FREE_ALLOCA(greet_packet);
	PACKET_FREE(auth_packet);
	PACKET_FREE_ALLOCA(ok_packet);

	if (errstr) {
		SET_CLIENT_ERROR(conn->error_info, errcode, UNKNOWN_SQLSTATE, errstr);

		php_error_docref(NULL TSRMLS_CC, E_WARNING, "[%d] %.64s (trying to connect via %s)", errcode, errstr, conn->scheme);

		efree(errstr);
	}
	if (conn->scheme) {
		pefree(conn->scheme, conn->persistent);
		conn->scheme = NULL;
	}


	/* This will also close conn->net.stream if it has been opened */
	conn->m->free_contents(conn TSRMLS_CC);

	if (self_alloced) {
		/*
		  We have alloced, thus there are no references to this
		  object - we are free to kill it!
		*/
		conn->m->dtor(conn TSRMLS_CC);
	} else {
		MYSQLND_INC_CONN_STATISTIC(&conn->stats, STAT_CONNECT_FAILURE);	
	}
	return NULL;
}
/* }}} */


/* {{{ mysqlnd_handle_numeric */
/*
  The following code is stolen from ZE - HANDLE_NUMERIC() macro from zend_hash.c
  and modified for the needs of mysqlnd.
*/
static
zend_bool mysqlnd_is_key_numeric(char *key, size_t length, long *idx)
{
	register char *tmp=key;

	if (*tmp=='-') {
		tmp++;
	}
	if ((*tmp>='0' && *tmp<='9')) {
		do { /* possibly a numeric index */
			char *end=key+length-1;

			if (*tmp++=='0' && length>2) { /* don't accept numbers with leading zeros */
				break;
			}
			while (tmp<end) {
				if (!(*tmp>='0' && *tmp<='9')) {
					break;
				}
				tmp++;
			}
			if (tmp==end && *tmp=='\0') { /* a numeric index */
				if (*key=='-') {
					*idx = strtol(key, NULL, 10);
					if (*idx!=LONG_MIN) {
						return TRUE;
					}
				} else {
					*idx = strtol(key, NULL, 10);
					if (*idx!=LONG_MAX) {
						return TRUE;
					}
				}
			}
		} while (0);
	}
	return FALSE;
}
/* }}} */


#if PHP_MAJOR_VERSION >= 6
/* {{{ mysqlnd_unicode_is_key_numeric */
static
zend_bool mysqlnd_unicode_is_key_numeric(UChar *key, size_t length, long *idx)
{
	register UChar *tmp=key;

	if (*tmp==0x2D /*'-'*/) {
		tmp++;
	}
	if ((*tmp>=0x30 /*'0'*/ && *tmp<=0x39 /*'9'*/)) { /* possibly a numeric index */
		do {
			UChar *end=key+length-1;

			if (*tmp++==0x30 && length>2) { /* don't accept numbers with leading zeros */
				break;
			}
			while (tmp<end) {
				if (!(*tmp>=0x30 /*'0'*/ && *tmp<=0x39 /*'9'*/)) {
					break;
				}
				tmp++;
			}
			if (tmp==end && *tmp==0) { /* a numeric index */
				if (*key==0x2D /*'-'*/) {
					*idx = zend_u_strtol(key, NULL, 10);
					if (*idx!=LONG_MIN) {
						return TRUE;
					}
				} else {
					*idx = zend_u_strtol(key, NULL, 10);
					if (*idx!=LONG_MAX) {
						return TRUE;
					}
				}
			}
		} while (0);
	}
	return FALSE;
}
/* }}} */
#endif


/* {{{ mysqlnd_read_result_metadata */
enum_func_status
mysqlnd_read_result_metadata(MYSQLND *conn, MYSQLND_RES *result TSRMLS_DC)
{
	int i = 0;
	php_mysql_packet_res_field field_packet;

	/*
	  Make it safe to call it repeatedly for PS -
	  better free and allocate a new because the number of field might change 
	  (select *) with altered table. Also for statements which skip the PS
	  infrastructure.
	*/
	if (result->meta) {
		mysqlnd_internal_free_result_metadata(result->meta, FALSE TSRMLS_CC);
		result->meta = NULL;
	}

	/* +1 is to have empty marker at the end */
	result->meta = ecalloc(1, sizeof(MYSQLND_RES_METADATA));
	result->meta->field_count = result->field_count;
	result->meta->fields = ecalloc(result->field_count + 1, sizeof(MYSQLND_FIELD));
	result->meta->zend_hash_keys = ecalloc(result->field_count,
									 		sizeof(struct mysqlnd_field_hash_key));

	/* 1. Read all fields metadata */

	/* It's safe to reread without freeing */
	PACKET_INIT_ALLOCA(field_packet, PROT_RSET_FLD_PACKET);
	for (;i < result->field_count; i++) {
		long idx;

		if (result->meta->fields[i].root) {
			/* We re-read metadata for PS */
			efree(result->meta->fields[i].root);
			result->meta->fields[i].root = NULL;
		}

		field_packet.metadata = &(result->meta->fields[i]);
		if (FAIL == PACKET_READ_ALLOCA(field_packet, conn)) {
			PACKET_FREE_ALLOCA(field_packet);
			goto error;
		}

		if (mysqlnd_ps_fetch_functions[result->meta->fields[i].type].func == NULL) {
			SET_CLIENT_ERROR(result->conn->error_info, CR_UNKNOWN_ERROR, UNKNOWN_SQLSTATE,
							 "Uknown field type sent by the server"); 
			php_error_docref(NULL TSRMLS_CC, E_WARNING,
							 "Unknown type %d sent by the server. "
							 "Please send a report to the developers",
							 result->meta->fields[i].type);
			goto error;
		}


#if PHP_MAJOR_VERSION >= 6
		if (UG(unicode)) {
			UChar *ustr;
			int ulen;
			zend_string_to_unicode(UG(utf8_conv), &ustr, &ulen,
								   result->meta->fields[i].name,
								   result->meta->fields[i].name_length TSRMLS_CC);
			if ((result->meta->zend_hash_keys[i].is_numeric =
				 			mysqlnd_unicode_is_key_numeric(ustr, ulen + 1, &idx)))
			{
				result->meta->zend_hash_keys[i].key = idx;
				efree(ustr);
			} else {
				result->meta->zend_hash_keys[i].ustr.u = ustr;
				result->meta->zend_hash_keys[i].ulen = ulen;
				result->meta->zend_hash_keys[i].key = zend_u_get_hash_value(IS_UNICODE, ZSTR(ustr), ulen + 1);
			}

		} else 
#endif
		{
			/* For BC we have to check whether the key is numeric and use it like this */
			if ((result->meta->zend_hash_keys[i].is_numeric =
						mysqlnd_is_key_numeric(field_packet.metadata->name,
									   		   field_packet.metadata->name_length + 1,
											   &idx)))
			{
				result->meta->zend_hash_keys[i].key = idx;
			} else {
				result->meta->zend_hash_keys[i].key =
						zend_get_hash_value(field_packet.metadata->name,
											field_packet.metadata->name_length + 1);
			}
		}
	}
	PACKET_FREE_ALLOCA(field_packet);

	/*
	  2. Follows an EOF packet, which the client of mysqlnd_read_result_metadata()
	     should consume.
	  3. If there is a result set, it follows. The last packet will have 'eof' set
	  	 If PS, then no result set follows.
	*/

	return PASS;
error:
	mysqlnd_internal_free_result_contents(result TSRMLS_CC);
	return FAIL;
}
/* }}} */


enum_func_status
mysqlnd_query_read_result_set_header(MYSQLND *conn, MYSQLND_STMT *stmt TSRMLS_DC)
{
	enum_func_status ret;
	php_mysql_packet_rset_header rset_header;

	ret = FAIL;
	PACKET_INIT_ALLOCA(rset_header, PROT_RSET_HEADER_PACKET);
	do {
		if (FAIL == (ret = PACKET_READ_ALLOCA(rset_header, conn))) {
			php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error reading result set's header");
			break;
		}

		if (rset_header.error_info.error_no) {
			/*
			  Cover a protocol design error: error packet does not
			  contain the server status. Therefore, the client has no way
			  to find out whether there are more result sets of
			  a multiple-result-set statement pending. Luckily, in 5.0 an
			  error always aborts execution of a statement, wherever it is
			  a multi-statement or a stored procedure, so it should be
			  safe to unconditionally turn off the flag here.
			*/
			conn->upsert_status.server_status &= ~SERVER_MORE_RESULTS_EXISTS;
			conn->upsert_status.affected_rows = -1;
			/*
			  This will copy the error code and the messages, as they
			  are buffers in the struct
			*/
			conn->error_info = rset_header.error_info;
			ret = FAIL;
			break;
		}
		conn->error_info.error_no = 0;

		switch (rset_header.field_count) {
			case MYSQLND_NULL_LENGTH: {	/* LOAD DATA LOCAL INFILE */
				zend_bool is_warning;
				conn->last_query_type = QUERY_LOAD_LOCAL;
				ret = mysqlnd_handle_local_infile(conn, rset_header.info_or_local_file, &is_warning TSRMLS_CC);

				conn->state = (ret == PASS || is_warning == TRUE)? CONN_READY:CONN_QUIT_SENT;
				MYSQLND_INC_CONN_STATISTIC(&conn->stats, STAT_NON_RSET_QUERY);
				break;
			}
			case 0:				/* UPSERT		*/
				conn->last_query_type = QUERY_UPSERT;
				conn->field_count = rset_header.field_count;
				conn->upsert_status.warning_count = rset_header.warning_count;
				conn->upsert_status.server_status = rset_header.server_status;
				conn->upsert_status.affected_rows = rset_header.affected_rows;
				conn->upsert_status.last_insert_id = rset_header.last_insert_id;
				SET_NEW_MESSAGE(conn->last_message, conn->last_message_len,
								rset_header.info_or_local_file, rset_header.info_or_local_file_len);
				/* Result set can follow UPSERT statement, check server_status */
				if (conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS) {
					conn->state = CONN_NEXT_RESULT_PENDING;
				} else {
					conn->state = CONN_READY;
				}
				ret = PASS;
				MYSQLND_INC_CONN_STATISTIC(&conn->stats, STAT_NON_RSET_QUERY);
				break;
			default:{			/* Result set	*/
				php_mysql_packet_eof fields_eof;
				MYSQLND_RES *result;
				uint stat = -1;

				MYSQLND_INC_CONN_STATISTIC(&conn->stats, STAT_RSET_QUERY);
				memset(&conn->upsert_status, 0, sizeof(conn->upsert_status));
				conn->last_query_type = QUERY_SELECT;
				conn->state = CONN_FETCHING_DATA;
				/* PS has already allocated it */
				if (!stmt) {
					conn->field_count = rset_header.field_count;
					result =
						conn->current_result=
							mysqlnd_result_init(rset_header.field_count,
												mysqlnd_palloc_get_cache_reference(conn->zval_cache));
				} else {
					if (!stmt->result) {
						/*
						  This is 'SHOW'/'EXPLAIN'-like query. Current implementation of
						  prepared statements can't send result set metadata for these queries
						  on prepare stage. Read it now.
						*/
						conn->field_count = rset_header.field_count;
						result =
							stmt->result =
								mysqlnd_result_init(rset_header.field_count,
													mysqlnd_palloc_get_cache_reference(conn->zval_cache));
					} else {
						/*
						  Update result set metadata if it for some reason changed between
						  prepare and execute, i.e.:
						  - in case of 'SELECT ?' we don't know column type unless data was
						    supplied to mysql_stmt_execute, so updated column type is sent
							now.
						  - if data dictionary changed between prepare and execute, for
							example a table used in the query was altered.
						  Note, that now (4.1.3) we always send metadata in reply to
						  COM_STMT_EXECUTE (even if it is not necessary), so either this or
						  previous branch always works.
						*/	
					}
					result = stmt->result;
				}

				if (FAIL == (ret = mysqlnd_read_result_metadata(conn, result TSRMLS_CC))) {
					/* For PS, we leave them in Prepared state */
					if (!stmt) {
						efree(conn->current_result);
						conn->current_result = NULL;
					}
					break;
				}

				/* Check for SERVER_STATUS_MORE_RESULTS if needed */
				PACKET_INIT_ALLOCA(fields_eof, PROT_EOF_PACKET);
				if (FAIL == (ret = PACKET_READ_ALLOCA(fields_eof, conn))) {
					mysqlnd_internal_free_result_contents(result TSRMLS_CC);
					efree(result);
					if (!stmt) {
						conn->current_result = NULL;
					} else {
						stmt->result = NULL;
						memset(stmt, 0, sizeof(MYSQLND_STMT));
						stmt->state = MYSQLND_STMT_INITTED;
					}
				} else {
					conn->upsert_status.warning_count = fields_eof.warning_count;
					conn->upsert_status.server_status = fields_eof.server_status;
				}

				if (fields_eof.server_status & MYSQLND_SERVER_QUERY_NO_GOOD_INDEX_USED) {
					stat = STAT_BAD_INDEX_USED;
				} else if (fields_eof.server_status & MYSQLND_SERVER_QUERY_NO_INDEX_USED) {
					stat = STAT_NO_INDEX_USED;
				}
				if (stat != -1) {
					MYSQLND_INC_CONN_STATISTIC(&conn->stats, stat);
				}
				PACKET_FREE_ALLOCA(fields_eof);

				break;
			}
		}
	} while (0);
	PACKET_FREE_ALLOCA(rset_header);
	return ret;
}


/* {{{ mysqlnd_query */
/* 
  If conn->error_info.error_no is not zero, then we had an error.
  Still the result from the query is PASS
*/
static enum_func_status
_mysqlnd_query(MYSQLND *conn, const char *query, unsigned int query_len TSRMLS_DC)
{

	if (PASS != mysqlnd_simple_command(conn, COM_QUERY, query, query_len,
									   PROT_LAST /* we will handle the OK packet*/,
									   FALSE TSRMLS_CC)) {
		return FAIL;		
	}

	/*
	  Here read the result set. We don't do it in simple_command because it need
	  information from the ok packet. We will fetch it ourselves.
	*/
	return mysqlnd_query_read_result_set_header(conn, NULL TSRMLS_CC);
}
/* }}} */


/* {{{ mysqlnd_fetch_lengths_unbuffered */
static
unsigned long * mysqlnd_fetch_lengths_unbuffered(MYSQLND_RES * const result)
{
	return result->lengths;
}
/* }}} */


/* {{{ mysqlnd_fetch_lengths_buffered */
/*
  Do lazy initialization for buffered results. As PHP strings have
  length inside, this function makes not much sense in the context
  of PHP, to be called as separate function. But let's have it for
  completeness.
*/
static 
unsigned long * mysqlnd_fetch_lengths_buffered(MYSQLND_RES * const result)
{
	int i;
	zval **previous_row;

	/*
	  If:
	  - unbuffered result
	  - first row hs not been read
	  - last_row has been read
	*/
	if (result->data->data_cursor == NULL ||
		result->data->data_cursor == result->data->data ||
		((result->data->data_cursor - result->data->data) > result->data->row_count))
	{
		return NULL;/* No rows or no more rows */
	}

	previous_row = *(result->data->data_cursor - 1);
	for (i = 0; i < result->field_count; i++) {
		result->lengths[i] = (Z_TYPE_P(previous_row[i]) == IS_NULL)? 0:Z_STRLEN_P(previous_row[i]);
	}

	return result->lengths;
}
/* }}} */


/* {{{ _mysqlnd_fetch_lengths */
PHPAPI unsigned long * _mysqlnd_fetch_lengths(MYSQLND_RES * const result)
{
	return result->m.fetch_lengths? result->m.fetch_lengths(result):NULL;
}
/* }}} */


/* {{{ mysqlnd_fetch_row_unbuffered */
static enum_func_status
mysqlnd_fetch_row_unbuffered(MYSQLND_RES *result, void *param, unsigned int flags,
							zend_bool *fetched_anything TSRMLS_DC)
{
	enum_func_status		ret;
	zval 					*row = (zval *) param;
	unsigned int			i,
							field_count = result->field_count;
	php_mysql_packet_row	*row_packet = result->row_packet;
	unsigned long			*lengths = result->lengths;

	if (result->unbuf->eof_reached) {
		/* No more rows obviously */
		*fetched_anything = FALSE;
		return PASS;
	}
	if (result->conn->state != CONN_FETCHING_DATA) {
		SET_CLIENT_ERROR(result->conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 mysqlnd_out_of_sync); 
		return FAIL;
	}
	/* Let the row packet fill our buffer and skip additional malloc + memcpy */
	row_packet->skip_extraction = row? FALSE:TRUE;

	/*
	  If we skip rows (row == NULL) we have to
	  mysqlnd_unbuffered_free_last_data() before it. The function returns always true.
	*/
	if (PASS == (ret = PACKET_READ(row_packet, result->conn)) && !row_packet->eof) {
		mysqlnd_unbuffered_free_last_data(result TSRMLS_CC);

		result->unbuf->last_row_data = row_packet->fields;
		result->unbuf->last_row_buffer = row_packet->row_buffer;
		row_packet->fields = NULL;
		row_packet->row_buffer = NULL;

		result->unbuf->row_count++;
		*fetched_anything = TRUE;

		MYSQLND_INC_CONN_STATISTIC(&result->conn->stats, STAT_ROWS_FETCHED_FROM_CLIENT);

		if (!row_packet->skip_extraction) {
			HashTable *row_ht = Z_ARRVAL_P(row);
			for (i = 0; i < field_count; i++) {
				zval *data = result->unbuf->last_row_data[i];
				int len = (Z_TYPE_P(data) == IS_NULL)? 0:Z_STRLEN_P(data);

				if (lengths) {
					lengths[i] = len;
				}

				/* Forbid ZE to free it, we will clean it */
				ZVAL_ADDREF(data);

				if ((flags & MYSQLND_FETCH_BOTH) == MYSQLND_FETCH_BOTH) {
					ZVAL_ADDREF(data);
				}
				if (flags & MYSQLND_FETCH_NUM) {
					zend_hash_next_index_insert(row_ht, &data, sizeof(zval *), NULL);
				}
				if (flags & MYSQLND_FETCH_ASSOC) {
					/* zend_hash_quick_update needs length + trailing zero */
					/* QQ: Error handling ? */
					/*
					  zend_hash_quick_update does not check, as add_assoc_zval_ex do, whether
					  the index is a numeric and convert it to it. This however means constant
					  hashing of the column name, which is not needed as it can be precomputed.
					*/
					if (result->meta->zend_hash_keys[i].is_numeric == FALSE) {
#if PHP_MAJOR_VERSION >= 6
						if (UG(unicode)) {
							zend_u_hash_quick_update(Z_ARRVAL_P(row), IS_UNICODE,
													 result->meta->zend_hash_keys[i].ustr,
													 result->meta->zend_hash_keys[i].ulen + 1,
													 result->meta->zend_hash_keys[i].key,
													 (void *) &data, sizeof(zval *), NULL);
						} else
#endif
						{
							zend_hash_quick_update(Z_ARRVAL_P(row),
												   result->meta->fields[i].name,
												   result->meta->fields[i].name_length + 1,
												   result->meta->zend_hash_keys[i].key,
												   (void *) &data, sizeof(zval *), NULL);
						}
					} else {
						zend_hash_index_update(Z_ARRVAL_P(row),
											   result->meta->zend_hash_keys[i].key,
											   (void *) &data, sizeof(zval *), NULL);
					}
				}
				if (result->meta->fields[i].max_length < len) {
					result->meta->fields[i].max_length = len;
				}
			}
		}
	} else if (row_packet->eof) {
		/* Mark the connection as usable again */

		result->unbuf->eof_reached = TRUE;
		result->conn->upsert_status.warning_count = row_packet->warning_count;
		result->conn->upsert_status.server_status = row_packet->server_status;
		/*
		  result->row_packet will be cleaned when
		  destroying the result object
		*/
		if (result->conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS) {
			result->conn->state = CONN_NEXT_RESULT_PENDING;
		} else {
			result->conn->state = CONN_READY;
		}
		mysqlnd_unbuffered_free_last_data(result TSRMLS_CC);
		*fetched_anything = FALSE;
	}

	return PASS;
}
/* }}} */


/* {{{ mysqlnd_use_result */
static
MYSQLND_RES * _mysqlnd_use_result(MYSQLND * const conn TSRMLS_DC)
{
	MYSQLND_RES *result;

	if (!conn->current_result) {
		return NULL;
	}

	/* Nothing to store for UPSERT/LOAD DATA */
	if (conn->last_query_type != QUERY_SELECT || conn->state != CONN_FETCHING_DATA) {
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 mysqlnd_out_of_sync); 
		return NULL;
	}

	MYSQLND_INC_CONN_STATISTIC(&conn->stats, STAT_UNBUFFERED_SETS);

	result = conn->current_result;
	conn->current_result = NULL;

	result->type			= MYSQLND_RES_NORMAL;
	result->m.fetch_row		= result->m.fetch_row_normal_unbuffered;
	result->m.fetch_lengths	= mysqlnd_fetch_lengths_unbuffered;
	result->unbuf			= ecalloc(1, sizeof(MYSQLND_RES_UNBUFFERED));

	result->conn = conn->m->get_reference(conn);
	/*
	  Will be freed in the mysqlnd_internal_free_result_contents() called
	  by the resource destructor. mysqlnd_fetch_row_unbuffered() expects
	  this to be not NULL.
	*/
	PACKET_INIT(result->row_packet, PROT_ROW_PACKET, php_mysql_packet_row *);
	result->row_packet->field_count = result->field_count;
	result->row_packet->binary_protocol = FALSE;
	result->row_packet->fields_metadata = result->meta->fields;
	result->lengths = ecalloc(result->field_count, sizeof(unsigned long));

	/* No multithreading issues as we don't share the connection :) */

	return result;
}
/* }}} */


/* {{{ mysqlnd_fetch_row_buffered */
static enum_func_status
mysqlnd_fetch_row_buffered(MYSQLND_RES *result, void *param, unsigned int flags,
						   zend_bool *fetched_anything TSRMLS_DC)
{
	unsigned int i;
	zval *row = (zval *) param;

	/* If we haven't read everything */
	if (result->data->data_cursor &&
		(result->data->data_cursor - result->data->data) < result->data->row_count)
	{
		zval **current_row = *result->data->data_cursor;
		for (i = 0; i < result->field_count; i++) {
			zval *data = current_row[i];

			/*
			  Let us later know what to do with this zval. If ref_count > 1, we will just
			  decrease it, otherwise free it. zval_ptr_dtor() make this very easy job.
			*/
			ZVAL_ADDREF(data);
			
			if ((flags & MYSQLND_FETCH_BOTH) == MYSQLND_FETCH_BOTH) {
				ZVAL_ADDREF(data);
			}
			if (flags & MYSQLND_FETCH_NUM) {
				zend_hash_next_index_insert(Z_ARRVAL_P(row), &data, sizeof(zval *), NULL);
			}
			if (flags & MYSQLND_FETCH_ASSOC) {
				/* zend_hash_quick_update needs length + trailing zero */
				/* QQ: Error handling ? */
				/*
				  zend_hash_quick_update does not check, as add_assoc_zval_ex do, whether
				  the index is a numeric and convert it to it. This however means constant
				  hashing of the column name, which is not needed as it can be precomputed.
				*/
				if (result->meta->zend_hash_keys[i].is_numeric == FALSE) {
#if PHP_MAJOR_VERSION >= 6
					if (UG(unicode)) {
						zend_u_hash_quick_update(Z_ARRVAL_P(row), IS_UNICODE,
												 result->meta->zend_hash_keys[i].ustr,
												 result->meta->zend_hash_keys[i].ulen + 1,
												 result->meta->zend_hash_keys[i].key,
												 (void *) &data, sizeof(zval *), NULL);
					} else
#endif
					{
						zend_hash_quick_update(Z_ARRVAL_P(row),
											   result->meta->fields[i].name,
											   result->meta->fields[i].name_length + 1,
											   result->meta->zend_hash_keys[i].key,
											   (void *) &data, sizeof(zval *), NULL);
					}
				} else {
					zend_hash_index_update(Z_ARRVAL_P(row),
										   result->meta->zend_hash_keys[i].key,
										   (void *) &data, sizeof(zval *), NULL);
				}
			}
		}
		result->data->data_cursor++;
		*fetched_anything = TRUE;
	} else {
		result->data->data_cursor = NULL;
		*fetched_anything = FALSE;
#ifndef MYSQLND_SILENT
		php_printf("NO MORE DATA\n ");
#endif
	}
	return PASS;
}
/* }}} */


#define STORE_RESULT_PREALLOCATED_SET 32

/* {{{ mysqlnd_store_result */
static
MYSQLND_RES *_mysqlnd_store_result(MYSQLND * const conn TSRMLS_DC)
{
	enum_func_status ret;
	unsigned int next_extend = STORE_RESULT_PREALLOCATED_SET, free_rows;
	php_mysql_packet_row row_packet;
	MYSQLND_RES *result;
	zend_bool to_cache = FALSE;

	if (!conn->current_result) {
		return NULL;
	}
	
	/* Nothing to store for UPSERT/LOAD DATA*/
	if (conn->last_query_type != QUERY_SELECT || conn->state != CONN_FETCHING_DATA) {
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 mysqlnd_out_of_sync); 
		return NULL;
	}

	MYSQLND_INC_CONN_STATISTIC(&conn->stats, STAT_BUFFERED_SETS);

	result = conn->current_result;
	conn->current_result = NULL;

	result->type			= MYSQLND_RES_NORMAL;
	result->m.fetch_row		= result->m.fetch_row_normal_buffered;
	result->m.fetch_lengths	= mysqlnd_fetch_lengths_buffered;

	conn->state = CONN_FETCHING_DATA;

	result->lengths = ecalloc(result->field_count, sizeof(unsigned long));

	/* Create room for 'next_extend' rows */

	result->conn 		= NULL;	/* store result does not reference  the connection */
	result->data 				= pecalloc(1, sizeof(MYSQLND_RES_BUFFERED), to_cache);
	result->data->data 		  	= pemalloc(next_extend * sizeof(zval **), to_cache);
	result->data->row_buffers 	= pemalloc(next_extend * sizeof(zend_uchar *), to_cache);
	result->data->persistent	= to_cache;
	result->data->qcache		= to_cache? mysqlnd_qcache_get_cache_reference(conn->qcache):NULL;
	result->data->references	= 1;


	free_rows = next_extend;

	PACKET_INIT_ALLOCA(row_packet, PROT_ROW_PACKET);
	row_packet.field_count = result->field_count;
	row_packet.binary_protocol = FALSE;
	row_packet.fields_metadata = result->meta->fields;
	/* Let the row packet fill our buffer and skip additional malloc + memcpy */
	while (FAIL != (ret = PACKET_READ_ALLOCA(row_packet, conn)) && !row_packet.eof) {
		int i;
		zval **current_row;

		if (!free_rows) {
			unsigned long total_rows = free_rows = next_extend = next_extend * 5 / 3; /* extend with 33% */
			total_rows += result->data->row_count;
			result->data->data = perealloc(result->data->data,
										   total_rows * sizeof(zval **), to_cache);

			result->data->row_buffers = perealloc(result->data->row_buffers,
												  total_rows * sizeof(zend_uchar *), to_cache);
		}
		free_rows--;
		current_row = result->data->data[result->data->row_count] = row_packet.fields;
		result->data->row_buffers[result->data->row_count] = row_packet.row_buffer;
		result->data->row_count++;

		/* So row_packet's destructor function won't efree() it */
		row_packet.fields = NULL;
		row_packet.row_buffer = NULL;

		for (i = 0; i < row_packet.field_count; i++) {
			unsigned long len;
			/*
			  NULL fields are 0 length, 0 is not more than 0
			  String of zero size, definitely can't be the next max_length.
			  Thus for NULL and zero-length we are quite efficient.
			*/
			if (Z_TYPE_P(current_row[i]) != IS_NULL &&
				(len = Z_STRLEN_P(current_row[i])) &&
				result->meta->fields[i].max_length < len)
			{
				result->meta->fields[i].max_length = len;
			}
		}
		/*
		  No need to FREE_ALLOCA as we can reuse the
		  'lengths' and 'fields' arrays. For lengths its absolutely safe.
		  'fields' is reused because the ownership of the strings has been
		  transfered above. 
		*/
	}
	MYSQLND_INC_CONN_STATISTIC_W_VALUE(&conn->stats, STAT_ROWS_FETCHED_FROM_CLIENT,
									   result->data->row_count);

	/* Finally clean */
	if (row_packet.eof) { 
		conn->upsert_status.warning_count = row_packet.warning_count;
		conn->upsert_status.server_status = row_packet.server_status;
	}
	PACKET_FREE_ALLOCA(row_packet);
	/* save some memory */
	if (free_rows) {
		result->data->data = perealloc(result->data->data,
									   result->data->row_count * sizeof(zval **),
									   to_cache);
		result->data->row_buffers = perealloc(result->data->row_buffers,
											  result->data->row_count * sizeof(zend_uchar *),
											  to_cache);
	}

	if (conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS) {
		conn->state = CONN_NEXT_RESULT_PENDING;
	} else {
		conn->state = CONN_READY;
	}

	if (PASS == ret) {
		/* Position at the first row */
		result->data->data_cursor = result->data->data;

		/* libmysql's documentation says it should be so for SELECT statements */
		conn->upsert_status.affected_rows = result->data->row_count;
	} else {
		mysqlnd_internal_free_result_contents(result TSRMLS_CC);
		efree(result);
		result = NULL;
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Pretty serious error");
	}
	return result;
}
/* }}} */


/* {{{ _mysqlnd_unbuffered_skip_result */
static enum_func_status
_mysqlnd_unbuffered_skip_result(MYSQLND_RES * const result TSRMLS_DC)
{
	zend_bool fetched_anything;

	/*
	  Unbuffered sets
	  A PS could be prepared - there is metadata and thus a stmt->result but the
	  fetch_row function isn't actually set (NULL), thus we have to skip these.
	*/
	if (!result->data && result->conn && result->unbuf &&
		!result->unbuf->eof_reached && result->m.fetch_row)
	{
		/* We have to fetch all data to clean the line */
		MYSQLND_INC_CONN_STATISTIC(&result->conn->stats,
									result->type == MYSQLND_RES_NORMAL? STAT_FLUSHED_NORMAL_SETS:
																		STAT_FLUSHED_PS_SETS);

		while ((PASS == result->m.fetch_row(result, NULL, 0, &fetched_anything TSRMLS_CC)) &&
			   fetched_anything == TRUE) {
			/* do nothing */;
		}
	}
	return PASS;
}
/* }}} */


/* {{{ _mysqlnd_free_result */
static enum_func_status
_mysqlnd_free_result(MYSQLND_RES *result, zend_bool implicit TSRMLS_DC)
{
	result->m.skip_result(result TSRMLS_CC);
	MYSQLND_INC_CONN_STATISTIC(result->conn? &result->conn->stats : NULL,
							   implicit == TRUE?	STAT_FREE_RESULT_EXPLICIT:
													STAT_FREE_RESULT_IMPLICIT);

	mysqlnd_internal_free_result(result TSRMLS_CC);
	return PASS;
}
/* }}} */


/* {{{ _mysqlnd_data_seek */
static enum_func_status
_mysqlnd_data_seek(MYSQLND_RES *result, mynd_ulonglong row)
{
	if (!result->data) {
		return FAIL;
	}

	/* libmysql just moves to the end, it does traversing of a linked list */
	if (row >= result->data->row_count) {
		result->data->data_cursor = NULL;
	} else {
		result->data->data_cursor = result->data->data + row;
	}

	return PASS;
}
/* }}} */


/* {{{ mysqlnd_errno */
static
unsigned int _mysqlnd_errno(const MYSQLND * const conn)
{
	return conn->error_info.error_no;
}
/* }}} */


/* {{{ mysqlnd_error */
static
const char * _mysqlnd_error(const MYSQLND * const conn)
{
	return conn->error_info.error;
}
/* }}} */


/* {{{ mysqlnd_sqlstate */
static
const char * _mysqlnd_sqlstate(const MYSQLND * const conn)
{
	return conn->error_info.sqlstate[0] ? conn->error_info.sqlstate:MYSQLND_SQLSTATE_NULL;
}
/* }}} */


/* {{{ _mysqlnd_real_escape_quotes */
static
ulong _mysqlnd_real_escape_quotes(const MYSQLND * const conn, char *newstr, const char *escapestr, int escapestr_len)
{
	const char 	*newstr_s = newstr;
	const char 	*newstr_e = newstr + 2 * escapestr_len;
	const char 	*end = escapestr + escapestr_len;
	zend_bool	escape_overflow = FALSE;
	
	for (;escapestr < end; escapestr++) {
		uint len;
		/* check unicode characters */
		if (conn->charset->char_maxlen > 1 && conn->charset->mb_charlen(*escapestr) > 1) {
			len = conn->charset->mb_valid(escapestr, end);
			/* check possible overflow */
			if ((newstr + len) > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			/* copy mb char without escaping it */
			while (len--) {
				*newstr++ = *escapestr++;
			}
			escapestr--;
			continue;
		}
		if (*newstr == '\'') {
			if (newstr + 2 > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			*newstr++ = '\'';
			*newstr++ = '\'';
		} else {
			if (newstr + 1 > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			*newstr++= *escapestr;
		}
	}
	*newstr = '\0';

	if (escape_overflow) {
		return (ulong)~0;
	}
	return (ulong)(newstr - newstr_s);
}
/* }}} */


/* {{{ _mysqlnd_real_escape_slashes */
static
ulong _mysqlnd_real_escape_slashes(const MYSQLND * const conn, char *newstr, const char *escapestr, int escapestr_len)
{
	const char 	*newstr_s = newstr;
	const char 	*newstr_e = newstr + 2 * escapestr_len;
	const char 	*end = escapestr + escapestr_len;
	zend_bool	escape_overflow = FALSE;
	
	for (;escapestr < end; escapestr++) {
		uint len;
		char	esc = '\0';

		/* check unicode characters */
		if (conn->charset->char_maxlen > 1 && conn->charset->mb_charlen(*escapestr) > 1) {
			len = conn->charset->mb_valid(escapestr, end);

			/* check possible overflow */
			if ((newstr + len) > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			/* copy mb char without escaping it */
			while (len--) {
				*newstr++ = *escapestr++;
			}
			escapestr--;
			continue;
		}
		if (conn->charset->char_maxlen > 1 && conn->charset->mb_charlen(*escapestr) > 1) {
			esc = *escapestr;
		} else {
			switch (*escapestr) {
				case 0:
					esc = '0';
					break;
				case '\n':
					esc = 'n';
					break;
				case '\r':
					esc = 'r';
					break;
				case '\\':
				case '\'':
				case '"':
					esc = *escapestr;
					break;
				case '\032':
					esc = 'Z';
					break;
			}
		}
		if (esc) {
			if (newstr + 2 > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			/* copy escaped character */
			*newstr++ = '\\';
			*newstr++ = esc;
		} else {
			if (newstr + 1 > newstr_e) {
				escape_overflow = TRUE;
				break;
			}
			/* copy non escaped character */
			*newstr++ = *escapestr;
		}
	}
	*newstr = '\0';
	if (escape_overflow) {
		return (ulong)~0;
	}
	return (ulong)(newstr - newstr_s);
}
/* }}} */


/* {{{ _mysqlnd_real_escape_string */
static
ulong _mysqlnd_real_escape_string(const MYSQLND * const conn, char *newstr, const char *escapestr, int escapestr_len)
{
	if (conn->upsert_status.server_status & SERVER_STATUS_NO_BACKSLASH_ESCAPES) {
		return _mysqlnd_real_escape_quotes(conn, newstr, escapestr, escapestr_len);
	}
	return _mysqlnd_real_escape_slashes(conn, newstr, escapestr, escapestr_len);
}


/* {{{ _mysqlnd_dump_debug_info */
static enum_func_status
_mysqlnd_dump_debug_info(MYSQLND * const conn TSRMLS_DC)
{
	return mysqlnd_simple_command(conn, COM_DEBUG, NULL, 0, PROT_EOF_PACKET, FALSE TSRMLS_CC);
}
/* }}} */


/* {{{ _mysqlnd_select_db */
static enum_func_status
_mysqlnd_select_db(MYSQLND * const conn, const char * const db, unsigned int db_len TSRMLS_DC)
{
	enum_func_status ret;
	ret = mysqlnd_simple_command(conn, COM_INIT_DB, db, db_len, PROT_OK_PACKET, FALSE TSRMLS_CC);
	/*
	  The server sends 0 but libmysql doesn't read it and has established
	  a protocol of giving back -1. Thus we have to follow it :(
	*/
	SET_ERROR_AFF_ROWS(conn);

	return ret;
}
/* }}} */


/* {{{ _mysqlnd_ping */
static enum_func_status
_mysqlnd_ping(MYSQLND * const conn TSRMLS_DC)
{
	enum_func_status ret;
	ret = mysqlnd_simple_command(conn, COM_PING, NULL, 0, PROT_OK_PACKET, FALSE TSRMLS_CC);
	/*
	  The server sends 0 but libmysql doesn't read it and has established
	  a protocol of giving back -1. Thus we have to follow it :(
	*/
	SET_ERROR_AFF_ROWS(conn);

	return ret;
}
/* }}} */


/* {{{ mysqlnd_stat */
static enum_func_status
_mysqlnd_stat(MYSQLND *conn, char **message, unsigned int * message_len TSRMLS_DC)
{
	enum_func_status ret;
	php_mysql_packet_stats stats_header;

	ret = mysqlnd_simple_command(conn, COM_STATISTICS, NULL, 0, PROT_LAST, FALSE TSRMLS_CC);
	if (FAIL == ret) {
		return FAIL;
	}
	PACKET_INIT_ALLOCA(stats_header, PROT_STATS_PACKET);
	if (FAIL == (ret = PACKET_READ_ALLOCA(stats_header, conn))) {
		return FAIL;
	}
	*message = stats_header.message;
	*message_len = stats_header.message_len;
	/* Ownership transfer */
	stats_header.message = NULL;
	PACKET_FREE_ALLOCA(stats_header);

	return PASS;
}
/* }}} */


/* {{{ _mysqlnd_kill */
static enum_func_status
_mysqlnd_kill(MYSQLND *conn, unsigned long pid TSRMLS_DC)
{
	enum_func_status ret;
	char buff[4];

	int4store(buff, pid);

	/* If we kill ourselves don't expect OK packet, PROT_LAST will skip it */
	if (pid != conn->thread_id) {
		ret = mysqlnd_simple_command(conn, COM_PROCESS_KILL, buff, 4, PROT_OK_PACKET, FALSE TSRMLS_CC);
		/*
		  The server sends 0 but libmysql doesn't read it and has established
		  a protocol of giving back -1. Thus we have to follow it :(
		*/
		conn->upsert_status.affected_rows = -1;	
	} else if (PASS == (ret = mysqlnd_simple_command(conn, COM_PROCESS_KILL, buff,
													 4, PROT_LAST, FALSE TSRMLS_CC))) {
		conn->state = CONN_QUIT_SENT;
	}
	return ret;
}
/* }}} */


/* {{{ _mysqlnd_set_charset */
static enum_func_status
_mysqlnd_set_charset(MYSQLND * const conn, const char * const csname TSRMLS_DC)
{
	char query[MAX_CHARSET_LEN + 12];
	const MYSQLND_CHARSET * const charset = mysqlnd_find_charset_name(csname);

	if (!charset) {
		SET_CLIENT_ERROR(conn->error_info, CR_CANT_FIND_CHARSET, UNKNOWN_SQLSTATE, "Invalid characterset or character set not supported");
		return FAIL;
	}

	strcpy(query, "SET NAMES ");
	strcat(query, csname);

	if (FAIL == conn->m->query(conn, query, strlen(query) TSRMLS_CC)) {
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error executing query");
	} else if (conn->error_info.error_no) {
		return FAIL;
	}
	conn->charset = charset;
	return PASS;
}
/* }}} */


/* {{{ _mysqlnd_refresh */
static enum_func_status
_mysqlnd_refresh(MYSQLND * const conn, unsigned long options TSRMLS_DC)
{
	zend_uchar bits[1];
	int1store(bits, options);

	return mysqlnd_simple_command(conn, COM_REFRESH, (char *)bits, 1, PROT_OK_PACKET, FALSE TSRMLS_CC);
}
/* }}} */


/* {{{ _mysqlnd_shutdown */
static enum_func_status
_mysqlnd_shutdown(MYSQLND * const conn, unsigned long level TSRMLS_DC)
{
	zend_uchar bits[1];
	int1store(bits, level);

	return mysqlnd_simple_command(conn, COM_SHUTDOWN, (char *)bits, 1, PROT_OK_PACKET, FALSE TSRMLS_CC);
}
/* }}} */


static enum_mysqlnd_collected_stats
close_type_to_stat_map[MYSQLND_CLOSE_LAST] = {
	STAT_CLOSE_EXPLICIT,
	STAT_CLOSE_IMPLICIT,
	STAT_CLOSE_DISCONNECT
};

/* {{{ _mysqlnd_send_close */
static enum_func_status
_mysqlnd_send_close(MYSQLND * conn TSRMLS_DC)
{
	enum_func_status ret = PASS;
	switch (conn->state) {
		case CONN_READY:
			ret =  mysqlnd_simple_command(conn, COM_QUIT, NULL, 0, PROT_LAST,
										  conn->tmp_int? TRUE : FALSE TSRMLS_CC);
			/* Do nothing */
			break;
		case CONN_NEXT_RESULT_PENDING:
		case CONN_QUERY_SENT:
		case CONN_FETCHING_DATA:
			MYSQLND_INC_CONN_STATISTIC(NULL, STAT_CLOSE_IN_MIDDLE);
#ifndef MYSQLND_SILENT
			php_printf("Brutally closing connection [%p][%s]\n", conn, conn->scheme);
#endif
			/*
			  Do nothing, the connection will be brutally closed
			  and the server will catch it and free close from its side.
			*/
		case CONN_ALLOCED:
			/*
			  Allocated but not connected or there was failure when trying
			  to connect with pre-allocated connect.

			  Fall-through
			*/
		case CONN_QUIT_SENT:
			/* The user has killed its own connection */
			break;
	}
	/*
	  We hold one reference, and every other object which needs the
	  connection does increase it by 1.
	*/
	conn->state = CONN_QUIT_SENT;

	return ret;
}
/* }}} */


/* {{{ _mysqlnd_close */
static enum_func_status
_mysqlnd_close(MYSQLND * conn, enum_connection_close_type close_type TSRMLS_DC)
{
	enum_func_status ret = PASS;
	enum_mysqlnd_collected_stats stat = close_type_to_stat_map[close_type];

	MYSQLND_INC_CONN_STATISTIC(NULL, stat);

	_mysqlnd_send_close(conn TSRMLS_CC);

	conn->m->free_reference(conn TSRMLS_CC);

	return ret;
}
/* }}} */


/* {{{ _mysqlnd_num_fields */
unsigned int _mysqlnd_num_fields(const MYSQLND_RES * const res)
{
	return res->field_count;
}
/* }}} */


/* {{{ _mysqlnd_num_fields */
mynd_ulonglong _mysqlnd_num_rows(const MYSQLND_RES * const res)
{
	/* Be compatible with libmysql. We count row_count, but will return 0 */
	return res->data? res->data->row_count:0;
}
/* }}} */


/* {{{ mysqlnd_field_count */
static
unsigned int _mysqlnd_field_count(const MYSQLND * const conn)
{
	return conn->field_count;
}
/* }}} */


/* {{{ mysqlnd_fetch_field */
static MYSQLND_FIELD *
_mysqlnd_fetch_field(MYSQLND_RES * const result)
{
	if (!result->meta || result->meta->current_field >= result->field_count)
		return NULL;
	return &result->meta->fields[result->meta->current_field++];
}
/* }}} */


/* {{{ _mysqlnd_fetch_field_direct */
static MYSQLND_FIELD *
_mysqlnd_fetch_field_direct(const MYSQLND_RES * const result, MYSQLND_FIELD_OFFSET fieldnr)
{
	return result->meta? &result->meta->fields[fieldnr]:NULL;
}
/* }}} */


/* {{{ mysqlnd_field_seek */
static MYSQLND_FIELD_OFFSET
_mysqlnd_field_seek(MYSQLND_RES * const result, MYSQLND_FIELD_OFFSET field_offset)
{
	MYSQLND_FIELD_OFFSET return_value = 0;
	if (result->meta) {
		return_value = result->meta->current_field;
		result->meta->current_field = field_offset;
	}
	return return_value;
}
/* }}} */


/* {{{ _mysqlnd_field_tell */
static MYSQLND_FIELD_OFFSET
_mysqlnd_field_tell(const MYSQLND_RES * const result)
{
	return result->meta? result->meta->current_field:0;
}
/* }}} */




/* {{{ _mysqlnd_insert_id */
static
mynd_ulonglong _mysqlnd_insert_id(const MYSQLND * const conn)
{
	return conn->upsert_status.last_insert_id;
}
/* }}} */


/* {{{ _mysqlnd_affected_rows */
static
mynd_ulonglong _mysqlnd_affected_rows(const MYSQLND * const conn)
{
	return conn->upsert_status.affected_rows;
}
/* }}} */


/* {{{ _mysqlnd_warning_count */
static
unsigned int _mysqlnd_warning_count(const MYSQLND * const conn)
{
	return conn->upsert_status.warning_count;
}
/* }}} */


/* {{{ _mysqlnd_info */
static
const char *_mysqlnd_info(const MYSQLND * const conn)
{
	return conn->last_message;
}
/* }}} */



/* {{{ _mysqlnd_client_info */
const char *_mysqlnd_get_client_info()
{
	return MYSQLND_VERSION;
}
/* }}} */


/* {{{ _mysqlnd_get_client_version */
unsigned int _mysqlnd_get_client_version()
{
	return MYSQLND_VERSION_ID;
}
/* }}} */


/* {{{ _mysqlnd_get_server_info */
static
const char * _mysqlnd_get_server_info(const MYSQLND * const conn)
{
	return conn->server_version;
}
/* }}} */



/* {{{ _mysqlnd_get_host_info */
static
const char * _mysqlnd_get_host_info(const MYSQLND * const conn)
{
	return conn->host_info;
}
/* }}} */


/* {{{ _mysqlnd_get_proto_info */
static
unsigned int _mysqlnd_get_proto_info(const MYSQLND *const conn)
{
	return conn->protocol_version;
}
/* }}} */


/* {{{ _mysqlnd_thread_id */
static
mynd_ulonglong _mysqlnd_thread_id(const MYSQLND * const conn)
{
	return conn->thread_id;
}
/* }}} */


/* {{{ _mysqlnd_get_server_version */
static
unsigned long _mysqlnd_get_server_version(const MYSQLND * const conn)
{
	long major, minor, patch;
	char *p = conn->server_version;

	if (!conn->server_version) {
		return 0;
	}

	major = strtol(p, &p, 10);
	p += 1; /* consume the dot */
	minor = strtol(p, &p, 10);
	p += 1; /* consume the dot */
	patch = strtol(p, &p, 10);
	
	return (unsigned long)(major * 10000L + (unsigned long)(minor * 100L + patch));
}
/* }}} */


/* {{{ _mysqlnd_more_results */
static
zend_bool _mysqlnd_more_results(const MYSQLND * const conn)
{
	/* (conn->state == CONN_NEXT_RESULT_PENDING) too */
	return conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS? TRUE:FALSE;
}
/* }}} */


/* {{{ _mysqlnd_next_result */
static enum_func_status
_mysqlnd_next_result(MYSQLND * const conn TSRMLS_DC)
{
	enum_func_status ret;

	if (conn->state != CONN_NEXT_RESULT_PENDING) {
		return FAIL;
	}

	SET_EMPTY_ERROR(conn->error_info);
	conn->upsert_status.affected_rows= ~(mynd_ulonglong) 0;
	/*
	  We are sure that there is a result set, since conn->state is set accordingly
	  in mysqlnd_store_result() or mysqlnd_fetch_row_unbuffered()
	*/
	if (FAIL == (ret = mysqlnd_query_read_result_set_header(conn, NULL TSRMLS_CC))) {
#ifndef MYSQLND_SILENT
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Serious error");
#endif
		conn->state = CONN_QUIT_SENT;
	}

	return ret;
}
/* }}} */


/* {{{ mysqlnd_field_type_name */
const char *mysqlnd_field_type_name(enum mysqlnd_field_types field_type)
{
	switch(field_type) {
		case FIELD_TYPE_STRING:
		case FIELD_TYPE_VAR_STRING:
			return "string";
		case FIELD_TYPE_TINY:
		case FIELD_TYPE_SHORT:
		case FIELD_TYPE_LONG:
		case FIELD_TYPE_LONGLONG:
		case FIELD_TYPE_INT24:
			return "int";
		case FIELD_TYPE_FLOAT:
		case FIELD_TYPE_DOUBLE:
		case FIELD_TYPE_DECIMAL:
		case FIELD_TYPE_NEWDECIMAL:
			return "real";
		case FIELD_TYPE_TIMESTAMP:
			return "timestamp";
		case FIELD_TYPE_YEAR:
			return "year";
		case FIELD_TYPE_DATE:
		case FIELD_TYPE_NEWDATE:
			return "date";
		case FIELD_TYPE_TIME:
			return "time";
		case FIELD_TYPE_SET:
			return "set";
		case FIELD_TYPE_ENUM:
			return "enum";
		case FIELD_TYPE_GEOMETRY:
			return "geometry";
		case FIELD_TYPE_DATETIME:
			return "datetime";
		case FIELD_TYPE_TINY_BLOB:
		case FIELD_TYPE_MEDIUM_BLOB:
		case FIELD_TYPE_LONG_BLOB:
		case FIELD_TYPE_BLOB:
			return "blob";
		case FIELD_TYPE_NULL:
			return "null";
		case FIELD_TYPE_BIT:
			return "bit";
		default:
			return "unknown";
	}
}
/* }}} */


/* {{{ _mysqlnd_change_user */
static enum_func_status
_mysqlnd_change_user(MYSQLND * const conn, const char *user, const char *passwd, const char *db TSRMLS_DC)
{
	/*
	  User could be max 16 * 3 (utf8), pass is 20 usually, db is up to 64*3
	  Stack space is not that expensive, so use a bit more to be protected against
	  stack overrungs.
	*/
	size_t user_len;
	enum_func_status ret;
	php_mysql_packet_chg_user_resp chg_user_resp;
	char buffer[MYSQLND_MAX_ALLOWED_USER_LEN + 1 + SCRAMBLE_LENGTH + MYSQLND_MAX_ALLOWED_DB_LEN + 1];
	char *p = buffer;

	if (!user) {
		user = "";
	}
	if (!passwd) {
		passwd = "";
	}
	if (!db) {
		db = "";
	}

	/* 1. user ASCIIZ */
	user_len = strlen(user);
	memcpy(p, user, MIN(user_len, MYSQLND_MAX_ALLOWED_DB_LEN));
	p += user_len;
	*p++ = '\0';

	/* 2. password SCRAMBLE_LENGTH followed by the scramble or \0 */
	if (passwd[0]) {
		*p++ = SCRAMBLE_LENGTH;
		php_mysqlnd_scramble((unsigned char *)p, conn->scramble, (unsigned char *)passwd);
		p += SCRAMBLE_LENGTH;
	} else {
		*p++ = '\0';
	}

	/* 3. db ASCIIZ */
	if (db[0]) {
		size_t db_len = strlen(db);
		memcpy(p, db, MIN(db_len, MYSQLND_MAX_ALLOWED_DB_LEN));
		p += db_len;
	}
	*p++ = '\0';

	if (PASS != mysqlnd_simple_command(conn, COM_CHANGE_USER, buffer, p - buffer,
									   PROT_LAST /* we will handle the OK packet*/,
									   FALSE TSRMLS_CC)) {
		return FAIL;
	}

	PACKET_INIT_ALLOCA(chg_user_resp, PROT_CHG_USER_PACKET);
	ret = PACKET_READ_ALLOCA(chg_user_resp, conn);
	conn->error_info = chg_user_resp.error_info;
	PACKET_FREE_ALLOCA(chg_user_resp);

	if (conn->error_info.error_no) {
		ret = FAIL;
		/*
		  COM_CHANGE_USER is broken in 5.1. At least in 5.1.15 and 5.1.14, 5.1.11 is immune.
		  bug#25371 mysql_change_user() triggers "packets out of sync"
		  When it gets fixed, there should be one more check here
		*/
		if (mysqlnd_get_server_version(conn) > 50113L) {
			php_mysql_packet_ok redundant_error_packet;
			PACKET_INIT_ALLOCA(redundant_error_packet, PROT_OK_PACKET);
			PACKET_READ_ALLOCA(redundant_error_packet, conn);
			PACKET_FREE_ALLOCA(redundant_error_packet);
		}
	}

	/*
	  Here we should close all statements. Unbuffered queries should not be a
	  problem as we won't allow sending COM_CHANGE_USER.
	*/

	return ret;
}
/* }}} */


/* {{{ _mysqlnd_set_client_option */
static enum_func_status
_mysqlnd_set_client_option(MYSQLND * const conn, enum mysqlnd_option option, const char * const value)
{
	switch (option) {
		case MYSQLND_OPT_NUMERIC_AND_DATETIME_AS_UNICODE:
			conn->options.numeric_and_datetime_as_unicode = *(uint*) value;
			break;
#ifdef MYSQLND_STRING_TO_INT_CONVERSION
		case MYSQLND_OPT_INT_AND_YEAR_AS_INT:
			conn->options.int_and_year_as_int = *(uint*) value;
			break;
#endif
		case MYSQL_OPT_CONNECT_TIMEOUT:
			conn->options.timeout_connect = *(uint*) value;
			break;
		case MYSQL_OPT_READ_TIMEOUT:
			conn->options.timeout_read = *(uint*) value;
			break;
		case MYSQL_OPT_WRITE_TIMEOUT:
			conn->options.timeout_write = *(uint*) value;
			break;
		case MYSQL_OPT_LOCAL_INFILE:
			if (!value || (*(uint*) value) ? 1 : 0) {
				conn->options.flags |= CLIENT_LOCAL_FILES;
			} else {
				conn->options.flags &= ~CLIENT_LOCAL_FILES;
			}
			break;
		case MYSQL_OPT_COMPRESS:
		case MYSQL_INIT_COMMAND:
		case MYSQL_READ_DEFAULT_FILE:
		case MYSQL_READ_DEFAULT_GROUP:
		case MYSQL_SET_CLIENT_IP:
		case MYSQL_REPORT_DATA_TRUNCATION:
		case MYSQL_OPT_SSL_VERIFY_SERVER_CERT:
			/* currently not supported. Todo!! */
			break;	
		case MYSQL_SET_CHARSET_DIR:
		case MYSQL_SET_CHARSET_NAME:
		case MYSQL_OPT_RECONNECT:
		case MYSQL_OPT_PROTOCOL:
			/* we don't need external character sets, all character sets are
			   compiled in. For compatibility we just ignore this setting.
			   Same for protocol, we don't support old protocol */
		case MYSQL_OPT_USE_REMOTE_CONNECTION:
		case MYSQL_OPT_USE_EMBEDDED_CONNECTION:
		case MYSQL_OPT_GUESS_CONNECTION:
			/* todo: throw an error, we don't support embedded */
			break;

		case MYSQL_OPT_NAMED_PIPE:
		case MYSQL_SHARED_MEMORY_BASE_NAME:
		case MYSQL_OPT_USE_RESULT:
		case MYSQL_SECURE_AUTH:
			/* not sure, todo ? */
		default:
			return FAIL;
	}
	return PASS;
}
/* }}} */


/* {{{ _mysqlnd_fetch_into */
static
void _mysqlnd_fetch_into(MYSQLND_RES *result, unsigned int flags, zval *return_value TSRMLS_DC ZEND_FILE_LINE_DC)
{
	zend_bool fetched_anything;

	if (!result->m.fetch_row) {
		RETURN_NULL();
	}
	/*
	  Hint Zend how many elements we will have in the hash. Thus it won't
	  extend and rehash the hash constantly.
	*/
	mysqlnd_array_init(return_value, mysqlnd_num_fields(result) * 2);
	if (FAIL == result->m.fetch_row(result, (void *)return_value, flags, &fetched_anything TSRMLS_CC)) {
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error while reading a row");
		RETURN_FALSE;
	} else if (fetched_anything == FALSE) {
		zval_dtor(return_value);
		RETURN_NULL();
	}
	/*
	  return_value is IS_NULL for no more data and an array for data. Thus it's ok
	  to return here.
	*/
}
/* }}} */


/* {{{ _mysqlnd_fetch_all */
static
void _mysqlnd_fetch_all(MYSQLND_RES *result, unsigned int flags, zval *return_value TSRMLS_DC ZEND_FILE_LINE_DC)
{
	zval	*row;
	ulong	i=0;

	/* mysqlnd_fetch_all works with buffered resultsets only */
	if (result->conn || !result->data ||
		!result->data->row_count || !result->data->data_cursor ||
		result->data->data_cursor >= result->data->data + result->data->row_count) {
		RETURN_NULL();
	}	

	mysqlnd_array_init(return_value, result->data->row_count);

	while (result->data->data_cursor &&
		   (result->data->data_cursor - result->data->data) < result->data->row_count)
	{
		MAKE_STD_ZVAL(row);
		mysqlnd_fetch_into(result, flags, row);
		add_index_zval(return_value, i++, row);
	}
}
/* }}} */


/* {{{ _mysqlnd_get_reference */
static
MYSQLND * _mysqlnd_get_reference(MYSQLND * const conn)
{
	++conn->refcount;
	return conn;
}
/* }}} */


/* {{{ _mysqlnd_free_reference */
static
void _mysqlnd_free_reference(MYSQLND * const conn TSRMLS_DC)
{
	if (!(--conn->refcount)) {
		/*
		  No multithreading issues as we don't share the connection :)
		  This will free the object too, of course because references has
		  reached zero.
		*/
		_mysqlnd_send_close(conn TSRMLS_CC);
		conn->m->dtor(conn TSRMLS_CC);
	}
}
/* }}} */


/* {{{ mysqlnd_result_metadata_clone_metadata */
static
MYSQLND_RES_METADATA * _mysqlnd_clone_metadata(const MYSQLND_RES_METADATA * const meta, zend_bool persistent)
{
	unsigned int i;
	/* +1 is to have empty marker at the end */
	MYSQLND_RES_METADATA *new_meta = pemalloc(sizeof(MYSQLND_RES_METADATA), persistent);
	MYSQLND_FIELD *new_fields = pecalloc(meta->field_count + 1, sizeof(MYSQLND_FIELD), persistent);
	MYSQLND_FIELD *original_fields = meta->fields;
	size_t len = meta->field_count * sizeof(struct mysqlnd_field_hash_key);

	new_meta->zend_hash_keys = pemalloc(len, persistent);
	memcpy(new_meta->zend_hash_keys, meta->zend_hash_keys, len);

	/*
	  This will copy also the strings and the root, which we will have
	  to adjust in the loop
	*/
	memcpy(new_fields, original_fields, (meta->field_count) * sizeof(MYSQLND_FIELD));
	for (i = 0; i < meta->field_count; i++) {
		/* First copy the root, then field by field adjust the pointers */
		new_fields[i].root = pemalloc(original_fields[i].root_len, persistent);
		memcpy(new_fields[i].root, original_fields[i].root, new_fields[i].root_len);

		new_fields[i].name		= new_fields[i].root + (original_fields[i].name - original_fields[i].root);
		new_fields[i].org_name	= new_fields[i].root + (original_fields[i].org_name - original_fields[i].root);

		new_fields[i].table		= new_fields[i].root + (original_fields[i].table - original_fields[i].root);
		new_fields[i].org_table	= new_fields[i].root + (original_fields[i].org_table - original_fields[i].root);

		new_fields[i].db		= new_fields[i].root + (original_fields[i].db - original_fields[i].root);
		new_fields[i].catalog	= new_fields[i].root + (original_fields[i].catalog - original_fields[i].root);
		/* def is not on the root, if allocated at all */
		if (original_fields[i].def) {
			new_fields[i].def	= pemalloc(original_fields[i].def_length + 1, persistent);
			/* copy the trailing \0 too */
			memcpy(new_fields[i].def, original_fields[i].def, original_fields[i].def_length + 1);
		}
#if PHP_MAJOR_VERSION >= 6
		if (new_meta->zend_hash_keys[i].ustr.u) {
			new_meta->zend_hash_keys[i].ustr.u =
					eustrndup(new_meta->zend_hash_keys[i].ustr.u, new_meta->zend_hash_keys[i].ulen);
		}
#endif 
	}
	new_meta->current_field = 0;
	new_meta->field_count = meta->field_count;

	new_meta->fields = new_fields;

	return new_meta;
}
/* }}} */


MYSQLND_STMT * _mysqlnd_stmt_init(MYSQLND * const conn);


static
struct st_mysqlnd_connection_methods mysqlnd_connection_methods = {
	_mysqlnd_real_escape_string,
	_mysqlnd_set_charset,
	_mysqlnd_query,
	_mysqlnd_use_result,
	_mysqlnd_store_result,
	_mysqlnd_next_result,
	_mysqlnd_more_results,

	_mysqlnd_stmt_init,

	_mysqlnd_shutdown,
	_mysqlnd_refresh,

	_mysqlnd_ping,
	_mysqlnd_kill,
	_mysqlnd_select_db,
	_mysqlnd_dump_debug_info,
	_mysqlnd_change_user,

	_mysqlnd_errno,
	_mysqlnd_error,
	_mysqlnd_sqlstate,
	_mysqlnd_thread_id,
	_mysqlnd_get_connection_stats,

	_mysqlnd_get_server_version,
	_mysqlnd_get_server_info,
	_mysqlnd_stat,
	_mysqlnd_get_host_info,
	_mysqlnd_get_proto_info,
	_mysqlnd_info,

	_mysqlnd_insert_id,
	_mysqlnd_affected_rows,
	_mysqlnd_warning_count,
	_mysqlnd_field_count,

	_mysqlnd_set_server_option,
	_mysqlnd_set_client_option,
	mysqlnd_conn_free_contents,
	_mysqlnd_close,
	_mysqlnd_conn_dtor,

	_mysqlnd_get_reference,
	_mysqlnd_free_reference
};


/* {{{ mysqlnd_init */
PHPAPI MYSQLND *mysqlnd_init(zend_bool persistent)
{
	MYSQLND *ret = pecalloc(1, sizeof(MYSQLND), persistent);

	ret->upsert_status.affected_rows = (mynd_ulonglong) ~0;
	ret->persistent = persistent;

	ret->m = & mysqlnd_connection_methods;
	ret->m->get_reference(ret);

	return ret;
}
/* }}} */


/* {{{ mysqlnd_result_init */
MYSQLND_RES *mysqlnd_result_init(unsigned int field_count, MYSQLND_ZVAL_PCACHE *cache)
{
	MYSQLND_RES *ret = ecalloc(1, sizeof(MYSQLND_RES));

	ret->field_count	= field_count;
	ret->zval_cache		= cache;
	ret->m.free_result	= _mysqlnd_free_result;
	ret->m.clone_metadata = _mysqlnd_clone_metadata;
	ret->m.free_result_buffers = mysqlnd_internal_free_result_buffers;
	ret->m.seek_data	= _mysqlnd_data_seek;
	ret->m.skip_result	= _mysqlnd_unbuffered_skip_result;
	ret->m.num_rows		= _mysqlnd_num_rows;
	ret->m.num_fields	= _mysqlnd_num_fields;
	ret->m.fetch_into	= _mysqlnd_fetch_into;
	ret->m.fetch_all	= _mysqlnd_fetch_all;
	ret->m.seek_field	= _mysqlnd_field_seek;
	ret->m.field_tell	= _mysqlnd_field_tell;
	ret->m.fetch_field	= _mysqlnd_fetch_field;
	ret->m.fetch_field_direct = _mysqlnd_fetch_field_direct;
	ret->m.fetch_row_normal_buffered = mysqlnd_fetch_row_buffered;
	ret->m.fetch_row_normal_unbuffered = mysqlnd_fetch_row_unbuffered;

	return ret;
}
/* }}} */

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
