// -*-Mode: C++;-*- 

/*
 * $Id: my_sql.h,v 2.20 1999/11/09 02:28:43 ral Exp $
 * Include file for the SQL-Wrapper class
 */
#ifndef _MYSQL_H
#define _MYSQL_H

#include "PatternPair.h"
#include "tabledescr.h"
#include "stringlist.h"
#include "sendmessage.h"
#include "change_type.h"
#include "my_sql_types.h"

//! Sql Wrapper class
/*!
 * This is the wrapper class for the mysql-c-lib
 * 
 * It contains the most routins to connect and deal with a sql-server
 * 
 * @author Rajko Albrecht
 * @version $Id: my_sql.h,v 2.20 1999/11/09 02:28:43 ral Exp $
 */

struct st_mysql;

typedef st_mysql MYSQL;

class CMySql {

protected:
    MYSQL*Sql_Connection;
    int nothing_else;
    CSendMessage*message_call_back;
    void do_message(const char*text)const{if (message_call_back) message_call_back->append_message(text);}
    void do_message(const std::string&text)const{if (message_call_back) message_call_back->append_message(text);}
    char*do_escape(const char*what);
    int do_refresh(int What, const char*ok_msg);
    int Server_MasterVersion;
    int Server_MinorVersion;
    int Server_PatchLevel;

    static const char* MessageArray[];

    static const my_sql_type f_arr[];

    static int MessageCount;
    std::string connect_info;
    int build_command(std::string&target,CColumnDescriptor&desc)const;
    static int build_key_command(std::string&target,const KeyList*keys,bool add = false);
    std::string m_Host;std::string m_Pass;std::string m_User;std::string m_Port;

public:
    //! Select Database
    /*!
     * Selects a database of server to use, mostly used internal
     */
    int connect_db(const char*db)const;
    //! Reload grant table
    int refresh_grants();
    //! flush the cache of sql server
    int refresh_cache();
    //! reload tables
    int refresh_tables();
    //! refreshs the log of sql server
    int refresh_log();
    //! shut down the sql server
    int shutdown();
    //! convert the password
    void scramble_password(std::string&pass)const;
    //! the count of field descriptors
    static int FieldDescriptCounts() ;
    //! get a specific field descriptor
    static const char* FieldDescript(int i);
    //! get specific type name
    static const char* TypeDescript(int i);
    //! is the given db name a system database?
    static bool is_system_db(const char*name);

    static const my_sql_type&sql_type(int i);

    //! Constructor
    /*!
     * Constructor - constructs an empty SQL-class
     *
     * @author Rajko Albrecht
     */
    CMySql();
    CMySql(const CMySql&old);
    //! Constructor
    /*!
     * Constructs an SQL-class and connects to the server with 
     * the given parameters.
     * 
     * \sa sql_connect
     * \param host contains the host to connect to, if empty, localhost is used
     * \param user the user - if empty, current loginname is used
     * \param pass - password of the user at host
     * \param port - the port the class should use. If connect to the localhost
     * via unix-socket, unused
     */
    CMySql(const char*host,const char*user,const char*pass,const char*port);
    /**
     * Destructor - disconnects and destruct the object
     */
    virtual ~CMySql();
    //! Connect to sql server
    /*!
      Connects to a mysql-server with the given parameter
      \param host contains the host to connect to, if empty, localhost is used
      \param user the user - if empty, current loginname is used
      \param pass password of the user at host
      \param port the port the class should use. If connect to the localhost
      via unix-socket, unused
      \sa success_connect, sql_disconnect
    */
    void sql_connect(const char*host,const char*user,const char*pass,const char*port);
    //! Disconnect
    /*!
      Disconnects from a server if connected
      \sa sql_connect, success_connect
    */
    void sql_disconnect();
    /**
     * Retrieves infos about the connected Sql-Server.
     *
     * \return A string, which contains the info
     */
    const std::string&get_info();
    //! Serverstatus
    /*!
     Returns the current status of the server (threads, open tables etc.)
     in a zero-terminated string
    */
    char*get_status();
    //! Get last error information
    /*!
      returns the last error statement. Most of them will printed out via the
      messagehandler.
     \sa set_message_call_back
    */
    char*get_last_error()const;
    //! connection status
    /*!
     returns the connection status of the class
     \return 1 if connected, 0 if not connected
    */
    int success_connect()const{return (Sql_Connection != 0);}
    /**
     * returns a list of the threads of the server. Means, which user
     * are connected, which ids etc.
     *
     * @see #kill_thread
     */
    bool get_threads(stringlist&target)const;
    //stringlist*get_threads();
    /**
     * kills the server thread with the id "pid". 
     *
     * If you kill your own, you've get an error-message!
     *
     * @param pid the id of the thread
     * @return 1 if successfull else 0
     * @see #get_threads
     */
    int kill_thread(unsigned long pid);
    /**
     * retrieves a list of databases in the server
     *
     * @param pattern a pattern to search for
     * @return a pointer to a list or 0, if no databases were found
     */
    bool get_db_list(const char*pattern,stringlist&target)const;
    //! Tablelist
    /**
      returns a list of tables in the given database
      @param pattern table matching pattern (% for all)
      @param db database to use
      @param target a stringlist, which will filled with the table information
      @return true if success else error
    */
    bool get_table_list(const char*pattern,const char*db,stringlist&target);
   //! Columnslist
    /*!
      returns a list of columns in a table
      \param db Database to use
      \param table table to use
      \return Pointer to a ColumnList or 0 if failure
    */
    bool get_table_fields(const char*db,const char*table,ColumnList&target)const;
    bool get_table_keys(const char*db, const char*table, KeyList&target)const;
    /**
     * Creates a new database
     * @param db name of database
     * @return 0 if success, else errorcode
    */
    int create_db(const char*db);
    //!Create a table
    /**
      Creates a new table
      @param db database to use
      @param name name of the new table
      @param columns a pointer to a list of CColumnDescriptor
      @param keys a pointer to a list of CKeyDescriptor, may be NULL
      @return true if success, false if failure
    */
    bool create_table(const char*db,const char*name,ColumnList*columns,KeyList*keys = 0);
    /**
     * retrieves a list of rows.
     * 
     * This function retrieves a list of rows from a table. The entrys are justified
     * to fit into the largest returned value of a column.
     *
     * @return 0 if success, -1 if not success
     * @param db      the name of the database to use
     * @param table   the table to use
     * @param sort_by sort by which value
     * @param which   which of the tablefields should retrieved
     * @param offset  if 0 start by first+offset element with search
     * @param count   if 0 show all found, if > 0, do not show more then count elements
     * @param Pattern a list with entrys to search for. Every pattern the wildcard is %, if %% is supplied, 
     *                the function rebuild it to \% and search for a percent-sign.
     */
    int fetch_rows(const char*db,const char*table,
		   const char*sort_by,stringlist*which,
		   unsigned int offset = 0,unsigned int count = 0,
		   PatternList*Pattern = 0,stringlist*sizes=0);
    /**
     * inserts a new row into table "table" in database db.
     *
     * @param db the database, if empty, current is used
     * @param table the table where to insert, can not be blank
     * @param Pattern a list of columnname-value pairs to insert into the table
     * @return 0 if success, -1 if error
     */
    int insert_row(const char*db,const char*table, PatternList*Pattern);
    /**
     * modifies a row
     *
     * @param db the database to use, if 0, current is used
     * @param table the table to work on, can not be 0
     * @param to_change list of values to modifiy. CPatternPair->entry the column to change, ->pattern the new value
     * @param identifiers a list of identifiers which rows are to change - they will with AND selected 
     *                    (means " where PairA->entry=PairA->pattern AND PairB->entry=PairB->pattern" )
     */
    int update_row(const char*db,const char*table,
		   PatternList*to_change,PatternList*identifiers);
    /**
     * doing a "DELETE" statement in a table. 
     *
     * @param db the database, if 0, current is used
     * @param table the table to use, cannot be null
     * @param identifiers list of identifiers to select the row(s) to delete. If 0, all rows in table will bedeleted
     *                    @see update_row
     */
    int delete_row(const char*db,const char*table,
		   PatternList*identifiers=0);
    //! Deleting table
    /*! 
      Deletes a table from a database
      \param db Database to use
      \param table Table to delete
      \return true if success false if error
      \sa create_table
    */
    bool delete_table(const char*db, const char*table);
    //! Renaming a table
    /*!
     * Renames a table in a given database
     *
     * \param db database to use
     * \param table table to rename
     * \param new_name the new name of the table
     * \return true if success else false
     */
    bool rename_table(const char*db,const char*table,const char*new_name)const;

    //! Alter tables
    /*!
     * Alter a table in a given database
     * \param db database to use
     * \param table table to alter
     * \param n_columns the columns of the table
     * \param n_keys the keys of the table
     * \param columns_change protocoll of changed columns
     * \param key_change protocoll of key_change
     * \return true if it was success, false if failure
     */
    bool alter_table(const char*db, const char*table,ColumnList&n_columns,const KeyList&n_keys,
		     change_list&columns_change,KeyList&key_orig)const;
    /**
     * Setup a callback handler where the class can store it
     * messages. 
     *
     * @param callback a Class based on CSendMessage or NULL if no message output wanted
     * @return none
     */
    void set_message_call_back(CSendMessage*callback){message_call_back=callback;}
    
    bool do_statement(const char*statement,stringlist&target,stringlist&sizes,stringlist&headers,
		      bool log_error_only = false);
};

inline CMySql::CMySql(const char*host,const char*user,const char*pass,const char*port)
{
    Sql_Connection = 0L;
    message_call_back = 0;
    sql_connect(host,user,pass,port);
}

#endif

