/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// Configuration.cpp: implementation of the Configuration class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "Engine.h"
#include "Configuration.h"
#include "SQLError.h"
#include "Log.h"

#ifdef STORAGE_ENGINE
#define CONFIG_FILE					"falcon.conf"
extern uint64		falcon_min_record_memory;
extern uint64		falcon_max_record_memory;
extern uint64		 falcon_page_cache_size;
extern int			falcon_log_mask;
#else
#define CONFIG_FILE					"netfraserver.conf"
#endif

static const char* RECORD_MEMORY_UPPER =	"20mb";
static const char* PAGE_CACHE_MEMORY =		"4mb";
static const int64 MIN_PAGE_CACHE =			2097152;
static const int64 MIN_RECORD_MEMORY =		5000000;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Configuration::Configuration(const char *configFile)
{
#ifdef STORAGE_ENGINE
	recordMemoryLower = falcon_min_record_memory;
	recordMemoryUpper = falcon_max_record_memory;
	pageCacheSize = falcon_page_cache_size;
#else
	recordMemoryLower = 0;
	recordMemoryUpper = getMemorySize(RECORD_MEMORY_UPPER);
	pageCacheSize = getMemorySize(PAGE_CACHE_MEMORY);
#endif
	javaInitialAllocation = 0;
	javaSecondaryAllocation = 0;
	maxThreads = 0;
	schedulerEnabled = true;
	gcSchedule = "0,30 * * * * *";
	scavengeSchedule = "15,45 * * * * *";
	checkpointSchedule = "7,37 * * * * *";
	useCount = 1;

	// Handle initialization file

	const char *fileName = (configFile) ? configFile : CONFIG_FILE;

	FILE *file = fopen (fileName, "r");
	
	if (!file)
		{
		if (configFile)
			throw SQLEXCEPTION (RUNTIME_ERROR, "can't open configuration file \"%s\"", configFile);
		return;
		}
		
	char line [1024];

	while (getLine (file, sizeof (line), line))
		{
		const char *p = line;
		
		while (*p == ' ' || *p == '\t')
			++p;
			
		if (*p == '#')
			continue;

		const char *start = p;

		while (*p && *p != ' ' && *p != '\t' && *p != '\n' && *p != '=' && *p != ':' && *p != '\r')
			++p;

		if (p == start)
			continue;

		JString parameter (start, p - start);

		while (*p == ' ' || *p == '\t' || *p == '=' || *p == ':')
			++p;

		start = p;

		while (*p && *p != '#' && *p != '\n' && *p != '\r')
			++p;

		JString value = JString (start, p - start);

		if (parameter.equalsNoCase ("classpath"))
			classpath = value;
		else if (parameter.equalsNoCase ("min_record_memory"))
			recordMemoryLower = getMemorySize (value);
		else if (parameter.equalsNoCase ("max_record_memory"))
			recordMemoryUpper = getMemorySize (value);
		else if (parameter.equalsNoCase ("page_cache_size"))
			pageCacheSize = getMemorySize (value);
		else if (parameter.equalsNoCase ("java_initial_allocation"))
			javaInitialAllocation = getMemorySize (value);
		else if (parameter.equalsNoCase ("java_secondary_allocation"))
			javaSecondaryAllocation = getMemorySize (value);
		else if (parameter.equalsNoCase ("gcSchedule"))
			gcSchedule = value;
		else if (parameter.equalsNoCase ("scavengeSchedule"))
			scavengeSchedule = value;
		else if (parameter.equalsNoCase ("checkpointSchedule"))
			checkpointSchedule = value;
		else if (parameter.equalsNoCase ("max_threads"))
			maxThreads = atoi(value);
		else if (parameter.equalsNoCase ("scrub"))
			Log::scrubWords (value);
		else if (parameter.equalsNoCase ("scheduler"))
			schedulerEnabled = enabled(value);
		else
			throw SQLEXCEPTION (DDL_ERROR, "unknown config parameter \"%s\"", 
								(const char*) parameter);
		}
		
	fclose (file);

	recordMemoryUpper = MAX(recordMemoryUpper, MIN_RECORD_MEMORY);
	pageCacheSize = MAX(pageCacheSize, MIN_PAGE_CACHE);
	
	if (recordMemoryLower && recordMemoryLower < MIN_RECORD_MEMORY/2)
		recordMemoryLower = MIN_RECORD_MEMORY/2;
	else
	if (!recordMemoryLower || recordMemoryLower >= recordMemoryUpper)
		recordMemoryLower = recordMemoryUpper / 2;

#ifdef STORAGE_ENGINE
	falcon_min_record_memory = recordMemoryLower;
	falcon_max_record_memory = recordMemoryUpper;
	falcon_page_cache_size = pageCacheSize;
#endif
}

Configuration::~Configuration()
{

}

bool Configuration::getLine(void *file, int length, char *line)
{
	char *p = line;
	char *end = line + length - 1;
	char buffer [1024];

	for (;;)
		{
		if (!fgets (buffer, sizeof (buffer), (FILE*) file))
			return false;
		const char *q = buffer;
		while (*q == ' ' || *q == '\t')
			++q;
		while (q < end && *q && *q != '\n')
			*p++ = *q++;
		if (p [-1] != '\\')
			{
			*p = 0;
			return true;
			}
		--p;
		}
}

int64 Configuration::getMemorySize(const char *string)
{
	int64 n = 0;

	for (const char *p = string; *p;)
		{
		char c = *p++;
		if (c >= '0' && c <= '9')
			n = n * 10 + c - '0';
		else if (c == 'g' || c == 'G')
			n *= 1024 * 1024 * 1024;
		else if (c == 'm' || c == 'M')
			n *= 1024 * 1024;
		else if (c == 'k' || c == 'K')
			n *= 1024;
		}

	return n;
}

void Configuration::addRef()
{
	++useCount;
}

void Configuration::release()
{
	if (--useCount == 0)
		delete this;
}

bool Configuration::enabled(JString string)
{
	if (string.equalsNoCase ("enabled") ||
		string.equalsNoCase ("yes") ||
		string.equalsNoCase ("on"))
		return true;

	if (string.equalsNoCase ("disabled") ||
		string.equalsNoCase ("no") ||
		string.equalsNoCase ("off"))
		return false;

	throw SQLEXCEPTION (DDL_ERROR, "unknown enable value \"%s\"", (const char*) string);
}
