/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// SerialLog.cpp: implementation of the SerialLog class.
//
//////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdio.h>
#include <memory.h>
#include "Engine.h"
#include "SerialLog.h"
#include "Sync.h"
#include "SQLError.h"
#include "Thread.h"
#include "SerialLogFile.h"
#include "SerialLogControl.h"
#include "SerialLogWindow.h"
#include "SerialLogTransaction.h"
#include "Threads.h"
#include "Thread.h"
#include "Database.h"
#include "Dbb.h"
#include "Scheduler.h"
#include "Bitmap.h"
#include "Log.h"
#include "RecoveryPage.h"
#include "RecoveryObjects.h"
#include "SRLVersion.h"

#ifdef _DEBUG
static char THIS_FILE[]=__FILE__;
#endif

static const int TRACE_PAGE = 0;

static const int windowBuffers = 10;
static bool debug;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

SerialLog::SerialLog(Database *db, JString schedule) : Schedule (schedule)
{
	database = db;
	dbb = database->dbb;
	creationTime = database->creationTime;
	file1 = new SerialLogFile();
	file2 = new SerialLogFile();
	logControl = new SerialLogControl(this);
	active = false;
	workerThread = NULL;
	nextBlockNumber = 0;
	firstWindow = NULL;
	lastWindow = NULL;
	freeWindows = NULL;
	//firstAction = NULL;
	//lastAction = NULL;
	memset(transactions, 0, sizeof(transactions));
	lastFlushBlock = 1;
	preFlushBlock = 1;
	lastReadBlock = 0;
	eventNumber = 0;
	endSrlQueue = NULL;
	writer = NULL;
	finishing = false;
	logicalFlushes = 0;
	physicalFlushes = 0;
	highWaterBlock = 0;
	bufferSpace = NULL;
	recoveryPages = NULL;
	recoveryIndexes = NULL;
	recoverySections = NULL;
	tracePage = TRACE_PAGE;
}

SerialLog::~SerialLog()
{
	delete file1;
	delete file2;
	delete logControl;
	delete recoveryIndexes;
	delete recoverySections;
	SerialLogWindow *window;

	while ( (window = firstWindow) )
		{
		firstWindow = window->next;
		delete window;
		}
		
	while (( window = freeWindows) )
		{
		freeWindows = window->next;
		delete window;
		}

	while (!buffers.isEmpty())
		buffers.pop();
		
	delete [] bufferSpace;
}


void SerialLog::gopherThread(void *arg)
{
	((SerialLog*) arg)->gopherThread();
}

void SerialLog::gopherThread()
{
	Sync deadMan(&syncGopher, "SerialLog::gopherThread");
	deadMan.lock(Exclusive);
	workerThread = Thread::getThread("SerialLog::gopherThread");
	active = true;
	Sync sync (&actions.syncObject, "SerialLog::gopherThread");

	while (!workerThread->shutdownInProgress)
		{
		if (!actions.first)
			{
			if (finishing)
				break;

			workerThread->sleep();
			
			if (workerThread->shutdownInProgress)
				continue;
			}

		sync.lock(Shared);
		SerialLogAction *transaction;

		for (transaction = actions.first; transaction; transaction = transaction->next)
			if (transaction->isRipe())
				break;

		sync.unlock();

		if (transaction)
			{
			transaction->doAction();
			sync.lock(Exclusive);
			actions.remove(transaction);
			
			// If transaction overlaps existing inactive transaction, expand range of existing transaction

			/***			
			for (SerialLogAction *action = inactions.first; action; action = action->next)
				if ((transaction->minBlockNumber >= action->minBlockNumber && transaction->minBlockNumber <= action->maxBlockNumber) ||
					(transaction->maxBlockNumber >= action->minBlockNumber && transaction->maxBlockNumber <= action->maxBlockNumber))
					{
					action->minBlockNumber = MIN(action->minBlockNumber, transaction->maxBlockNumber);
					action->maxBlockNumber = MAX(action->maxBlockNumber, transaction->maxBlockNumber);
					delete transaction;
					transaction = NULL;
					
					break;
					}
			***/
			
			if (transaction)
				inactions.append(transaction);
				
			sync.unlock();
			}
		else
			workerThread->sleep();
		}

	active = false;
	workerThread = NULL;
}

void SerialLog::open(JString fileRoot, bool createFlag)
{
	file1->open(fileRoot + ".fl1", createFlag);
	file2->open(fileRoot + ".fl2", createFlag);

	int sectorSize = file1->sectorSize;
	bufferSpace = new UCHAR[windowBuffers * srlWindowSize + sectorSize];
	UCHAR *space = (UCHAR*) (((IPTR) bufferSpace + sectorSize - 1) / sectorSize * sectorSize);

	for (int n = 0; n < windowBuffers; ++n, space += srlWindowSize)
		buffers.push(space);

	if (createFlag)
		initializeLog(1);
}

void SerialLog::start()
{
	database->threads->start("SerialLog::start", gopherThread, this);
}

void SerialLog::recover()
{
	Log::log("Recovering database %s ...\n", (const char*) dbb->fileName);
	dbb->setRecovering(true);
	recoveryPhase = 0;
	
	// See if either or both files have valid blocks

	SerialLogWindow *window1 = allocWindow(file1, 0);
	SerialLogWindow *window2 = allocWindow(file2, 0);
	SerialLogBlock *block1 = window1->readFirstBlock();
	SerialLogBlock *block2 = window2->readFirstBlock();

	if (!block1)
		{
		window1->deactivateWindow();
		release(window1);
		window1 = NULL;
		}

	if (!block2)
		{
		window2->deactivateWindow();
		release(window2);
		window2 = NULL;
		}

	// Pick a window to start the search for the last block

	SerialLogWindow *recoveryWindow = NULL;
	SerialLogWindow *otherWindow = NULL;

	if (block1)
		{
		if (block2)
			{
			if (block1->blockNumber > block2->blockNumber)
				{
				recoveryWindow = window1;
				otherWindow = window2;
				window1->next = NULL;
				lastWindow = window2->next = window1;
				firstWindow = window1->prior = window2;
				window2->prior = NULL;
				}
			else
				{
				recoveryWindow = window2;
				otherWindow = window1;
				}
			}
		else
			recoveryWindow = window1;
		}
	else if (block2)
		recoveryWindow = window2;
	else
		{
		Log::log("No recovery block found\n");
		initializeLog(1);
		
		return;
		}

	// Look through windows looking for the very last block

	SerialLogBlock *recoveryBlock = recoveryWindow->firstBlock();
	uint64 recoveryBlockNumber = recoveryBlock->blockNumber;
	SerialLogBlock *lastBlock = findLastBlock(recoveryWindow);
	Log::log("first recovery block is " I64FORMAT "\n", 
			(otherWindow) ? otherWindow->firstBlock()->blockNumber : recoveryBlockNumber);
	Log::log("last recovery block is " I64FORMAT "\n", lastBlock->blockNumber);

	if (otherWindow)
		{
		SerialLogBlock *block = findLastBlock(otherWindow);

		if (block && block->blockNumber != (recoveryBlockNumber - 1))
			throw SQLError(LOG_ERROR, "corrupted serial log");
		
		SerialLogWindow *window = findWindow(block->blockNumber);
		window->deactivateWindow();
		}

	// Find read block

	uint64 readBlockNumber = lastBlock->readBlockNumber;
	Log::log("recovery read block is " I64FORMAT "\n", readBlockNumber);
	nextBlockNumber = lastBlock->blockNumber + 1;
	lastWindow->deactivateWindow();
	SerialLogWindow *window = findWindow(readBlockNumber);
	
	if (!window)
		throw SQLError(LOG_ERROR, "can't find recovery block %d\n", (int) readBlockNumber);
		
	window->activateWindow(true);
	SerialLogBlock *block = window->findBlock(readBlockNumber);
	SerialLogControl control(this);
	control.debug = debug;
	control.setWindow(window, block, 0);
	SerialLogRecord *record;
	writeWindow = NULL;
	writeBlock = NULL;
	pass1 = true;
	recoveryPages = new RecoveryObjects(this);
	recoverySections = new RecoveryObjects(this);
	recoveryIndexes = new RecoveryObjects(this);
	recoveryPhase = 1;
	
	// Make a first pass finding records, transactions, etc.

	while ( (record = control.nextRecord()) )
		record->pass1();

	control.debug = false;
	pass1 = false;
	control.setWindow(window, block, 0);

	recoveryPages->reset();
	recoveryIndexes->reset();
	recoverySections->reset();
	recoveryPhase = 2;

	// Next, make a second pass to reallocate any necessary pages

	while ( (record = control.nextRecord()) )
		record->pass2();

	recoveryPages->reset();
	recoveryIndexes->reset();
	recoverySections->reset();

	// Now mark any transactions still pending as rolled back

	control.setWindow(window, block, 0);
	SerialLogAction *transaction;
	recoveryPhase = 3;

	for (transaction = actions.first; transaction; transaction = transaction->next)
		transaction->preRecovery();

	// Make a third pass doing things

	while ( (record = control.nextRecord()) )
		record->redo();
		
	// Get rid of completed transactions

	/***
	for (SerialLogAction **ptr = &firstAction; (transaction = *ptr);)
		if (transaction->completedRecovery())
			delete transaction;
		else
			ptr = &transaction->next;
	***/

	for (SerialLogAction *action, **ptr = &actions.first; (action = *ptr);)
		if (action->completedRecovery())
			{
			actions.remove(action);
			delete action;
			}
		else
			ptr = &action->next;

	control.fini();
	window->deactivateWindow();
	window = findWindow(lastBlock->blockNumber);
	window->activateWindow(true);
	
	if ( (writeBlock = window->nextAvailableBlock(lastBlock)) )
		{
		writeWindow = window;
		initializeWriteBlock(writeBlock);
		}
	else
		{
		SerialLogFile *file = (window->file == file1) ? file2 : file1;
		//writeWindow = allocWindow(file, window->getNextFileOffset());
		writeWindow = allocWindow(file, 0);
		writeBlock = writeWindow->firstBlock();
		window->deactivateWindow();
		initializeWriteBlock(writeBlock);
		writeWindow->firstBlockNumber = writeBlock->blockNumber;
		}

	preFlushBlock = writeBlock->blockNumber;
	delete recoveryPages;
	delete recoverySections;
	delete recoveryIndexes;
	recoveryPages = NULL;
	recoveryIndexes = NULL;
	recoverySections = NULL;
	
	for (window = firstWindow; window; window = window->next)
		if (!(window->inUse == 0 || window == writeWindow))
			ASSERT(false);
		
	dbb->setRecovering(false);
	checkpoint(true);
	sectionUseVector.zap();
	indexUseVector.zap();
	Log::log("Recovery complete\n");
}

void SerialLog::overflowFlush(void)
{
	ASSERT(syncWrite.getExclusiveThread() == Thread::getThread("SerialLog::overflowFlush"));
	int event = ++eventNumber;		
	++logicalFlushes;

	// OK, we're going to do some writing.  Start by locking the serial log

	*writePtr++ = srlEnd | LOW_BYTE_FLAG;
	writeBlock->blockNumber = nextBlockNumber++;
	writeBlock->creationTime = (uint32) creationTime;
	writeBlock->length = writePtr - (UCHAR*) writeBlock;
	lastReadBlock = writeBlock->readBlockNumber = getReadBlock();

	// Keep track of what needs to be written

	SerialLogWindow *flushWindow = writeWindow;
	SerialLogBlock *flushBlock = writeBlock;
	createNewWindow();

	mutex.lock();
	flushWindow->write(flushBlock);
	++physicalFlushes;
	mutex.unlock();		
	
	highWaterBlock = flushBlock->blockNumber;
	ASSERT(writer || !srlQueue);
}

uint64 SerialLog::flush(bool forceNewWindow, uint64 commitBlockNumber, Sync *clientSync)
{
	Sync sync(&syncWrite, "SerialLog::flush");
	Sync *syncPtr = clientSync;
	
	if (!syncPtr)
		{
		sync.lock(Exclusive);
		syncPtr = &sync;
		}

	int event = ++eventNumber;		
	Thread *thread = Thread::getThread("SerialLog::flush");
	++logicalFlushes;
	thread->commitBlockNumber = commitBlockNumber;

	// Add ourselves to the queue to preserve our place in order

	thread->srlQueue = NULL;

	if (endSrlQueue)
		endSrlQueue->srlQueue = thread;
	else
		srlQueue = thread;

	endSrlQueue = thread;
	thread->wakeupType = None;

	// If there's a writer and it's not use, go to sleep
	
	if (writer && writer != thread)
		{
		syncPtr->unlock();

		for (;;)
			{
			thread->sleep();
			
			if (thread->wakeupType != None)
				break;
			}

		syncPtr->lock(Exclusive);

		if (commitBlockNumber <= highWaterBlock)
			{
			if (writer == thread || (!writer && srlQueue))
				wakeupFlushQueue(thread);

			ASSERT(writer || !srlQueue);
			syncPtr->unlock();
			
			return nextBlockNumber;
			}
		}

	// OK, we're going to do some writing.  Start by locking the serial log

	ASSERT(writer == NULL || writer == thread);
	writer = thread;
	*writePtr++ = srlEnd | LOW_BYTE_FLAG;
	writeBlock->blockNumber = nextBlockNumber++;
	writeBlock->creationTime = (uint32) creationTime;
	writeBlock->length = writePtr - (UCHAR*) writeBlock;
	lastReadBlock = writeBlock->readBlockNumber = getReadBlock();

	// Keep track of what needs to be written

	SerialLogWindow *flushWindow = writeWindow;
	SerialLogBlock *flushBlock = writeBlock;

	// Prepare the next write block for use while we're writing

	if (!forceNewWindow && (writeBlock = writeWindow->nextAvailableBlock(writeBlock)))
		{
		writePtr = writeBlock->data;
		writeBlock->blockNumber = nextBlockNumber;
		writeBlock->readBlockNumber = lastReadBlock;
		writeBlock->creationTime = (uint32) creationTime;
		writeBlock->length = writePtr - (UCHAR*) writeBlock;
		writeWindow->currentLength = writePtr - writeWindow->buffer;
		writeWindow->lastBlockNumber = writeBlock->blockNumber;
		}
	else
		createNewWindow();

	// Everything is ready to go.  Release the locks and start writing

	mutex.lock();
	syncPtr->unlock();
	//Log::debug("Flushing log block %d, read block %d\n", (int) flushBlock->blockNumber, (int) flushBlock->readBlockNumber);
	flushWindow->write(flushBlock);
	++physicalFlushes;
	mutex.unlock();
	
	// We're done.  Wake up anyone who got flushed and pick the next writer

	syncPtr->lock(Exclusive);
	highWaterBlock = MAX(highWaterBlock, flushBlock->blockNumber);
	wakeupFlushQueue(thread);
	ASSERT(writer != thread);
	ASSERT(writer || !srlQueue);
	syncPtr->unlock();

	return nextBlockNumber;
}

void SerialLog::createNewWindow(void)
{
	// We need a new window.  Start by purging any unused windows

	while (firstWindow->lastBlockNumber < lastReadBlock && firstWindow->inUse == 0 && firstWindow->useCount == 0)
		release(firstWindow);

	// Then figure out which file to use.  If the other isn't in use, take it

	SerialLogFile *file = (writeWindow->file == file1) ? file2 : file1;
	int64 fileOffset = 0;
	int64 fileLength = file->highWater;
	
	for (SerialLogWindow *window = firstWindow; window; window = window->next)
		if (window->file == file)
			{
			file = writeWindow->file;
			fileOffset = writeWindow->getNextFileOffset();
			break;
			}

	if (fileOffset == 0)
		Log::log("Switching log files (%d used)\n", fileLength);

	writeWindow->deactivateWindow();
	writeWindow = allocWindow(file, fileOffset);
	initializeWriteBlock(writeWindow->firstBlock());
	writeWindow->firstBlockNumber = writeBlock->blockNumber;
}

void SerialLog::shutdown()
{
	finishing = true;
	wakeup();
	Sync sync(&syncGopher, "SerialLog::shutdown");
	sync.lock(Shared);
	dbb->flush();
	checkpoint(false);
	
	if (workerThread)
		workerThread->shutdown();
}

void SerialLog::close()
{

}

void SerialLog::putData(uint32 length, const UCHAR *data)
{
	if ((writePtr + length) < writeWarningTrack)
		{
		memcpy(writePtr, data, length);
		writePtr += length;
		writeBlock->length = writePtr - (UCHAR*) writeBlock;
		writeWindow->currentLength = writePtr - writeWindow->buffer;
		ASSERT(writePtr < writeWindow->warningTrack);

		return;
		}

	// Data didn't fit in block -- find out how much didn't fit, then flush the rest
	// Note: the record code is going to be overwritten by an srlEnd byte.

	int tailLength = writePtr - recordStart;
	UCHAR recordCode = *recordStart;						
	writeBlock->length = recordStart - (UCHAR*) writeBlock;
	writePtr = recordStart;
	//flush(true, 0, NULL);
	overflowFlush();

	while (writePtr + length + tailLength >= writeWarningTrack)
		//flush(true, 0, NULL);
		overflowFlush();

	putVersion();

	// Restore the initial part of the record

	if (tailLength)
		{
		*writePtr++ = recordCode;

		if (tailLength > 1)
			{
			memcpy(writePtr, recordStart + 1, tailLength - 1);
			writePtr += tailLength - 1;
			}
		}

	// And finally finish the copy of data

	memcpy(writePtr, data, length);
	writePtr += length;
	writeBlock->length = writePtr - (UCHAR*) writeBlock;
	recordStart = writeBlock->data;
	ASSERT(writePtr < writeWindow->warningTrack);
}

void SerialLog::startRecord()
{
	ASSERT(!dbb->recovering);

	if (writePtr == writeBlock->data)
		putVersion();

	recordStart = writePtr;
}

void SerialLog::wakeup()
{
	if (workerThread)
		workerThread->wake();
}

void SerialLog::release(SerialLogWindow *window)
{
	ASSERT(window->inUse == 0);
	
	if (window->buffer)
		{
		releaseBuffer(window->buffer);
		window->buffer = NULL;
		}

	if (window->next)
		window->next->prior = window->prior;
	else
		lastWindow = window->prior;

	if (window->prior)
		window->prior->next = window->next;
	else
		firstWindow = window->next;

	window->next = freeWindows;
	freeWindows = window;
}

SerialLogWindow* SerialLog::allocWindow(SerialLogFile *file, int64 origin)
{
	SerialLogWindow *window = freeWindows;

	if (window)
		{
		freeWindows = window->next;
		window->setPosition(file, origin);
		}
	else
		window = new SerialLogWindow(this, file, origin);

	window->next = NULL;

	if ( (window->prior = lastWindow) )
		lastWindow->next = window;
	else
		firstWindow = window;

	lastWindow = window;
	window->activateWindow(false);

	return window;
}

void SerialLog::initializeLog(int64 blockNumber)
{
	nextBlockNumber = blockNumber;
	writeWindow = allocWindow(file1, 0);
	writeWindow->firstBlockNumber = nextBlockNumber;
	writeWindow->lastBlockNumber = nextBlockNumber;
	initializeWriteBlock((SerialLogBlock*) writeWindow->buffer);
}

SerialLogWindow* SerialLog::findWindow(uint64 blockNumber)
{
	for (SerialLogWindow *window = firstWindow; window; window = window->next)
		if (blockNumber >= window->firstBlockNumber && 
			 blockNumber <= window->lastBlockNumber)
			{
			//window->activateWindow(true);
			return window;
			}

	return NULL;
}

SerialLogTransaction* SerialLog::findTransaction(TransId transactionId)
{
	Sync sync (&actions.syncObject, "SerialLog::findTransaction");
	sync.lock(Shared);

	for (SerialLogTransaction *transaction = transactions[transactionId % SLT_HASH_SIZE];
		  transaction; transaction = transaction->collision)
		if (transaction->transactionId == transactionId)
			return transaction;

	return NULL;
}

SerialLogTransaction* SerialLog::getTransaction(TransId transactionId)
{
	ASSERT(transactionId > 0);
	SerialLogTransaction *transaction = findTransaction(transactionId);

	if (transaction)
		return transaction;

	Sync sync (&actions.syncObject, "SerialLog::findTransaction");
	sync.lock(Exclusive);
	
	/***
	if (transactionId == 0 || transactionId == 41)
		printf ("SerialLog::getTransaction id "TXIDFORMAT"\n", transactionId);
	***/
	
	int slot = (int) (transactionId % SLT_HASH_SIZE);

	for (transaction = transactions[transactionId % SLT_HASH_SIZE];
		  transaction; transaction = transaction->collision)
		if (transaction->transactionId == transactionId)
			return transaction;

	transaction = new SerialLogTransaction(this, transactionId);
	transaction->collision = transactions[slot];
	transactions[slot] = transaction;

	/***
	if ( (transaction->prior = lastAction) )
		lastAction->next = transaction;
	else
		firstAction = transaction;

	transaction->next = NULL;
	lastAction = transaction;
	***/
	
	actions.append(transaction);
	
	if (writeWindow)
		transaction->setStart(recordStart, writeBlock, writeWindow);

	return transaction;
}

uint64 SerialLog::getReadBlock()
{
	Sync sync (&actions.syncObject, "SerialLog::getReadBlock");
	sync.lock(Shared);
	uint64 blockNumber = lastFlushBlock;
	SerialLogAction *transaction;
	
	for (transaction = actions.first; transaction; transaction = transaction->next)
		{
		uint64 number = transaction->getBlockNumber();
		
		if (number == 1)
			transaction->getBlockNumber();
			
		if (number > 0) 
			blockNumber = MIN(blockNumber, number);
		}
	
	for (transaction = inactions.first; transaction; transaction = transaction->next)
		{
		uint64 number = transaction->minBlockNumber;
			
		if (number > 0) 
			blockNumber = MIN(blockNumber, number);
		}
	
	return blockNumber; 
}

/***
void SerialLog::actionDelete(SerialLogAction *action)
{
	Sync sync (&syncTransactions, "SerialLog::actionDelete");
	sync.lock(Exclusive);

	if (action->prior)
		action->prior->next = action->next;
	else
		firstAction = action->next;

	if (action->next)
		action->next->prior = action->prior;
	else
		lastAction = action->prior;

}
***/

void SerialLog::transactionDelete(SerialLogTransaction *transaction)
{
	Sync sync (&actions.syncObject, "SerialLog::transactionDelete");
	sync.lock(Exclusive);
	int slot = (int) (transaction->transactionId % SLT_HASH_SIZE);

	for (SerialLogTransaction **ptr = transactions + slot; *ptr; ptr = &(*ptr)->collision)
		if (*ptr == transaction)
			{
			*ptr = transaction->collision;
			break;
			}

}

void SerialLog::execute(Scheduler *scheduler)
{
	try
		{
		checkpoint(false);
		}
	catch (SQLException& exception)
		{
		Log::log("SerialLog checkpoint failed: %s\n", exception.getText());
		}

	getNextEvent();
	scheduler->addEvent (this);
}

void SerialLog::checkpoint(bool force)
{
	if (force || writePtr != writeBlock->data || getReadBlock() > lastReadBlock + 1 || inactions.first)
		{
		dbb->flush();
		dbb->sync();
		logControl->checkpoint.append();
		//flush(false, 0, NULL);				// flush now handled by SRLCheckpoint
		}
}

void SerialLog::initializeWriteBlock(SerialLogBlock *block)
{
	writeBlock = block;
	writePtr = writeBlock->data;
	writeBlock->blockNumber = nextBlockNumber;
	writeBlock->creationTime = (uint32) creationTime;
	writeBlock->length = writePtr - (UCHAR*) writeBlock;
	writeWindow->setLastBlock(writeBlock);
	
	writeWarningTrack = writeWindow->warningTrack;
}


SerialLogBlock* SerialLog::findLastBlock(SerialLogWindow *window)
{
	SerialLogBlock *lastBlock = window->firstBlock();

	for (;;)
		{
		lastBlock = window->findLastBlock(lastBlock);
		SerialLogWindow *newWindow = allocWindow(window->file, window->origin + window->currentLength);

		if (window->next != newWindow)
			{
			lastWindow = newWindow->prior;
			lastWindow->next = NULL;
			newWindow->next = window->next;
			newWindow->prior = window;
			newWindow->next->prior = newWindow;
			window->next = newWindow;
			}

		SerialLogBlock *newBlock = newWindow->readFirstBlock();

		if (!newBlock || newBlock->blockNumber != lastBlock->blockNumber + 1)
			{
			newWindow->deactivateWindow();
			release(newWindow);
			break;
			}

		window->deactivateWindow();
		window = newWindow;
		lastBlock = newBlock;
		}

	return lastBlock;
}

bool SerialLog::isIndexActive(int indexId)
{
	if (!recoveryIndexes)
		return true;
		
	return recoveryIndexes->isObjectActive(indexId);
}


bool SerialLog::isSectionActive(int sectionId)
{
	if (!recoverySections)
		return true;
		
	return recoverySections->isObjectActive(sectionId);
}

uint32 SerialLog::appendLog(IO *shadow, int lastPage)
{
	Sync sync(&syncWrite, "SerialLog::appendLog");
	sync.lock(Exclusive);
	shadow->seek(lastPage);
	uint32 totalLength = 0;
	uint64 endBlock = flush(false, 0, NULL);

	for (uint64 blockNumber = getReadBlock(); blockNumber < endBlock; ++blockNumber)
		{
		SerialLogWindow *window = findWindow(blockNumber);
		
		if (window)
			{
			window->activateWindow(true);
			SerialLogBlock *block = window->findBlock(blockNumber);
			uint32 length = ROUNDUP(block->length, window->sectorSize);
			shadow->write(length, block->data);
			totalLength += length;
			window->deactivateWindow();
			}
		}

	return totalLength;
}

void SerialLog::dropDatabase()
{
	if (file1)
		file1->dropDatabase();

	if (file2)
		file2->dropDatabase();
}

void SerialLog::shutdownNow()
{

}

UCHAR* SerialLog::allocBuffer()
{
	if (!buffers.isEmpty())
		return (UCHAR*) buffers.pop();

	for (SerialLogWindow *window = firstWindow; window; window = window->next)
		if (window->buffer && window->inUse == 0)
			{
			ASSERT(window != writeWindow);
			UCHAR *buffer = window->buffer;
			window->setBuffer(NULL);

			return buffer;
			}

	ASSERT(false);

	return NULL;
}

void SerialLog::releaseBuffer(UCHAR *buffer)
{
	buffers.push(buffer);
}

void SerialLog::copyClone(JString fileRoot, int logOffset, int logLength)
{
	file1->open(fileRoot + ".nl1", true);
	dbb->seek(logOffset);
	uint32 bufferLength = 32768;
	UCHAR *buffer = new UCHAR [bufferLength];
	uint32 position = 0;
	
	for (uint32 length = logLength; length > 0;)
		{
		uint32 len = MIN(length, bufferLength);
		dbb->read(len, buffer);
		file1->write(position, len, (SerialLogBlock*) buffer);
		length -= len;
		position += len;
		}
	
	delete [] buffer;
	file1->close();
	file2->open(fileRoot + ".nl2", true);
	file2->close();
}

bool SerialLog::bumpPageIncarnation(int32 pageNumber, int state)
{
	bool ret = recoveryPages->bumpIncarnation(pageNumber, state, pass1);
	
	if (ret)
		{
		if (pass1)
			dbb->reallocPage(pageNumber);
		else if (pageNumber == tracePage)
			printf("trace page %d\n", tracePage);
		}

	return ret;
}

bool SerialLog::bumpSectionIncarnation(int sectionId, int state)
{
	return recoverySections->bumpIncarnation(sectionId, state, pass1);
}

bool SerialLog::bumpIndexIncarnation(int indexId, int state)
{
	return recoveryIndexes->bumpIncarnation(indexId, state, pass1);
}

void SerialLog::preFlush(void)
{
	preFlushBlock = writeBlock->blockNumber - 1;
	Sync sync(&actions.syncObject, "SerialLog::preFlush");
	sync.lock(Shared);
	
	for (SerialLogAction *action = inactions.first; action; action = action->next)
		action->flushing = true;
}

void SerialLog::pageCacheFlushed()
{
	lastFlushBlock = preFlushBlock;
	Sync sync(&actions.syncObject, "SerialLog::pageCacheFlushed");
	sync.lock(Exclusive);
	
	for (SerialLogAction *action, **ptr = &inactions.first; (action = *ptr);)
		if (action->flushing && action->maxBlockNumber < lastFlushBlock)
			{
			inactions.remove(action);
			delete action;
			}
		else
			ptr = &action->next;
}

int SerialLog::recoverLimboTransactions(void)
{
	int count = 0;
	
	for (SerialLogAction *action = actions.first; action; action = action->next)
		if (action->isTransaction())
			{
			SerialLogTransaction *transaction = (SerialLogTransaction*) action;
			
			if (transaction->state == sltPrepared)
				++count;
			}
	
	if (count)
		Log::log("Warning: Recovery found %d prepared transactions in limbo\n", count);

	return count;
	/***
	SerialLogAction *action;
	int64 blockNumber = 0;
	int64 bn;
		
	for (action = actions.first; action; action = action->next)
		if (action->isTransaction() && (bn = action->getBlockNumber()))
			blockNumber = MIN(blockNumber, bn);
	
	if (blockNumber == 0)
		return;	
		
	SerialLogWindow *window = findWindow(blockNumber);
	SerialLogBlock *block = window->findBlock(blockNumber);
	SerialLogControl control(this);
	control.setWindow(window, block, 0);
	SerialLogRecord *record;

	// Make a first pass finding records, transactions, etc.

	while ( (record = control.nextRecord()) )
		record->pass1();

	window->deactivateWindow();
	***/
}

void SerialLog::putVersion()
{
	*writePtr++ = srlVersion | LOW_BYTE_FLAG;
	*writePtr++ = srlCurrentVersion | LOW_BYTE_FLAG;
}

void SerialLog::wakeupFlushQueue(Thread *ourThread)
{
	ASSERT(syncWrite.getExclusiveThread() == Thread::getThread("SerialLog::wakeupFlushQueue"));
	writer = NULL;
	Thread *thread = srlQueue;
	
	// Start by making sure we're out of the que

	for (Thread *prior = NULL; thread; prior = thread, thread = thread->srlQueue)
		if (thread == ourThread)
			{
			if (prior)
				prior->srlQueue = thread->srlQueue;
			else
				srlQueue = thread->srlQueue;
			
			if (endSrlQueue == thread)
				endSrlQueue = prior;
			
			break;
			}
	
	while ( (thread = srlQueue) )
		{
		srlQueue = thread->srlQueue;
		thread->eventNumber = ++eventNumber;

		if (thread->commitBlockNumber <= highWaterBlock)
			{
			thread->wakeupType = Shared;
			thread->wake();
			}
		else
			{
			writer = thread;
			thread->wakeupType = Exclusive;
			thread->wake();
			break;
			}
		}

	if (!srlQueue)
		endSrlQueue = NULL;
}

void SerialLog::setSectionActive(int id)
{
	if (!recoverySections)
		return;
	
	if (dbb->recovering)
		recoverySections->bumpIncarnation(id, objInUse, false);
	else
		recoverySections->deleteObject(id);
}

void SerialLog::setSectionInactive(int id)
{
	if (!recoverySections)
		recoverySections = new RecoveryObjects(this);
	
	recoverySections->bumpIncarnation(id, objDeleted, true);
}

void SerialLog::setIndexActive(int id)
{
	if (!recoveryIndexes)
		return;
	
	if (dbb->recovering)
		recoveryIndexes->setActive(id);
		//recoveryIndexes->bumpIncarnation(id, false);
	else
		recoveryIndexes->deleteObject(id);
}

void SerialLog::setIndexInactive(int id)
{
	if (!recoveryIndexes)
		recoveryIndexes = new RecoveryObjects(this);
	
	recoveryIndexes->setInactive(id);
}

void SerialLog::extendSectionVector(int newLength)
{
	sectionUseVector.extend(newLength);
}

bool SerialLog::sectionInUse(int sectionId)
{
	return sectionUseVector.get(sectionId) > 0;
}

void SerialLog::extendIndexVector(int newLength)
{
	indexUseVector.extend(newLength);
}

bool SerialLog::indexInUse(int indexId)
{
	return indexUseVector.get(indexId) > 0;
}

int SerialLog::getPageState(int32 pageNumber)
{
	return recoveryPages->getCurrentState(pageNumber);
}

void SerialLog::redoFreePage(int32 pageNumber)
{
	if (pageNumber == tracePage)
		printf("Redoing free of page %d\n", pageNumber);
	
	dbb->redoFreePage(pageNumber);
}

void SerialLog::setPhysicalBlock(TransId transId)
{
	if (transId)
		{
		SerialLogTransaction *transaction = findTransaction(transId);
		
		if (transaction)
			transaction->setPhysicalBlock();
		}
}

void SerialLog::logInfo(void)
{
	Sync sync(&actions.syncObject, "SerialLog::preFlush");
	sync.lock(Shared);
	int count = 0;
	uint64 minBlockNumber = writeBlock->blockNumber;
		
	for (SerialLogAction *action = inactions.first; action; action = action->next)
		{
		++count;
		
		if (action->minBlockNumber < minBlockNumber)
			minBlockNumber = action->minBlockNumber;
		}
	
	int64 delta = writeBlock->blockNumber - minBlockNumber;
	
	if (count || delta)
		Log::log(LogInfo, "Post-commit queue: %d transactions, " I64FORMAT " blocks\n", count, delta);
}

void SerialLog::commitByXid(int xidLength, const UCHAR* xid)
{
	Sync sync(&actions.syncObject, "SerialLog::commitByXid");
	sync.lock(Shared);
		
	for (SerialLogAction *action = inactions.first; action; action = action->next)
		if (action->isTransaction())
			{
			SerialLogTransaction *transaction = (SerialLogTransaction*) action;
			
			if (transaction->isXidEqual(xidLength, xid))
				transaction->commit();
			}
}

void SerialLog::rollbackByXid(int xidLength, const UCHAR* xid)
{
	Sync sync(&actions.syncObject, "SerialLog::rollbackByXid");
	sync.lock(Shared);
		
	for (SerialLogAction *action = inactions.first; action; action = action->next)
		if (action->isTransaction())
			{
			SerialLogTransaction *transaction = (SerialLogTransaction*) action;
			
			if (transaction->isXidEqual(xidLength, xid))
				transaction->rollback();
			}
}
