/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// SerialLog.h: interface for the SerialLog class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_SERIALLOG_H__D2A71E6B_A3B0_41C8_9C73_628DA067F722__INCLUDED_)
#define AFX_SERIALLOG_H__D2A71E6B_A3B0_41C8_9C73_628DA067F722__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "SyncObject.h"
#include "Schedule.h"
#include "Stack.h"
#include "DenseArray.h"
#include "Queue.h"

static const unsigned int altLogFlag	= 0x80000000;
static const int srlSignature			= 123456789;
static const int SLT_HASH_SIZE			= 101;
static const int srlWindowSize			= 1048576;

// States for recovery objects

static const int objUnknown				= 0;
static const int objInUse				= 1;
static const int objDeleted				= 2;

struct SerialLogBlock 
{
	uint64	blockNumber;
	uint64	readBlockNumber;
	uint32	length;
	uint32	creationTime;
	UCHAR	data[1];
};

class SerialLogFile;
class Database;
class Dbb;
class Thread;
class SerialLogControl;
class SerialLogWindow;
class SerialLogTransaction;
class SerialLogAction;
class Bitmap;
class IO;
class RecoveryObjects;
class Sync;

class SerialLog : public Schedule, public SyncObject
{
public:
	SerialLog(Database *db, JString schedule);
	virtual ~SerialLog();

	void			putVersion();
	void			pageCacheFlushed();

	void			releaseBuffer (UCHAR *buffer);
	UCHAR*			allocBuffer();
	void			shutdownNow();
	void			dropDatabase();
	uint32			appendLog (IO *shadow, int lastPage);
	bool			isSectionActive(int sectionId);
	bool			isIndexActive(int indexId);
	//void			actionDelete(SerialLogAction *action);
	SerialLogBlock* findLastBlock (SerialLogWindow *window);
	void			initializeWriteBlock (SerialLogBlock *block);
	void			checkpoint(bool force);
	virtual void	execute(Scheduler * scheduler);
	void			transactionDelete (SerialLogTransaction *transaction);
	uint64			getReadBlock();
	SerialLogTransaction* getTransaction (TransId transactionId);
	SerialLogTransaction* findTransaction (TransId transactionId);
	SerialLogWindow* findWindow (uint64 blockNumber);
	SerialLogWindow* allocWindow(SerialLogFile *file, int64 origin);
	void			initializeLog(int64 blockNumber);
	void			release (SerialLogWindow *window);
	void			wakeup();
	void			startRecord();
	void			putData(uint32 length, const UCHAR *data);
	void			close();
	void			shutdown();
	uint64			flush(bool forceNewWindow, uint64 commitBlockNumber, Sync *syncPtr);
	void			recover();
	void			start();
	void			open(JString fileRoot, bool createFlag);
	void			gopherThread();
	void			copyClone(JString fileRoot, int logOffset, int logLength);
	int				recoverLimboTransactions(void);
	void			preFlush(void);
	void			wakeupFlushQueue(Thread *ourThread);
	
	void			setSectionActive(int id);
	void			setSectionInactive(int id);
	void			setIndexActive(int id);
	void			setIndexInactive(int id);
	void			extendSectionVector(int newLength);
	bool			sectionInUse(int sectionId);
	bool			bumpIndexIncarnation(int indexId, int state);
	bool			bumpSectionIncarnation (int sectionId, int state);
	bool			bumpPageIncarnation (int32 pageNumber, int state);

	int				getPageState(int32 pageNumber);
	//void			reallocPage(int32 pageNumber);
	void			redoFreePage(int32 pageNumber);
	
	void			extendIndexVector(int newLength);
	bool			indexInUse(int indexId);
	
	static void		gopherThread(void *arg);

	SerialLogFile		*file1;
	SerialLogFile		*file2;
	SerialLogWindow		*firstWindow;
	SerialLogWindow		*lastWindow;
	SerialLogWindow		*freeWindows;
	SerialLogWindow		*writeWindow;
	SerialLogTransaction	*transactions[SLT_HASH_SIZE];
	Database			*database;
	RecoveryObjects		*recoveryPages;
	RecoveryObjects		*recoverySections;
	RecoveryObjects		*recoveryIndexes;
	Dbb					*dbb;
	Thread				*workerThread;
	Thread				*srlQueue;
	Thread				*endSrlQueue;
	SerialLogControl	*logControl;
	uint64				nextBlockNumber;
	uint64				highWaterBlock;
	uint64				lastFlushBlock;
	uint64				lastReadBlock;
	uint64				preFlushBlock;
	UCHAR				*recordStart;
	UCHAR				*writePtr;
	UCHAR				*writeWarningTrack;
	SerialLogBlock		*writeBlock;
	SyncObject			syncWrite;
	SyncObject			syncSections;
	SyncObject			syncIndexes;
	SyncObject			syncGopher;
	Stack				buffers;
	UCHAR				*bufferSpace;
	time_t				creationTime;
	bool				active;
	bool				pass1;
	bool				finishing;
	int					logicalFlushes;
	int					physicalFlushes;
	int					recoveryPhase;
	int					eventNumber;
	int32				tracePage;
	DenseArray<int, 200>	sectionUseVector;
	DenseArray<int, 200>	indexUseVector;
	Queue<SerialLogAction>	actions;
	Queue<SerialLogAction>	inactions;
	
private:
	Thread *volatile	writer;
public:
	void overflowFlush(void);
	void createNewWindow(void);
	void setPhysicalBlock(TransId transactionId);
	void logInfo(void);
	void commitByXid(int xidLength, const UCHAR* xid);
	void rollbackByXid(int xidLength, const UCHAR* xid);
};

#endif // !defined(AFX_SERIALLOG_H__D2A71E6B_A3B0_41C8_9C73_628DA067F722__INCLUDED_)
