/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <memory.h>
#include <string.h>
#include <stdio.h>
#include "Engine.h"
#include "StorageTableShare.h"
#include "StorageDatabase.h"
#include "SyncObject.h"
#include "Sync.h"
#include "Sequence.h"
#include "Index.h"
#include "Table.h"
#include "SyncObject.h"
#include "CollationManager.h"
#include "MySQLCollation.h"

//static const char *SCHEMA_NAME			= "mysql";

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

StorageTableShare::StorageTableShare(StorageDatabase *db, const char *tableName, const char *schema, int impureSize)
{
	storageDatabase = db;
	impure = new UCHAR[impureSize];
	initialized = false;
	table = NULL;
	name = JString::upcase(tableName);
	indexes = NULL;
	syncObject = new SyncObject;
	sequence = NULL;
	schemaName = schema;
	tempTable = false;
}

StorageTableShare::~StorageTableShare(void)
{
	delete syncObject;
	delete [] impure;
	delete [] indexes;
}

void StorageTableShare::lock(bool exclusiveLock)
{
	syncObject->lock(NULL, (exclusiveLock) ? Exclusive : Shared);
}

void StorageTableShare::unlock(void)
{
	syncObject->unlock();
}

int StorageTableShare::open(void)
{
	if (!table)
		{
		table = storageDatabase->findTable(name, schemaName);
		sequence = storageDatabase->findSequence(name, schemaName);
		}
	
	if (!table)
		return StorageErrorTableNotFound;
	
	return 0;
}

int StorageTableShare::create(StorageConnection *storageConnection, const char* sql, int64 autoIncrementValue)
{
	if (!(table = storageDatabase->createTable(storageConnection, name, schemaName, sql, autoIncrementValue)))
		return StorageErrorTableExits;
	
	if (autoIncrementValue)
		sequence = storageDatabase->findSequence(name, schemaName);
		
	return 0;
}

int StorageTableShare::deleteTable(StorageConnection *storageConnection)
{
	int res = storageDatabase->deleteTable(storageConnection, name, schemaName);
	
	if (res == 0)
		{
		table = NULL;
		delete [] indexes;
		indexes = NULL;
		}

	return res;
}

void StorageTableShare::cleanupFieldName(const char* name, char* buffer, int bufferLength)
{
	char *q = buffer;
	char *end = buffer + bufferLength - 1;
	const char *p = name;
	
	for (; *p && q < end; ++p)
		*q++ = UPPER(*p);
	
	*q = 0;
}

const char* StorageTableShare::cleanupName(const char* name, char* buffer, int bufferLength, char *schema, int schemaLength)
{
	return cleanupTableName(name, buffer, bufferLength, schema, schemaLength);
}


const char* StorageTableShare::cleanupTableName(const char* name, char* buffer, int bufferLength, char *schema, int schemaLength)
{
	char *q = buffer;
	const char *p = name;
	
	while (*p == '.')
		++p;

	for (; *p; ++p)
		if (*p == '/' || *p == '\\')
			{
			*q = 0;
			strcpy(schema, buffer);
			q = buffer;
			}
		else
			*q++ = UPPER(*p);
	
	*q = 0;
	
	if ( (q = strchr(buffer, '.')) )
		*q = 0;
	
	return buffer;
}

int StorageTableShare::createIndex(StorageConnection *storageConnection, const char* name, const char* sql)
{
	if (!table)
		open();

	return storageDatabase->createIndex(storageConnection, table, name, sql);
}

int StorageTableShare::renameTable(StorageConnection *storageConnection, const char* newName)
{
	char tableName[256];
	char schemaName[256];
	StorageTableShare::cleanupName(newName, tableName, sizeof(tableName), schemaName, sizeof(schemaName));
	int ret = storageDatabase->renameTable(storageConnection, table, tableName, schemaName);
	
	if (ret)
		return ret;
	
	storageDatabase->remove(this);
	name = tableName;
	storageDatabase->add(this);
	
	return ret;
}

StorageIndexDesc* StorageTableShare::getIndex(int indexCount, int indexId, StorageIndexDesc* indexDesc)
{
	if (!indexes)
		{
		indexes = new StorageIndexDesc*[indexCount];
		memset(indexes, 0, indexCount * sizeof(StorageIndexDesc*));
		numberIndexes = indexCount;
		}
	
	if (indexId >= numberIndexes)
		return NULL;
	
	StorageIndexDesc *index = indexes[indexId];
	
	if (index)
		return index;

	indexes[indexId] = index = new StorageIndexDesc;
	*index = *indexDesc;
	
	if (indexDesc->primaryKey)
		index->index = table->primaryKey;
	else
		{
		char indexName[150];
		sprintf(indexName, "%s$%d", (const char*) name, indexId);
		index->index = storageDatabase->findIndex(table, indexName);
		}
	
	return index;
}

StorageIndexDesc* StorageTableShare::getIndex(int indexId)
{
	if (!indexes || indexId >= numberIndexes)
		return NULL;
	
	return indexes[indexId];
}

INT64 StorageTableShare::getSequenceValue(int delta)
{
	if (!sequence)
		return 0;

	return sequence->update(delta, NULL);
}

int StorageTableShare::setSequenceValue(INT64 value)
{
	if (!sequence)
		return StorageErrorNoSequence;
		
	Sync sync(syncObject, "StorageTableShare::setSequenceValue");
	sync.lock(Exclusive);
	INT64 current = sequence->update(0, NULL);
	
	if (value > current)
		sequence->update(value - current, NULL);

	return 0;
}

int StorageTableShare::getIndexId(const char* schemaName, const char* indexName)
{
	if (indexes)
		for (int n = 0; n < numberIndexes; ++n)
			{
			Index *index = indexes[n]->index;
			
			if (strcmp(index->getIndexName(), indexName) == 0 &&
				strcmp(index->getSchemaName(), schemaName) == 0)
				return n;
			}
		
	return -1;
}

int StorageTableShare::haveIndexes(void)
{
	if (indexes == NULL)
		return false;
	
	for (int n = 0; n < numberIndexes; ++n)
		if (indexes[n]== NULL)
			return false;
	
	return true;
}

void StorageTableShare::setTablePath(const char* path, bool tmp)
{
	if (pathName.IsEmpty())
		pathName = path;
	
	tempTable = tmp;
}

void StorageTableShare::registerCollation(const char* collationName, void* arg)
{
	JString name = JString::upcase(collationName);
	Collation *collation = CollationManager::findCollation(name);
	
	if (collation)
		{
		collation->release();
		
		return;
		}
	
	collation = new MySQLCollation(name, arg);
	CollationManager::addCollation(collation);
}
