/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include <memory.h>
#include "Engine.h"
#include "TransactionManager.h"
#include "Transaction.h"
#include "Sync.h"
#include "Interlock.h"
#include "SQLError.h"
#include "Database.h"
#include "Connection.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


TransactionManager::TransactionManager(Database *db)
{
	database = db;
	transactionSequence = 1;
}

TransactionManager::~TransactionManager(void)
{
	for (Transaction *transaction; (transaction = activeTransactions.first);)
		{
		activeTransactions.first = transaction->next;
		transaction->release();
		}
}

TransId TransactionManager::findOldestActive()
{
	TransId oldest = transactionSequence;
	Sync sync (&activeTransactions.syncObject, "TransactionManager::findOldestActive");
	sync.lock (Shared);
	
	for (Transaction *transaction = activeTransactions.first; transaction; transaction = transaction->next)
		if (transaction->isActive())
			{
			oldest = transaction->oldestActive;
			transaction->scavenged = false;
			break;
			}

	sync.unlock();
	
	return oldest;
}

Transaction* TransactionManager::startTransaction(Connection* connection)
{
	Sync sync (&activeTransactions.syncObject, "Database::startTransaction");
	sync.lock (Shared);
	Transaction *transaction;
	
	for (transaction = activeTransactions.first; transaction; transaction = transaction->next)
		if (transaction->state == Available && transaction->dependencies == 0)
			if (COMPARE_EXCHANGE(&transaction->state, Available, Initializing))
				{
				transaction->initialize(connection, transactionSequence++);
				
				return transaction;
				}
				
	transaction = new Transaction (connection, transactionSequence++);
	sync.unlock();
	sync.lock(Exclusive);
	activeTransactions.append(transaction);

	return transaction;	
}

void TransactionManager::dropTable(Table* table, Transaction* transaction)
{
	Sync committedTrans (&committedTransactions.syncObject, "TransactionManager::dropTable");
	committedTrans.lock (Shared);
	
	for (Transaction *trans = committedTransactions.first; trans; trans = trans->next)
		trans->dropTable(table);
	
	committedTrans.unlock();
}

bool TransactionManager::hasUncommittedRecords(Table* table, Transaction* transaction)
{
	Sync syncTrans (&activeTransactions.syncObject, "TransactionManager::hasUncommittedRecords");
	syncTrans.lock (Shared);
	
	for (Transaction *trans = activeTransactions.first; trans; trans = trans->next)
		if (trans != transaction && trans->isActive() && trans->hasUncommittedRecords(table))
			return true;

	return false;
}

void TransactionManager::scavengeRecords(int threshold)
{
	Sync sync (&activeTransactions.syncObject, "TransactionManager::scavengeRecord");
	sync.lock (Shared);
	
	for (bool again = true; again;)
		{
		again = false;
		
		for (Transaction *transaction = activeTransactions.first; transaction; transaction = transaction->next)
			if (transaction != database->systemConnection->transaction && transaction->isActive() &&!transaction->scavenged)
				{
				transaction->addRef();
				sync.unlock();
				transaction->scavengeRecords(threshold);
				transaction->release();
				sync.lock(Shared);
				again = true;
				break;
				}
		}
}

void TransactionManager::commitByXid(int xidLength, const UCHAR* xid)
{
	Sync sync (&activeTransactions.syncObject, "TransactionManager::commitByXid");
	sync.lock (Shared);
	
	for (bool again = true; again;)
		{
		again = false;
		
		for (Transaction *transaction = activeTransactions.first; transaction; transaction = transaction->next)
			if (transaction->state == Limbo && transaction->isXidEqual(xidLength, xid))
				{
				sync.unlock();
				transaction->commit();
				sync.lock(Shared);
				again = true;
				break;
				}
		}
}

void TransactionManager::rollbackByXid(int xidLength, const UCHAR* xid)
{
	Sync sync (&activeTransactions.syncObject, "TransactionManager::rollbackByXid");
	sync.lock (Shared);
	
	for (bool again = true; again;)
		{
		again = false;
		
		for (Transaction *transaction = activeTransactions.first; transaction; transaction = transaction->next)
			if (transaction->state == Limbo && transaction->isXidEqual(xidLength, xid))
				{
				sync.unlock();
				transaction->rollback();
				sync.lock(Shared);
				again = true;
				break;
				}
		}
}
