/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"


// Code ----------------------------------------------------------------------

// CDBSession::CDBSession --------------------------------------------------
//
// @mfunc Constructor for this class
//
// @rdesc NONE
//
CDBSession::CDBSession
    (
    LPUNKNOWN pUnkOuter         //@parm IN | Outer Unkown Pointer
    )
{
    CLEAR_CONSTRUCT( CDBSession );
	
	//  Initialize simple member vars
	m_pUnkOuter			= pUnkOuter ? pUnkOuter : this;

    // Increment global object count.
    OBJECT_CONSTRUCTED();
}


// CDBSession::~CDBSession -------------------------------------------------
//
// @mfunc Destructor for this class
//
// @rdesc NONE
//
CDBSession:: ~CDBSession
    (
    void
    )
{
	// HRESULT hr;

    // Free properties management object
    delete m_pUtilProp;

    // Free contained interfaces
    delete m_pIGetDataSource;
    delete m_pIOpenRowset;
    delete m_pISessionProperties;
	delete m_pIDBCreateCommand;
	delete m_pIDBSchemaRowset;
	//delete m_pIIndexDefinition;

	// 
	m_pCDataSource->Release();
	m_pCDataSource->RemoveSession(m_cNSession);

    // Decrement global object count.
    OBJECT_DESTRUCTED();

    return;
}


// CDBSession::FInit --------------------------------------------------------
//
// @mfunc Initialize the command Object
//
// @rdesc Did the Initialization Succeed
//      @flag  TRUE | Initialization succeeded
//      @flag  FALSE | Initialization failed
//
BOOL CDBSession::FInit
    (
	ULONG		ulSession,		//@parm IN | # of Session
//    char*		szPath,			//@parm IN | Path of Data Dictionary
	CDataSource	*pCDataSource	//@parm IN | session's parent
    )
{
	TRACE( "CDBSession::FInit" );

    LPUNKNOWN   pIUnknown = (LPUNKNOWN) this;

    if (m_pUnkOuter)
        pIUnknown = m_pUnkOuter;

    // just make a copy for ourselves.
    //if (!lstrcpyn( m_szPath, szPath, MAX_PATH ))
    //    return FALSE;

	// Establish # of session
	m_cNSession = ulSession;

	// Establish parent object pointer
	assert( pCDataSource );
	m_pCDataSource	= pCDataSource;
	m_pCDataSource->AddRef();

    // Allocate properties management object
    m_pUtilProp = new CUtilProp( m_pCDataSource, m_pCDataSource->m_pUtilProp );

    // Allocate contained interface objects
    m_pIGetDataSource		= new CImpIGetDataSource( this, pIUnknown );
    m_pIOpenRowset			= new CImpIOpenRowset( this, pIUnknown );
    m_pISessionProperties	= new CImpISessionProperties( this, pIUnknown );
	m_pIDBCreateCommand		= new CImpIDBCreateCommand( this, pIUnknown );
	m_pIDBSchemaRowset		= new CImpIDBSchemaRowset(this, pIUnknown);
	//m_pIIndexDefinition		= new CImpIIndexDefinition(this, pIUnknown);

	return (BOOL) (m_pIGetDataSource && 
				   m_pIOpenRowset && 
				   m_pISessionProperties &&
				   m_pIDBCreateCommand &&
				   m_pIDBSchemaRowset /*&&
				   m_pIIndexDefinition*/ );
}



// CDBSession::QueryInterface -----------------------------------------------
//
// @mfunc Returns a pointer to a specified interface. Callers use
// QueryInterface to determine which interfaces the called object
// supports.
//
// @rdesc HRESULT indicating the status of the method
//      @flag S_OK          | Interface is supported and ppvObject is set.
//      @flag E_NOINTERFACE | Interface is not supported by the object
//      @flag E_INVALIDARG  | One or more arguments are invalid.
//
STDMETHODIMP CDBSession::QueryInterface
    (
    REFIID riid,        //@parm IN | Interface ID of the interface being queried for.
    LPVOID * ppv        //@parm OUT | Pointer to interface that was instantiated
    )
{
	TRACE( "CDBSession::QueryInterface" );

    // Is the pointer bad?
    if (ppv == NULL)
        return E_INVALIDARG;

    //  Place NULL in *ppv in case of failure
    *ppv = NULL;

    //  This is the non-delegating IUnknown implementation
    if (riid == IID_IUnknown)
        *ppv = (LPVOID) this;
    else if (riid == IID_IGetDataSource)
        *ppv = (LPVOID) m_pIGetDataSource;
    else if (riid == IID_IOpenRowset)
        *ppv = (LPVOID) m_pIOpenRowset;
    else if (riid == IID_ISessionProperties)
        *ppv = (LPVOID) m_pISessionProperties;
	else if (riid == IID_IDBSchemaRowset)
		*ppv = (LPVOID) m_pIDBSchemaRowset;
	//else if (riid == IID_IIndexDefinition)
	//	*ppv = (LPVOID) m_pIIndexDefinition;
    else if (riid == IID_IDBCreateCommand)
        *ppv = (LPVOID) m_pIDBCreateCommand;

	if (*ppv == NULL)
		return E_NOINTERFACE;

    //  If we're going to return an interface, AddRef it first
	((LPUNKNOWN) *ppv)->AddRef();
    return S_OK;
}


// CDBSession::AddRef -------------------------------------------------------
//
// @mfunc Increments a persistence count for the object
//
// @rdesc Current reference count
//
STDMETHODIMP_( ULONG ) CDBSession::AddRef
     (
     void
     )
{
    return ++m_cRef;
}


// CDBSession::Release ------------------------------------------------------
//
// @mfunc Decrements a persistence count for the object and if
// persistence count is 0, the object destroys itself.
//
// @rdesc Current reference count
//
STDMETHODIMP_( ULONG ) CDBSession::Release
     (
     void
     )
{
    if (!--m_cRef)
        {
        delete this;
        return 0;
        }

    return m_cRef;
}

//-----------------------------------------------------------------------------
// CImpIGetDataSource::GetDataSource 
//
// @mfunc Retrieve an interface pointer on the session object
//
// @rdesc 
//		@flag S_OK | Session Object Interface returned
//		@flag E_INVALIDARG | ppDataSource was NULL
//		@flag E_NOINTERFACE | IID not supported
//
STDMETHODIMP CImpIGetDataSource::GetDataSource
	(
	REFIID		riid,			// @parm IN  | IID desired
	IUnknown**	ppDataSource	// @parm OUT | ptr to interface
	)
{
	INTERFACE_METHOD_START( "IGetDataSource::GetDataSource" );

	// Check Function Arguments
	if( ppDataSource == NULL )
		return E_INVALIDARG;

	assert(m_pObj->m_pCDataSource);
	assert(m_pObj->m_pCDataSource->m_pUnkOuter);
	
	//Handle Aggregated DataSource (if aggregated)
	return m_pObj->m_pCDataSource->m_pUnkOuter->QueryInterface(riid, (LPVOID*)ppDataSource);

	INTERFACE_METHOD_END();
}

// ISessionProperties::GetProperties ----------------------------------------------------
//
// @mfunc Returns current settings of all properties in the DBPROPFLAGS_SESSION property 
//			group
// @rdesc HRESULT
//      @flag S_OK          | The method succeeded
//      @flag E_INVALIDARG  | pcProperties or prgPropertyInfo was NULL
//      @flag E_OUTOFMEMORY | Out of memory
//
STDMETHODIMP CImpISessionProperties::GetProperties
    (
	    ULONG				cPropertySets,		//@parm IN | count of restiction guids
		const DBPROPIDSET	rgPropertySets[],	//@parm IN | restriction guids
		ULONG*              pcPropertySets,		//@parm OUT | count of property sets returned
		DBPROPSET**			prgPropertySets		//@parm OUT | property sets information returned
    )
{
    INTERFACE_METHOD_START( "ISessionProperties::GetProperties" );

	assert( m_pObj );
    assert( m_pObj->m_pUtilProp );

	// Check Arguments
	HRESULT hr = m_pObj->m_pUtilProp->GetPropertiesArgChk(PROPSET_SESSION, cPropertySets, 
								rgPropertySets, pcPropertySets, prgPropertySets);
	if ( FAILED(hr) )
		return hr;

    // just pass this call on to the utility object that manages our properties
    return m_pObj->m_pUtilProp->GetProperties(
									PROPSET_SESSION,
									cPropertySets, 
									rgPropertySets,
									pcPropertySets, 
									prgPropertySets );

	INTERFACE_METHOD_END();
}


// CImpISessionProperties::SetProperties  --------------------------------------------
//
// @mfunc Set properties in the DBPROPFLAGS_SESSION property group
//
// @rdesc HRESULT
//      @flag E_INVALIDARG  | cProperties was not equal to 0 and rgProperties was NULL
//      @flag E_NOTIMPL		| this method is not implemented
//
STDMETHODIMP    CImpISessionProperties::SetProperties
    (
	    ULONG		cProperties,
		DBPROPSET	rgProperties[]
	)
{
    INTERFACE_METHOD_START( "ISessionProperties::SetProperties" );

	assert( m_pObj );
    assert( m_pObj->m_pUtilProp );

	// Quick return if the Count of Properties is 0
	if( cProperties == 0 )
		return S_OK;

	// Check Arguments for use by properties
	HRESULT hr = m_pObj->m_pUtilProp->SetPropertiesArgChk(cProperties, rgProperties);
	if( FAILED(hr) )
		return hr;

    // just pass this call on to the utility object that manages our properties
    return m_pObj->m_pUtilProp->SetProperties( 
									PROPSET_SESSION,
									cProperties, 
									rgProperties);

	INTERFACE_METHOD_END();
}


// CImpIDBCreateCommand::CreateCommand ------------------------------------------------
//
// @mfunc Creates a new Command object, and returns the
// requested interface on the newly created object.
//
// @rdesc HRESULT
//      @flag S_OK                  | The method succeeded.
//      @flag E_INVALIDARG          | ppCommand was NULL
//      @flag E_NOINTERFACE         | Could not obtain requested interface on DBSession object
//      @flag E_OUTOFMEMORY         | Out of memory
//      @flag E_FAIL                | Could not initialize command
//      @flag DB_E_NOAGGREGATION    | pUnkOuter was not NULL (this object does not support
//                                    being aggregated)
//      @flag DB_E_OBJECTOPEN       | The provider would have to open a new connection to 
//									  support the operation and DBPROP_MULTIPLECONNECTIONS is set to VARIANT_FALSE
//
STDMETHODIMP CImpIDBCreateCommand::CreateCommand
    (
    IUnknown*   pUnkOuter,  //@parm IN  | Controlling IUnknown if being aggregated 
    REFIID      riid,       //@parm IN  | The ID of the interface 
    IUnknown**  ppCommand	//@parm OUT | A pointer to memory in which to return the interface pointer
    )
{
    INTERFACE_METHOD_START( "IDBCreateCommand::CreateCommand" );
	
	CCommand* pCommand = NULL;

    // check in-params and NULL out-params in case of error
    if (ppCommand)
        *ppCommand = NULL;
    else
        return E_INVALIDARG;
	
	if (pUnkOuter != NULL && riid != IID_IUnknown)
		return DB_E_NOAGGREGATION;

    assert(m_pObj);
	
	// Create session:

	// allocate memory for new session
    pCommand = new CCommand(pUnkOuter);
    if (!pCommand)
        return E_OUTOFMEMORY;

	// Initialize new command
    if (!pCommand->Init(m_pObj))
	{
        delete pCommand;
        return E_FAIL;
	}

    // get requested interface pointer to Command
    HRESULT hr = pCommand->QueryInterface(riid, (void**)ppCommand);
	if (hr != S_OK)
	{
        delete pCommand;
        return hr;
	}
	
	return S_OK;

	INTERFACE_METHOD_END();
}

//------------------------------------------------------------------------
// CDBSession::SetActiveRowset
//
// @mfunc MARK Rowset ulRowset As Active
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT CDBSession::SetActiveRowset
	(
	ULONG ulRowset	//@parm IN | # of Rowset
	)
{
	TRACE( "CDBSession::SetActiveRowset" );

	m_rbRowsetMonitor[ulRowset] = true;
	return (S_OK);
}


// CDBSession::SetInActiveRowset
//
// @mfunc MARK Rowset ulRowset As InActive
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT CDBSession::SetInActiveRowset
	(
	ULONG ulRowset	//@parm IN | # of Rowset
	)
{
	m_rbRowsetMonitor[ulRowset] = false;
	return (S_OK);
}


// CDBSession::GetFirstInactiveRowset
//
// @mfunc Get first inactive Rowset
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT CDBSession::GetFirstInactiveRowset
	(
	ULONG *ulRowset		//@parm IN | # of Rowset
	)
{
	TRACE( "CDBSession::GetFirstInactiveRowset" );

	ULONG ul;
	for (ul = 0; ul < NUM_SUPPORTED_ROWSETS_PER_SESSION; ul++)
	{
		if (m_rbRowsetMonitor[ul] == false)
		{
			*ulRowset = ul;
			return (S_OK);
		}
	}

	return (S_FALSE);
}


// CDBSession::CountActiveRowsets
//
// @mfunc Count Active Rowsets
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT CDBSession::CountActiveRowsets()
{
	TRACE( "CDBSession::CountActiveRowsets" );

	ULONG ul;
	ULONG ulRC = 0;
	for (ul = 0; ul < NUM_SUPPORTED_ROWSETS_PER_SESSION; ul++)
	{
		if (m_rbRowsetMonitor[ul] == true)
		{
			ulRC++;
		}
	}

	m_ulTotalRowsets = ulRC;

	return (S_OK);
}


// CDBSession::RemoveSession
//
// @mfunc Performs all operations, connected with removing session
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT	CDBSession::RemoveRowset
	(
	ULONG ulRowset	//@parm IN | # of Rowset
	)
{
	TRACE( "CDBSession::RemoveRowset" );

	HRESULT hr = (S_OK);
	
	// some cautions
	assert(ulRowset < NUM_SUPPORTED_ROWSETS_PER_SESSION);

	// mark session # ulRowset as inactive
	hr = SetInActiveRowset(ulRowset);
	if (hr == S_FALSE)
		return E_FAIL;	

	// shortcut for (m_pObj->m_pSessions)[ulSess]
	m_pRowsets[ulRowset] = NULL;

	// refresh the number of active sessions
	hr = CountActiveRowsets();	

	return (hr);
}

