/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

#include "hfiles.h"
#include "headers.h"
#include "extbuff.h"


//--------------------------------------------------------------------
// @mfunc Constructor for this class
//
// @rdesc NONE
//
CExtBuffer::CExtBuffer
    (
    void
    )
{
    CLEAR_CONSTRUCT( CExtBuffer );
}


//--------------------------------------------------------------------
// @mfunc Destructor for this class
//
// @rdesc NONE
//
CExtBuffer:: ~CExtBuffer
    (
    void
    )
{
    if (m_cbAlloc)
        VirtualFree((VOID *) m_rgItem, m_cbAlloc, MEM_DECOMMIT );

    if (m_rgItem)
        VirtualFree((VOID *) m_rgItem, 0, MEM_RELEASE );
}


//--------------------------------------------------------------------
// @mfunc Allocate and Initialize Buffer
//
// @rdesc HRESULT indicating routines status
//      @flag  S_OK | Initialization succeeded
//      @flag  E_OUTOFMEMORY | Not enough memory to allocate buffer
//
STDMETHODIMP CExtBuffer::FInit
    (
    ULONG cItemMax,     //@parm IN | Maximum number of items ever
    ULONG cbItem,       //@parm IN | Size of each item, in bytes
    ULONG cbPage        //@parm IN | Size of system page size (from SysInfo)
    )
{
    BYTE  *pb;

    m_cbReserved = ((cbItem *cItemMax) / cbPage + 1) *cbPage;
    m_rgItem = (BYTE *) VirtualAlloc( NULL, m_cbReserved, MEM_RESERVE, PAGE_READWRITE );

    if (m_rgItem == NULL)
        return E_OUTOFMEMORY;

    m_cbItem  = cbItem;
    m_dbAlloc = (cbItem / cbPage + 1) *cbPage;
    pb = (BYTE *) VirtualAlloc( m_rgItem, m_dbAlloc, MEM_COMMIT, PAGE_READWRITE );
    if (pb == NULL)
    {
        VirtualFree((VOID *) m_rgItem, 0, MEM_RELEASE );
        m_rgItem = NULL;
        return E_OUTOFMEMORY;
    }

    m_cbAlloc = m_dbAlloc;
    return S_OK;
}


//--------------------------------------------------------------------
// @mfunc Retrieves a pointer to the value at given index
//
// @rdesc If index is within the range of 1 to m_cItems then a valid
// pointer is returned, else NULL is returned.
//
void* CExtBuffer::operator[]
    (
    ULONG hItem          //@parm IN | Index of element in buffer
    )
{
    // Return ptr to element [n], where n = 1...m_cItem.
    // Returns NULL if 'n' is out of range.
    //
    // You must use InsertIntoExtBuffer to add new elements.
    // Thereafter you can use this operator to retrieve the address of the item.
    // (You can't delete an element, but you can overwrite its space.)

    if (1 <= hItem && hItem <= m_cItem)
        return m_rgItem + (hItem - 1) *m_cbItem;
    else
        return NULL;
}


//--------------------------------------------------------------------
// @mfunc Add Data to the fixed buffers and return the index it was
// added at.
//
// @rdesc HRESULT indicating routines status
//      @flag  S_OK | Data copied successfully
//      @flag  E_OUTOFMEMORY | Not enough memory to allocate buffer
//
STDMETHODIMP CExtBuffer::InsertIntoExtBuffer
    (
    VOID* pvItem,       //@parm IN | Pointer to buffer to copy
    ULONG &hItem        //@parm OUT | Index of where data was placed
    )
{
    ULONG cbOffset;

    cbOffset = m_cItem*m_cbItem;
    if ((cbOffset + m_cbItem) > m_cbAlloc)
    {
        BYTE *pb;

        if ((m_cbAlloc + m_dbAlloc) > m_cbReserved)
            pb = NULL;
        else
            pb = (BYTE *) VirtualAlloc( m_rgItem + m_cbAlloc,
                                        m_dbAlloc,
                                        MEM_COMMIT,
                                        PAGE_READWRITE );
        if (pb == NULL)
			return E_OUTOFMEMORY;
 
        m_cbAlloc += m_dbAlloc;
    }

    memcpy((m_rgItem + cbOffset), (BYTE *) pvItem, m_cbItem );
    m_cItem++;
    hItem = m_cItem;

    return S_OK;
}



//--------------------------------------------------------------------
// @mfunc Obtain a pointer to the data at a given index into the buffer
//
// @rdesc HRESULT indicating routines status
//      @flag  S_OK | pvItem contains a pointer to the data requested
//      @flag  E_INVALIDARG | Invalid Index passed in
//
STDMETHODIMP CExtBuffer::GetItemOfExtBuffer
    (
    ULONG hItem,        //@parm IN | Index of item to get
    VOID* pvItem        //@parm OUT | Pointer to block at index
    )
{
    if (hItem > m_cItem || hItem == 0)
        return E_INVALIDARG;

    memcpy((BYTE *) pvItem, (m_rgItem + (hItem - 1) *m_cbItem), m_cbItem );
    return S_OK;
}


//--------------------------------------------------------------------
// @mfunc Get the extents of the currently allocated buffers
//
// @rdesc HRESULT indicating routines status
//      @flag  S_OK | Extents were obtained successfuly
//
STDMETHODIMP CExtBuffer::GetFirstLastItemH
    (
    ULONG &hItemFirst,      //@parm OUT | First item allocated
    ULONG &hItemLast        //@parm OUT | Last item allocated
    )
{
    hItemFirst = 1;
    hItemLast  = m_cItem;
    return S_OK;
}

