/* Copyright (C) 2000 MySQL AB & MySQL Finland AB & TCX DataKonsult AB
   
   This file was taken from MySQL 3.23.11 sources and modified for
   use in Colibri.SQL by SWsoft. Changes are (C) 2000, 2001, SWsoft.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA */

/*
** SETUP.C - This is the ODBC sample driver code for
** setup.
**
**	This code is furnished on an as-is basis as part of the ODBC SDK and is
**	intended for example purposes only.
**
*/
/*--------------------------------------------------------------------------
  setup.c -- Sample ODBC setup

  This code demonstrates how to interact with the ODBC Installer.  These
  functions may be part of your ODBC driver or in a separate DLL.

  The ODBC Installer allows a driver to control the management of
  data sources by calling the ConfigDSN entry point in the appropriate
  DLL.  When called, ConfigDSN receives four parameters:

    hwndParent ---- Handle of the parent window for any dialogs which
                    may need to be created.  If this handle is NULL,
                    then no dialogs should be displayed (that is, the
                    request should be processed silently).

    fRequest ------ Flag indicating the type of request (add, configure
                    (edit), or remove).

    lpszDriver ---- Far pointer to a null-terminated string containing
                    the name of your driver.  This is the same string you
                    supply in the ODBC.INF file as your section header
                    and which ODBC Setup displays to the user in lieu
                    of the actual driver filename.  This string needs to
                    be passed back to the ODBC Installer when adding a
                    new data source name.

    lpszAttributes- Far pointer to a list of null-terminated attribute
                    keywords.  This list is similar to the list passed
                    to SQLDriverConnect, except that each key-value
                    pair is separated by a null-byte rather than a
                    semicolon.  The entire list is then terminated with
                    a null-byte (that is, two consecutive null-bytes
                    mark the end of the list).  The keywords accepted
                    should be those for SQLDriverConnect which are
                    applicable, any new keywords you define for ODBC.INI,
                    and any additional keywords you decide to document.

  ConfigDSN should return TRUE if the requested operation succeeds and
  FALSE otherwise.  The complete prototype for ConfigDSN is:

  BOOL FAR PASCAL ConfigDSN(HWND    hwndParent,
                            WORD    fRequest,
                            LPSTR   lpszDriver,
                            LPCSTR  lpszAttributes)

  Your setup code should not write to ODBC.INI directly to add or remove
  data source names.  Instead, link with ODBCINST.LIB (the ODBC Installer
  library) and call SQLWriteDSNToIni and SQLRemoveDSNFromIni.
  Use SQLWriteDSNToIni to add data source names.  If the data source name
  already exists, SQLWriteDSNToIni will delete it (removing all of its
  associated keys) and rewrite it.  SQLRemoveDSNToIni removes a data
  source name and all of its associated keys.

  For NT compatibility, the driver code should not use the
  Get/WritePrivateProfileString windows functions for ODBC.INI, but instead,
  use SQLGet/SQLWritePrivateProfileString functions that are macros (16 bit) or
  calls to the odbcinst.dll (32 bit).

--------------------------------------------------------------------------*/


// Includes ----------------------------------------------------------------


#include  "myodbc.h"					// Local include files
#undef VOID                                             // Because of ctl3d.h
#define VOID void
//#include <ctl3d.h>
#include "odbcinst.h"					// ODBC installer prototypes
#include <string.h>					// C include files
#include "dialogs.h"
#include <winuser.h>
#include <htmlhelp.h>

#ifdef SWSOFT
#define COBJMACROS
#include <shlobj.h>
#endif

// Constants ---------------------------------------------------------------

const char EMPTYSTR  []= "";
const char OPTIONON  []= "Yes";
const char OPTIONOFF []= "No";

// ODBC.INI keywords
const char ODBC_INI    []="ODBC.INI";     // ODBC initialization file
const char INI_KDESC   []="Description";  // Data source description
const char INI_KDB     []="Database";
#ifdef SWSOFT
extern int AddRemoveTable( HWND hwnd );
const char INI_KSYSDB  []="SystemDb";
int g_iInitSize = 0;
extern int CheckDSN(HWND hdlg);
#endif
const char INI_KSERVER []="Server";
const char INI_KUSER   []="User";	  // Second option
const char INI_KPASSWORD[] = "Password";
const char INI_KPORT   []  = "Port";
const char INI_KFLAG   []  = "Option";
const char INI_KSTMT   []  = "Stmt";
const char INI_SDEFAULT[]= "Default";
const char szTranslateName[] = "TranslationName";
const char szTranslateDLL[] = "TranslationDLL";
const char szTranslateOption[] = "TranslationOption";
const char szUnkTrans[] = "Unknown Translator";

#if defined (SWSOFT_LIB)
extern int mysql_applic_getport();
#endif



// Attribute string look-up table (maps keys to associated indexes)

static struct {
  char  szKey[MAXKEYLEN];
  int    iKey;
} s_aLookup[] = { "DSN",	KEY_DSN,
		  "DESC",	KEY_DESC,
		  "Description",KEY_DESC,
		  "Database",	KEY_DB,
		  "DB",		KEY_DB,
#ifdef SWSOFT
		  "SYSDB",		KEY_SYSDB,
#endif
		  "Server",	KEY_SERVER,
		  "UID",	KEY_USER,
		  "User",	KEY_USER,
		  "Password",	KEY_PASSWORD,
		  "PWD",	KEY_PASSWORD,
		  "Port",	KEY_PORT,
		  "Option",	KEY_FLAG,
                  "",0
};

#ifdef SWSOFT
void INTFUNC CenterDialog (HWND   hdlg);

static int HasSystemTable( const char* path )
{
	char szFile[ MAX_PATH ];
	
	if( GetFileAttributes( path ) == -1 )
		return -1; // error
	
	strcpy( szFile, path );
	strcat( szFile, "\\system.tbl" );

	if( GetFileAttributes( szFile ) != -1 )
		return 1; // exists
	else
		return 0; // not exist
}

void ShowHelp(HWND hdlg, const char * topic)
{
		  char lpHelpString[255];
		  DWORD ret;
		  char * pdest;
		  
		  TCHAR szFileName[_MAX_PATH];
		  
		  memset(szFileName, 0, _MAX_PATH);
		  ret = GetModuleFileName( s_hModule, szFileName,_MAX_PATH);
		  
		  LoadString(s_hModule,IDS_HELPFILE_NAME,lpHelpString,255);
		  
		  if( ret > 0)
		  {
			  
			  pdest = strrchr( szFileName, '\\' );

			  
			  if(pdest > 0)
			  {
				  *(pdest+1) = 0;
				  strcat(szFileName,lpHelpString);

				  if (strlen(topic))
				  {
					  strcat(szFileName,"\\");
					  strcat(szFileName,topic);
				  }
				  
#ifndef _LIB				  
				  HtmlHelp(NULL,/*hdlg,*/ szFileName, HH_DISPLAY_TOPIC,0);
#endif
			  }
		  }
}


int CALLBACK AskTableNameDlg (HWND hdlg,WORD wMsg,WPARAM wParam,LPARAM lParam)
{
 switch (wMsg) {
    // Initialize the dialog
  case WM_INITDIALOG:
  {
	char szDefName[ 64 ];
	char * pchar, * lchar1, * lchar2, * lchar;

	pchar = (LPSTR) lParam;
	lchar1 = strrchr( pchar, '\\' );
	lchar2 = strrchr( pchar, '/' );
	if( !lchar1 ) lchar1 = pchar; else lchar1++;
	if( !lchar2 ) lchar2 = pchar; else lchar2++;

	for( pchar = szDefName, lchar = max( lchar1, lchar2 ); *lchar && *lchar != '.'; lchar++, pchar++ )
		*pchar = *lchar;
	*pchar = 0;

//    Ctl3dRegister (s_hModule);
    SetWindowLong(hdlg, DWL_USER, lParam);
//    Ctl3dSubclassDlg(hdlg, CTL3D_ALL);
    CenterDialog(hdlg); 				// Center dialog

    SendDlgItemMessage(hdlg, IDC_BTRVTABLE_NAME, EM_LIMITTEXT, 64, 0L);
    SetDlgItemText( hdlg, IDC_BTRVTABLE_NAME, szDefName );
    SetDlgItemText( hdlg, IDC_BTRVTABLE_PATH, (LPCSTR) lParam );
    return TRUE;				/* Focus was not set */
  }
/*
#ifdef WIN32
//  case WM_CTLCOLORBTN:
  case WM_CTLCOLORDLG:
  case WM_CTLCOLOREDIT:
  case WM_CTLCOLORLISTBOX:
  case WM_CTLCOLORMSGBOX:
  case WM_CTLCOLORSCROLLBAR:
  case WM_CTLCOLORSTATIC:
    return DefDlgProc(hdlg,wMsg, wParam, lParam);  //(BOOL)Ctl3dCtlColorEx(wMsg, wParam, lParam);

  case WM_SETTEXT:
  case WM_NCPAINT:
  case WM_NCACTIVATE:
    //SetWindowLong(hdlg, DWL_MSGRESULT,
	//	  Ctl3dDlgFramePaint(hdlg, wMsg, wParam, lParam));
    return TRUE;
#endif
*/
//  case WM_SYSCOLORCHANGE:
//    return Ctl3dColorChange();

	case WM_HELP:
	  {
			ShowHelp(hdlg,"Overview.htm");
		  return TRUE;
	  }
    // Process buttons
  case WM_COMMAND:
    switch (GET_WM_COMMAND_ID(wParam, lParam)) 
	{
		// Accept results
		case IDOK:
		{
		  LPSTR tname = (LPSTR)GetWindowLong(hdlg, DWL_USER);
   		  GetDlgItemText(hdlg, IDC_BTRVTABLE_NAME, tname, 64 );

//		  Ctl3dUnregister (s_hModule);
		  EndDialog(hdlg, wParam);
		  return TRUE;
		}

		// Return to caller
		case IDCANCEL:
		case IDC_SKIP:
//		  Ctl3dUnregister (s_hModule);
		  EndDialog(hdlg, wParam);
		  return TRUE;

		case IDHELP:
		{
			ShowHelp(hdlg,"Overview.htm");
			return TRUE;
		}
    }
    break;
  }

  // Message not processed
  return FALSE;
}

int AskTableName( char* tname, const char* fname, HWND hwnd )
{
	int ret;
	
	strcpy( tname, fname );	
	ret = DialogBoxParam(s_hModule,
					 MAKEINTRESOURCE(BTRV_TBLNAME),
					 hwnd,
					 (DLGPROC) AskTableNameDlg,
					 (LONG)tname);
	return ret;
}

int ExistsFrb( const char* pszMkd )
{
	char szFrb[ MAX_PATH ];
	char* point;

	strcpy( szFrb, pszMkd );
	point = strrchr( szFrb, '.' );
	if( !point )
		point = szFrb + strlen( szFrb );
	strcpy( point, ".frb" );
	
	return GetFileAttributes( szFrb ) != ~(DWORD)0;
}

int RegisterTables(BOOL ask, const char* path, HWND hwnd)
{
	SQLHENV henv;
	SQLHDBC hdbc;
	DBC FAR * dbc;
	char dsn[1024];
	STARTUPINFO si;
	PROCESS_INFORMATION processInfo;
	TCHAR szPath[MAX_PATH],lpTextQuest[MAX_PATH];
	BOOL bRet;
	LPVOID lpMsgBuf;

	TCHAR szFileName[MAX_PATH];
	char lpstrFile[ MAX_PATH ] = "";	
	DWORD ret;
	char* lpPos;
		
	OPENFILENAME of;

	memset(szFileName, 0, MAX_PATH);
	
	//////////
	if( ask && 
		MessageBox( hwnd, "Do you want to register tables now?", "List of tables", MB_ICONQUESTION | MB_YESNO ) != IDYES )
			return 1;

	///////////////
	if( SQLAllocEnv(&henv) != SQL_SUCCESS )
		return -1;
	if( SQLAllocConnect(henv, &hdbc) != SQL_SUCCESS )
	{
		SQLFreeEnv( henv );
		return -1;
	}
	strcpy( dsn, "DSN=temp;SYSDB=" );
	strcat( dsn, path );

	if( SQLDriverConnect( hdbc, hwnd, dsn, (SQLSMALLINT)strlen(dsn), (SQLCHAR*)NULL, (SQLSMALLINT)0, (SQLSMALLINT*)NULL, (SQLUSMALLINT)SQL_DRIVER_NOPROMPT ) != SQL_SUCCESS )
	{
		SQLFreeConnect( hdbc );
		SQLFreeEnv( henv );
		return -1;
	}
	dbc = (DBC FAR*) hdbc;

	////////////////
	memset( &of, 0, sizeof(of) );
	of.lStructSize = sizeof(of);
	of.hwndOwner = hwnd;
	of.lpstrFilter = "Btrieve files (*.mkd;*.dat;*.btr;*.dta)\0*.mkd;*.dat;*.btr;*.dta\0All files\0*.*\0";
	of.lpstrInitialDir = path;
	of.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST /*| OFN_NOCHANGEDIR*/ | OFN_NONETWORKBUTTON | OFN_HIDEREADONLY;
	of.lpstrFile = lpstrFile;
	of.nMaxFile = MAX_PATH;
	of.lpstrTitle = "Register a Btrieve table";

	if (GetOpenFileName( &of ) )
	{
		char table[ 128 ];
		switch( AskTableName( table, lpstrFile, hwnd ) )
		{
		case IDCANCEL:
			goto quit;
		case IDOK:
			GetShortPathName( lpstrFile, lpstrFile, MAX_PATH );
			if( !ExistsFrb( lpstrFile ) )
			{
				wsprintf(lpTextQuest, "Cannot register table: %s \nTable definition file (FRB) does not exist\n\nDo you want to start DataAnalyzer now to create FRB file?", lpstrFile);

				if (IDYES == MessageBox( hwnd, lpTextQuest, table, MB_YESNO | MB_ICONQUESTION ))
				{
					ZeroMemory(&si, sizeof(si));
					si.cb           = sizeof(si);
					si.dwFlags      = STARTF_USESHOWWINDOW;
					si.wShowWindow  = SW_SHOW;
					
					ret = GetModuleFileName( s_hModule, szFileName,_MAX_PATH);
					lpPos = strrchr(szFileName,'\\');
					if (lpPos)
					{
						*lpPos = 0;
					}

					wsprintf(szPath, "%s\\%s %s", szFileName, "Analyzer.exe", lpstrFile);
					bRet = CreateProcess(NULL, szPath, NULL, NULL, FALSE, 0, NULL, NULL, &si, &processInfo);
					if (!bRet)
					{
						FormatMessage( 
							FORMAT_MESSAGE_ALLOCATE_BUFFER | 
							FORMAT_MESSAGE_FROM_SYSTEM | 
							FORMAT_MESSAGE_IGNORE_INSERTS,
							NULL,
							GetLastError(),
							MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
							(LPTSTR) &lpMsgBuf,
							0,
							NULL 
							);
						// Display the string.
						MessageBox( NULL, (LPCTSTR)lpMsgBuf, "ODBC Setup", MB_OK | MB_ICONEXCLAMATION );
						// Free the buffer.
						LocalFree( lpMsgBuf );

						//MessageBox(NULL,"Can't start DataAnalyzer","ODBC setup",MB_OK | MB_ICONEXCLAMATION);
					}
					//else
					//{
					//	MessageBox(hwnd,"DataAnalyzer has been started.","ODBC Setup",MB_OK);
					//}
  				}
			}
			else if( mysql_register_table(&dbc->mysql, table, lpstrFile) )
				MessageBox( hwnd, "Cannot register the table\nThe table may be already registered", table, MB_ICONWARNING );
		}
	}

quit:
	//////////////
	SQLDisconnect( hdbc );
	SQLFreeConnect( hdbc );
	SQLFreeEnv( henv );

	return 0;
}

int UnRegisterTables(BOOL ask, const char* path, HWND hwnd, char* lpTableName)
{
	SQLHENV henv;
	SQLHDBC hdbc;
	DBC FAR * dbc;
	char dsn[1024];
	char lpstrFile[ MAX_PATH ] = "";	

	char szMessage[ MAX_PATH ] ;

	sprintf(szMessage,"Do you want to unregister table %s from database?",lpTableName);
	
	//////////
	if( ask && 
		MessageBox( hwnd, szMessage , "List of tables", MB_ICONQUESTION | MB_YESNO ) != IDYES )
			return 1;

	///////////////
	if( SQLAllocEnv(&henv) != SQL_SUCCESS )
		return -1;
	if( SQLAllocConnect(henv, &hdbc) != SQL_SUCCESS )
	{
		SQLFreeEnv( henv );
		return -1;
	}
	strcpy( dsn, "DSN=temp;SYSDB=" );
	strcat( dsn, path );

	if( SQLDriverConnect( hdbc, hwnd, dsn, (SQLSMALLINT)strlen(dsn), (SQLCHAR*)NULL, (SQLSMALLINT)0, (SQLSMALLINT*)NULL, (SQLUSMALLINT)SQL_DRIVER_NOPROMPT ) != SQL_SUCCESS )
	{
		SQLFreeConnect( hdbc );
		SQLFreeEnv( henv );
		return -1;
	}
	dbc = (DBC FAR*) hdbc;

	mysql_unregister_table(&dbc->mysql, lpTableName);


	//////////////
	SQLFreeConnect( hdbc );
	SQLFreeEnv( henv );

	return 0;
}



int CreateSystemTable(BOOL ask, BOOL regTbls, const char* path, HWND hwnd)
{
	SQLHENV henv;
	SQLHDBC hdbc;
	DBC FAR * dbc;
	int ret;
	char dsn[1024];
	char _path[ MAX_PATH + 1 ];

	////////////
	if( ask && 
		MessageBox( hwnd, "The directory doesn't contain any system table. Click 'OK' to create system table or 'Cancel' to return to previous screen.", "System table", MB_OKCANCEL | MB_ICONEXCLAMATION ) != IDOK )
			return 1;

	////////////
	if( SQLAllocEnv(&henv) != SQL_SUCCESS )
	{
		MessageBox( hwnd, "Cannot allocate resources", path, MB_ICONWARNING );
		return -1;
	}
	if( SQLAllocConnect(henv, &hdbc) != SQL_SUCCESS )
	{
		SQLFreeEnv( henv );
		MessageBox( hwnd, "Cannot allocate resources", path, MB_ICONWARNING );
		return -1;
	}
	strcpy( dsn, "DSN=temp;SYSDB=" );
	strcat( dsn, path );

	if( SQLDriverConnect( hdbc, hwnd, dsn, (SQLSMALLINT)strlen(dsn), (SQLCHAR*)NULL, (SQLSMALLINT)0, (SQLSMALLINT*)NULL, (SQLUSMALLINT)SQL_DRIVER_NOPROMPT ) != SQL_SUCCESS )
	{
		SQLFreeConnect( hdbc );
		SQLFreeEnv( henv );
		MessageBox( hwnd, "Cannot connect to driver", path, MB_ICONWARNING );
		return -1;
	}
	dbc = (DBC FAR*) hdbc;
	GetShortPathName( path, _path, MAX_PATH );
	ret = mysql_create_systable(&dbc->mysql, _path);

	SQLFreeConnect( hdbc );
	SQLFreeEnv( henv );

	/////
	if( ret && ask )
		MessageBox( hwnd, "Cannot create a system table", path, MB_ICONWARNING );

	/*
	//////////////
	if( !ret && regTbls )
	{
		ret = RegisterTables( TRUE, path, hwnd );
		if( ret > 0 ) // 0 == OK, -1 == error, 1 == cancel
			ret = 0;
	}
	*/
	return ret;
}

void DoRegister( HWND hdlg )
{
  char szPath[ MAX_PATH ];
  GetDlgItemText( hdlg, IDC_SYSDB, szPath, sizeof(szPath) );

  switch( HasSystemTable( szPath ) )
  {
  case -1:
	  MessageBox( hdlg, "Wrong path", szPath, MB_ICONWARNING );
	  return;
  case 0:
	  if( CreateSystemTable( TRUE, FALSE, szPath, hdlg ) != 0 )
		  return;
  }
  RegisterTables( FALSE, szPath, hdlg );
}


static void DoBrowse( HWND hdlg )
{
	char strPath[ MAX_PATH ];
	BROWSEINFO bi;
	LPITEMIDLIST pidl = NULL;
	LPMALLOC pMalloc = NULL;

	CoInitialize(NULL);

	if( SHGetMalloc(&pMalloc) == NOERROR )
	{
		memset( &bi, 0, sizeof(bi));
		bi.hwndOwner = hdlg;
		bi.ulFlags = BIF_RETURNONLYFSDIRS;
		bi.lpszTitle = "Choose a folder of system database";
		bi.pszDisplayName = strPath;
		pidl = SHBrowseForFolder( &bi );

		if( pidl && SHGetPathFromIDList(pidl, strPath) )
		{
			  SetDlgItemText(hdlg, IDC_SYSDB, strPath);

			  if( !HasSystemTable( strPath )  )
				  CreateSystemTable( TRUE, TRUE, strPath, hdlg );
		}

		if( pidl )
			pMalloc->lpVtbl->Free( pMalloc, pidl );
		pMalloc->lpVtbl->Release( pMalloc );
	}

	CoUninitialize();
}

#endif

#ifdef SWSOFT_LIB
void set_attributes(LPSETUPDLG lpsetupdlg,BOOL bSetup)
#else
void set_attributes(LPSETUPDLG lpsetupdlg)
#endif
{
  LPCSTR lpszDSN=lpsetupdlg->aAttr[KEY_DSN].szAttr;

  if (!lpsetupdlg->aAttr[KEY_DESC].fSupplied)
    SQLGetPrivateProfileString(lpszDSN, INI_KDESC,
			       EMPTYSTR,
			       lpsetupdlg->aAttr[KEY_DESC].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_DESC].szAttr),
			       ODBC_INI);
  if (!lpsetupdlg->aAttr[KEY_DB].fSupplied)
    SQLGetPrivateProfileString(lpszDSN, INI_KDB,
			       lpszDSN,
			       lpsetupdlg->aAttr[KEY_DB].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_DB].szAttr),
			       ODBC_INI);
#ifdef SWSOFT
  if (!lpsetupdlg->aAttr[KEY_SYSDB].fSupplied)
    SQLGetPrivateProfileString(lpszDSN, INI_KSYSDB,
			       EMPTYSTR,
			       lpsetupdlg->aAttr[KEY_SYSDB].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_SYSDB].szAttr),
			       ODBC_INI);
#endif
  if (!lpsetupdlg->aAttr[KEY_SERVER].fSupplied)
    SQLGetPrivateProfileString(lpszDSN, INI_KSERVER,
			       EMPTYSTR,
			       lpsetupdlg->aAttr[KEY_SERVER].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_SERVER].szAttr),
			       ODBC_INI);
  if (!lpsetupdlg->aAttr[KEY_USER].fSupplied)
    SQLGetPrivateProfileString(lpszDSN, INI_KUSER,
			       EMPTYSTR,
			       lpsetupdlg->aAttr[KEY_USER].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_USER].szAttr),
			       ODBC_INI);
  if (!lpsetupdlg->aAttr[KEY_PASSWORD].fSupplied)
    SQLGetPrivateProfileString(lpszDSN, INI_KPASSWORD,
			       EMPTYSTR,
			       lpsetupdlg->aAttr[KEY_PASSWORD].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_PASSWORD].
				      szAttr),
			       ODBC_INI);
  if (!lpsetupdlg->aAttr[KEY_PORT].fSupplied)
  {
    char buff[10];
#if defined (SWSOFT_LIB)
	*buff = 0;
	if( !bSetup )
	{
		int port = mysql_applic_getport();
		if( port )
			int2str(port,buff,10);
	}		
#else
    int2str(MYSQL_PORT,buff,10);
#endif
    SQLGetPrivateProfileString(lpszDSN, INI_KPORT,
			       buff,
			       lpsetupdlg->aAttr[KEY_PORT].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_PORT].
				      szAttr),
			       ODBC_INI);
  }
  if (!lpsetupdlg->aAttr[KEY_FLAG].fSupplied)
#ifdef SWSOFT
	  if( !*lpszDSN )
		  strcpy( lpsetupdlg->aAttr[KEY_FLAG].szAttr, "16384" );
	  else
#endif
    SQLGetPrivateProfileString(lpszDSN, INI_KFLAG,
#ifdef SWSOFT
				   "16384",
#else
			       "0",
#endif
			       lpsetupdlg->aAttr[KEY_FLAG].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_FLAG].
				      szAttr),
			       ODBC_INI);

  if (!lpsetupdlg->aAttr[KEY_STMT].fSupplied)
    SQLGetPrivateProfileString(lpszDSN, INI_KSTMT,
			       EMPTYSTR,
			       lpsetupdlg->aAttr[KEY_STMT].szAttr,
			       sizeof(lpsetupdlg->aAttr[KEY_STMT].
				      szAttr),
			       ODBC_INI);
}


#ifdef SWSOFT_LIB
void set_dlg_items(HWND hdlg ,LPSETUPDLG lpsetupdlg, BOOL bSetup)
#else
void set_dlg_items(HWND hdlg ,LPSETUPDLG lpsetupdlg)
#endif
{
  uint i;
  ulong flag;
#ifdef SWSOFT_LIB
  set_attributes(lpsetupdlg, bSetup);
#else
  set_attributes(lpsetupdlg);
#endif
  SetDlgItemText(hdlg, IDC_DSNAME,	lpsetupdlg->aAttr[KEY_DSN].szAttr);
  SetDlgItemText(hdlg, IDC_DB,		lpsetupdlg->aAttr[KEY_DB].szAttr);
#ifdef SWSOFT
  SetDlgItemText(hdlg, IDC_SYSDB,	lpsetupdlg->aAttr[KEY_SYSDB].szAttr);
  SetDlgItemText(hdlg, IDC_DESC,	lpsetupdlg->aAttr[KEY_DESC].szAttr);
#endif
  SetDlgItemText(hdlg, IDC_SERVER,	lpsetupdlg->aAttr[KEY_SERVER].szAttr);
  SetDlgItemText(hdlg, IDC_USER,	lpsetupdlg->aAttr[KEY_USER].szAttr);
  SetDlgItemText(hdlg, IDC_PASSWORD,	lpsetupdlg->aAttr[KEY_PASSWORD].szAttr);
  SetDlgItemText(hdlg, IDC_PORT,	lpsetupdlg->aAttr[KEY_PORT].szAttr);
  SetDlgItemText(hdlg, IDC_STMT,	lpsetupdlg->aAttr[KEY_STMT].szAttr);
  flag=(ulong) atol(lpsetupdlg->aAttr[KEY_FLAG].szAttr);
#ifdef DBUG_OFF
  flag&= ~FLAG_DEBUG;
#endif
  for (i=0 ; i < NUMOFBITS ; i++)
    CheckDlgButton(hdlg, IDC_CHECK1+i,
		   flag & (1 << i) ? BST_CHECKED : BST_UNCHECKED);
}


void INTFUNC CenterDialog (HWND   hdlg);
int  CALLBACK ConfigDlgProc (HWND hdlg,WORD wMsg,WPARAM wParam, LPARAM lParam);
BOOL CALLBACK SetDSNAttributes(HWND hwnd, LPSETUPDLG lpsetupdlg);

/* ConfigDSN ---------------------------------------------------------------
  Description:  ODBC Setup entry point
                This entry point is called by the ODBC Installer
                (see file header for more details)
  Input      :  hwnd ----------- Parent window handle
                fRequest ------- Request type (i.e., add, config, or remove)
                lpszDriver ----- Driver name
                lpszAttributes - data source attribute string
  Output     :  TRUE success, FALSE otherwise
--------------------------------------------------------------------------*/
#ifndef _LIB
BOOL EXPFUNC ConfigDSN (HWND	 hwnd,
			 WORD	 fRequest,
			 LPCSTR  lpszDriver,
			 LPCSTR  lpszAttributes)
{
  BOOL  fSuccess;				// Success/fail flag
  GLOBALHANDLE hglbAttr;
  LPSETUPDLG lpsetupdlg;

  // Allocate attribute array
  hglbAttr = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT, sizeof(SETUPDLG));
  if (!hglbAttr)
    return FALSE;
  lpsetupdlg = (LPSETUPDLG)GlobalLock(hglbAttr);

  // Parse attribute string
  if (lpszAttributes)
    ParseAttributes(lpszAttributes, lpsetupdlg);

  // Save original data source name
  if (lpsetupdlg->aAttr[KEY_DSN].fSupplied)
    lstrcpy(lpsetupdlg->szDSN, lpsetupdlg->aAttr[KEY_DSN].szAttr);
  else
    lpsetupdlg->szDSN[0] = '\0';

  // Remove data source
  if (ODBC_REMOVE_DSN == fRequest) {
    // Fail if no data source name was supplied
    if (!lpsetupdlg->aAttr[KEY_DSN].fSupplied)
      fSuccess = FALSE;

    // Otherwise remove data source from ODBC.INI
    else
      fSuccess = SQLRemoveDSNFromIni(lpsetupdlg->aAttr[KEY_DSN].szAttr);
  }

  // Add or Configure data source
  else {
    // Save passed variables for global access (e.g., dialog access)
    lpsetupdlg->hwndParent = hwnd;
    lpsetupdlg->lpszDrvr   = lpszDriver;
    lpsetupdlg->fNewDSN	   = (ODBC_ADD_DSN == fRequest);
    lpsetupdlg->fDefault   = !lstrcmpi(lpsetupdlg->aAttr[KEY_DSN].szAttr,
				       INI_SDEFAULT);

    // Display the appropriate dialog (if parent window handle supplied)
    if (hwnd) {
      // Display dialog(s)
#ifdef SWSOFT
		BOOL bSimple = TRUE;
		//DWORD resId = bSimple ? BTRV_DSN_SIMPLE2 : BTRV_DSN;
		DWORD resId = BTRV_DSN;
		int iRet = DialogBoxParam(s_hModule,
			MAKEINTRESOURCE(resId),
			hwnd,
			(DLGPROC) ConfigDlgProc,
			(LONG)(LPSTR)lpsetupdlg);
		
			/*
			if( iRet != IDC_SWITCH )
			{
			fSuccess = iRet == IDOK;
			break;
			}
		*/
		bSimple = !bSimple;
		
#else
      fSuccess = (IDOK == DialogBoxParam(s_hModule,
					 MAKEINTRESOURCE(CONFIGDSN),
					 hwnd,
					 (DLGPROC) ConfigDlgProc,
					 (LONG)(LPSTR)lpsetupdlg));
#endif
    }

    else if (lpsetupdlg->aAttr[KEY_DSN].fSupplied)
      fSuccess = SetDSNAttributes(hwnd, lpsetupdlg);
    else
      fSuccess = FALSE;
  }

  GlobalUnlock(hglbAttr);
  GlobalFree(hglbAttr);
  return fSuccess;
}
#endif // _LIB


/* CenterDialog ------------------------------------------------------------
	Description:  Center the dialog over the frame window
	Input	   :  hdlg -- Dialog window handle
	Output	   :  None
--------------------------------------------------------------------------*/
void INTFUNC CenterDialog(HWND hdlg)
{
  HWND	hwndFrame;
  RECT	rcDlg, rcScr, rcFrame;
  int		cx, cy;

  hwndFrame = GetParent(hdlg);
  if( !hwndFrame )
	  hwndFrame = GetDesktopWindow();

  GetWindowRect(hdlg, &rcDlg);
  cx = rcDlg.right  - rcDlg.left;
  cy = rcDlg.bottom - rcDlg.top;

  GetClientRect(hwndFrame, &rcFrame);

  ClientToScreen(hwndFrame, (LPPOINT)(&rcFrame.left));
  ClientToScreen(hwndFrame, (LPPOINT)(&rcFrame.right));
  rcDlg.top    = rcFrame.top  + (((rcFrame.bottom - rcFrame.top) - cy) >> 1);
  rcDlg.left   = rcFrame.left + (((rcFrame.right - rcFrame.left) - cx) >> 1);
  rcDlg.bottom = rcDlg.top  + cy;
  rcDlg.right  = rcDlg.left + cx;

  GetWindowRect(GetDesktopWindow(), &rcScr);
  if (rcDlg.bottom > rcScr.bottom)
  {
    rcDlg.bottom = rcScr.bottom;
    rcDlg.top    = rcDlg.bottom - cy;
  }
  if (rcDlg.right  > rcScr.right)
  {
    rcDlg.right = rcScr.right;
    rcDlg.left  = rcDlg.right - cx;
  }

  if (rcDlg.left < 0) rcDlg.left = 0;
  if (rcDlg.top  < 0) rcDlg.top  = 0;

  MoveWindow(hdlg, rcDlg.left, rcDlg.top, cx, cy, TRUE);
  return;
}


/* ConfigDlgProc -----------------------------------------------------------
  Description:  Manage add data source name dialog
  Input      :  hdlg --- Dialog window handle
                wMsg --- Message
                wParam - Message parameter
                lParam - Message parameter
  Output     :  TRUE if message processed, FALSE otherwise
--------------------------------------------------------------------------*/
int CALLBACK ConfigDlgProc (HWND hdlg,WORD wMsg,WPARAM wParam,LPARAM lParam)
{
 switch (wMsg) {
    // Initialize the dialog
  case WM_INITDIALOG:
  {
    LPSETUPDLG lpsetupdlg;
	RECT rcIniti;
	RECT rcFrame;
	HWND hFrame;


	GetWindowRect(hdlg,&rcIniti);
	g_iInitSize = rcIniti.bottom - rcIniti.top;

	hFrame = GetDlgItem(hdlg, IDC_FRM);
	GetWindowRect(hFrame,&rcFrame);
	if (rcFrame.top - rcIniti.top != rcIniti.bottom - rcIniti.top)
	{
		SetWindowPos(hdlg,NULL,0,0,rcIniti.right-rcIniti.left, rcFrame.top - rcIniti.top,SWP_NOMOVE);
		SetDlgItemText(hdlg, IDC_SWITCH,"More >>");
	}
	



//    Ctl3dRegister (s_hModule);
    SetWindowLong(hdlg, DWL_USER, lParam);
/*
#ifdef WIN32
    Ctl3dSubclassDlg(hdlg, CTL3D_ALL);
#else
    Ctl3dSubclassDlgEx(hdlg, CTL3D_ALL);
#endif
*/
   CenterDialog(hdlg); 				// Center dialog



    lpsetupdlg = (LPSETUPDLG) lParam;
#ifdef SWSOFT_LIB
    set_attributes(lpsetupdlg, TRUE);
    set_dlg_items(hdlg,lpsetupdlg, TRUE);
#else
    set_attributes(lpsetupdlg);
    set_dlg_items(hdlg,lpsetupdlg);
#endif

    if (lpsetupdlg->fDefault)
    {
      EnableWindow(GetDlgItem(hdlg, IDC_DSNAME), FALSE);
      EnableWindow(GetDlgItem(hdlg, IDC_DSNAMETEXT), FALSE);
    }
    else
    {
      SendDlgItemMessage(hdlg, IDC_DSNAME,
			 EM_LIMITTEXT, (WPARAM)(MAXDSNAME-1), 0L);
      SendDlgItemMessage(hdlg, IDC_DESC,
			 EM_LIMITTEXT, (WPARAM)(MAXDESC-1), 0L);
      SendDlgItemMessage(hdlg, IDC_DB,
			 EM_LIMITTEXT, (WORD)(NAME_LEN), 0L);
      SendDlgItemMessage(hdlg, IDC_SERVER,
			 EM_LIMITTEXT, (WORD)(MAXSERVER-1), 0L);
      SendDlgItemMessage(hdlg, IDC_USER,
			 EM_LIMITTEXT, (WORD)(MAXUSER-1), 0L);
      SendDlgItemMessage(hdlg, IDC_PASSWORD,
			 EM_LIMITTEXT, (WORD)(MAXPASSWORD-1), 0L);
      SendDlgItemMessage(hdlg, IDC_PORT,
			 EM_LIMITTEXT, (WORD) 5, 0L);
      SendDlgItemMessage(hdlg, IDC_STMT,
			 EM_LIMITTEXT, (WORD) (MAXSTMT-1), 0L);
#if defined(DBUG_OFF) && defined(NOT_YET)
      /* HELP: How can one disable this button ? */
      SendDlgItemMessage(hdlg, IDC_CHECK3,
			 EM_SETREADONLY, (WORD) TRUE, 0L);
#endif
    }
    return TRUE;				/* Focus was not set */
  }
/*
#ifdef WIN32
  case WM_CTLCOLORBTN:
  case WM_CTLCOLORDLG:
  case WM_CTLCOLOREDIT:
  case WM_CTLCOLORLISTBOX:
  case WM_CTLCOLORMSGBOX:
  case WM_CTLCOLORSCROLLBAR:
  case WM_CTLCOLORSTATIC:
    return (BOOL)Ctl3dCtlColorEx(wMsg, wParam, lParam);

  case WM_SETTEXT:
  case WM_NCPAINT:
  case WM_NCACTIVATE:
    SetWindowLong(hdlg, DWL_MSGRESULT,
		  Ctl3dDlgFramePaint(hdlg, wMsg, wParam, lParam));
    return TRUE;
#endif

  case WM_SYSCOLORCHANGE:
    return Ctl3dColorChange();
*/
	case WM_HELP:
	{
		RECT cr,rcFrame;
		HWND hFrame;

		GetWindowRect(hdlg,&cr);
		hFrame = GetDlgItem(hdlg, IDC_FRM);
		GetWindowRect(hFrame,&rcFrame);
		if (rcFrame.top - cr.top != cr.bottom - cr.top)
		{
			ShowHelp(hdlg,"Overview.htm");
		}
		else
		{
			ShowHelp(hdlg,"Overview.htm");
		}

		  return TRUE;
	}
    // Process buttons
  case WM_COMMAND:
    switch (GET_WM_COMMAND_ID(wParam, lParam)) {
      // Ensure the OK button is enabled only when a data source name
      // is entered
    case IDC_DSNAME:
		if (GET_WM_COMMAND_CMD(wParam, lParam) == EN_CHANGE)
		{
			char	szItem[MAXDSNAME];		// Edit control text
			int sdnlen = GetDlgItemText(hdlg, IDC_DSNAME,  szItem, sizeof(szItem));
			int systablen = setupdlg_db_type() == SETUPDLG_BTRIEVE ? GetDlgItemText(hdlg, IDC_SYSDB,  szItem, sizeof(szItem)) : 1;
			// Enable/disable the OK button
			EnableWindow(GetDlgItem(hdlg, IDOK),  sdnlen && systablen);
			
			//Hide table button if path to system table is empty
			EnableWindow(GetDlgItem(hdlg, IDC_ADD), sdnlen && systablen);
			//Hide test button if path to system table is empty
			EnableWindow(GetDlgItem(hdlg, IDC_TEST),sdnlen && systablen);
			
			return TRUE;
		}
      break;

case IDC_SYSDB:
      if (GET_WM_COMMAND_CMD(wParam, lParam) == EN_CHANGE)
      {
		  
		  char	szItem[MAXDSNAME];		// Edit control text
		  int sdnlen = GetDlgItemText(hdlg, IDC_DSNAME,  szItem, sizeof(szItem));
		  int systablen = setupdlg_db_type() == SETUPDLG_BTRIEVE ? GetDlgItemText(hdlg, IDC_SYSDB,  szItem, sizeof(szItem)) : 1;
		  // Enable/disable the OK button
		  EnableWindow(GetDlgItem(hdlg, IDOK),  sdnlen && systablen);
		  
		  //Hide table button if path to system table is empty
		  EnableWindow(GetDlgItem(hdlg, IDC_ADD), sdnlen && systablen);
		  //Hide test button if path to system table is empty
		  EnableWindow(GetDlgItem(hdlg, IDC_TEST),sdnlen && systablen);

	return TRUE;
      }
      break;


    // Accept results
#ifdef SWSOFT
	case IDC_SWITCH:
	{
		RECT cr,rcFrame;
		HWND hFrame;

		GetWindowRect(hdlg,&cr);
		hFrame = GetDlgItem(hdlg, IDC_FRM);
		GetWindowRect(hFrame,&rcFrame);
		if (rcFrame.top - cr.top != cr.bottom - cr.top)
		{
			SetWindowPos(hdlg,NULL,0,0,cr.right-cr.left, rcFrame.top - cr.top,SWP_NOMOVE);
			SetDlgItemText(hdlg, IDC_SWITCH,"More >>");
		}
		else
		{
			SetWindowPos(hdlg,NULL,0,0,cr.right-cr.left, g_iInitSize ,SWP_NOMOVE);
			SetDlgItemText(hdlg, IDC_SWITCH,"<< Less");
		}


		return TRUE;
	}
#endif
    case IDOK:
    {
      LPSETUPDLG lpsetupdlg;
      ulong flag;
      uint i;

      lpsetupdlg = (LPSETUPDLG)GetWindowLong(hdlg, DWL_USER);
      // Retrieve dialog values
      if (!lpsetupdlg->fDefault)
	GetDlgItemText(hdlg, IDC_DSNAME,
		       lpsetupdlg->aAttr[KEY_DSN].szAttr,
		       sizeof(lpsetupdlg->aAttr[KEY_DSN].szAttr));
      GetDlgItemText(hdlg, IDC_DESC,
		     lpsetupdlg->aAttr[KEY_DESC].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_DESC].szAttr));
      GetDlgItemText(hdlg, IDC_DB,
		     lpsetupdlg->aAttr[KEY_DB].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_DB].szAttr));
#ifdef SWSOFT
      GetDlgItemText(hdlg, IDC_SYSDB,
		     lpsetupdlg->aAttr[KEY_SYSDB].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_SYSDB].szAttr));
#endif
      GetDlgItemText(hdlg, IDC_SERVER,
		     lpsetupdlg->aAttr[KEY_SERVER].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_SERVER].szAttr));
      GetDlgItemText(hdlg, IDC_USER,
		     lpsetupdlg->aAttr[KEY_USER].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_USER].szAttr));
      GetDlgItemText(hdlg, IDC_PASSWORD,
		     lpsetupdlg->aAttr[KEY_PASSWORD].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_PASSWORD].szAttr));
      GetDlgItemText(hdlg, IDC_PORT,
		     lpsetupdlg->aAttr[KEY_PORT].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_PORT].szAttr));
      GetDlgItemText(hdlg, IDC_STMT,
		     lpsetupdlg->aAttr[KEY_STMT].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_STMT].szAttr));

      flag=0;
      for (i=0 ; i < NUMOFBITS ; i++)
	if (IsDlgButtonChecked(hdlg, IDC_CHECK1+i) == BST_CHECKED)
	  flag|= 1 << i;
      sprintf(lpsetupdlg->aAttr[KEY_FLAG].szAttr,"%lu",flag);

#ifdef SWSOFT
	  if( setupdlg_db_type() == SETUPDLG_BTRIEVE && GET_WM_COMMAND_ID(wParam, lParam) == IDOK )
	  switch( HasSystemTable(lpsetupdlg->aAttr[KEY_SYSDB].szAttr) )
	  {
	  case -1:
		  if( MessageBox( hdlg, "Wrong path. Do you want to continue?", lpsetupdlg->aAttr[KEY_SYSDB].szAttr, MB_ICONWARNING | MB_YESNO) == IDNO )
			  return TRUE;
		  break;
	  case 0:
		  if (CreateSystemTable( TRUE, TRUE, lpsetupdlg->aAttr[KEY_SYSDB].szAttr, hdlg ) != 0);
			return FALSE;
		  break;
	  }

	  if( GET_WM_COMMAND_ID(wParam, lParam) == IDOK || MessageBox(hdlg, "Do you want to save changes?", "Colibri", MB_ICONQUESTION | MB_YESNO | MB_DEFBUTTON2 ) == IDYES )
#endif
      // Update ODBC.INI
      SetDSNAttributes(hdlg, lpsetupdlg);

#ifdef SWSOFT
//      Ctl3dUnregister (s_hModule);
      EndDialog(hdlg, wParam);
      return TRUE;
#endif
    }

#ifdef SWSOFT
	case IDC_BROWSE:
		DoBrowse( hdlg );
		return TRUE;
	case IDC_ADD:
		{
			char szPath[255];
			GetDlgItemText(hdlg, IDC_SYSDB,szPath,	255);
			
			switch( HasSystemTable( szPath) )
			{
			case -1:
				MessageBox( hdlg, "Wrong path",  szPath, MB_ICONWARNING );
				return TRUE;
			case 0:
				if( CreateSystemTable( TRUE, FALSE, szPath, hdlg ) != 0 )
					return TRUE;
			}
			AddRemoveTable(hdlg);
			
			return TRUE;
		}
#endif

	// Return to caller
    case IDCANCEL:
//      Ctl3dUnregister (s_hModule);
      EndDialog(hdlg, wParam);
      return TRUE;
	case IDHELP:
		{
			RECT cr,rcFrame;
			HWND hFrame;

			GetWindowRect(hdlg,&cr);
			hFrame = GetDlgItem(hdlg, IDC_FRM);
			GetWindowRect(hFrame,&rcFrame);
			if (rcFrame.top - cr.top != cr.bottom - cr.top)
			{
				ShowHelp(hdlg,"Overview.htm");
			}
			else
			{
				ShowHelp(hdlg,"Overview.htm");
			}
			return TRUE;
		}
	case IDC_TEST:
		{
			BOOL bIsSucc,bIsHasSysTab;
			char szPath[255];
			GetDlgItemText(hdlg, IDC_SYSDB,szPath,	255);

			if( !HasSystemTable(szPath) )
				CreateSystemTable( TRUE, FALSE, szPath, hdlg );

			SetCursor( LoadCursor(NULL, IDC_WAIT) );			
			bIsHasSysTab = HasSystemTable(szPath);				
			bIsSucc = CheckDSN(hdlg);
			SetCursor( LoadCursor(NULL, IDC_ARROW) );

			if (bIsSucc && bIsHasSysTab)
			{
				MessageBox(hdlg,"Connection successfull","Test DSN", MB_OK | MB_ICONINFORMATION);
			}
			else
			{
				MessageBox(hdlg,"Connection failed","Test DSN", MB_OK | MB_ICONEXCLAMATION);
			}
		
		}

    }
    break;
  }

  // Message not processed
  return FALSE;
}


/* ParseAttributes ---------------------------------------------------------
  Description:  Parse attribute string moving values into the aAttr array
  Input      :  lpszAttributes - Pointer to attribute string
  Output     :  None (global aAttr normally updated)
--------------------------------------------------------------------------*/
void INTFUNC ParseAttributes(LPCSTR lpszAttributes, LPSETUPDLG lpsetupdlg)
{
  LPCSTR	lpsz;
  LPCSTR	lpszStart;
  char	aszKey[MAXKEYLEN];
  int		iElement;
  int		cbKey;
  DBUG_ENTER("ParseAttributes");
  DBUG_PRINT("enter",("attributes: %s",lpszAttributes));

  for (lpsz=lpszAttributes; *lpsz;)
  {  //  Extract key name (e.g., DSN), it must be terminated by an equals
    lpszStart = lpsz;
    if (!(lpsz=strchr(lpsz,'=')))
      DBUG_VOID_RETURN;
    // Determine the keys index in the key table (-1 if not found)
    iElement = -1;
    cbKey	 = lpsz - lpszStart;
    if (cbKey < sizeof(aszKey))
    {
      register int j;

      _fmemcpy(aszKey, lpszStart, cbKey);
      aszKey[cbKey] = '\0';
      for (j = 0; *s_aLookup[j].szKey; j++)
      {
	if (!lstrcmpi(s_aLookup[j].szKey, aszKey))
	{
	  iElement = s_aLookup[j].iKey;
	  break;
	}
      }
    }

    // Locate end of key value
    lpszStart = ++lpsz;
    lpsz=strcend(lpsz,';');

    // Save value if key is known
    // NOTE: This code assumes the szAttr buffers in aAttr have been
    //	   zero initialized
    if (iElement >= 0)
    {
      lpsetupdlg->aAttr[iElement].fSupplied = TRUE;
      strmake(lpsetupdlg->aAttr[iElement].szAttr,lpszStart,
	      min(lpsz-lpszStart, sizeof(lpsetupdlg->aAttr[0].szAttr)-1));
    }
    if (*lpsz)
      lpsz++;               /* Skipp ';' */
  }
  DBUG_VOID_RETURN;
}


/* SetDSNAttributes --------------------------------------------------------
  Description:  Write data source attributes to ODBC.INI
  Input      :  hwnd - Parent window handle (plus globals)
  Output     :  TRUE if successful, FALSE otherwise
--------------------------------------------------------------------------*/
BOOL INTFUNC SetDSNAttributes(HWND hwndParent, LPSETUPDLG lpsetupdlg)
{
  LPCSTR lpszDSN=lpsetupdlg->aAttr[KEY_DSN].szAttr;  // Pointer to data source name

  // Validate arguments
  if (lpsetupdlg->fNewDSN && !*lpsetupdlg->aAttr[KEY_DSN].szAttr)
    return FALSE;

  // Write the data source name
  if (!SQLWriteDSNToIni(lpszDSN, lpsetupdlg->lpszDrvr))
  {
    if (hwndParent)
    {
      char  szBuf[MAXPATHLEN];
      char  szMsg[MAXPATHLEN];

      LoadString(s_hModule, IDS_BADDSN, szBuf, sizeof(szBuf));
      wsprintf(szMsg, szBuf, lpszDSN);
      LoadString(s_hModule, IDS_MSGTITLE, szBuf, sizeof(szBuf));
      MessageBox(hwndParent, szMsg, szBuf, MB_ICONEXCLAMATION | MB_OK);
    }
    return FALSE;
  }

  // Update ODBC.INI
  // Save the value if the data source is new, if it was edited, or if
  // it was explicitly supplied
  if (hwndParent || lpsetupdlg->aAttr[KEY_DESC].fSupplied )
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KDESC,
				 lpsetupdlg->aAttr[KEY_DESC].szAttr,
				 ODBC_INI);
  if (hwndParent || lpsetupdlg->aAttr[KEY_DB].fSupplied )
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KDB,
				 lpsetupdlg->aAttr[KEY_DB].szAttr,
				 ODBC_INI);
#ifdef SWSOFT
  if (hwndParent || lpsetupdlg->aAttr[KEY_SYSDB].fSupplied )
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KSYSDB,
				 lpsetupdlg->aAttr[KEY_SYSDB].szAttr,
				 ODBC_INI);
#endif
  if (hwndParent || lpsetupdlg->aAttr[KEY_SERVER].fSupplied )
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KSERVER,
				 lpsetupdlg->aAttr[KEY_SERVER].szAttr,
				 ODBC_INI);
  if (hwndParent || lpsetupdlg->aAttr[KEY_USER].fSupplied )
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KUSER,
				 lpsetupdlg->aAttr[KEY_USER].szAttr,
				 ODBC_INI);
  if (hwndParent || lpsetupdlg->aAttr[KEY_PASSWORD].fSupplied )
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KPASSWORD,
				 lpsetupdlg->aAttr[KEY_PASSWORD].szAttr,
				 ODBC_INI);
  if (hwndParent || lpsetupdlg->aAttr[KEY_PORT].fSupplied )
  {
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KPORT,
				 lpsetupdlg->aAttr[KEY_PORT].szAttr,
				 ODBC_INI);
  }
  if (hwndParent || lpsetupdlg->aAttr[KEY_FLAG].fSupplied )
  {
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KFLAG,
				 lpsetupdlg->aAttr[KEY_FLAG].szAttr,
				 ODBC_INI);
  }
  if (hwndParent || lpsetupdlg->aAttr[KEY_STMT].fSupplied )
  {
    SQLWritePrivateProfileString(lpszDSN,
				 INI_KSTMT,
				 lpsetupdlg->aAttr[KEY_STMT].szAttr,
				 ODBC_INI);
  }

  // If the data source name has changed, remove the old name
  if (lpsetupdlg->aAttr[KEY_DSN].fSupplied &&
      lstrcmpi(lpsetupdlg->szDSN, lpsetupdlg->aAttr[KEY_DSN].szAttr))
  {
    SQLRemoveDSNFromIni(lpsetupdlg->szDSN);
  }
  return TRUE;
}


/*****************************************************************************
** Connect dialog for driver connect
*****************************************************************************/

BOOL FAR PASCAL DriverConnectProc(HWND hdlg, WORD wMsg,	WPARAM wParam,
				  LPARAM lParam)
{
  LPSETUPDLG lpsetupdlg;

  switch (wMsg) {
  case WM_INITDIALOG:
	  {
		  RECT rcIniti;
		  RECT rcFrame;
		  HWND hFrame;
		  
		  
		  GetWindowRect(hdlg,&rcIniti);
		  g_iInitSize = rcIniti.bottom - rcIniti.top;
		  
		  hFrame = GetDlgItem(hdlg, IDC_FRM);
		  GetWindowRect(hFrame,&rcFrame);
		  if (rcFrame.top - rcIniti.top != rcIniti.bottom - rcIniti.top)
		  {
			  SetWindowPos(hdlg,NULL,0,0,rcIniti.right-rcIniti.left, rcFrame.top - rcIniti.top,SWP_NOMOVE);
			  SetDlgItemText(hdlg, IDC_SWITCH,"More >>");
		  }
		  
		  //    Ctl3dRegister (s_hModule);
		  SetWindowLong(hdlg, DWL_USER, lParam);
		  /*
		  #ifdef WIN32
		  Ctl3dSubclassDlg(hdlg, CTL3D_ALL);
		  #else
		  Ctl3dSubclassDlgEx(hdlg, CTL3D_ALL);
		  #endif
		  */
		  CenterDialog(hdlg);
		  lpsetupdlg = (LPSETUPDLG) lParam;
#ifdef SWSOFT_LIB
		  set_dlg_items(hdlg,lpsetupdlg,TRUE);
#else
		  set_dlg_items(hdlg,lpsetupdlg);
#endif
		  
		  SendDlgItemMessage(hdlg, IDC_DSNAME,
			  EM_LIMITTEXT, (WPARAM)(MAXDSNAME-1), 0L);
		  SendDlgItemMessage(hdlg, IDC_DESC,
			  EM_LIMITTEXT, (WPARAM)(MAXDESC-1), 0L);
		  SendDlgItemMessage(hdlg, IDC_DB,
			  EM_LIMITTEXT, (WORD)(NAME_LEN), 0L);
		  SendDlgItemMessage(hdlg, IDC_SERVER,
			  EM_LIMITTEXT, (WORD)(MAXSERVER-1), 0L);
		  SendDlgItemMessage(hdlg, IDC_USER,
			  EM_LIMITTEXT, (WORD)(MAXUSER-1), 0L);
		  SendDlgItemMessage(hdlg, IDC_PASSWORD,
			  EM_LIMITTEXT, (WORD)(MAXPASSWORD-1), 0L);
		  SendDlgItemMessage(hdlg, IDC_PORT,
			  EM_LIMITTEXT, (WORD)5, 0L);
		  SendDlgItemMessage(hdlg, IDC_STMT,
			  EM_LIMITTEXT, (WORD) (MAXSTMT-1), 0L);
		  return TRUE;
	  }
/*
#ifdef WIN32
  case WM_CTLCOLORBTN:
  case WM_CTLCOLORDLG:
  case WM_CTLCOLOREDIT:
  case WM_CTLCOLORLISTBOX:
  case WM_CTLCOLORMSGBOX:
  case WM_CTLCOLORSCROLLBAR:
  case WM_CTLCOLORSTATIC:
    return (BOOL)Ctl3dCtlColorEx(wMsg, wParam, lParam);

  case WM_SETTEXT:
  case WM_NCPAINT:
  case WM_NCACTIVATE:
    SetWindowLong(hdlg, DWL_MSGRESULT,
		  Ctl3dDlgFramePaint(hdlg, wMsg, wParam, lParam));
    return TRUE;
#endif

  case WM_SYSCOLORCHANGE:
    return Ctl3dColorChange();
*/
  case WM_HELP:
	  {
		  RECT cr,rcFrame;
		  HWND hFrame;
		  
		  GetWindowRect(hdlg,&cr);
		  hFrame = GetDlgItem(hdlg, IDC_FRM);
		  GetWindowRect(hFrame,&rcFrame);
		  if (rcFrame.top - cr.top != cr.bottom - cr.top)
		  {
			  ShowHelp(hdlg,"Overview.htm");
		  }
		  else
		  {
			  ShowHelp(hdlg,"Overview.htm");
		  }
		  
		  return TRUE;
	  }
  case WM_COMMAND:
    switch (GET_WM_COMMAND_ID(wParam, lParam)) {
    case IDC_DSNAME:
		if (GET_WM_COMMAND_CMD(wParam, lParam) == EN_CHANGE)
		{
			char	szItem[MAXDSNAME];		// Edit control text
			int sdnlen = GetDlgItemText(hdlg, IDC_DSNAME,  szItem, sizeof(szItem));
			int systablen = setupdlg_db_type() == SETUPDLG_BTRIEVE ? GetDlgItemText(hdlg, IDC_SYSDB,  szItem, sizeof(szItem)) : 1;
			// Enable/disable the OK button
			EnableWindow(GetDlgItem(hdlg, IDOK),  sdnlen && systablen);
			
			//Hide table button if path to system table is empty
			EnableWindow(GetDlgItem(hdlg, IDC_ADD), sdnlen && systablen);
			//Hide test button if path to system table is empty
			EnableWindow(GetDlgItem(hdlg, IDC_TEST),sdnlen && systablen);
			
			return TRUE;
		}
		break;
		
	case IDC_SYSDB:
		if (GET_WM_COMMAND_CMD(wParam, lParam) == EN_CHANGE)
		{
			
			char	szItem[MAXDSNAME];		// Edit control text
			int sdnlen = GetDlgItemText(hdlg, IDC_DSNAME,  szItem, sizeof(szItem));
			int systablen = setupdlg_db_type() == SETUPDLG_BTRIEVE ? GetDlgItemText(hdlg, IDC_SYSDB,  szItem, sizeof(szItem)) : 1;
			// Enable/disable the OK button
			EnableWindow(GetDlgItem(hdlg, IDOK),  sdnlen && systablen);
			
			//Hide table button if path to system table is empty
			EnableWindow(GetDlgItem(hdlg, IDC_ADD), sdnlen && systablen);
			//Hide test button if path to system table is empty
			EnableWindow(GetDlgItem(hdlg, IDC_TEST),sdnlen && systablen);
			
			return TRUE;
		}
		break;

#ifdef SWSOFT
	case IDC_SWITCH:
		{
			RECT cr,rcFrame;
			HWND hFrame;
			
			GetWindowRect(hdlg,&cr);
			hFrame = GetDlgItem(hdlg, IDC_FRM);
			GetWindowRect(hFrame,&rcFrame);
			if (rcFrame.top - cr.top != cr.bottom - cr.top)
			{
				SetWindowPos(hdlg,NULL,0,0,cr.right-cr.left, rcFrame.top - cr.top,SWP_NOMOVE);
				SetDlgItemText(hdlg, IDC_SWITCH,"More >>");
			}
			else
			{
				SetWindowPos(hdlg,NULL,0,0,cr.right-cr.left, g_iInitSize ,SWP_NOMOVE);
				SetDlgItemText(hdlg, IDC_SWITCH,"<< Less");
			}
			
			
			return TRUE;
		}
		break;
#endif
    case IDOK:
    {
      ulong flag;
      uint i;
      lpsetupdlg = (LPSETUPDLG)GetWindowLong(hdlg, DWL_USER);

      GetDlgItemText(hdlg, IDC_DSNAME,
		     lpsetupdlg->aAttr[KEY_DSN].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_DSN].szAttr));
      GetDlgItemText(hdlg, IDC_DB,
		     lpsetupdlg->aAttr[KEY_DB].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_DB].szAttr));
#ifdef SWSOFT
      GetDlgItemText(hdlg, IDC_SYSDB,
		     lpsetupdlg->aAttr[KEY_SYSDB].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_SYSDB].szAttr));
#endif
      GetDlgItemText(hdlg, IDC_SERVER,
		     lpsetupdlg->aAttr[KEY_SERVER].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_SERVER].szAttr));
      GetDlgItemText(hdlg, IDC_USER,
		     lpsetupdlg->aAttr[KEY_USER].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_USER].szAttr));
      GetDlgItemText(hdlg, IDC_PASSWORD,
		     lpsetupdlg->aAttr[KEY_PASSWORD].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_PASSWORD].szAttr));
      GetDlgItemText(hdlg, IDC_PORT,
		     lpsetupdlg->aAttr[KEY_PORT].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_PORT].szAttr));
      GetDlgItemText(hdlg, IDC_STMT,
		     lpsetupdlg->aAttr[KEY_STMT].szAttr,
		     sizeof(lpsetupdlg->aAttr[KEY_STMT].szAttr));

      flag=0;
      for (i=0 ; i < NUMOFBITS ; i++)
	if (IsDlgButtonChecked(hdlg, IDC_CHECK1+i) == BST_CHECKED)
	  flag|= 1 << i;
      sprintf(lpsetupdlg->aAttr[KEY_FLAG].szAttr,"%lu",flag);

#ifdef SWSOFT
	  if( setupdlg_db_type() == SETUPDLG_BTRIEVE && ( GET_WM_COMMAND_ID(wParam, lParam) == IDOK || MessageBox(hdlg, "Do you want to save changes?", "Colibri", MB_ICONQUESTION | MB_YESNO | MB_DEFBUTTON2 ) == IDYES ) )
	  switch( HasSystemTable(lpsetupdlg->aAttr[KEY_SYSDB].szAttr) )
	  {
	  case -1:
		  if( MessageBox( hdlg, "Wrong path. Do you want to continue?", lpsetupdlg->aAttr[KEY_SYSDB].szAttr, MB_ICONWARNING | MB_YESNO) == IDNO )
			  return TRUE;
		  break;
	  case 0:
		  CreateSystemTable( TRUE, TRUE, lpsetupdlg->aAttr[KEY_SYSDB].szAttr, hdlg );
		  break;
	  }

//      Ctl3dUnregister (s_hModule);
      EndDialog(hdlg, wParam);
      return TRUE;
#endif
    }
    case IDCANCEL:
//      Ctl3dUnregister (s_hModule);
      EndDialog(hdlg, GET_WM_COMMAND_ID(wParam, lParam) == IDOK);
      return TRUE;
#ifdef SWSOFT
	case IDC_BROWSE:
		DoBrowse( hdlg );
		return TRUE;
	case IDC_ADD:
		{
			char szPath[255];
			GetDlgItemText(hdlg, IDC_SYSDB,szPath,	255);
			
			switch( HasSystemTable( szPath) )
			{
			case -1:
				MessageBox( hdlg, "Wrong path",  szPath, MB_ICONWARNING );
				return TRUE;
			case 0:
				if( CreateSystemTable( TRUE, FALSE, szPath, hdlg ) != 0 )
					return TRUE;
			}
			AddRemoveTable(hdlg);
			
			return TRUE;
		}
	case IDHELP:
		{
			RECT cr,rcFrame;
			HWND hFrame;

			GetWindowRect(hdlg,&cr);
			hFrame = GetDlgItem(hdlg, IDC_FRM);
			GetWindowRect(hFrame,&rcFrame);
			if (rcFrame.top - cr.top != cr.bottom - cr.top)
			{
				ShowHelp(hdlg,"Overview.htm");
			}
			else
			{
				ShowHelp(hdlg,"Overview.htm");
			}
			return TRUE;
		}
	case IDC_TEST:
		{
			BOOL bIsSucc,bIsHasSysTab;
			char szPath[255];
			GetDlgItemText(hdlg, IDC_SYSDB,szPath,	255);

			if( !HasSystemTable(szPath) )
				CreateSystemTable( TRUE, FALSE, szPath, hdlg );

			SetCursor( LoadCursor(NULL, IDC_WAIT) );			
			bIsHasSysTab = HasSystemTable(szPath);				
			bIsSucc = CheckDSN(hdlg);
			SetCursor( LoadCursor(NULL, IDC_ARROW) );

			if (bIsSucc && bIsHasSysTab)
			{
				MessageBox(hdlg,"Connection successfull","Test DSN", MB_OK | MB_ICONINFORMATION);
			}
			else
			{
				MessageBox(hdlg,"Connection failed","Test DSN", MB_OK | MB_ICONEXCLAMATION);
			}
		
		}
#endif
    }
  }
  return FALSE;
}
