/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"

static const MAX_BITS = 1008;

static const TYPE_CHAR = 1;
static const TYPE_SLONG = 3;

// Code ----------------------------------------------------------------------

// CRowset::CRowset ----------------------------------------------------------
//
// @mfunc Constructor for this class
//
// @rdesc NONE
//
CRowset::CRowset
    (
    LPUNKNOWN pUnkOuter         //@parm IN | Outer Unkown Pointer
    )
{
	CLEAR_CONSTRUCT( CRowset );
	TRACE( "CRowset::CRowset" );

    //  Initialize simple member vars
	m_pUnkOuter		= pUnkOuter ? pUnkOuter : this;
	m_bStartPosition	= true;

    // Increment global object count.
    OBJECT_CONSTRUCTED();

    return;
}


// CRowset::~CRowset ---------------------------------------------------------
//
// @mfunc Destructor for this class
//
// @rdesc NONE
//

CRowset:: ~CRowset
    (
    void
    )
{
	TRACE( "CRowset:: ~CRowset" );
	//First of all proceed notification
	// Send notification on DBREASON_ROWSET_RELEASE
	OnRowsetChange(DBREASON_ROWSET_RELEASE, DBEVENTPHASE_DIDEVENT, TRUE);

	//Decrement count of recordset in command
	if (m_pCommand)
		m_pCommand->DecrementRowset();

    // Release all column names (pointed to by m_rgdbcolinfo).
	if (m_rgdbcolinfo != NULL)
	{
		for (DWORD i = 1; i < m_cCols; i++)
			delete [] m_rgdbcolinfo[i].pwszName;
	}

    // Free pointers.
    // (Note delete is safe for NULL ptr.)
    delete [] m_rgdbcolinfo;
    delete [] m_rgdwDataOffsets;
    delete    m_prowbitsIBuffer;
    delete    m_pUtilProp;

    if (NULL != m_pIBuffer)
        ReleaseSlotList( m_pIBuffer );

    // Free accessors.
    // Each accessor is allocated via new/delete.
    // We store an array of ptrs to each accessor (m_pextbufferAccessor).
    if (NULL != m_pExtBufferAccessor)
    {
        ULONG       hAccessor, hAccessorLast;
        PACCESSOR   pAccessor;

        m_pExtBufferAccessor->GetFirstLastItemH( hAccessor, hAccessorLast );
        for (; hAccessor <= hAccessorLast; hAccessor++)
        {
            m_pExtBufferAccessor->GetItemOfExtBuffer( hAccessor, &pAccessor );
            delete [] pAccessor;
        }
    
		delete m_pExtBufferAccessor;
    }

    //  Free contained interfaces
    delete m_pIColumnsInfo;
    delete m_pIRowset;
    delete m_pIRowsetChange;
    delete m_pIAccessor;
    delete m_pIRowsetInfo;
    delete m_pIConvertType;
	delete m_pIConnectionPointContainer;
	delete m_pIRowsetLocate;
	delete m_pIRowsetFind;
	delete m_pIRowsetScroll;
	delete m_pIRowsetIdentity;
	delete m_pIRowsetIndex;

	// free Connection Point
	if (m_pIRowsetNotifyConnectionPoint)
		delete m_pIRowsetNotifyConnectionPoint;

    // free CData object
    if (m_pData)
        delete m_pData;

	// Decrement the DBSession Count.  GetSpecification is not possible anymore
	if (m_pSession != NULL)
	{
		// Mark the session as not having an open rowset anymore
		// m_pCreator->RowsetDestroyed();
		m_pSession->RemoveRowset(m_cNRowset);
		m_pSession->Release();
	}

	//Decrement creator object count
	if (m_pCreator != NULL)
		m_pCreator->Release();

    // Decrement global object count.
    OBJECT_DESTRUCTED();

    return;
}

// CRowset::Init ------------------------------------------------------------
//
// @mfunc Initialize the rowset Object
//
// @rdesc Did the Initialization Succeed
//      @flag  TRUE | Initialization succeeded
//      @flag  FALSE | Initialization failed
//
BOOL CRowset::Init
    (
	ULONG		ulRowset,		//@parm IN	| # of Rowset
    CData*		pData,			//@parm IN	| pointer to CData object
	UINT		enumAdvanced,	//@parm IN	| Does rowset support advanced interfaces?
	CCommand*	pCommand		//@parm IN	| ptr to parent command
    )
{
	TRACE( "CRowset::Init" );
	//Decrement count of recordset in command
	if (pCommand != NULL)
	{
		m_pCommand = pCommand;
		m_pCommand->IncrementRowset();
	}

    LPUNKNOWN   pIUnknown = (LPUNKNOWN) this;
	ULONG		PropIndex;

    if (m_pUnkOuter)
        pIUnknown = m_pUnkOuter;

	// Establish # of Rowset
	m_cNRowset = ulRowset;

	// Establish pointer to CData
    m_pData = pData;

    // allocate utility object that manages our properties
	if( m_pSession )
		m_pUtilProp = new CUtilProp( m_pSession->m_pCDataSource, m_pSession->m_pUtilProp );
	else
		m_pUtilProp = new CUtilProp();

    // Find # of columns in the result set + Column for bookmark
	m_pData->GetColumnCnt(&m_cCols);
    if (m_cCols <= 0)
        return FALSE;
    m_cCols++;

	for (int i = 0; i < ADVISELIMIT; i++)
		m_Connections[i] = NULL;

    //---------------------
    // Allocate info arrays
    //---------------------

    m_rgdbcolinfo     = new DBCOLUMNINFO[m_cCols];
    m_rgdwDataOffsets = new ULONG[m_cCols];

    if (m_rgdbcolinfo == NULL || m_rgdwDataOffsets == NULL)
        return FALSE;

	m_fBookmarkCol = false;

	// Check if we need to provide user with bookmarks.
	if (m_pUtilProp -> GetPropIndex(DBPROP_BOOKMARKS, &PropIndex) == TRUE)
		m_fBookmarkCol = m_pUtilProp -> m_rgproperties[PropIndex].boolVal != 0;

	// Get maximum number of rows
	if (m_pUtilProp -> GetPropIndex(DBPROP_MAXROWS, &PropIndex) == TRUE)
	{
		m_cMaxRows = m_pUtilProp->m_rgproperties[PropIndex].longVal;
		if( !m_cMaxRows )
			m_cMaxRows = MAXLONG;
	}

    if (FAILED( GatherColumnInfo()))
        return FALSE;

    if (FAILED( CreateHelperFunctions()))
        return FALSE;

    // Allocate contained interface objects
    // Note that our approach is simple - we always create *all* of the Rowset interfaces
    // If our properties were read\write (i.e., could be set), we would need to
    // consult properties to known which interfaces to create.
    // Also, none of our interfaces conflict. If any did conflict, then we could
    // not just blindly create them all.
    m_pIColumnsInfo             = new CImpIColumnsInfo(&m_cCols, &m_fBookmarkCol, &m_rgdbcolinfo, NULL, pIUnknown);
    m_pIRowset                  = new CImpIRowset(this, pIUnknown);
    m_pIAccessor                = new CImpIAccessor(m_pExtBufferAccessor, this, pIUnknown);
    m_pIRowsetInfo              = new CImpIRowsetInfo(this, pIUnknown);
	m_pIConvertType				= new CImpIConvertType(pIUnknown);
	m_pIConnectionPointContainer = new CImpIConnectionPointContainer(this, pIUnknown);
	m_pIRowsetLocate			= new CImpIRowsetLocate(this, pIUnknown);
	m_pIRowsetIdentity			= new CImpIRowsetIdentity(this, pIUnknown);
	
	if (enumAdvanced & ROWSET_FIND)
	{
		m_pIRowsetFind = new CImpIRowsetFind(this, pIUnknown);
		if (m_pIRowsetFind == NULL)
			return false;
	}

	if (enumAdvanced & ROWSET_CHANGE)
	{
		m_pIRowsetChange = new CImpIRowsetChange(this, pIUnknown);
		if (m_pIRowsetChange == NULL)
			return false;

	}

	if (enumAdvanced & ROWSET_SCROLL)
	{
		m_pIRowsetScroll = new CImpIRowsetScroll(this, pIUnknown);	
		if (m_pIRowsetScroll == NULL)
			return false;
	}

	if (enumAdvanced & ROWSET_INDEX)
	{
		m_pIRowsetIndex	 = new CImpIRowsetIndex(this, pIUnknown);	
		if (m_pIRowsetIndex == NULL)
			return false;
	}

    // if all interfaces were created, return success
    return (BOOL) (m_pIColumnsInfo &&
                   m_pIRowset &&
                   m_pIAccessor &&
                   m_pIRowsetInfo &&
				   m_pIConvertType &&
				   m_pIConnectionPointContainer &&
				   m_pIRowsetLocate &&
				   m_pIRowsetIdentity);
}


// CRowset::QueryInterface ---------------------------------------------------
//
// @mfunc Returns a pointer to a specified interface. Callers use
// QueryInterface to determine which interfaces the called object
// supports.
//
// @rdesc HRESULT indicating the status of the method
//      @flag S_OK | Interface is supported and ppvObject is set.
//      @flag E_NOINTERFACE | Interface is not supported by the object
//      @flag E_INVALIDARG | One or more arguments are invalid.
//
STDMETHODIMP CRowset::QueryInterface
    (
    REFIID riid,
    LPVOID * ppv
    )
{
    TRACE( "CRowset::QueryInterface" );
	if (NULL == ppv)
        return E_INVALIDARG;

    //  Place NULL in *ppv in case of failure
    *ppv = NULL;

    //  This is the non-delegating IUnknown implementation
    if (riid == IID_IUnknown)
        *ppv = (LPVOID) this;
    else if (riid == IID_IRowset)
        *ppv = (LPVOID) m_pIRowset;
    else if (riid == IID_IColumnsInfo)
        *ppv = (LPVOID) m_pIColumnsInfo;
    else if (riid == IID_IRowsetChange)
        *ppv = (LPVOID) m_pIRowsetChange;
    else if (riid == IID_IAccessor)
        *ppv = (LPVOID) m_pIAccessor;
    else if (riid == IID_IRowsetInfo)
        *ppv = (LPVOID) m_pIRowsetInfo;
    else if (riid == IID_IConvertType)
        *ppv = (LPVOID) m_pIConvertType;
	else if (riid == IID_IConnectionPointContainer)
		*ppv = (LPVOID) m_pIConnectionPointContainer;
	else if (riid == IID_IRowsetLocate)
		*ppv = (LPVOID) m_pIRowsetLocate;
	else if (riid == IID_IRowsetScroll)
		*ppv = (LPVOID) m_pIRowsetScroll;
	else if (riid == IID_IRowsetFind)
		*ppv = (LPVOID) m_pIRowsetFind;
	else if (riid == IID_IRowsetIdentity)
		*ppv = (LPVOID) m_pIRowsetIdentity;
	else if (riid == IID_IRowsetIndex)
		*ppv = (LPVOID) m_pIRowsetIndex;

    if (*ppv == NULL)
		return E_NOINTERFACE;

    //  If we're going to return an interface, AddRef it first
    ((LPUNKNOWN) *ppv)->AddRef();
    return S_OK;
}


// CRowset::AddRef -----------------------------------------------------------
//
// @mfunc Increments a persistence count for the object
//
// @rdesc Current reference count
//
STDMETHODIMP_( ULONG ) CRowset::AddRef
     (
     void
     )
{
    TRACE( "CRowset::Release" );
	return ++m_cRef;
}


// CRowset::Release ----------------------------------------------------------
//
// @mfunc Decrements a persistence count for the object and if
// persistence count is 0, the object destroys itself.
//
// @rdesc Current reference count
//
STDMETHODIMP_( ULONG ) CRowset::Release
     (
     void
     )
{
    TRACE( "CRowset::Release" );
	if (!--m_cRef)
    {
        if( m_pSession != NULL )
			m_pSession->DecRowsetCount();
        
		delete this;
        return 0;
    }

    return m_cRef;
}

// CRowset::CreateHelperFunctions --------------------------------------------
//
// @mfunc Creates Helper classes that are needed to manage the Rowset Object
//
// @rdesc HRESULT
//      @flag S_OK | Helper classes created
//      @flag E_FAIL | Helper classes were not created
//
HRESULT CRowset::CreateHelperFunctions
    (
    void
    )
{
    TRACE( "CRowset::CreateHelperFunctions" );
	//----------------------
    // Create helper objects
    //----------------------

    // Bit array to track presence/absense of rows.
    m_prowbitsIBuffer = new CBitArray;
    if (m_prowbitsIBuffer == NULL || FAILED( m_prowbitsIBuffer->FInit( MAX_BITS, g_dwPageSize )))
        return E_FAIL;

    // List of free slots.
    // This manages the allocation of sets of contiguous rows.
    if (FAILED( InitializeSlotList( MAX_TOTAL_ROWBUFF_SIZE / m_cbRowSize,
                         m_cbRowSize, g_dwPageSize, m_prowbitsIBuffer,
                         &m_pIBuffer, &m_rgbRowData )))
        return E_FAIL;

    // Create the ExtBuffer array.
    // This is an array of pointers to malloc'd accessors.
    m_pExtBufferAccessor = (LPEXTBUFFER) new CExtBuffer;
    if (m_pExtBufferAccessor == NULL ||
		FAILED( m_pExtBufferAccessor->FInit( 1, sizeof( PACCESSOR ), g_dwPageSize )))
        return E_FAIL;

	// Here we are allocating buffer for our rowset.
	// It will will be static and its size will be ACTIVE_ROWS_LIMIT
	if (FAILED( GetNextSlots(m_pIBuffer, ACTIVE_ROWS_LIMIT + 1, &m_irowMin) ))
		return E_FAIL;

	for ( ULONG i = 0; i < ACTIVE_ROWS_LIMIT; i++ )
	{
		m_SlotRef[ i ].hRow = m_irowMin + i + 1;
		m_SlotRef[ i ].NextSlot = i + 1;
	}
	m_SlotRef[ACTIVE_ROWS_LIMIT - 1].NextSlot = 0xFFFFFFFF;
	m_FirstFreeSlot = 0;

    return S_OK;
}

// CRowset::IsSlotLimitReached -------------------------------------------------
//
// @mfunc
//
// @rdesc NONE
//
BOOL	CRowset::IsSlotLimitReached()
{
	TRACE( "CRowset::IsSlotLimitReached" );
	return (m_FirstFreeSlot == 0xFFFFFFFF)? TRUE : FALSE;
}

// CRowset::GetFreeSlot -------------------------------------------------
//
// @mfunc
//
// @rdesc NONE
//
ULONG	CRowset::GetFreeSlot()
{
	TRACE( "CRowset::GetFreeSlot" );
	ULONG	iSlot;

	iSlot = m_FirstFreeSlot;
	if (iSlot != 0xFFFFFFFF)
	{
		m_FirstFreeSlot = m_SlotRef[iSlot].NextSlot;
		m_SlotRef[iSlot].Status = 1;
		m_prowbitsIBuffer->SetSlots( iSlot, iSlot );
		m_ulRowRefCount++;
	}

	return iSlot;
}

// CRowset::ReleaseSlot -------------------------------------------------
//
// @mfunc
//
// @rdesc NONE
//
void	CRowset::ReleaseSlot(ULONG iSlot)
{
	TRACE( "CRowset::ReleaseSlot" );
	// assert(m_SlotRef[ iSlot ].Status);
	// assert(m_ulRowRefCount);

	m_SlotRef[ iSlot ].NextSlot = m_FirstFreeSlot;
	m_FirstFreeSlot = iSlot;
	// status: slot is free
	m_SlotRef[iSlot].Status = 0;

	m_prowbitsIBuffer->ResetSlots( iSlot, iSlot );
	
	// ATTENTION !!!
	m_ulRowRefCount--;

	return;
}

// CRowset::OnFieldChange
//
// @mfunc	Call consumer which previousely registered itself via IConnectionPoint
//			Inform it about changes made for the Filed(s)
HRESULT	CRowset::OnFieldChange
(
    HROW			hRow,
    ULONG			cColumns,
    ULONG			rgColumns[],
	DBREASON		eReason,
	DBEVENTPHASE	ePhase,
	BOOL			fCantDeny
)
{
	TRACE( "CRowset::OnFieldChange" );
	ULONG	iAdvise;
	HRESULT	hr = S_OK;

	if (!CheckReason(eReason, ePhase))
		return S_OK;

	m_bNotReentrant = true;

	for (iAdvise = 0; iAdvise < ADVISELIMIT; iAdvise++)
	{
		if (m_Connections[iAdvise])
		{
			hr = ((IRowsetNotify*)m_Connections[iAdvise])->OnFieldChange(
							m_pIRowset, hRow, cColumns, rgColumns, eReason, ePhase, fCantDeny );

			if (!fCantDeny && hr == S_FALSE)
				break;
		}
	}

	if (hr == S_FALSE)
	{
		if (ePhase != DBEVENTPHASE_OKTODO || iAdvise > ADVISELIMIT - 1)
			iAdvise = ADVISELIMIT - 1;

		for (int i = 0; i <= iAdvise; i++)
		{
			if (m_Connections[i])
			{
				((IRowsetNotify*)m_Connections[i])->OnFieldChange(
							m_pIRowset, hRow, cColumns, rgColumns, eReason, ePhase, fCantDeny );
			}
		}

		m_bNotReentrant = false;
		return DB_E_CANCELED;
	}

	m_bNotReentrant = false;
	return S_OK;
}

// CRowset::OnRowChange
//
// @mfunc	Call consumer which previousely registered itself via IConnectionPoint
//			Inform it about changes made for the Row(s)
HRESULT	CRowset::OnRowChange
(
	ULONG			cRows,
    const HROW		rghRows[],
	DBREASON		eReason,
	DBEVENTPHASE	ePhase,
	BOOL			fCantDeny
)
{
    TRACE( "CRowset::OnRowChange" );
	ULONG	iAdvise;
	HRESULT	hr = S_OK;

	if (!CheckReason(eReason, ePhase))
		return S_OK;

	m_bNotReentrant = true;

	for ( iAdvise = 0; iAdvise < ADVISELIMIT; iAdvise++)
	{
		if (m_Connections[iAdvise])
		{
			hr = ((IRowsetNotify*)m_Connections[iAdvise])->OnRowChange(
							m_pIRowset, cRows, rghRows, eReason, ePhase, fCantDeny);

			if (!fCantDeny && hr == S_FALSE)
				break;
		}
	}

	if (hr == S_FALSE)
	{
		if (ePhase != DBEVENTPHASE_OKTODO || iAdvise > ADVISELIMIT - 1)
			iAdvise = ADVISELIMIT - 1;

		for (int i = 0; i <= iAdvise; i++)
		{
			if (m_Connections[i])
			{
				((IRowsetNotify*)m_Connections[i])->OnRowChange(
								m_pIRowset, cRows, rghRows, eReason, DBEVENTPHASE_FAILEDTODO, TRUE);
			}
		}

		m_bNotReentrant = false;
		return DB_E_CANCELED;
	}

	m_bNotReentrant = false;
	return S_OK;
}



// CRowset::OnRowsetChange
//
// @mfunc	Call consumer which previousely registered itself via IConnectionPoint
//			Inform it about changes made for the Rowset
HRESULT	CRowset::OnRowsetChange
(
	DBREASON		eReason,
	DBEVENTPHASE	ePhase,
	BOOL			fCantDeny
)
{
	TRACE( "CRowset::OnRowsetChange" );
	ULONG	iAdvise;
	HRESULT	hr = S_OK;

	if (!CheckReason(eReason, ePhase))
		return S_OK;

	m_bNotReentrant = true;

	//Notificate
	for (iAdvise = 0; iAdvise < ADVISELIMIT; iAdvise++)
	{
		if (m_Connections[iAdvise])
		{
			hr = ((IRowsetNotify*)m_Connections[iAdvise])->OnRowsetChange(
							m_pIRowset, eReason, ePhase, fCantDeny );
			
			if (!fCantDeny && hr == S_FALSE)
				break;
		}
	}

	if (hr == S_FALSE)
	{
		if (ePhase != DBEVENTPHASE_OKTODO || iAdvise > ADVISELIMIT - 1)
			iAdvise = ADVISELIMIT - 1;

		for (int i = 0; i <= iAdvise; i++)
		{
			if (m_Connections[i])
			{
				((IRowsetNotify*)m_Connections[i])->OnRowsetChange(
								m_pIRowset, eReason, DBEVENTPHASE_FAILEDTODO, TRUE);
			}
		}

		m_bNotReentrant = false;
		return DB_E_CANCELED;
	}

	m_bNotReentrant = false;
	return S_OK;
}


//Check if we need to continue doing notifications
bool CRowset::CheckReason
		(
		DBREASON		eReason,
		DBEVENTPHASE	ePhase
		)
{
	TRACE( "CRowset::CheckReason" );
	ULONG nProperty = 0;
	ULONG ulPropIndex;

	switch (eReason)
	{
	case DBREASON_ROWSET_FETCHPOSITIONCHANGE:
		nProperty = DBPROP_NOTIFYROWSETFETCHPOSITIONCHANGE;
		break;
	
	case DBREASON_ROWSET_RELEASE:
		nProperty = DBPROP_NOTIFYROWSETRELEASE;
		break;

	case DBREASON_ROW_RELEASE:
	case DBREASON_ROW_ACTIVATE:
		break;
	
	case DBREASON_COLUMN_SET:
		nProperty = DBPROP_NOTIFYCOLUMNSET;
		break;

	case DBREASON_ROW_DELETE:
		nProperty = DBPROP_NOTIFYROWDELETE;
		break;
	
	case DBREASON_ROW_INSERT:
		nProperty = DBPROP_NOTIFYROWINSERT;
		break;

	default:
		assert("Unknown reason");
	}

	//Check if we need to notificate
	if (nProperty != 0 && m_pUtilProp->GetPropIndex(nProperty, &ulPropIndex))
	{
		ULONG npPhase = 0;

		switch (ePhase)
		{
		case DBEVENTPHASE_OKTODO:
			npPhase = DBPROPVAL_NP_OKTODO;
			break;

		case DBEVENTPHASE_ABOUTTODO:
			npPhase = DBPROPVAL_NP_ABOUTTODO;
			break;

		case DBEVENTPHASE_FAILEDTODO:
			npPhase = DBPROPVAL_NP_FAILEDTODO;
			break;

		case DBEVENTPHASE_SYNCHAFTER:
			npPhase = DBPROPVAL_NP_SYNCHAFTER;
			break;

		case DBEVENTPHASE_DIDEVENT:
			npPhase = DBPROPVAL_NP_DIDEVENT;
			break;
		
		default:
			assert("Unknown phase");
		}

		return (m_pUtilProp->m_rgproperties[ulPropIndex].longVal & npPhase) == npPhase;
	}

	return true;
}

// CRowset::GatherColumnInfo -------------------------------------------------
//
// @mfunc Creates DBCOLINFO structures for each column in the result set.
//
// @rdesc HRESULT
//      @flag S_OK | Column Info Obtained
//      @flag E_FAIL | Problems getting Column Info
//
HRESULT CRowset::GatherColumnInfo
    (
    void
    )
{
    TRACE( "CRowset::GatherColumnInfo" );
	DBCOLUMNINFO*   pcolinfo;
    ULONG           dwOffset;
    UWORD           icol;
	static WCHAR*	wszBookmarkName= L"Bookmark";

    //----------------------------------
    // Gather column info
    //----------------------------------


	// If yes then we are to fill out boomark column information
	if (m_fBookmarkCol)
	{
		ULONG ulBmks = m_pData->bookmarks();
		
		pcolinfo = &m_rgdbcolinfo[0];
		pcolinfo->iOrdinal		= 0;
        pcolinfo->pTypeInfo     = NULL;
		pcolinfo->bPrecision	= (BYTE) ~0;
		pcolinfo->bScale		= (BYTE) ~0;
        
		pcolinfo->wType			= ulBmks == 1 ? DBTYPE_I4 : DBTYPE_BYTES;
        pcolinfo->ulColumnSize  = BMKS_SIZE * ulBmks;
		pcolinfo->dwFlags		= DBCOLUMNFLAGS_ISBOOKMARK;
		if( ulBmks == 1 )
			pcolinfo->dwFlags |= DBCOLUMNFLAGS_ISFIXEDLENGTH;

	    // We use ordinal numbers for our columns
        pcolinfo->columnid.eKind          = DBKIND_PROPID;
        pcolinfo->columnid.uGuid.guid     = GUID_NULL;
        pcolinfo->columnid.uName.ulPropid = 0;
		pcolinfo->pwszName = wszBookmarkName;

		m_rgdwDataOffsets[0]	= offsetof( ROWBUFF, cbBmk );
	}

    pcolinfo = &m_rgdbcolinfo[1];
    dwOffset = offsetof( ROWBUFF, cdData );

    for (icol = 1; icol < m_cCols; icol++, pcolinfo++)
    {
		m_pData->GetColumnInfo(icol, pcolinfo );

        // Set the offset from the start of the row,
        // for this column, then advance past.
        dwOffset = ROUND_UP( dwOffset, COLUMN_ALIGNVAL );
        m_rgdwDataOffsets[icol] = dwOffset;
        dwOffset += offsetof(COLUMNDATA, bData) + pcolinfo->ulColumnSize;
    }

    m_cbRowSize = ROUND_UP( dwOffset, COLUMN_ALIGNVAL );
    return S_OK;
}


// CRowset::GetRowBuff--------------------------------------------
//
// @mfunc Shorthand way to get the address of a row buffer.
// Later we may extend this so that it can span several
// non-contiguous blocks of memory.
//
// @rdesc Pointer to the buffer.
//
ROWBUFF* CRowset::GetRowBuff
    (
    ULONG iRow                  //@parm IN | Row to get address of.
    )
{
	// This assumes iRow is valid...
    // How can we calculate maximum row?
    // Should we return NULL if it's out of range?
    assert( m_rgbRowData );
    assert( m_cbRowSize );
    assert( iRow > 0 );
    return (ROWBUFF *) (m_rgbRowData + m_cbRowSize*iRow);
}


// CRowset::Row2Bookmark--------------------------------------------
//
// @mfunc Returns row handle which corresponds to a bookmark
//
// @desc NONE
//
ULONG*	CRowset::Row2Bookmark(HROW hRow)
{
	if (hRow > m_cRows)
		return NULL;
	
	return ((PROWBUFF)GetRowBuff((ULONG) hRow))->dwBmks;
}

// CRowset::Bookmark2Row--------------------------------------------
//
// @mfunc Returns bookmark of a row handle
//
// @desc NONE
//
HROW	CRowset::Bookmark2Row(const ULONG* pBookmark)
{
TRACE( "CRowset::Bookmark2Row" );
HROW		hRow;
PROWBUFF	pRowBuff;

	if (pBookmark)
		for (hRow = 0; hRow < m_cRows; hRow++)
		{
			pRowBuff = (PROWBUFF)GetRowBuff((ULONG) hRow);
			if( !memcmp( pRowBuff->dwBmks, pBookmark, pRowBuff->cbBmk ) )
				return hRow;
		}

	return NULL;
}


// CRowset::IfRowMarkedAsDeleted--------------------------------------------
//
// @mfunc IfRowMarkedAsDeleted
//
// @desc Checks if row with given row handle marked as deleted
//
bool CRowset::IfRowMarkedAsDeleted
	(
	const HROW hRow
	)
{
	    TRACE( "CRowset::IfRowMarkedAsDeleted" );
		BYTE*           pbProvRow;

        // Get RowBuffer to look at which row this applies to
        pbProvRow = (BYTE *) GetRowBuff((ULONG) hRow );

        return (S_OK == IsDeleted((ULONG) PROWBUFF(pbProvRow)->dwBmkStatus));
}



//--------------------------------------------------------------------
// @mfunc Check if the row has already been deleted
//
// @rdesc HRESULT
//      @flag S_OK | Row already deleted
//      @flag S_FALSE | Row not deleted
//
HRESULT CRowset::IsDeleted
    (
    ULONG ulRowBmkStatus                 //@parm IN | Row to Check
    )
{
    TRACE( "CRowset::IsDeleted" );
	// Already deleted
    if (ulRowBmkStatus == BMKS_ROW_DELETED)
        return S_OK;
    else
        return S_FALSE;
}

//--------------------------------------------------------------------
// @mfunc MarkAsDeleted 
// Mark the row as already deleted
//
//		@rdesc HRESULT
//      @flag S_OK | Row marked as deleted
//
HRESULT CRowset::MarkAsDeleted
    (
    ULONG* pulRowBmkStatus                 //@parm IN | Row to Check
    )
{
    TRACE( "CRowset::MarkAsDeleted" );
	// mark as deleted
    *pulRowBmkStatus = BMKS_ROW_DELETED;
    return S_OK;
}


HRESULT CRowset::GoPending(LONG nMove, GoPendingFunction func)
{
	TRACE2( "CRowset::Move %d ", nMove );
	if (nMove == 0)
		return S_OK;

	m_pData->SetMoveInfo( E_FAIL );

	if (m_bStartPosition)
	{
		//Correct position for first move
		m_bStartPosition = false;
		m_bEOF = nMove < 0;
		m_bBOF = nMove > 0;
	}

	if (m_bEOF)
	{
		if (nMove > 0)
			return m_pData->SetMoveInfo( DB_S_ENDOFROWSET );

		HRESULT hr = m_pData->MoveLast();
		if (hr != S_OK)
			return m_pData->SetMoveInfo( hr );

		m_nPreviousMove = 1;
		m_bEOF = false;
	}
	else if (m_bBOF)
	{
		if (nMove < 0)
			return m_pData->SetMoveInfo( DB_S_ENDOFROWSET );

		HRESULT hr = m_pData->MoveFirst();
		if (hr != S_OK)
			return m_pData->SetMoveInfo( hr );

		m_nPreviousMove = -1;
		m_bBOF = false;
	}
	
	HRESULT hr;

	//Correct position for next moves if direction is changed
	if (nMove > 0 && (m_nPreviousMove < 0 || m_bSkipRow))
		hr = (m_pData->*func)(nMove - 1, TRUE);
	else if (nMove < 0 && (m_nPreviousMove > 0 || m_bSkipRow))
		hr = (m_pData->*func)(nMove + 1, TRUE);
	else
		hr = (m_pData->*func)(nMove, FALSE); //Do not correct position

	//Save moving direction
	m_nPreviousMove = nMove;
	m_bSkipRow = false;
	
	if (hr == DB_S_ENDOFROWSET)
	{
		m_bEOF = nMove > 0;
		m_bBOF = nMove < 0;
	}

	return hr;
}


HRESULT CRowset::GetFetchedData(LONG nPosAfterLastMove, DWORD* pdwBmk)
{ 
	HRESULT hr = m_pData->GetFetchedData( nPosAfterLastMove, pdwBmk );

	// This is necessary for internal SQL support (every move is very pending there :) )
	if (hr == DB_S_ENDOFROWSET)
	{
		m_bEOF = m_nPreviousMove > 0;
		m_bBOF = m_nPreviousMove < 0;
	}

	return hr;
}



void CRowset::InitPosition()
{
	TRACE( "CRowset::InitPosition" );
	m_nPreviousMove = 0;
	m_bStartPosition = true;
}


// Move cursor to row specified by bookmark
// This function takes care of predefined bookmarks
HRESULT CRowset::MoveToBookmark
			(
			void *pBookmark,		// IN | ptr to bookmark
			ULONG cbBookmark		// IN | size of bookmark
			)
{
	TRACE( "CRowset::MoveToBookmark" );
	// if pBookmark == NULL, we have not to do anything
	if (pBookmark == NULL)
		return S_OK;

	// Invalidate buffers
	m_pData->SetMoveInfo( E_FAIL );

	m_bStartPosition = false;
	m_nPreviousMove  = 0;

	if (cbBookmark == 1 && *((BYTE*)pBookmark) == DBBMK_FIRST)
	{
		// Set position before first row in the table
		m_bBOF = true;
		m_bEOF = false;
		TRACE("DBBMK_FIRST");
		return S_OK;
	}

	if (cbBookmark == 1 && *((BYTE*)pBookmark) == DBBMK_LAST)
	{
		// Set position after last row in the table
		m_bBOF = false;
		m_bEOF = true;
		TRACE("DBBMK_LAST");
		return S_OK;
	}

	// Set position at the bookmark
	m_bSkipRow = true;
	m_bBOF = false;
	m_bEOF = false;
	return m_pData->MoveBookmark( (ULONG*)pBookmark );
}


// HRow and slot number are different functions
// This function allows to obtain slot number from HROW value
// Returns UNVALID_SLOT_NUMBER if fails.
ULONG CRowset::GetSlotNumber( HROW hRow)
{
	TRACE( "CRowset::GetSlotNumber" );
	for ( ULONG i = 0; i < ACTIVE_ROWS_LIMIT; i++ )
		if ( hRow == m_SlotRef[ i ].hRow )
				return i;
	return INVALID_SLOT_NUMBER;
}


HRESULT CRowset::ApplyAccessor
	(
    HACCESSOR   hAccessor,  //@parm IN | Handle to the accessor to use
	BYTE*		pRow,		//@parm IN | Pointer to the row buffer
    void*		pData,      //@parm IN | Pointer to the new data
	ULONG		ulMaxBind	//@parm IN | Maximum binding to apply
	)
{
#ifdef _DEBUG
	TRACE("IRowsetChange::ApplyAccessor()\n");
#endif

    PACCESSOR	pAccessor;
	PCOLUMNDATA pColumnData;
    DBTYPE      dwSrcType;
    DBTYPE      dwDstType;
    void*       pSrc;
    void*       pDst;
    ULONG       dwSrcLength;
    ULONG*      pdwDstLength;
    ULONG       dwDstMaxLength;
    DWORD       dwSrcStatus;
    DWORD*      pdwDstStatus;
    DWORD       dwPart;
	BYTE        b;
	HRESULT		hr;

    if (m_pExtBufferAccessor == NULL  ||   
        FAILED( m_pExtBufferAccessor->GetItemOfExtBuffer((ULONG) hAccessor, &pAccessor)) ||
        pAccessor == NULL)
    {
        return DB_E_BADACCESSORHANDLE;
    }

	// Nothing to do for null accessors
	if( pAccessor->bNullAccessor )
		return S_OK;

    ULONG		cBindings = pAccessor->cBindings;
    DBBINDING*	pBinding  = pAccessor->rgBindings;

    // Apply accessor to data.
    for (ULONG ibind = 0, dwErrorCount = 0, ulBind = 0; ibind < cBindings && ulBind < ulMaxBind; ibind++)
    {
        ULONG icol = pBinding[ibind].iOrdinal;
		// Note, that icol == 0 is for bookmark
		if (icol == 0)
			continue;

		// Not a self bookmark
		ulBind++;

        pColumnData = (COLUMNDATA *) (pRow + m_rgdwDataOffsets[icol]);

        dwDstType       = m_rgdbcolinfo[icol].wType;
        pDst            = &(pColumnData->bData);
		pdwDstLength	= (ULONG*)&pColumnData->dwLength;
        pdwDstStatus    = &(pColumnData->dwStatus);
        dwDstMaxLength  = m_rgdbcolinfo[icol].ulColumnSize;

        dwPart          = pBinding[ibind].dwPart;
        dwSrcType       = pBinding[ibind].wType;

        if ((dwPart & DBPART_VALUE) == 0)
        {
            if (((dwPart & DBPART_STATUS)
                && (*(ULONG *) ((BYTE*) pData + pBinding[ibind].obStatus) & DBSTATUS_S_ISNULL))
               || ((dwPart & DBPART_LENGTH) && *(ULONG *) ((BYTE*) pData + pBinding[ibind].obLength) == 0))
            {
                pSrc = &b;
                b = 0x00;
            }
            else
                return E_FAIL;
        }
        else
        {
			pSrc = (void *) ((BYTE*) pData + pBinding[ibind].obValue);
        }

        dwSrcLength = (dwPart & DBPART_LENGTH) ? *(ULONG *) ((BYTE*) pData + pBinding[ibind].obLength) : 0;

        dwSrcStatus = (dwPart & DBPART_STATUS) ? *(ULONG *) ((BYTE*) pData + pBinding[ibind].obStatus)
            : DBSTATUS_S_OK;

		//Convert data
		hr = g_pIDataConvert->DataConvert(
				dwSrcType,
				dwDstType,
				dwSrcLength,
				pdwDstLength,
				pSrc,
				pDst,
				dwDstMaxLength,
				dwSrcStatus,
				pdwDstStatus,
				0,	// bPrecision for conversion to DBNUMERIC
				0,	// bScale for conversion to DBNUMERIC
				//DBDATACONVERT_SETDATABEHAVIOR
				//DBDATACONVERT_DEFAULT
				DBDATACONVERT_LENGTHFROMNTS);

	    if (FAILED( hr ))
		{
#ifdef _DEBUG
		switch(hr)
			{
			case E_FAIL:			// fatal error
				TRACE("fatal error");
	            break;  
			case E_OUTOFMEMORY:		// unable to allocate memory
				TRACE("unable to allocate memory");
	            break;  
			case DB_E_BADBINDINFO:	// the source or destination data type is invalid
				TRACE("the source or destination data type is invalid");
	            break;  
			case DB_E_DATAOVERFLOW:	// the request conversion result in overflow
				TRACE("the request conversion result in overflow");
	            break;  
			case DB_E_ERRORSOCCURRED:			// errors occurd while converting data
				TRACE("errors occured while converting data");
				break;  
			case DB_E_UNSUPPORTEDCONVERSION:	// errors occured while converting data
				TRACE("errors occured while converting data");
	            break;  
			default:	
				TRACE("unrecognizable error");
				break;				
			}
#endif
            return hr;  // fatal error
		}

        if (hr != S_OK)
            dwErrorCount++; // rounding or truncation or can't coerce
	} //for

	return dwErrorCount ? DB_S_ERRORSOCCURRED : S_OK;
}




