<?php

namespace App\Exceptions;

use Exception;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Database\QueryException;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use LibreNMS\Exceptions\DatabaseConnectException;

class Handler extends ExceptionHandler
{
    /**
     * A list of the exception types that should not be reported.
     *
     * @var array
     */
    protected $dontReport = [
        \Illuminate\Auth\AuthenticationException::class,
        \Illuminate\Auth\Access\AuthorizationException::class,
        \Symfony\Component\HttpKernel\Exception\HttpException::class,
        \Illuminate\Database\Eloquent\ModelNotFoundException::class,
        \Illuminate\Session\TokenMismatchException::class,
        \Illuminate\Validation\ValidationException::class,
    ];

    /**
     * Report or log an exception.
     *
     * This is a great spot to send exceptions to Sentry, Bugsnag, etc.
     *
     * @param  \Exception  $exception
     * @return void
     */
    public function report(Exception $exception)
    {
        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Exception  $exception
     * @return \Illuminate\Http\Response
     */
    public function render($request, Exception $exception)
    {
        if ($exception instanceof QueryException) {
            // connect exception, convert to our standard connection exception
            if (config('app.debug')) {
                // get message form PDO exception, it doesn't contain the query
                $message = $exception->getMessage();
            } else {
                $message = $exception->getPrevious()->getMessage();
            }

            if (in_array($exception->getCode(), [1044, 1045, 2002])) {
                throw new DatabaseConnectException($message, $exception->getCode(), $exception);
            }
            return response('Unhandled MySQL Error [' . $exception->getCode() . "] $message");
        }

        // emulate Laravel 5.5 renderable exceptions
        if (method_exists($exception, 'render')) {
            return $exception->render($request);
        }

        if (config('app.debug')) {
            return parent::render($request, $exception);
        }

        return view('errors.generic')->with('exception', $exception);
    }

    /**
     * Convert an authentication exception into an unauthenticated response.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Illuminate\Auth\AuthenticationException  $exception
     * @return \Illuminate\Http\Response
     */
    protected function unauthenticated($request, AuthenticationException $exception)
    {
        if ($request->expectsJson()) {
            return response()->json(['error' => 'Unauthenticated.'], 401);
        }

        return redirect()->guest(route('login'));
    }
}
