# Copyright (C) 2007-2012, Parrot Foundation.

=head1 NAME

Parrot::Manifest - Re-create MANIFEST and MANIFEST.SKIP

=head1 SYNOPSIS

    use Parrot::Manifest;

    $mani = Parrot::Manifest->new($0);

    $manifest_lines_ref = $mani->prepare_manifest();
    $need_for_files     = $mani->determine_need_for_manifest($manifest_lines_ref);
    $mani->print_manifest($manifest_lines_ref) if $need_for_files;

    $print_str     = $mani->prepare_manifest_skip();
    $need_for_skip = $mani->determine_need_for_manifest_skip($print_str);
    $mani->print_manifest_skip($print_str) if $need_for_skip;

=head1 DESCRIPTION

This package exports no functions.  A Parrot::Manifest object is used in
F<tools/dev/mk_manifest_and_skip.pl>, which in turn is the basis of C<make
manifest_tests>.

=cut

package Parrot::Manifest;
use strict;
use warnings;
use Carp;
use File::Basename;
use Parrot::BuildUtil;

=head1 METHODS

=head2 new

    $mani = Parrot::Manifest->new({
        script    => $0,
        file      => $filename,
        skip      => $skipfilename,
    })

Creates a Parrot::Manifest object by asking C<git status> for verbose output,
and parsing the results.

C<file> is the name of the file that the manifest will eventually be written
to, and defaults to F<MANIFEST>.  C<skip> is the name of the file that will
hold the list of files to be skipped, and defaults to F<MANIFEST.SKIP>.

=cut

sub new {
    my $class   = shift;
    my $argsref = shift;

    my %data = (
        time       => scalar gmtime,
        cmd        => 'git',
        script     => $argsref->{script},
        file       => $argsref->{file}      ? $argsref->{file}      : q{MANIFEST},
        skip       => $argsref->{skip}      ? $argsref->{skip}      : q{MANIFEST.SKIP},
    );

    my $lsfiles = qx($data{cmd} ls-files );

    # grab the versioned resources:
    my @versioned_files;
    my @versioned_output = split /\n/, $lsfiles;
    for my $filename (@versioned_output) {
        next if $filename =~ m[/\.git|^blib|^ports];
        push @versioned_files, $filename;
    }
    $data{versioned_files} = \@versioned_files;

    return bless( \%data, $class );

}

=head2 prepare_manifest

    $manifest_lines_ref = $mani->prepare_manifest();

Prepares the manifest from the data read in by the C<new()> method, and
returns a hash of the files.  The keys of the hash are the filenames, and the
values are strings representing the package and a list of the meta flags.

=cut

sub prepare_manifest {
    my $self = shift;

    my %manifest_lines;
    for my $file ( @{ $self->{versioned_files} } ) {
        $manifest_lines{$file} = _get_manifest_entry($file);
    }

    return \%manifest_lines;
}

=head2 determine_need_for_manifest

    $need_for_files =
        $mani->determine_need_for_manifest($manifest_lines_ref);

Determines the need for the manifest.  The checks are:

=over 4

=item *

If there's no manifest yet, we need one.

=item *

If there's a difference between what's already there and what's in the list,
we need a new one.

=back

If a new manifest is needed, the return value is C<1>; otherwise it is
undefined.  The value passed in is the hash as returned from I<e.g.>,
C<prepare_manifest()>.

=cut

sub determine_need_for_manifest {
    my $self               = shift;
    my $proposed_files_ref = shift;

    return 1 unless -f $self->{file};

    my $current_files_ref        = $self->_get_current_files();
    my $different_patterns_count = 0;
    my %missing;
    foreach my $cur ( keys %{$current_files_ref} ) {
        unless ($proposed_files_ref->{$cur}) {
            $different_patterns_count++;
            $missing{"+".$cur}++;
        }
    }
    foreach my $pro ( keys %{$proposed_files_ref} ) {
        unless ($current_files_ref->{$pro}) {
            $different_patterns_count++;
            $missing{"-".$pro}++;
        }
    }

    $different_patterns_count ? return \%missing : return;
}

=head2 print_manifest

    $mani->print_manifest($manifest_lines_ref) if $need_for_files;

Writes the manifest to a file.  The example above does so only if an update is
needed.

=cut

my $text_file_coda = <<'CODA';
# Local variables:
#   mode: text
#   buffer-read-only: t
# End:
CODA

sub print_manifest {
    my $self               = shift;
    my $manifest_lines_ref = shift;

    my $print_str          = <<"END_HEADER";
# ex: set ro:
#
# generated by $self->{script}
#
# See below for documentation on the format of this file.
#
# See docs/submissions.pod and the documentation in
# $self->{script}.
END_HEADER

    for my $k ( sort keys %{$manifest_lines_ref} ) {
        $print_str .= sprintf "%- 59s %s\n", ( $k, $manifest_lines_ref->{$k} );
    }
    $print_str .= $text_file_coda;
    open my $MANIFEST, '>', $self->{file}
        or croak "Unable to open $self->{file} for writing";
    print {$MANIFEST} $print_str;
    close $MANIFEST or croak "Unable to close $self->{file} after writing";

    return 1;
}

# Gets the package and the meta flags for the given file.  This function does
# it based on the directory the file is in.  If a particular file is needed,
# then _get_special (below) provides that functionality.
sub _get_manifest_entry {
    my $file    = shift;

    my $special = _get_special();
    my $loc     = '[]';
    for ($file) {
        $loc =
              exists( $special->{$_} ) ? $special->{$_}
            : !m[/]                           ? '[]'
            : m[^docs/book/draft/]            ? '[]'
            : m[^docs/]                       ? '[doc]'
            : m[^examples/]                   ? '[examples]'
            : m[README]                       ? '[]doc'
            : m[^include/]                    ? '[main]include'
            : m[^t/]                          ? '[test]'
            : m[/t/]                          ? '[test]'
            : ( m[^languages/(\w+)/] and $1 ne 'conversion' ) ? "[$1]"
            : ( m[^compilers/(\w+)/] and $1 ne 'conversion' ) ? "[$1]"
            : m[^lib/Parrot/]                 ? '[devel]lib'
            : m[^runtime/]                    ? '[library]'
            : m[^src/pmc/.*\.h]               ? '[devel]include'
            : m[^runtime/parrot/library/PCT]  ? '[pct]'
            : m[^runtime/parrot/library/PCT]  ? '[pge]'
            : m[^runtime/parrot/library/TGE]  ? '[tge]'
            : m[^runtime/parrot/library/JSON] ? '[json]'
            :                                   '[]'; # default
    }

    return $loc;
}

# See comments for _get_manifest_entry, above
sub _get_special {
    my %special = qw(
        CREDITS                                         [main]doc
        DEPRECATED.yaml                                 [devel]doc
        DONORS.pod                                      [main]doc
        LICENSE                                         [main]doc
        PBC_COMPAT                                      [main]doc
        PLATFORMS                                       [devel]doc
        README.pod                                      [devel]doc
        README_win32.pod                                [devel]doc
        README_cygwin.pod                               [devel]doc
        RESPONSIBLE_PARTIES                             [main]doc
        TODO                                            [devel]doc
        VERSION                                         [devel]
        languages/t/harness                             [test]
        lib/File/Which.pm                               [devel]lib
        src/vtable.tbl                                  [devel]src
        tools/build/ops2c.pl                            [devel]
        tools/build/pmc2c.pl                            [devel]
        tools/dev/create_language.pl                    [devel]
        tools/dev/gen_makefile.pl                       [devel]
        tools/dev/mk_language_shell.pl                  [devel]
        tools/dev/pbc_to_exe.pir                        [devel]
        tools/dev/pprof2cg.pl                           [devel]
        tools/dev/reconfigure.pl                        [devel]
    );

    return \%special;
}

# Gets files currently listed in manifest, and returns a hash
sub _get_current_files {
    my $self          = shift;

    my %current_files;
    open my $FILE, "<", $self->{file}
        or die "Unable to open $self->{file} for reading";
    while ( my $line = <$FILE> ) {
        chomp $line;

        next if $line =~ /^\s*$/;

        next if $line =~ /^#/;

        my ($file) = split /\s+/, $line;
        $current_files{ $file }++;
    }
    close $FILE or die "Unable to close $self->{file} after reading";

    return \%current_files;
}

=head2 prepare_manifest_skip

    $print_str = $mani->prepare_manifest_skip();

Gets a list of the files that Git ignores, and returns a string that can be
put into F<MANIFEST.SKIP>.

=cut

sub prepare_manifest_skip {
    my $self      = shift;

    my $ignores_ref = $self->_get_ignores();

    return $self->_compose_manifest_skip($ignores_ref);
}


=head2 determine_need_for_manifest_skip

    $need_for_skip =
        $mani->determine_need_for_manifest_skip($print_str);

Determines whether F<MANIFEST.SKIP> is needed.  The tests used are:

=over 4

=item *

If the file doesn't exist, we need one.

=item *

If the proposed and existing contents differ, we need one.

=back

=cut

sub determine_need_for_manifest_skip {
    my $self      = shift;
    my $print_str = shift;

    if ( !-f $self->{skip} ) {
        return 1;
    }
    else {
        my $current_skips_ref        = $self->_get_current_skips();
        my $proposed_skips_ref       = _get_proposed_skips($print_str);
        my $different_patterns_count = 0;
        foreach my $cur ( keys %{$current_skips_ref} ) {
            $different_patterns_count++ unless $proposed_skips_ref->{$cur};
        }
        foreach my $pro ( keys %{$proposed_skips_ref} ) {
            $different_patterns_count++ unless $current_skips_ref->{$pro};
        }

        $different_patterns_count ? return 1 : return;
    }
}

=head2 print_manifest_skip

    $mani->print_manifest_skip($print_str) if $need_for_skip;

Writes F<MANIFEST.SKIP> to a file.  The example above does so only if needed.

=cut

sub print_manifest_skip {
    my $self      = shift;
    my $print_str = shift;

    open my $MANIFEST_SKIP, '>', $self->{skip}
        or die "Unable to open $self->{skip} for writing";
    $print_str .= $text_file_coda;
    print $MANIFEST_SKIP $print_str;
    close $MANIFEST_SKIP
        or die "Unable to close $self->{skip} after writing";

    return 1;
}

# Gets a list of files that Git ignores
sub _get_ignores {
    my $self      = shift;

    my $gitignore = Parrot::BuildUtil::slurp_file('.gitignore');

    my %ignores;
    my @ignore = sort grep { $_ !~ /^#/ } split( /\n/, $gitignore );

    for my $ignore (@ignore) {
         my ($dirname, $basename) = (dirname($ignore), basename($ignore));
         # .gitignore has different regexen than MANIFEST
         $ignore =~ s/\./\\./g;
         $ignore =~ s/\*/.\*/g;
         # printf "%s:%s:%s\n", $ignore, $dirname, $basename;
         $ignores{$ignore} = "";
    }

    return \%ignores;
}

# Turns list of ignored files into F<MANIFEST.SKIP> format
sub _compose_manifest_skip {
    my $self       = shift;
    my $ignore_ref = shift;

    my %ignore     = %{$ignore_ref};
    my $print_str  = <<"END_HEADER";
# ex: set ro:
#
# generated by $self->{script}
#
# This file should contain a list of directories and files created in Parrot's
# configuration and build processes which are not meant for inclusion in
# distributions.
#
# Ignore the .git directory
\\B\\.git\\b

# ports/ should not go into release tarballs
^ports\$
^ports/
END_HEADER

    foreach my $file ( sort keys %ignore ) {
        my $dir = $file;
        # printf "dir=$dir,file=$ignore{$file}\n";
        foreach ( $ignore{$file} ) {
            $print_str .=
                ( $dir ne '.' )
                ? "^$dir$_\$\n^$dir$_/\n"
                : "^$_\$\n^$_/\n";
        }
    }

    return $print_str;
}

# Gets a list of the currently skipped files from F<MANIFEST.SKIP>
sub _get_current_skips {
    my $self          = shift;

    my %current_skips;
    open my $SKIP, "<", $self->{skip}
        or die "Unable to open $self->{skip} for reading";
    while ( my $line = <$SKIP> ) {
        chomp $line;
        next if $line =~ /^\s*$/;
        next if $line =~ /^#/;
        $current_skips{$line}++;
    }
    close $SKIP or die "Unable to close $self->{skip} after reading";

    return \%current_skips;
}

# Gets list of files we're proposing to skip
sub _get_proposed_skips {
    my $print_str      = shift;

    my @proposed_lines = split /\n/, $print_str;
    my %proposed_skips = ();
    for my $line (@proposed_lines) {
        next if $line =~ /^\s*$/;
        next if $line =~ /^#/;
        $proposed_skips{$line}++;
    }

    return \%proposed_skips;
}

1;

=head1 MANIFEST FORMAT

The format of the F<MANIFEST> (currently F<MANIFEST> and F<MANIFEST.generated>
are used) is:

    source_path <whitespace> [package]meta1,meta2,...

or you may optionally specify a different destination path:

    source_path <whitespace> [package]meta1,meta2,... <whitespace> destination

Additionally, there may be a C<*> in front of the whole line to designate
a generated file:

    source_path <whitespace> *[package]meta1,meta2,... <whitespace> destination

The square brackets around C<package> are literal. C<package> gives
the name of the RPM that the given file will be installed for, and is
only used by this script to skip files that are not members of any
package.

The various meta flags recognized are:

=over 4

=item C<doc>

Tag this file with C<%doc> in the RPM, and omit the leading path (because
F<rpm> will put it into a directory of its choosing).

=item C<include>

Write this file to the location given by the C<--includedir> option.

=item C<lib>

Write this file to the location given by the C<--libdir> option.

=item C<bin>

Write this file to the location given by the C<--bindir> option.

=back

The optional C<destination> field provides a general way to change
where a file will be written to. It will be applied before any
metadata tags.

Example: if this line is in the F<MANIFEST.generated> file

  languages/snorkfest/snork-compile        [main]bin

and the C<--bindir=/usr/parroty/bin>, then the generated
F<parrot-E<lt>VERSIONE<gt>-1.E<lt>archE<gt>.rpm> file will contain the file
C</usr/parroty/bin/snork-compile>.

=head1 SEE ALSO

F<tools/dev/mk_manifest_and_skip.pl>.

=head1 AUTHOR

James E. Keenan (jkeenan@cpan.org) refactored code from earlier versions of
F<tools/dev/mk_manifest_and_skip.pl>.

=head1 LICENSE

This is free software which you may distribute under the same terms as Perl
itself.

=cut

# Local Variables:
#   mode: cperl
#   cperl-indent-level: 4
#   fill-column: 100
# End:
# vim: expandtab shiftwidth=4:
