/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.snapshot.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Interner;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileSystemLoopException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.EnumSet;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Predicate;
import javax.annotation.Nullable;
import org.gradle.internal.file.FileMetadata;
import org.gradle.internal.file.impl.DefaultFileMetadata;
import org.gradle.internal.hash.FileHasher;
import org.gradle.internal.hash.HashCode;
import org.gradle.internal.snapshot.CompleteFileSystemLocationSnapshot;
import org.gradle.internal.snapshot.MerkleDirectorySnapshotBuilder;
import org.gradle.internal.snapshot.MissingFileSnapshot;
import org.gradle.internal.snapshot.RegularFileSnapshot;
import org.gradle.internal.snapshot.SnapshottingFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectorySnapshotter {
    private static final Logger LOGGER = LoggerFactory.getLogger(DirectorySnapshotter.class);
    private static final EnumSet<FileVisitOption> DONT_FOLLOW_SYMLINKS = EnumSet.noneOf(FileVisitOption.class);
    private final FileHasher hasher;
    private final Interner<String> stringInterner;
    private final DefaultExcludes defaultExcludes;

    public DirectorySnapshotter(FileHasher hasher, Interner<String> stringInterner, String ... defaultExcludes) {
        this.hasher = hasher;
        this.stringInterner = stringInterner;
        this.defaultExcludes = new DefaultExcludes(defaultExcludes);
    }

    public CompleteFileSystemLocationSnapshot snapshot(String absolutePath, @Nullable SnapshottingFilter.DirectoryWalkerPredicate predicate, AtomicBoolean hasBeenFiltered) {
        try {
            Path rootPath = Paths.get(absolutePath, new String[0]);
            PathVisitor visitor = new PathVisitor(predicate, hasBeenFiltered, this.hasher, this.stringInterner, this.defaultExcludes);
            Files.walkFileTree(rootPath, DONT_FOLLOW_SYMLINKS, Integer.MAX_VALUE, visitor);
            return visitor.getResult();
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Could not list contents of directory '%s'.", absolutePath), e);
        }
    }

    private static class PathVisitor
    implements FileVisitor<Path> {
        private final MerkleDirectorySnapshotBuilder builder;
        private final SnapshottingFilter.DirectoryWalkerPredicate predicate;
        private final AtomicBoolean hasBeenFiltered;
        private final FileHasher hasher;
        private final Interner<String> stringInterner;
        private final DefaultExcludes defaultExcludes;
        private final Deque<SymbolicLinkMapping> symbolicLinkMappings = new ArrayDeque<SymbolicLinkMapping>();
        private final Deque<String> parentDirectories = new ArrayDeque<String>();

        public PathVisitor(@Nullable SnapshottingFilter.DirectoryWalkerPredicate predicate, AtomicBoolean hasBeenFiltered, FileHasher hasher, Interner<String> stringInterner, DefaultExcludes defaultExcludes) {
            this.builder = MerkleDirectorySnapshotBuilder.sortingRequired();
            this.predicate = predicate;
            this.hasBeenFiltered = hasBeenFiltered;
            this.hasher = hasher;
            this.stringInterner = stringInterner;
            this.defaultExcludes = defaultExcludes;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
            String fileName = this.getFilename(dir);
            String internedName = this.intern(fileName);
            if (this.builder.isRoot() || this.shouldVisit(dir, internedName, true, this.builder.getRelativePath())) {
                this.builder.preVisitDirectory(this.intern(this.remapAbsolutePath(dir)), internedName);
                this.parentDirectories.addFirst(dir.toString());
                return FileVisitResult.CONTINUE;
            }
            return FileVisitResult.SKIP_SUBTREE;
        }

        private String getFilename(Path dir) {
            return Optional.ofNullable(dir.getFileName()).map(Object::toString).orElse("");
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
            if (attrs.isSymbolicLink()) {
                BasicFileAttributes targetAttributes = this.readAttributesOfSymlinkTarget(file, attrs);
                if (targetAttributes.isDirectory()) {
                    try {
                        Path targetDir = file.toRealPath(new LinkOption[0]);
                        String targetDirString = targetDir.toString();
                        if (this.introducesCycle(targetDirString)) {
                            return FileVisitResult.CONTINUE;
                        }
                        this.symbolicLinkMappings.addFirst(new SymbolicLinkMapping(file.toString(), targetDirString));
                        Files.walkFileTree(targetDir, EnumSet.noneOf(FileVisitOption.class), Integer.MAX_VALUE, this);
                        this.symbolicLinkMappings.removeFirst();
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(String.format("Could not list contents of directory '%s'.", file), e);
                    }
                } else {
                    this.visitResolvedFile(file, targetAttributes, FileMetadata.AccessType.VIA_SYMLINK);
                }
            } else {
                this.visitResolvedFile(file, attrs, FileMetadata.AccessType.DIRECT);
            }
            return FileVisitResult.CONTINUE;
        }

        private boolean introducesCycle(String targetDirString) {
            return this.parentDirectories.contains(targetDirString);
        }

        private String remapAbsolutePath(Path dir) {
            String targetAbsolutePath = dir.toString();
            return this.symbolicLinkMappings.stream().map(mapping -> mapping.remapPath(targetAbsolutePath)).filter(Optional::isPresent).map(Optional::get).findFirst().orElse(targetAbsolutePath);
        }

        private void visitResolvedFile(Path file, BasicFileAttributes targetAttributes, FileMetadata.AccessType accessType) {
            String internedName = this.intern(file.getFileName().toString());
            if (this.shouldVisit(file, internedName, false, this.builder.getRelativePath())) {
                this.builder.visitFile(this.snapshotFile(file, internedName, targetAttributes, accessType));
            }
        }

        private BasicFileAttributes readAttributesOfSymlinkTarget(Path symlink, BasicFileAttributes symlinkAttributes) {
            try {
                return Files.readAttributes(symlink, BasicFileAttributes.class, new LinkOption[0]);
            }
            catch (IOException ioe) {
                return symlinkAttributes;
            }
        }

        private CompleteFileSystemLocationSnapshot snapshotFile(Path absoluteFilePath, String internedName, BasicFileAttributes attrs, FileMetadata.AccessType accessType) {
            String internedAbsoluteFilePath = this.intern(this.remapAbsolutePath(absoluteFilePath));
            if (attrs.isRegularFile()) {
                try {
                    long lastModified = attrs.lastModifiedTime().toMillis();
                    long fileLength = attrs.size();
                    FileMetadata metadata = DefaultFileMetadata.file((long)lastModified, (long)fileLength, (FileMetadata.AccessType)accessType);
                    HashCode hash = this.hasher.hash(absoluteFilePath.toFile(), fileLength, lastModified);
                    return new RegularFileSnapshot(internedAbsoluteFilePath, internedName, hash, metadata);
                }
                catch (UncheckedIOException e) {
                    LOGGER.info("Could not read file path '{}'.", (Object)absoluteFilePath, (Object)e);
                }
            }
            return new MissingFileSnapshot(internedAbsoluteFilePath, internedName, accessType);
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) {
            boolean isDirectory;
            String internedName;
            if (this.isNotFileSystemLoopException(exc) && this.shouldVisit(file, internedName = this.intern(file.getFileName().toString()), isDirectory = Files.isDirectory(file, new LinkOption[0]), this.builder.getRelativePath())) {
                LOGGER.info("Could not read file path '{}'.", (Object)file);
                String internedAbsolutePath = this.intern(file.toString());
                this.builder.visitFile(new MissingFileSnapshot(internedAbsolutePath, internedName, FileMetadata.AccessType.DIRECT));
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, @Nullable IOException exc) {
            if (this.isNotFileSystemLoopException(exc)) {
                throw new UncheckedIOException(String.format("Could not read directory path '%s'.", dir), exc);
            }
            FileMetadata.AccessType accessType = FileMetadata.AccessType.viaSymlink((!this.symbolicLinkMappings.isEmpty() && this.symbolicLinkMappings.getFirst().target.equals(dir.toString()) ? 1 : 0) != 0);
            this.builder.postVisitDirectory(accessType);
            this.parentDirectories.removeFirst();
            return FileVisitResult.CONTINUE;
        }

        private boolean isNotFileSystemLoopException(@Nullable IOException e) {
            return e != null && !(e instanceof FileSystemLoopException);
        }

        private String intern(String string) {
            return (String)this.stringInterner.intern((Object)string);
        }

        private boolean shouldVisit(Path path, String internedName, boolean isDirectory, Iterable<String> relativePath) {
            if (isDirectory ? this.defaultExcludes.excludeDir(internedName) : this.defaultExcludes.excludeFile(internedName)) {
                return false;
            }
            if (this.predicate == null) {
                return true;
            }
            boolean allowed = this.predicate.test(path, internedName, isDirectory, relativePath);
            if (!allowed) {
                this.hasBeenFiltered.set(true);
            }
            return allowed;
        }

        public CompleteFileSystemLocationSnapshot getResult() {
            return this.builder.getResult();
        }
    }

    @VisibleForTesting
    static class DefaultExcludes {
        private final ImmutableSet<String> excludeFileNames;
        private final ImmutableSet<String> excludedDirNames;
        private final Predicate<String> excludedFileNameSpec;

        public DefaultExcludes(String[] defaultExcludes) {
            ArrayList excludeFiles = Lists.newArrayList();
            ArrayList excludeDirs = Lists.newArrayList();
            ArrayList excludeFileSpecs = Lists.newArrayList();
            for (String defaultExclude : defaultExcludes) {
                if (defaultExclude.startsWith("**/")) {
                    defaultExclude = defaultExclude.substring(3);
                }
                int length = defaultExclude.length();
                if (defaultExclude.endsWith("/**")) {
                    excludeDirs.add(defaultExclude.substring(0, length - 3));
                    continue;
                }
                int firstStar = defaultExclude.indexOf(42);
                if (firstStar == -1) {
                    excludeFiles.add(defaultExclude);
                    continue;
                }
                StartMatcher start = firstStar == 0 ? it -> true : new StartMatcher(defaultExclude.substring(0, firstStar));
                EndMatcher end = firstStar == length - 1 ? it -> true : new EndMatcher(defaultExclude.substring(firstStar + 1, length));
                excludeFileSpecs.add(start.and(end));
            }
            this.excludeFileNames = ImmutableSet.copyOf((Collection)excludeFiles);
            this.excludedFileNameSpec = excludeFileSpecs.stream().reduce(it -> false, Predicate::or);
            this.excludedDirNames = ImmutableSet.copyOf((Collection)excludeDirs);
        }

        public boolean excludeDir(String name) {
            return this.excludedDirNames.contains((Object)name);
        }

        public boolean excludeFile(String name) {
            return this.excludeFileNames.contains((Object)name) || this.excludedFileNameSpec.test(name);
        }

        private static class StartMatcher
        implements Predicate<String> {
            private final String start;

            public StartMatcher(String start) {
                this.start = start;
            }

            @Override
            public boolean test(String element) {
                return element.startsWith(this.start);
            }
        }

        private static class EndMatcher
        implements Predicate<String> {
            private final String end;

            public EndMatcher(String end) {
                this.end = end;
            }

            @Override
            public boolean test(String element) {
                return element.endsWith(this.end);
            }
        }
    }

    private static class SymbolicLinkMapping {
        private final String source;
        private final String target;

        private SymbolicLinkMapping(String source, String target) {
            this.source = source;
            this.target = target;
        }

        Optional<String> remapPath(String absolutePath) {
            if (absolutePath.equals(this.target)) {
                return Optional.of(this.source);
            }
            if (absolutePath.startsWith(this.target) && absolutePath.charAt(this.target.length()) == File.separatorChar) {
                return Optional.of(this.source + File.separatorChar + absolutePath.substring(this.target.length() + 1));
            }
            return Optional.empty();
        }
    }
}

