"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.traverseNodes = traverseNodes;
exports.isMakeUsageCollectorFunction = isMakeUsageCollectorFunction;
exports.sourceHasUsageCollector = sourceHasUsageCollector;
exports.parseUsageCollection = parseUsageCollection;

var ts = _interopRequireWildcard(require("typescript"));

var _devUtils = require("@kbn/dev-utils");

var path = _interopRequireWildcard(require("path"));

var _utils = require("./utils");

var _serializer = require("./serializer");

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
function* traverseNodes(maybeNodes) {
  const nodes = Array.isArray(maybeNodes) ? maybeNodes : [maybeNodes];

  for (const node of nodes) {
    const children = [];
    yield node;
    ts.forEachChild(node, child => {
      children.push(child);
    });

    for (const child of children) {
      yield* traverseNodes(child);
    }
  }
}

function isMakeUsageCollectorFunction(node, sourceFile) {
  if (ts.isCallExpression(node)) {
    const isMakeUsageCollector = /makeUsageCollector$/.test(node.expression.getText(sourceFile));

    if (isMakeUsageCollector) {
      return true;
    }
  }

  return false;
}

function getCollectionConfigNode(collectorNode, sourceFile) {
  if (collectorNode.arguments.length > 1) {
    throw Error(`makeUsageCollector does not accept more than one argument.`);
  }

  const collectorConfig = collectorNode.arguments[0];

  if (ts.isObjectLiteralExpression(collectorConfig)) {
    return collectorConfig;
  }

  const variableDefintionName = collectorConfig.getText();

  for (const node of traverseNodes(sourceFile)) {
    if (ts.isVariableDeclaration(node)) {
      const declarationName = node.name.getText();

      if (declarationName === variableDefintionName) {
        if (!node.initializer) {
          throw Error(`Unable to parse collector configs.`);
        }

        if (ts.isObjectLiteralExpression(node.initializer)) {
          return node.initializer;
        }

        if (ts.isCallExpression(node.initializer)) {
          const functionName = node.initializer.expression.getText(sourceFile);

          for (const sfNode of traverseNodes(sourceFile)) {
            if (ts.isFunctionDeclaration(sfNode)) {
              var _sfNode$name;

              const fnDeclarationName = (_sfNode$name = sfNode.name) === null || _sfNode$name === void 0 ? void 0 : _sfNode$name.getText();

              if (fnDeclarationName === functionName) {
                const returnStatements = [];

                for (const fnNode of traverseNodes(sfNode)) {
                  if (ts.isReturnStatement(fnNode) && fnNode.parent === sfNode.body) {
                    returnStatements.push(fnNode);
                  }
                }

                if (returnStatements.length > 1) {
                  throw Error(`Collector function cannot have multiple return statements.`);
                }

                if (returnStatements.length === 0) {
                  throw Error(`Collector function must have a return statement.`);
                }

                if (!returnStatements[0].expression) {
                  throw Error(`Collector function return statement must be an expression.`);
                }

                return returnStatements[0].expression;
              }
            }
          }
        }
      }
    }
  }

  throw Error(`makeUsageCollector argument must be an object.`);
}

function extractCollectorDetails(collectorNode, program, sourceFile) {
  if (collectorNode.arguments.length > 1) {
    throw Error(`makeUsageCollector does not accept more than one argument.`);
  }

  const collectorConfig = getCollectionConfigNode(collectorNode, sourceFile);
  const typeProperty = (0, _utils.getProperty)(collectorConfig, 'type');

  if (!typeProperty) {
    throw Error(`usageCollector.type must be defined.`);
  }

  const typePropertyValue = (0, _utils.getPropertyValue)(typeProperty, program);

  if (!typePropertyValue || typeof typePropertyValue !== 'string') {
    throw Error(`usageCollector.type must be be a non-empty string literal.`);
  }

  const fetchProperty = (0, _utils.getProperty)(collectorConfig, 'fetch');

  if (!fetchProperty) {
    throw Error(`usageCollector.fetch must be defined.`);
  }

  const schemaProperty = (0, _utils.getProperty)(collectorConfig, 'schema');

  if (!schemaProperty) {
    throw Error(`usageCollector.schema must be defined.`);
  }

  const schemaPropertyValue = (0, _utils.getPropertyValue)(schemaProperty, program, {
    chaseImport: true
  });

  if (!schemaPropertyValue || typeof schemaPropertyValue !== 'object') {
    throw Error(`usageCollector.schema must be be an object.`);
  }

  const collectorNodeType = collectorNode.typeArguments;

  if (!collectorNodeType || (collectorNodeType === null || collectorNodeType === void 0 ? void 0 : collectorNodeType.length) === 0) {
    throw Error(`makeUsageCollector requires a Usage type makeUsageCollector<Usage>({ ... }).`);
  }

  const usageTypeNode = collectorNodeType[0];
  const usageTypeName = usageTypeNode.getText();
  const usageType = (0, _serializer.getDescriptor)(usageTypeNode, program);
  return {
    collectorName: typePropertyValue,
    schema: {
      value: schemaPropertyValue
    },
    fetch: {
      typeName: usageTypeName,
      typeDescriptor: usageType
    }
  };
}

function sourceHasUsageCollector(sourceFile) {
  if (sourceFile.isDeclarationFile === true || sourceFile.identifierCount === 0) {
    return false;
  }

  const identifiers = sourceFile.identifiers;

  if (!identifiers.get('makeUsageCollector') && !identifiers.get('type') || !identifiers.get('fetch')) {
    return false;
  }

  return true;
}

function* parseUsageCollection(sourceFile, program) {
  const relativePath = path.relative(process.cwd(), sourceFile.fileName);

  if (sourceHasUsageCollector(sourceFile)) {
    for (const node of traverseNodes(sourceFile)) {
      if (isMakeUsageCollectorFunction(node, sourceFile)) {
        try {
          const collectorDetails = extractCollectorDetails(node, program, sourceFile);
          yield [relativePath, collectorDetails];
        } catch (err) {
          throw (0, _devUtils.createFailError)(`Error extracting collector in ${relativePath}\n${err}`);
        }
      }
    }
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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