/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import React from 'react';
import { DevToolsSettingsModal } from '../components'; // @ts-ignore

import { retrieveAutoCompleteInfo } from '../../lib/mappings/mappings';
import { useServicesContext, useEditorActionContext } from '../contexts';

var getAutocompleteDiff = function getAutocompleteDiff(newSettings, prevSettings) {
  return Object.keys(newSettings.autocomplete).filter(function (key) {
    // @ts-ignore
    return prevSettings.autocomplete[key] !== newSettings.autocomplete[key];
  });
};

var _refreshAutocompleteSettings = function refreshAutocompleteSettings(settings, selectedSettings) {
  retrieveAutoCompleteInfo(settings, selectedSettings);
};

var fetchAutocompleteSettingsIfNeeded = function fetchAutocompleteSettingsIfNeeded(settings, newSettings, prevSettings) {
  // We'll only retrieve settings if polling is on. The expectation here is that if the user
  // disables polling it's because they want manual control over the fetch request (possibly
  // because it's a very expensive request given their cluster and bandwidth). In that case,
  // they would be unhappy with any request that's sent automatically.
  if (newSettings.polling) {
    var autocompleteDiff = getAutocompleteDiff(newSettings, prevSettings);
    var isSettingsChanged = autocompleteDiff.length > 0;
    var isPollingChanged = prevSettings.polling !== newSettings.polling;

    if (isSettingsChanged) {
      // If the user has changed one of the autocomplete settings, then we'll fetch just the
      // ones which have changed.
      var changedSettings = autocompleteDiff.reduce(function (changedSettingsAccum, setting) {
        changedSettingsAccum[setting] = newSettings.autocomplete[setting];
        return changedSettingsAccum;
      }, {});
      retrieveAutoCompleteInfo(settings, changedSettings);
    } else if (isPollingChanged && newSettings.polling) {
      // If the user has turned polling on, then we'll fetch all selected autocomplete settings.
      retrieveAutoCompleteInfo(settings, settings.getAutocomplete());
    }
  }
};

export function Settings(_ref) {
  var onClose = _ref.onClose;

  var _useServicesContext = useServicesContext(),
      settings = _useServicesContext.services.settings;

  var dispatch = useEditorActionContext();

  var onSaveSettings = function onSaveSettings(newSettings) {
    var prevSettings = settings.toJSON();
    fetchAutocompleteSettingsIfNeeded(settings, newSettings, prevSettings); // Update the new settings in localStorage

    settings.updateSettings(newSettings); // Let the rest of the application know settings have updated.

    dispatch({
      type: 'updateSettings',
      payload: newSettings
    });
    onClose();
  };

  return /*#__PURE__*/React.createElement(DevToolsSettingsModal, {
    onClose: onClose,
    onSaveSettings: onSaveSettings,
    refreshAutocompleteSettings: function refreshAutocompleteSettings(selectedSettings) {
      return _refreshAutocompleteSettings(settings, selectedSettings);
    },
    settings: settings.toJSON()
  });
}