/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var JSON_COLON = ':';
var JSON_STRING_DELIMITER = '"';
var JSON_STRING_ESCAPE = '\\';
/**
 * Accepts JSON (as a string) and extracts the positions of all JSON string
 * values.
 *
 * For example:
 *
 * '{ "my_string_value": "is this", "my_number_value": 42 }'
 *
 * Would extract one result:
 *
 * [ { startIndex: 21, endIndex: 29 } ]
 *
 * This result maps to `"is this"` from the example JSON.
 *
 */

export var extractJSONStringValues = function extractJSONStringValues(input) {
  var position = 0;
  var currentStringStartPos;
  var isInsideString = false;
  var stringValues = [];

  function read() {
    return input[position];
  }

  function peekNextNonWhitespace() {
    var peekPosition = position + 1;

    while (peekPosition < input.length) {
      var peekChar = input[peekPosition];

      if (peekChar.match(/[^\s\r\n]/)) {
        return peekChar;
      }

      ++peekPosition;
    }
  }

  function advance() {
    ++position;
  }

  while (position < input.length) {
    var char = read();

    if (!isInsideString) {
      if (char === JSON_STRING_DELIMITER) {
        currentStringStartPos = position;
        isInsideString = true;
      } // else continue scanning for JSON_STRING_DELIMITER

    } else {
      if (char === JSON_STRING_ESCAPE) {
        // skip ahead - we are still inside of a string
        advance();
      } else if (char === JSON_STRING_DELIMITER) {
        if (peekNextNonWhitespace() !== JSON_COLON) {
          stringValues.push({
            startIndex: currentStringStartPos,
            endIndex: position
          });
        }

        isInsideString = false;
      }
    }

    advance();
  }

  return {
    stringValues: stringValues
  };
};