/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { Observable } from 'rxjs';
import { map, share } from 'rxjs/operators';
import { createKbnUrlControls, getStateFromKbnUrl, setStateToKbnUrl } from '../../state_management/url';
/**
 * KbnUrlStateStorage is a state storage for {@link syncState} utility which:
 *
 * 1. Keeps state in sync with the URL.
 * 2. Serializes data and stores it in the URL in one of the supported formats:
 *   * Rison encoded.
 *   * Hashed URL: In URL we store only the hash from the serialized state, but the state itself is stored in sessionStorage. See Kibana's `state:storeInSessionStorage` advanced option for more context.
 * 3. Takes care of listening to the URL updates and notifies state about the updates.
 * 4. Takes care of batching URL updates to prevent redundant browser history records.
 *
 * {@link https://github.com/elastic/kibana/blob/master/src/plugins/kibana_utils/docs/state_sync/storages/kbn_url_storage.md | Refer to this guide for more info}
 * @public
 */

/**
 * Creates {@link IKbnUrlStateStorage} state storage
 * @returns - {@link IKbnUrlStateStorage}
 * @public
 */
export var createKbnUrlStateStorage = function createKbnUrlStateStorage() {
  var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {
    useHash: false
  },
      _ref$useHash = _ref.useHash,
      useHash = _ref$useHash === void 0 ? false : _ref$useHash,
      history = _ref.history;

  var url = createKbnUrlControls(history);
  return {
    set: function set(key, state) {
      var _ref2 = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {
        replace: false
      },
          _ref2$replace = _ref2.replace,
          replace = _ref2$replace === void 0 ? false : _ref2$replace;

      // syncState() utils doesn't wait for this promise
      return url.updateAsync(function (currentUrl) {
        return setStateToKbnUrl(key, state, {
          useHash: useHash
        }, currentUrl);
      }, replace);
    },
    get: function get(key) {
      // if there is a pending url update, then state will be extracted from that pending url,
      // otherwise current url will be used to retrieve state from
      return getStateFromKbnUrl(key, url.getPendingUrl());
    },
    change$: function change$(key) {
      return new Observable(function (observer) {
        var unlisten = url.listen(function () {
          observer.next();
        });
        return function () {
          unlisten();
        };
      }).pipe(map(function () {
        return getStateFromKbnUrl(key);
      }), share());
    },
    flush: function flush() {
      var _ref3 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref3$replace = _ref3.replace,
          replace = _ref3$replace === void 0 ? false : _ref3$replace;

      return !!url.flush(replace);
    },
    cancel: function cancel() {
      url.cancel();
    }
  };
};