"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;

var fs = _interopRequireWildcard(require("fs"));

var _wreck = _interopRequireDefault(require("@hapi/wreck"));

var _http = _interopRequireDefault(require("http"));

var _https = _interopRequireDefault(require("https"));

var _routes = require("./routes");

var _authentication_type = require("../authentication_type");

var _helper = require("./helper");

var _next_url = require("../../../utils/next_url");

var _common = require("../../../../common");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;

    super(config, sessionStorageFactory, router, esClient, core, logger);

    _defineProperty(this, "type", _common.AuthType.OPEN_ID);

    _defineProperty(this, "openIdAuthConfig", void 0);

    _defineProperty(this, "authHeaderName", void 0);

    _defineProperty(this, "openIdConnectUrl", void 0);

    _defineProperty(this, "wreckClient", void 0);

    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;

    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }

    this.openIdAuthConfig.scope = scope;
  }

  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info

      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }

  createWreckClient() {
    var _this$config$openid3, _this$config$openid4;

    const wreckHttpsOption = {};

    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
    }

    if (((_this$config$openid4 = this.config.openid) === null || _this$config$openid4 === void 0 ? void 0 : _this$config$openid4.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);

      wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }

    if (Object.keys(wreckHttpsOption).length > 0) {
      return _wreck.default.defaults({
        agents: {
          http: new _http.default.Agent(),
          https: new _https.default.Agent(wreckHttpsOption),
          httpsAllowUnauthorized: new _https.default.Agent({
            rejectUnauthorized: false
          })
        }
      });
    } else {
      return _wreck.default;
    }
  }

  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }

  async getAdditionalAuthHeader(request) {
    return {};
  }

  getCookie(request, authInfo) {
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValue: request.headers.authorization
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  } // TODO: Add token expiration check here


  async isValidCookie(cookie) {
    var _cookie$credentials, _cookie$credentials2, _cookie$credentials3;

    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) || !((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.expires_at)) {
      return false;
    }

    if (((_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.expires_at) > Date.now()) {
      return true;
    } // need to renew id token


    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid5, _this$config$openid6;

        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.client_id,
          client_secret: (_this$config$openid6 = this.config.openid) === null || _this$config$openid6 === void 0 ? void 0 : _this$config$openid6.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient); // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token

        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValue: `Bearer ${refreshTokenResponse.idToken}`,
            refresh_token: refreshTokenResponse.refreshToken,
            expires_at: (0, _helper.getExpirationDate)(refreshTokenResponse) // expiresIn is in second

          };
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      // nextUrl is a key value pair
      const nextUrl = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);
      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}${_common.OPENID_AUTH_LOGIN}?${nextUrl}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }

  buildAuthHeaderFromCookie(cookie) {
    var _cookie$credentials4;

    const header = {};
    const authHeaderValue = (_cookie$credentials4 = cookie.credentials) === null || _cookie$credentials4 === void 0 ? void 0 : _cookie$credentials4.authHeaderValue;

    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }

    return header;
  }

}

exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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