"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthentication = void 0;

var _querystring = require("querystring");

var _security_cookie = require("../../../session/security_cookie");

var _routes = require("./routes");

var _authentication_type = require("../authentication_type");

var _common = require("../../../../common");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class SamlAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);

    _defineProperty(this, "type", 'saml');

    _defineProperty(this, "redirectSAMlCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
  }

  generateNextUrl(request) {
    const path = this.coreSetup.http.basePath.serverBasePath + (request.url.pathname || '/app/opensearch-dashboards');
    return (0, _querystring.escape)(path);
  } // Check if we can get the previous tenant information from the expired cookie.


  async init() {
    const samlAuthRoutes = new _routes.SamlAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.securityClient, this.coreSetup);
    samlAuthRoutes.setupRoutes();
  }

  requestIncludesAuthInfo(request) {
    return request.headers[SamlAuthentication.AUTH_HEADER_NAME] ? true : false;
  }

  async getAdditionalAuthHeader(request) {
    return {};
  }

  getCookie(request, authInfo) {
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValue: request.headers[SamlAuthentication.AUTH_HEADER_NAME]
      },
      authType: _common.AuthType.SAML,
      expiryTime: Date.now() + this.config.session.ttl
    };
  } // Can be improved to check if the token is expiring.


  async isValidCookie(cookie) {
    var _cookie$credentials;

    return cookie.authType === _common.AuthType.SAML && cookie.username && cookie.expiryTime && ((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue);
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectSAMlCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }

  buildAuthHeaderFromCookie(cookie) {
    var _cookie$credentials2;

    const headers = {};
    headers[SamlAuthentication.AUTH_HEADER_NAME] = (_cookie$credentials2 = cookie.credentials) === null || _cookie$credentials2 === void 0 ? void 0 : _cookie$credentials2.authHeaderValue;
    return headers;
  }

}

exports.SamlAuthentication = SamlAuthentication;

_defineProperty(SamlAuthentication, "AUTH_HEADER_NAME", 'authorization');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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