/*
 * Decompiled with CFR 0.152.
 */
package ghidra.server;

import ghidra.framework.remote.InetNameLookup;
import ghidra.framework.store.local.IndexedLocalFileSystem;
import ghidra.framework.store.local.LocalFileSystem;
import ghidra.server.Repository;
import ghidra.server.UserManager;
import ghidra.server.remote.RepositoryServerHandleImpl;
import ghidra.util.NamingUtilities;
import ghidra.util.StringUtilities;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.exception.UserAccessException;
import java.io.File;
import java.io.IOException;
import java.net.UnknownHostException;
import java.rmi.server.ServerNotActiveException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import sun.rmi.transport.tcp.TCPTransport;
import utilities.util.FileUtilities;

public class RepositoryManager {
    static final Logger log = LogManager.getLogger(RepositoryManager.class);
    private static Map<Thread, String> clientNameMap = new WeakHashMap<Thread, String>();
    private File rootDirFile;
    private HashMap<String, Repository> repositoryMap;
    private ArrayList<RepositoryServerHandleImpl> handleList = new ArrayList();
    private UserManager userMgr;
    private boolean anonymousAccessAllowed;

    public RepositoryManager(File rootDir, boolean supportLocalPasswords, boolean requireExplicitPasswordReset, int defaultPasswordExpirationDays, boolean anonymousAccessAllowed) throws IOException {
        this.rootDirFile = rootDir;
        log.info("Instantiating Repository Manager for " + this.rootDirFile.getAbsolutePath());
        if (!this.rootDirFile.isDirectory()) {
            throw new IOException(this.rootDirFile + " is not a directory");
        }
        if (!this.rootDirFile.canWrite()) {
            throw new IOException(this.rootDirFile + " can not be written to");
        }
        this.anonymousAccessAllowed = anonymousAccessAllowed;
        this.userMgr = new UserManager(this, supportLocalPasswords, requireExplicitPasswordReset, defaultPasswordExpirationDays);
        this.repositoryMap = new HashMap();
        this.initialize();
    }

    public boolean anonymousAccessAllowed() {
        return this.anonymousAccessAllowed;
    }

    private boolean isAnonymousUser(String user) {
        if (this.anonymousAccessAllowed) {
            return "-anonymous-".equals(user);
        }
        return false;
    }

    public synchronized void dispose() {
        for (Repository rep : this.repositoryMap.values()) {
            rep.dispose();
        }
    }

    File getRootDir() {
        return this.rootDirFile;
    }

    public synchronized Repository createRepository(String currentUser, String name) throws IOException {
        if (this.isAnonymousUser(currentUser)) {
            throw new UserAccessException("Anonymous user not permitted to create repository");
        }
        this.validateUser(currentUser);
        if (!NamingUtilities.isValidName((String)name)) {
            throw new IOException("Invalid repository name: " + name);
        }
        if (this.repositoryMap.containsKey(name)) {
            throw new DuplicateFileException("Repository named " + name + " already exists");
        }
        File f = new File(this.rootDirFile, NamingUtilities.mangle((String)name));
        if (!f.mkdir()) {
            throw new IOException("Failed to make directory for " + f.getAbsolutePath());
        }
        Repository rep = new Repository(this, currentUser, f, name);
        RepositoryManager.log(name, null, "repository created", currentUser);
        this.repositoryMap.put(name, rep);
        return rep;
    }

    public synchronized Repository getRepository(String currentUser, String name) throws UserAccessException {
        Repository rep;
        if (!this.isAnonymousUser(currentUser)) {
            this.validateUser(currentUser);
        }
        if ((rep = this.repositoryMap.get(name)) != null) {
            rep.validateReadPrivilege(currentUser);
        }
        return rep;
    }

    synchronized Repository getRepository(String name) {
        return this.repositoryMap.get(name);
    }

    public synchronized void deleteRepository(String currentUser, String name) throws IOException {
        if (this.isAnonymousUser(currentUser)) {
            throw new UserAccessException("Anonymous user not permitted to delete repository");
        }
        this.validateUser(currentUser);
        Repository rep = this.repositoryMap.get(name);
        if (rep == null) {
            return;
        }
        rep.delete(currentUser);
        File f = new File(this.rootDirFile, NamingUtilities.mangle((String)name));
        if (!FileUtilities.deleteDir((File)f)) {
            throw new IOException("Failed to remove directory for " + f.getAbsolutePath());
        }
        this.repositoryMap.remove(name);
    }

    public synchronized String[] getRepositoryNames(String currentUser) {
        ArrayList<String> list = new ArrayList<String>();
        for (Repository rep : this.repositoryMap.values()) {
            if (this.isAnonymousUser(currentUser)) {
                if (!rep.anonymousAccessAllowed()) continue;
                list.add(rep.getName());
                continue;
            }
            if (rep.getUser(currentUser) == null) continue;
            list.add(rep.getName());
        }
        Collections.sort(list);
        String[] names = new String[list.size()];
        return list.toArray(names);
    }

    public synchronized String[] getAllUsers(String currentUser) throws IOException {
        if (this.isAnonymousUser(currentUser)) {
            return new String[0];
        }
        try {
            return this.userMgr.getUsers();
        }
        catch (IOException e) {
            log.error("Error while accessing user list: " + e.getMessage());
            throw new IOException("Failed to read user list");
        }
    }

    public UserManager getUserManager() {
        return this.userMgr;
    }

    private void validateUser(String currentUser) throws UserAccessException {
        if (!this.userMgr.isValidUser(currentUser)) {
            throw new UserAccessException(currentUser + " is unknown to this repository manager");
        }
    }

    private void initialize() throws IOException {
        int i;
        log.info("Known Repositories:");
        String[] names = RepositoryManager.getRepositoryNames(this.rootDirFile);
        for (i = 0; i < names.length; ++i) {
            log.info("   " + names[i]);
        }
        if (names.length == 0) {
            log.info("   <none>");
        }
        for (i = 0; i < names.length; ++i) {
            File f = new File(this.rootDirFile, NamingUtilities.mangle((String)names[i]));
            if (!f.isDirectory()) {
                log.error("Error while processing repository " + names[i] + ", directory not found: " + f);
                continue;
            }
            if (!f.canWrite()) {
                throw new IOException(f.getAbsolutePath() + " can not be written to");
            }
            try {
                Repository rep = new Repository(this, null, f, names[i]);
                String name = rep.getName();
                this.repositoryMap.put(name, rep);
                continue;
            }
            catch (UserAccessException rep) {
                continue;
            }
            catch (Exception e) {
                log.error("Error while processing repository " + names[i] + ", " + e.getMessage());
            }
        }
        this.userMgr.updateUserList(true);
    }

    static String getElapsedTimeSince(long t) {
        t = System.currentTimeMillis() - t;
        if (t < 1000L) {
            return null;
        }
        int hours = (int)(t / 3600000L);
        int mins = (int)((t - (long)(hours * 3600000)) / 60000L);
        int secs = (int)(t % 60000L / 1000L);
        StringBuilder tbuf = new StringBuilder();
        Object units = "secs";
        tbuf.append(StringUtilities.pad((String)Integer.toString(secs), (char)'0', (int)2));
        if (t >= 60000L) {
            units = "mins:" + (String)units;
            tbuf.insert(0, ":");
            tbuf.insert(0, StringUtilities.pad((String)Integer.toString(mins), (char)'0', (int)2));
            if (t >= 3600000L) {
                units = "hours:" + (String)units;
                tbuf.insert(0, ":");
                tbuf.insert(0, StringUtilities.pad((String)Integer.toString(hours), (char)'0', (int)2));
            }
        }
        return tbuf.toString() + " (" + (String)units + ")";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addHandle(RepositoryServerHandleImpl handle) {
        ArrayList<RepositoryServerHandleImpl> arrayList = this.handleList;
        synchronized (arrayList) {
            this.handleList.add(handle);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dropHandle(RepositoryServerHandleImpl handle) {
        ArrayList<RepositoryServerHandleImpl> arrayList = this.handleList;
        synchronized (arrayList) {
            this.handleList.remove(handle);
        }
    }

    static String[] getRepositoryNames(File rootDirFile) {
        File[] dirList = rootDirFile.listFiles();
        if (dirList == null) {
            return new String[0];
        }
        ArrayList<String> list = new ArrayList<String>(dirList.length);
        for (int i = 0; i < dirList.length; ++i) {
            if (!dirList[i].isDirectory() || LocalFileSystem.isHiddenDirName((String)dirList[i].getName())) continue;
            if (!NamingUtilities.isValidMangledName((String)dirList[i].getName())) {
                log.warn("Ignoring repository directory with bad name: " + dirList[i]);
                continue;
            }
            list.add(NamingUtilities.demangle((String)dirList[i].getName()));
        }
        Collections.sort(list);
        String[] names = new String[list.size()];
        return list.toArray(names);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getRMIClient() {
        String host;
        Thread currentThread = Thread.currentThread();
        if (!currentThread.getName().startsWith("RMI TCP Connection")) {
            return null;
        }
        Map<Thread, String> map = clientNameMap;
        synchronized (map) {
            host = clientNameMap.get(currentThread);
            if (host != null) {
                return host;
            }
        }
        try {
            host = TCPTransport.getClientHost();
            try {
                host = InetNameLookup.getCanonicalHostName((String)host);
            }
            catch (UnknownHostException e) {
                log.warn("Failed to resolve hostname: " + host);
            }
            map = clientNameMap;
            synchronized (map) {
                clientNameMap.put(currentThread, host);
            }
        }
        catch (ServerNotActiveException serverNotActiveException) {
            // empty catch block
        }
        return host;
    }

    public static void log(String repositoryName, String path, String msg, String user) {
        StringBuffer buf = new StringBuffer();
        if (repositoryName != null) {
            buf.append("[");
            buf.append(repositoryName);
            buf.append("]");
        }
        String host = RepositoryManager.getRMIClient();
        Object userStr = user;
        if (userStr != null) {
            if (host != null) {
                userStr = (String)userStr + "@" + host;
            }
        } else {
            userStr = host;
        }
        if (path != null) {
            buf.append(path);
        }
        if (repositoryName != null || path != null) {
            buf.append(": ");
        }
        buf.append(msg);
        if (userStr != null) {
            buf.append(" (");
            buf.append((String)userStr);
            buf.append(")");
        }
        log.info(buf.toString());
    }

    static void listRepositories(File repositoriesRootDir, boolean includeUserAccessDetails) {
        String[] names = RepositoryManager.getRepositoryNames(repositoriesRootDir);
        System.out.println("\nRepositories:");
        if (names.length == 0) {
            System.out.println("   <No repositories have been created>");
        } else {
            for (String name : names) {
                Object type;
                File repoDir;
                block8: {
                    repoDir = new File(repositoriesRootDir, NamingUtilities.mangle((String)name));
                    String rootPath = repoDir.getAbsolutePath();
                    boolean isIndexed = IndexedLocalFileSystem.isIndexed((String)rootPath);
                    if (isIndexed || IndexedLocalFileSystem.hasIndexedStructure((String)rootPath)) {
                        type = "Indexed Filesystem";
                        try {
                            int indexVersion = IndexedLocalFileSystem.readIndexVersion((String)rootPath);
                            if (indexVersion == 1) {
                                type = null;
                                break block8;
                            }
                            type = (String)type + " (V" + indexVersion + ")";
                        }
                        catch (IOException e) {
                            type = (String)type + "(unknown)";
                        }
                    } else {
                        type = "Mangled Filesystem";
                    }
                }
                System.out.println("  " + name + (String)(type == null ? "" : " - uses " + (String)type));
                if (!includeUserAccessDetails) continue;
                Repository.listUserPermissions(repoDir, "    ");
            }
        }
    }

    static void markAllRepositoriesForIndexMigration(File serverDir) {
        String[] names = RepositoryManager.getRepositoryNames(serverDir);
        if (names.length == 0) {
            System.err.println("No repositories found!");
            return;
        }
        int count = 0;
        for (String name : names) {
            if (!Repository.markRepositoryForIndexMigration(serverDir, name, true)) continue;
            ++count;
        }
        if (count == 0) {
            System.out.println("All repositories are already indexed");
        }
    }
}

