# -*- coding: utf-8 -*- #
# Copyright 2020 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""'notebooks instances update' command."""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

from googlecloudsdk.api_lib.notebooks import instances as instance_util
from googlecloudsdk.api_lib.notebooks import util
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.notebooks import flags

DETAILED_HELP = {
    'DESCRIPTION':
        """
        Request for updating notbook instance.
    """,
    'EXAMPLES':
        """
    To update machine type for an instance, run:

        $ {command} /projects/example-project/locations/us-central1-b/instances/example-instance --machine-type=n1-standard-8

    To update labels for an instance, run:

        $ {command} /projects/example-project/locations/us-central1-b/instances/example-instance --labels=k1=v1,k2=v2

    To update labels and accelerator cores, run:

        $ {command} /projects/example-project/locations/us-central1-b/instances/example-instance --labels=k1=v1,k2=v2 --accelerator-core-count=2
    """,
}


@base.ReleaseTracks(base.ReleaseTrack.ALPHA, base.ReleaseTrack.BETA)
class Update(base.Command):
  """Request for updating instances."""

  @staticmethod
  def Args(parser):
    """Register flags for this command."""
    flags.AddUpdateInstanceFlags(parser)

  def Run(self, args):
    instance_service = util.GetClient().projects_locations_instances
    if args.IsSpecified('accelerator_type') or args.IsSpecified(
        'accelerator_core_count'):
      operation = instance_service.SetAccelerator(
          instance_util.CreateSetAcceleratorRequest(args))
      instance_util.HandleLRO(
          operation,
          args,
          instance_service,
          operation_type=instance_util.OperationType.UPDATE)
    if args.IsSpecified('labels'):
      operation = instance_service.SetLabels(
          instance_util.CreateSetLabelsRequest(args))
      instance_util.HandleLRO(
          operation,
          args,
          instance_service,
          operation_type=instance_util.OperationType.UPDATE)
    if args.IsSpecified('machine_type'):
      operation = instance_service.SetMachineType(
          instance_util.CreateSetMachineTypeRequest(args))
      instance_util.HandleLRO(
          operation,
          args,
          instance_service,
          operation_type=instance_util.OperationType.UPDATE)


Update.detailed_help = DETAILED_HELP
