/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import React from 'react';
import { EuiBetaBadge, EuiButton, EuiEmptyPrompt, EuiIcon, EuiLink } from '@elastic/eui';
import { i18n } from '@kbn/i18n';
import { FormattedMessage } from '@kbn/i18n/react';

var getBadge = function getBadge(item) {
  if (item.stage === 'beta') {
    return /*#__PURE__*/React.createElement(EuiBetaBadge, {
      className: "visListingTable__betaIcon",
      label: "B",
      title: i18n.translate('visualize.listing.betaTitle', {
        defaultMessage: 'Beta'
      }),
      tooltipContent: i18n.translate('visualize.listing.betaTooltip', {
        defaultMessage: 'This visualization is in beta and is subject to change. The design and code is less mature than official GA ' + 'features and is being provided as-is with no warranties. Beta features are not subject to the support SLA of official GA ' + 'features'
      })
    });
  } else if (item.stage === 'experimental') {
    return /*#__PURE__*/React.createElement(EuiBetaBadge, {
      className: "visListingTable__experimentalIcon",
      label: "E",
      title: i18n.translate('visualize.listing.experimentalTitle', {
        defaultMessage: 'Experimental'
      }),
      tooltipContent: i18n.translate('visualize.listing.experimentalTooltip', {
        defaultMessage: 'This visualization might be changed or removed in a future release and is not subject to the support SLA.'
      })
    });
  }
};

var renderItemTypeIcon = function renderItemTypeIcon(item) {
  var icon;

  if (item.image) {
    icon = /*#__PURE__*/React.createElement("img", {
      className: "visListingTable__typeImage",
      "aria-hidden": "true",
      alt: "",
      src: item.image
    });
  } else {
    icon = /*#__PURE__*/React.createElement(EuiIcon, {
      className: "visListingTable__typeIcon",
      "aria-hidden": "true",
      type: item.icon || 'empty',
      size: "m"
    });
  }

  return icon;
};

export var getTableColumns = function getTableColumns(application, history) {
  return [{
    field: 'title',
    name: i18n.translate('visualize.listing.table.titleColumnName', {
      defaultMessage: 'Title'
    }),
    sortable: true,
    render: function render(field, _ref) {
      var editApp = _ref.editApp,
          editUrl = _ref.editUrl,
          title = _ref.title;
      return /*#__PURE__*/React.createElement(EuiLink, {
        onClick: function onClick() {
          if (editApp) {
            application.navigateToApp(editApp, {
              path: editUrl
            });
          } else if (editUrl) {
            history.push(editUrl);
          }
        },
        "data-test-subj": "visListingTitleLink-".concat(title.split(' ').join('-'))
      }, field);
    }
  }, {
    field: 'typeTitle',
    name: i18n.translate('visualize.listing.table.typeColumnName', {
      defaultMessage: 'Type'
    }),
    sortable: true,
    render: function render(field, record) {
      return /*#__PURE__*/React.createElement("span", null, renderItemTypeIcon(record), record.typeTitle, getBadge(record));
    }
  }, {
    field: 'description',
    name: i18n.translate('visualize.listing.table.descriptionColumnName', {
      defaultMessage: 'Description'
    }),
    sortable: true,
    render: function render(field, record) {
      return /*#__PURE__*/React.createElement("span", null, record.description);
    }
  }];
};
export var getNoItemsMessage = function getNoItemsMessage(createItem) {
  return /*#__PURE__*/React.createElement(EuiEmptyPrompt, {
    iconType: "visualizeApp",
    title: /*#__PURE__*/React.createElement("h1", {
      id: "visualizeListingHeading"
    }, /*#__PURE__*/React.createElement(FormattedMessage, {
      id: "visualize.listing.createNew.title",
      defaultMessage: "Create your first visualization"
    })),
    body: /*#__PURE__*/React.createElement("p", null, /*#__PURE__*/React.createElement(FormattedMessage, {
      id: "visualize.listing.createNew.description",
      defaultMessage: "You can create different visualizations based on your data."
    })),
    actions: /*#__PURE__*/React.createElement(EuiButton, {
      onClick: createItem,
      fill: true,
      iconType: "plusInCircle",
      "data-test-subj": "createVisualizationPromptButton"
    }, /*#__PURE__*/React.createElement(FormattedMessage, {
      id: "visualize.listing.createNew.createButtonLabel",
      defaultMessage: "Create new visualization"
    }))
  });
};