# Copyright 2000 by Iddo Friedberg
#
# This file is part of the Biopython distribution and governed by your
# choice of the "Biopython License Agreement" or the "BSD 3-Clause License".
# Please see the LICENSE file that should have been included as part of this
# package.

"""Substitution matrices for use in alignments, etc.

The information on this page was originally obtained from Dr. Gerhard Vogt's
page http://www.embl-heidelberg.de/~vogt/matrices/mlist1.html (dead link),
and was extracted using a script.

You can view an archive copy of this webpage from 1999 here;
https://web.archive.org/web/19991014010917/http://www.embl-heidelberg.de/%7Evogt/matrices/mlist1.html
"""

# The data rich dictionaries do not lend themselves to black style, turn it off:
# fmt: off

# a list of all available substitution matrices
available_matrices = ["benner6", "benner22", "benner74", "blosum100",
                      "blosum30", "blosum35", "blosum40", "blosum45",
                      "blosum50", "blosum55", "blosum60", "blosum62",
                      "blosum65", "blosum70", "blosum75", "blosum80",
                      "blosum85", "blosum90", "blosum95", "feng",
                      "fitch", "genetic", "gonnet", "grant",
                      "ident", "johnson", "levin", "mclach",
                      "miyata", "nwsgappep", "pam120", "pam180",
                      "pam250", "pam30", "pam300", "pam60",
                      "pam90", "rao", "risler", "structure",
                      ]

# http://www.embl-heidelberg.de/~vogt/matrices/benner6.cmp
benner6 = {
    ("W", "F"): -1.6, ("L", "R"): -3.2, ("I", "I"): 4.4, ("Q", "Q"): 5.3,
    ("W", "N"): -4.4, ("V", "I"): 3.9, ("H", "T"): -1.7, ("H", "P"): -0.4,
    ("W", "V"): -4.8, ("Q", "E"): 2.1, ("W", "R"): 2.0, ("Q", "A"): -1.7,
    ("H", "H"): 6.1, ("H", "D"): 0.1, ("L", "N"): -3.4, ("Y", "M"): -3.6,
    ("Y", "I"): -3.3, ("Y", "E"): -4.1, ("E", "S"): -1.2, ("Y", "A"): -4.0,
    ("Y", "Y"): 9.5, ("T", "C"): -1.5, ("E", "C"): -4.7, ("Y", "Q"): -1.4,
    ("E", "G"): 0.5, ("V", "A"): 0.7, ("C", "C"): 12.1, ("M", "R"): -3.0,
    ("P", "T"): 0.6, ("V", "E"): -3.0, ("P", "P"): 6.5, ("I", "T"): 0.7,
    ("K", "S"): -1.2, ("R", "G"): -0.1, ("I", "P"): -2.0, ("R", "C"): -0.4,
    ("A", "T"): 1.7, ("K", "K"): 5.6, ("A", "P"): 1.1, ("V", "M"): 3.3,
    ("I", "D"): -4.2, ("K", "C"): -2.8, ("K", "G"): -1.4, ("R", "S"): -0.9,
    ("F", "Q"): -4.4, ("F", "A"): -3.2, ("V", "V"): 4.0, ("M", "N"): -2.5,
    ("F", "E"): -6.7, ("D", "N"): 2.5, ("F", "I"): 0.0, ("F", "M"): -0.1,
    ("M", "S"): -1.3, ("S", "S"): 2.1, ("L", "Q"): -2.4, ("W", "E"): -5.6,
    ("W", "A"): -4.3, ("W", "M"): -4.4, ("H", "S"): -0.9, ("W", "I"): -5.0,
    ("S", "C"): 0.9, ("L", "A"): -1.3, ("L", "E"): -5.0, ("W", "Q"): -2.6,
    ("H", "G"): -2.1, ("Q", "N"): 0.1, ("H", "C"): -1.2, ("L", "M"): -2.9,
    ("W", "Y"): -0.3, ("Y", "N"): -0.9, ("E", "P"): -2.6, ("Y", "F"): 5.6,
    ("E", "T"): -1.6, ("A", "A"): 2.5, ("I", "N"): -2.5, ("G", "A"): 0.8,
    ("Y", "V"): -3.8, ("E", "D"): 4.4, ("W", "H"): -2.8, ("Y", "R"): -2.6,
    ("M", "Q"): -3.1, ("P", "S"): 1.4, ("R", "H"): 1.8, ("A", "C"): -1.7,
    ("R", "D"): -1.5, ("K", "P"): -2.3, ("L", "D"): -5.3, ("K", "T"): -1.1,
    ("V", "N"): -2.4, ("M", "A"): -0.2, ("K", "H"): 0.9, ("V", "R"): -3.7,
    ("P", "C"): -2.7, ("M", "E"): -4.1, ("A", "S"): 1.4, ("T", "T"): 2.4,
    ("R", "T"): -1.3, ("I", "G"): -3.4, ("R", "P"): -1.3, ("K", "D"): -0.2,
    ("I", "C"): -3.6, ("F", "R"): -4.9, ("F", "V"): -0.5, ("L", "C"): -3.8,
    ("F", "F"): 8.3, ("D", "A"): -0.6, ("F", "N"): -3.5, ("W", "D"): -6.3,
    ("L", "P"): -0.2, ("Q", "S"): -1.4, ("N", "C"): -1.6, ("N", "G"): -0.1,
    ("H", "N"): 1.4, ("W", "T"): -2.6, ("Q", "G"): -1.6, ("W", "P"): -4.8,
    ("Q", "C"): -3.2, ("N", "S"): 1.2, ("L", "H"): -2.2, ("L", "L"): 4.8,
    ("G", "T"): -0.5, ("M", "M"): 4.8, ("G", "P"): -1.7, ("Y", "K"): -4.0,
    ("Y", "G"): -4.9, ("Y", "C"): 2.6, ("E", "A"): -0.7, ("E", "E"): 5.2,
    ("Y", "S"): -1.8, ("M", "P"): -1.8, ("V", "C"): -3.1, ("M", "T"): 0.6,
    ("V", "G"): -2.3, ("R", "E"): -0.4, ("V", "K"): -3.8, ("K", "Q"): 2.5,
    ("R", "A"): -1.7, ("I", "R"): -3.8, ("N", "A"): 0.0, ("V", "S"): -0.9,
    ("M", "D"): -4.3, ("M", "H"): -3.4, ("K", "A"): -1.9, ("R", "Q"): 2.5,
    ("K", "E"): 0.9, ("F", "S"): -1.8, ("I", "K"): -3.8, ("D", "P"): -2.8,
    ("D", "T"): -1.2, ("I", "M"): 4.0, ("F", "C"): -0.1, ("W", "L"): -3.0,
    ("F", "G"): -5.7, ("F", "K"): -6.3, ("F", "T"): -2.4, ("D", "D"): 5.2,
    ("Q", "T"): -1.7, ("W", "G"): -1.7, ("Q", "P"): 0.1, ("W", "C"): 1.6,
    ("W", "K"): -1.4, ("H", "Q"): 3.2, ("Q", "D"): 0.6, ("W", "W"): 14.7,
    ("V", "L"): 1.9, ("L", "G"): -4.6, ("W", "S"): -2.9, ("L", "K"): -4.1,
    ("N", "P"): -1.1, ("H", "E"): -0.2, ("N", "T"): 0.5, ("H", "A"): -2.1,
    ("Y", "L"): -1.6, ("Y", "H"): 4.4, ("G", "S"): 0.8, ("Y", "D"): -2.3,
    ("V", "Q"): -3.5, ("L", "T"): -0.4, ("G", "G"): 5.8, ("G", "C"): -1.3,
    ("E", "N"): 1.1, ("Y", "T"): -3.4, ("Y", "P"): -3.8, ("R", "N"): -0.1,
    ("V", "D"): -3.3, ("K", "R"): 4.3, ("V", "H"): -3.8, ("I", "Q"): -3.8,
    ("V", "P"): -1.6, ("M", "C"): -3.7, ("K", "N"): 1.0, ("V", "T"): 0.6,
    ("M", "G"): -3.7, ("T", "S"): 1.5, ("I", "E"): -4.1, ("M", "K"): -2.9,
    ("I", "A"): 0.1, ("N", "N"): 3.6, ("R", "R"): 5.1, ("F", "P"): -3.2,
    ("L", "I"): 2.4, ("I", "S"): -1.2, ("D", "S"): -0.4, ("L", "S"): -1.5,
    ("I", "H"): -3.7, ("F", "D"): -5.7, ("D", "C"): -3.7, ("F", "H"): 0.1,
    ("D", "G"): 0.8, ("F", "L"): 2.4
}


# http://www.embl-heidelberg.de/~vogt/matrices/benner22.cmp
benner22 = {
    ("W", "F"): 0.5, ("L", "R"): -2.9, ("I", "I"): 4.2, ("Q", "Q"): 4.2,
    ("W", "N"): -5.2, ("V", "I"): 3.6, ("H", "T"): -1.1, ("H", "P"): -0.4,
    ("W", "V"): -4.5, ("Q", "E"): 1.7, ("W", "R"): -1.1, ("Q", "A"): -0.9,
    ("H", "H"): 6.1, ("H", "D"): 0.3, ("L", "N"): -3.5, ("Y", "M"): -1.8,
    ("Y", "I"): -2.2, ("Y", "E"): -4.0, ("E", "S"): -0.5, ("Y", "A"): -3.5,
    ("Y", "Y"): 9.0, ("T", "C"): -1.1, ("E", "C"): -4.3, ("Y", "Q"): -1.9,
    ("E", "G"): 0.5, ("V", "A"): 0.4, ("C", "C"): 12.6, ("M", "R"): -2.1,
    ("P", "T"): 0.4, ("V", "E"): -2.7, ("P", "P"): 7.0, ("I", "T"): 0.3,
    ("K", "S"): -0.4, ("R", "G"): -0.7, ("I", "P"): -2.3, ("R", "C"): -1.6,
    ("A", "T"): 1.4, ("K", "K"): 4.4, ("A", "P"): 0.8, ("V", "M"): 2.5,
    ("I", "D"): -4.0, ("K", "C"): -3.3, ("K", "G"): -1.0, ("R", "S"): -0.5,
    ("F", "Q"): -3.6, ("F", "A"): -3.1, ("V", "V"): 3.7, ("M", "N"): -2.6,
    ("F", "E"): -5.7, ("D", "N"): 2.4, ("F", "I"): 0.5, ("F", "M"): 0.7,
    ("M", "S"): -1.5, ("S", "S"): 2.0, ("L", "Q"): -2.0, ("W", "E"): -6.3,
    ("W", "A"): -5.5, ("W", "M"): -2.8, ("H", "S"): -0.5, ("W", "I"): -4.4,
    ("S", "C"): 0.3, ("L", "A"): -1.7, ("L", "E"): -4.4, ("W", "Q"): -3.3,
    ("H", "G"): -2.0, ("Q", "N"): 0.5, ("H", "C"): -1.5, ("L", "M"): 3.2,
    ("W", "Y"): 1.5, ("Y", "N"): -1.2, ("E", "P"): -1.7, ("Y", "F"): 5.9,
    ("E", "T"): -0.9, ("A", "A"): 2.5, ("I", "N"): -2.7, ("G", "A"): 0.8,
    ("Y", "V"): -2.6, ("E", "D"): 3.9, ("W", "H"): -2.7, ("Y", "R"): -2.7,
    ("M", "Q"): -1.7, ("P", "S"): 1.1, ("R", "H"): 1.5, ("A", "C"): -1.2,
    ("R", "D"): -1.0, ("K", "P"): -1.6, ("L", "D"): -4.9, ("K", "T"): -0.4,
    ("V", "N"): -2.3, ("M", "A"): -0.8, ("K", "H"): 0.8, ("V", "R"): -2.9,
    ("P", "C"): -3.1, ("M", "E"): -3.4, ("A", "S"): 1.3, ("T", "T"): 2.5,
    ("R", "T"): -0.7, ("I", "G"): -3.8, ("R", "P"): -1.2, ("K", "D"): 0.2,
    ("I", "C"): -2.4, ("F", "R"): -4.3, ("F", "V"): -0.1, ("L", "C"): -2.6,
    ("F", "F"): 7.7, ("D", "A"): -0.2, ("F", "N"): -3.5, ("W", "D"): -6.4,
    ("L", "P"): -1.3, ("Q", "S"): -0.6, ("N", "C"): -1.9, ("N", "G"): 0.4,
    ("H", "N"): 1.4, ("W", "T"): -4.5, ("Q", "G"): -1.4, ("W", "P"): -5.8,
    ("Q", "C"): -3.3, ("N", "S"): 1.1, ("L", "H"): -2.1, ("L", "L"): 4.6,
    ("G", "T"): -0.7, ("M", "M"): 4.9, ("G", "P"): -1.8, ("Y", "K"): -3.6,
    ("Y", "G"): -4.8, ("Y", "C"): 0.6, ("E", "A"): -0.3, ("E", "E"): 4.6,
    ("Y", "S"): -1.9, ("M", "P"): -2.0, ("V", "C"): -1.7, ("M", "T"): 0.1,
    ("V", "G"): -2.5, ("R", "E"): -0.1, ("V", "K"): -2.7, ("K", "Q"): 2.2,
    ("R", "A"): -1.2, ("I", "R"): -3.2, ("N", "A"): 0.0, ("V", "S"): -0.9,
    ("M", "D"): -3.9, ("M", "H"): -2.4, ("K", "A"): -1.0, ("R", "Q"): 2.2,
    ("K", "E"): 1.0, ("F", "S"): -2.2, ("I", "K"): -3.0, ("D", "P"): -1.8,
    ("D", "T"): -0.7, ("I", "M"): 3.1, ("F", "C"): -0.1, ("W", "L"): -1.8,
    ("F", "G"): -5.8, ("F", "K"): -5.1, ("F", "T"): -2.6, ("D", "D"): 4.8,
    ("Q", "T"): -0.7, ("W", "G"): -4.5, ("Q", "P"): -0.1, ("W", "C"): 0.5,
    ("W", "K"): -3.7, ("H", "Q"): 2.4, ("Q", "D"): 0.6, ("W", "W"): 15.7,
    ("V", "L"): 2.0, ("L", "G"): -4.9, ("W", "S"): -3.9, ("L", "K"): -3.3,
    ("N", "P"): -1.1, ("H", "E"): -0.2, ("N", "T"): 0.5, ("H", "A"): -1.6,
    ("Y", "L"): -0.7, ("Y", "H"): 3.7, ("G", "S"): 0.6, ("Y", "D"): -3.0,
    ("V", "Q"): -2.4, ("L", "T"): -1.0, ("G", "G"): 6.2, ("G", "C"): -1.7,
    ("E", "N"): 1.2, ("Y", "T"): -3.0, ("Y", "P"): -3.5, ("R", "N"): 0.4,
    ("V", "D"): -3.0, ("K", "R"): 3.9, ("V", "H"): -3.0, ("I", "Q"): -2.7,
    ("V", "P"): -1.7, ("M", "C"): -2.5, ("K", "N"): 1.0, ("V", "T"): 0.4,
    ("M", "G"): -3.8, ("T", "S"): 1.5, ("I", "E"): -3.6, ("M", "K"): -2.0,
    ("I", "A"): -0.4, ("N", "N"): 3.3, ("R", "R"): 5.0, ("F", "P"): -3.4,
    ("L", "I"): 2.7, ("I", "S"): -1.4, ("D", "S"): 0.1, ("L", "S"): -2.1,
    ("I", "H"): -3.2, ("F", "D"): -5.4, ("D", "C"): -3.7, ("F", "H"): 0.3,
    ("D", "G"): 0.7, ("F", "L"): 2.2
}


assert benner6 != benner22


# http://www.embl-heidelberg.de/~vogt/matrices/benner74.cmp
benner74 = {
    ("W", "F"): 3.0, ("L", "R"): -2.4, ("I", "I"): 4.0, ("Q", "Q"): 3.0,
    ("W", "N"): -4.0, ("V", "I"): 3.2, ("H", "T"): -0.5, ("H", "P"): -1.0,
    ("W", "V"): -2.9, ("Q", "E"): 1.7, ("W", "R"): -1.6, ("Q", "A"): -0.3,
    ("H", "H"): 6.1, ("H", "D"): 0.4, ("L", "N"): -3.1, ("Y", "M"): -0.5,
    ("Y", "I"): -1.0, ("Y", "E"): -3.0, ("E", "S"): 0.1, ("Y", "A"): -2.6,
    ("Y", "Y"): 8.1, ("T", "C"): -0.6, ("E", "C"): -3.2, ("Y", "Q"): -1.8,
    ("E", "G"): -0.5, ("V", "A"): 0.1, ("C", "C"): 11.8, ("M", "R"): -1.8,
    ("P", "T"): 0.1, ("V", "E"): -2.1, ("P", "P"): 7.5, ("I", "T"): -0.3,
    ("K", "S"): 0.0, ("R", "G"): -1.0, ("I", "P"): -2.6, ("R", "C"): -2.2,
    ("A", "T"): 0.7, ("K", "K"): 3.4, ("A", "P"): 0.4, ("V", "M"): 1.8,
    ("I", "D"): -3.9, ("K", "C"): -2.9, ("K", "G"): -1.1, ("R", "S"): -0.2,
    ("F", "Q"): -2.8, ("F", "A"): -2.6, ("V", "V"): 3.4, ("M", "N"): -2.2,
    ("F", "E"): -4.3, ("D", "N"): 2.2, ("F", "I"): 0.9, ("F", "M"): 1.3,
    ("M", "S"): -1.4, ("S", "S"): 2.1, ("L", "Q"): -1.7, ("W", "E"): -4.7,
    ("W", "A"): -4.1, ("W", "M"): -1.3, ("H", "S"): -0.3, ("W", "I"): -2.3,
    ("S", "C"): 0.1, ("L", "A"): -1.4, ("L", "E"): -3.1, ("W", "Q"): -2.8,
    ("H", "G"): -1.6, ("Q", "N"): 0.7, ("H", "C"): -1.3, ("L", "M"): 2.9,
    ("W", "Y"): 3.6, ("Y", "N"): -1.4, ("E", "P"): -0.7, ("Y", "F"): 5.3,
    ("E", "T"): -0.2, ("A", "A"): 2.4, ("I", "N"): -2.8, ("G", "A"): 0.6,
    ("Y", "V"): -1.4, ("E", "D"): 2.9, ("W", "H"): -1.0, ("Y", "R"): -2.0,
    ("M", "Q"): -1.0, ("P", "S"): 0.5, ("R", "H"): 1.0, ("A", "C"): 0.3,
    ("R", "D"): -0.5, ("K", "P"): -0.8, ("L", "D"): -4.2, ("K", "T"): 0.1,
    ("V", "N"): -2.2, ("M", "A"): -0.8, ("K", "H"): 0.6, ("V", "R"): -2.2,
    ("P", "C"): -3.1, ("M", "E"): -2.2, ("A", "S"): 1.1, ("T", "T"): 2.5,
    ("R", "T"): -0.3, ("I", "G"): -4.3, ("R", "P"): -0.1, ("K", "D"): 0.4,
    ("I", "C"): -1.2, ("F", "R"): -3.5, ("F", "V"): 0.1, ("L", "C"): -1.6,
    ("F", "F"): 7.2, ("D", "A"): -0.3, ("F", "N"): -3.2, ("W", "D"): -5.5,
    ("L", "P"): -2.2, ("Q", "S"): 0.1, ("N", "C"): -1.8, ("N", "G"): 0.4,
    ("H", "N"): 1.2, ("W", "T"): -3.7, ("Q", "G"): -1.1, ("W", "P"): -5.2,
    ("Q", "C"): -2.6, ("N", "S"): 0.9, ("L", "H"): -1.9, ("L", "L"): 4.2,
    ("G", "T"): -1.0, ("M", "M"): 4.5, ("G", "P"): -1.7, ("Y", "K"): -2.4,
    ("Y", "G"): -4.3, ("Y", "C"): -0.4, ("E", "A"): -0.1, ("E", "E"): 3.7,
    ("Y", "S"): -1.9, ("M", "P"): -2.4, ("V", "C"): -0.2, ("M", "T"): -0.4,
    ("V", "G"): -3.1, ("R", "E"): 0.3, ("V", "K"): -1.9, ("K", "Q"): 1.7,
    ("R", "A"): -0.8, ("I", "R"): -2.6, ("N", "A"): -0.2, ("V", "S"): -1.0,
    ("M", "D"): -3.2, ("M", "H"): -1.5, ("K", "A"): -0.4, ("R", "Q"): 1.6,
    ("K", "E"): 1.2, ("F", "S"): -2.6, ("I", "K"): -2.3, ("D", "P"): -1.0,
    ("D", "T"): -0.2, ("I", "M"): 2.6, ("F", "C"): -0.7, ("W", "L"): -0.9,
    ("F", "G"): -5.4, ("F", "K"): -3.6, ("F", "T"): -2.2, ("D", "D"): 4.8,
    ("Q", "T"): -0.1, ("W", "G"): -4.1, ("Q", "P"): -0.2, ("W", "C"): -0.9,
    ("W", "K"): -3.6, ("H", "Q"): 1.4, ("Q", "D"): 0.8, ("W", "W"): 14.7,
    ("V", "L"): 1.9, ("L", "G"): -4.6, ("W", "S"): -3.4, ("L", "K"): -2.4,
    ("N", "P"): -1.0, ("H", "E"): 0.2, ("N", "T"): 0.4, ("H", "A"): -1.0,
    ("Y", "L"): -0.1, ("Y", "H"): 2.5, ("G", "S"): 0.4, ("Y", "D"): -2.8,
    ("V", "Q"): -1.7, ("L", "T"): -1.1, ("G", "G"): 6.6, ("G", "C"): -2.0,
    ("E", "N"): 1.0, ("Y", "T"): -2.1, ("Y", "P"): -3.4, ("R", "N"): 0.3,
    ("V", "D"): -2.9, ("K", "R"): 2.9, ("V", "H"): -2.1, ("I", "Q"): -2.0,
    ("V", "P"): -1.9, ("M", "C"): -1.2, ("K", "N"): 0.9, ("V", "T"): 0.2,
    ("M", "G"): -3.5, ("T", "S"): 1.4, ("I", "E"): -2.9, ("M", "K"): -1.5,
    ("I", "A"): -0.8, ("N", "N"): 3.6, ("R", "R"): 4.8, ("F", "P"): -3.8,
    ("L", "I"): 2.8, ("I", "S"): -1.8, ("D", "S"): 0.4, ("L", "S"): -2.2,
    ("I", "H"): -2.3, ("F", "D"): -4.7, ("D", "C"): -3.2, ("F", "H"): 0.0,
    ("D", "G"): 0.2, ("F", "L"): 2.1
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum100.cmp
blosum100 = {
    ("W", "F"): 0, ("L", "R"): -4, ("S", "P"): -2, ("V", "T"): -1,
    ("Q", "Q"): 7, ("N", "A"): -2, ("Z", "Y"): -4, ("W", "R"): -4,
    ("Q", "A"): -1, ("S", "D"): -1, ("H", "H"): 9, ("S", "H"): -2,
    ("H", "D"): -2, ("L", "N"): -5, ("W", "A"): -4, ("Y", "M"): -3,
    ("G", "R"): -4, ("Y", "I"): -3, ("Y", "E"): -4, ("B", "Y"): -4,
    ("Y", "A"): -4, ("V", "D"): -5, ("B", "S"): -1, ("Y", "Y"): 8,
    ("G", "N"): -2, ("E", "C"): -6, ("Y", "Q"): -3, ("Z", "Z"): 4,
    ("V", "A"): -1, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): -1, ("P", "P"): 8, ("V", "I"): 2, ("V", "S"): -3,
    ("Z", "P"): -3, ("V", "M"): 0, ("T", "F"): -3, ("V", "Q"): -3,
    ("K", "K"): 6, ("P", "D"): -3, ("I", "H"): -5, ("I", "D"): -6,
    ("T", "R"): -2, ("P", "L"): -4, ("K", "G"): -3, ("M", "N"): -4,
    ("P", "H"): -3, ("F", "Q"): -4, ("Z", "G"): -4, ("X", "L"): -2,
    ("T", "M"): -2, ("Z", "C"): -6, ("X", "H"): -2, ("D", "R"): -3,
    ("B", "W"): -6, ("X", "D"): -3, ("Z", "K"): 0, ("F", "A"): -4,
    ("Z", "W"): -4, ("F", "E"): -5, ("D", "N"): 1, ("B", "K"): -1,
    ("X", "X"): -2, ("F", "I"): -1, ("B", "G"): -2, ("X", "T"): -1,
    ("F", "M"): -1, ("B", "C"): -5, ("Z", "I"): -4, ("Z", "V"): -3,
    ("S", "S"): 6, ("L", "Q"): -3, ("W", "E"): -5, ("Q", "R"): 0,
    ("N", "N"): 7, ("W", "M"): -3, ("Q", "C"): -5, ("W", "I"): -4,
    ("S", "C"): -2, ("L", "A"): -3, ("S", "G"): -1, ("L", "E"): -5,
    ("W", "Q"): -3, ("H", "G"): -4, ("S", "K"): -1, ("Q", "N"): -1,
    ("N", "R"): -1, ("H", "C"): -5, ("Y", "N"): -3, ("G", "Q"): -3,
    ("Y", "F"): 3, ("C", "A"): -1, ("V", "L"): 0, ("G", "E"): -4,
    ("G", "A"): -1, ("K", "R"): 2, ("E", "D"): 1, ("Y", "R"): -3,
    ("M", "Q"): -1, ("T", "I"): -2, ("C", "D"): -5, ("V", "F"): -2,
    ("T", "A"): -1, ("T", "P"): -3, ("B", "P"): -3, ("T", "E"): -2,
    ("V", "N"): -4, ("P", "G"): -4, ("M", "A"): -2, ("K", "H"): -2,
    ("V", "R"): -4, ("P", "C"): -5, ("M", "E"): -4, ("K", "L"): -4,
    ("V", "V"): 5, ("M", "I"): 1, ("T", "Q"): -2, ("I", "G"): -6,
    ("P", "K"): -2, ("M", "M"): 8, ("K", "D"): -2, ("I", "C"): -2,
    ("Z", "D"): 0, ("F", "R"): -4, ("X", "K"): -2, ("Q", "D"): -2,
    ("X", "G"): -3, ("Z", "L"): -4, ("X", "C"): -3, ("Z", "H"): -1,
    ("B", "L"): -5, ("B", "H"): -1, ("F", "F"): 7, ("X", "W"): -4,
    ("B", "D"): 4, ("D", "A"): -3, ("S", "L"): -4, ("X", "S"): -1,
    ("F", "N"): -5, ("S", "R"): -2, ("W", "D"): -7, ("V", "Y"): -3,
    ("W", "L"): -4, ("H", "R"): -1, ("W", "H"): -3, ("H", "N"): 0,
    ("W", "T"): -5, ("T", "T"): 6, ("S", "F"): -3, ("W", "P"): -6,
    ("L", "D"): -6, ("B", "I"): -5, ("L", "H"): -4, ("S", "N"): 0,
    ("B", "T"): -2, ("L", "L"): 5, ("Y", "K"): -4, ("E", "Q"): 1,
    ("Y", "G"): -6, ("Z", "S"): -1, ("Y", "C"): -4, ("G", "D"): -3,
    ("B", "V"): -5, ("E", "A"): -2, ("Y", "W"): 1, ("E", "E"): 6,
    ("Y", "S"): -3, ("C", "N"): -4, ("V", "C"): -2, ("T", "H"): -3,
    ("P", "R"): -3, ("V", "G"): -5, ("T", "L"): -3, ("V", "K"): -4,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -4, ("T", "D"): -2,
    ("P", "F"): -5, ("I", "N"): -5, ("K", "I"): -4, ("M", "D"): -5,
    ("V", "W"): -4, ("W", "W"): 11, ("M", "H"): -3, ("P", "N"): -4,
    ("K", "A"): -2, ("M", "L"): 2, ("K", "E"): 0, ("Z", "E"): 5,
    ("X", "N"): -2, ("Z", "A"): -2, ("Z", "M"): -3, ("X", "F"): -3,
    ("K", "C"): -5, ("B", "Q"): -1, ("X", "B"): -2, ("B", "M"): -4,
    ("F", "C"): -3, ("Z", "Q"): 3, ("X", "Z"): -2, ("F", "G"): -5,
    ("B", "E"): 0, ("X", "V"): -2, ("F", "K"): -4, ("B", "A"): -3,
    ("X", "R"): -2, ("D", "D"): 7, ("W", "G"): -5, ("Z", "F"): -5,
    ("S", "Q"): -1, ("W", "C"): -5, ("W", "K"): -5, ("H", "Q"): 0,
    ("L", "C"): -3, ("W", "N"): -6, ("S", "A"): 1, ("L", "G"): -5,
    ("W", "S"): -4, ("S", "E"): -1, ("H", "E"): -1, ("S", "I"): -4,
    ("H", "A"): -3, ("S", "M"): -3, ("Y", "L"): -3, ("Y", "H"): 1,
    ("Y", "D"): -5, ("E", "R"): -2, ("X", "P"): -3, ("G", "G"): 6,
    ("G", "C"): -5, ("E", "N"): -1, ("Y", "T"): -3, ("Y", "P"): -5,
    ("T", "K"): -2, ("A", "A"): 5, ("P", "Q"): -2, ("T", "C"): -2,
    ("V", "H"): -5, ("T", "G"): -3, ("I", "Q"): -4, ("Z", "T"): -2,
    ("C", "R"): -5, ("V", "P"): -4, ("P", "E"): -3, ("M", "C"): -3,
    ("K", "N"): -1, ("I", "I"): 5, ("P", "A"): -1, ("M", "G"): -5,
    ("T", "S"): 1, ("I", "E"): -5, ("P", "M"): -4, ("M", "K"): -2,
    ("I", "A"): -3, ("P", "I"): -4, ("R", "R"): 7, ("X", "M"): -2,
    ("L", "I"): 1, ("X", "I"): -2, ("Z", "B"): 1, ("X", "E"): -2,
    ("Z", "N"): -1, ("X", "A"): -1, ("B", "R"): -2, ("B", "N"): 4,
    ("F", "D"): -5, ("X", "Y"): -3, ("Z", "R"): -1, ("F", "H"): -2,
    ("B", "F"): -5, ("F", "L"): 0, ("X", "Q"): -2, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum30.cmp
blosum30 = {
    ("W", "F"): 1, ("L", "R"): -2, ("S", "P"): -1, ("V", "T"): 1,
    ("Q", "Q"): 8, ("N", "A"): 0, ("Z", "Y"): -2, ("W", "R"): 0,
    ("Q", "A"): 1, ("S", "D"): 0, ("H", "H"): 14, ("S", "H"): -1,
    ("H", "D"): -2, ("L", "N"): -2, ("W", "A"): -5, ("Y", "M"): -1,
    ("G", "R"): -2, ("Y", "I"): -1, ("Y", "E"): -2, ("B", "Y"): -3,
    ("Y", "A"): -4, ("V", "D"): -2, ("B", "S"): 0, ("Y", "Y"): 9,
    ("G", "N"): 0, ("E", "C"): 1, ("Y", "Q"): -1, ("Z", "Z"): 4,
    ("V", "A"): 1, ("C", "C"): 17, ("M", "R"): 0, ("V", "E"): -3,
    ("T", "N"): 1, ("P", "P"): 11, ("V", "I"): 4, ("V", "S"): -1,
    ("Z", "P"): 0, ("V", "M"): 0, ("T", "F"): -2, ("V", "Q"): -3,
    ("K", "K"): 4, ("P", "D"): -1, ("I", "H"): -2, ("I", "D"): -4,
    ("T", "R"): -3, ("P", "L"): -3, ("K", "G"): -1, ("M", "N"): 0,
    ("P", "H"): 1, ("F", "Q"): -3, ("Z", "G"): -2, ("X", "L"): 0,
    ("T", "M"): 0, ("Z", "C"): 0, ("X", "H"): -1, ("D", "R"): -1,
    ("B", "W"): -5, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -2,
    ("Z", "W"): -1, ("F", "E"): -4, ("D", "N"): 1, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): 0, ("X", "T"): 0,
    ("F", "M"): -2, ("B", "C"): -2, ("Z", "I"): -3, ("Z", "V"): -3,
    ("S", "S"): 4, ("L", "Q"): -2, ("W", "E"): -1, ("Q", "R"): 3,
    ("N", "N"): 8, ("W", "M"): -3, ("Q", "C"): -2, ("W", "I"): -3,
    ("S", "C"): -2, ("L", "A"): -1, ("S", "G"): 0, ("L", "E"): -1,
    ("W", "Q"): -1, ("H", "G"): -3, ("S", "K"): 0, ("Q", "N"): -1,
    ("N", "R"): -2, ("H", "C"): -5, ("Y", "N"): -4, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): -3, ("V", "L"): 1, ("G", "E"): -2,
    ("G", "A"): 0, ("K", "R"): 1, ("E", "D"): 1, ("Y", "R"): 0,
    ("M", "Q"): -1, ("T", "I"): 0, ("C", "D"): -3, ("V", "F"): 1,
    ("T", "A"): 1, ("T", "P"): 0, ("B", "P"): -2, ("T", "E"): -2,
    ("V", "N"): -2, ("P", "G"): -1, ("M", "A"): 1, ("K", "H"): -2,
    ("V", "R"): -1, ("P", "C"): -3, ("M", "E"): -1, ("K", "L"): -2,
    ("V", "V"): 5, ("M", "I"): 1, ("T", "Q"): 0, ("I", "G"): -1,
    ("P", "K"): 1, ("M", "M"): 6, ("K", "D"): 0, ("I", "C"): -2,
    ("Z", "D"): 0, ("F", "R"): -1, ("X", "K"): 0, ("Q", "D"): -1,
    ("X", "G"): -1, ("Z", "L"): -1, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -1, ("B", "H"): -2, ("F", "F"): 10, ("X", "W"): -2,
    ("B", "D"): 5, ("D", "A"): 0, ("S", "L"): -2, ("X", "S"): 0,
    ("F", "N"): -1, ("S", "R"): -1, ("W", "D"): -4, ("V", "Y"): 1,
    ("W", "L"): -2, ("H", "R"): -1, ("W", "H"): -5, ("H", "N"): -1,
    ("W", "T"): -5, ("T", "T"): 5, ("S", "F"): -1, ("W", "P"): -3,
    ("L", "D"): -1, ("B", "I"): -2, ("L", "H"): -1, ("S", "N"): 0,
    ("B", "T"): 0, ("L", "L"): 4, ("Y", "K"): -1, ("E", "Q"): 2,
    ("Y", "G"): -3, ("Z", "S"): -1, ("Y", "C"): -6, ("G", "D"): -1,
    ("B", "V"): -2, ("E", "A"): 0, ("Y", "W"): 5, ("E", "E"): 6,
    ("Y", "S"): -2, ("C", "N"): -1, ("V", "C"): -2, ("T", "H"): -2,
    ("P", "R"): -1, ("V", "G"): -3, ("T", "L"): 0, ("V", "K"): -2,
    ("K", "Q"): 0, ("R", "A"): -1, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): 0, ("K", "I"): -2, ("M", "D"): -3,
    ("V", "W"): -3, ("W", "W"): 20, ("M", "H"): 2, ("P", "N"): -3,
    ("K", "A"): 0, ("M", "L"): 2, ("K", "E"): 2, ("Z", "E"): 5,
    ("X", "N"): 0, ("Z", "A"): 0, ("Z", "M"): -1, ("X", "F"): -1,
    ("K", "C"): -3, ("B", "Q"): -1, ("X", "B"): -1, ("B", "M"): -2,
    ("F", "C"): -3, ("Z", "Q"): 4, ("X", "Z"): 0, ("F", "G"): -3,
    ("B", "E"): 0, ("X", "V"): 0, ("F", "K"): -1, ("B", "A"): 0,
    ("X", "R"): -1, ("D", "D"): 9, ("W", "G"): 1, ("Z", "F"): -4,
    ("S", "Q"): -1, ("W", "C"): -2, ("W", "K"): -2, ("H", "Q"): 0,
    ("L", "C"): 0, ("W", "N"): -7, ("S", "A"): 1, ("L", "G"): -2,
    ("W", "S"): -3, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -1,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): 3, ("Y", "H"): 0,
    ("Y", "D"): -1, ("E", "R"): -1, ("X", "P"): -1, ("G", "G"): 8,
    ("G", "C"): -4, ("E", "N"): -1, ("Y", "T"): -1, ("Y", "P"): -2,
    ("T", "K"): -1, ("A", "A"): 4, ("P", "Q"): 0, ("T", "C"): -2,
    ("V", "H"): -3, ("T", "G"): -2, ("I", "Q"): -2, ("Z", "T"): -1,
    ("C", "R"): -2, ("V", "P"): -4, ("P", "E"): 1, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 6, ("P", "A"): -1, ("M", "G"): -2,
    ("T", "S"): 2, ("I", "E"): -3, ("P", "M"): -4, ("M", "K"): 2,
    ("I", "A"): 0, ("P", "I"): -3, ("R", "R"): 8, ("X", "M"): 0,
    ("L", "I"): 2, ("X", "I"): 0, ("Z", "B"): 0, ("X", "E"): -1,
    ("Z", "N"): -1, ("X", "A"): 0, ("B", "R"): -2, ("B", "N"): 4,
    ("F", "D"): -5, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -3,
    ("B", "F"): -3, ("F", "L"): 2, ("X", "Q"): 0, ("B", "B"): 5
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum35.cmp
blosum35 = {
    ("W", "F"): 1, ("L", "R"): -2, ("S", "P"): -2, ("V", "T"): 1,
    ("Q", "Q"): 7, ("N", "A"): -1, ("Z", "Y"): -1, ("W", "R"): 0,
    ("Q", "A"): 0, ("S", "D"): -1, ("H", "H"): 12, ("S", "H"): -1,
    ("H", "D"): 0, ("L", "N"): -2, ("W", "A"): -2, ("Y", "M"): 0,
    ("G", "R"): -2, ("Y", "I"): 0, ("Y", "E"): -1, ("B", "Y"): -2,
    ("Y", "A"): -1, ("V", "D"): -2, ("B", "S"): 0, ("Y", "Y"): 8,
    ("G", "N"): 1, ("E", "C"): -1, ("Y", "Q"): 0, ("Z", "Z"): 4,
    ("V", "A"): 0, ("C", "C"): 15, ("M", "R"): 0, ("V", "E"): -2,
    ("T", "N"): 0, ("P", "P"): 10, ("V", "I"): 4, ("V", "S"): -1,
    ("Z", "P"): 0, ("V", "M"): 1, ("T", "F"): -1, ("V", "Q"): -3,
    ("K", "K"): 5, ("P", "D"): -1, ("I", "H"): -3, ("I", "D"): -3,
    ("T", "R"): -2, ("P", "L"): -3, ("K", "G"): -1, ("M", "N"): -1,
    ("P", "H"): -1, ("F", "Q"): -4, ("Z", "G"): -2, ("X", "L"): 0,
    ("T", "M"): 0, ("Z", "C"): -2, ("X", "H"): -1, ("D", "R"): -1,
    ("B", "W"): -3, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -2,
    ("Z", "W"): -1, ("F", "E"): -3, ("D", "N"): 1, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 1, ("B", "G"): 0, ("X", "T"): 0,
    ("F", "M"): 0, ("B", "C"): -2, ("Z", "I"): -3, ("Z", "V"): -2,
    ("S", "S"): 4, ("L", "Q"): -2, ("W", "E"): -1, ("Q", "R"): 2,
    ("N", "N"): 7, ("W", "M"): 1, ("Q", "C"): -3, ("W", "I"): -1,
    ("S", "C"): -3, ("L", "A"): -2, ("S", "G"): 1, ("L", "E"): -1,
    ("W", "Q"): -1, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 1,
    ("N", "R"): -1, ("H", "C"): -4, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): -2, ("V", "L"): 2, ("G", "E"): -2,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 2, ("Y", "R"): 0,
    ("M", "Q"): -1, ("T", "I"): -1, ("C", "D"): -3, ("V", "F"): 1,
    ("T", "A"): 0, ("T", "P"): 0, ("B", "P"): -1, ("T", "E"): -1,
    ("V", "N"): -2, ("P", "G"): -2, ("M", "A"): 0, ("K", "H"): -2,
    ("V", "R"): -1, ("P", "C"): -4, ("M", "E"): -2, ("K", "L"): -2,
    ("V", "V"): 5, ("M", "I"): 1, ("T", "Q"): 0, ("I", "G"): -3,
    ("P", "K"): 0, ("M", "M"): 6, ("K", "D"): -1, ("I", "C"): -4,
    ("Z", "D"): 1, ("F", "R"): -1, ("X", "K"): 0, ("Q", "D"): -1,
    ("X", "G"): -1, ("Z", "L"): -2, ("X", "C"): -2, ("Z", "H"): -1,
    ("B", "L"): -2, ("B", "H"): 0, ("F", "F"): 8, ("X", "W"): -1,
    ("B", "D"): 5, ("D", "A"): -1, ("S", "L"): -2, ("X", "S"): 0,
    ("F", "N"): -1, ("S", "R"): -1, ("W", "D"): -3, ("V", "Y"): 0,
    ("W", "L"): 0, ("H", "R"): -1, ("W", "H"): -4, ("H", "N"): 1,
    ("W", "T"): -2, ("T", "T"): 5, ("S", "F"): -1, ("W", "P"): -4,
    ("L", "D"): -2, ("B", "I"): -2, ("L", "H"): -2, ("S", "N"): 0,
    ("B", "T"): -1, ("L", "L"): 5, ("Y", "K"): -1, ("E", "Q"): 2,
    ("Y", "G"): -2, ("Z", "S"): 0, ("Y", "C"): -5, ("G", "D"): -2,
    ("B", "V"): -2, ("E", "A"): -1, ("Y", "W"): 3, ("E", "E"): 6,
    ("Y", "S"): -1, ("C", "N"): -1, ("V", "C"): -2, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -3, ("T", "L"): 0, ("V", "K"): -2,
    ("K", "Q"): 0, ("R", "A"): -1, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -1, ("K", "I"): -2, ("M", "D"): -3,
    ("V", "W"): -2, ("W", "W"): 16, ("M", "H"): 1, ("P", "N"): -2,
    ("K", "A"): 0, ("M", "L"): 3, ("K", "E"): 1, ("Z", "E"): 5,
    ("X", "N"): 0, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -1,
    ("K", "C"): -2, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -2,
    ("F", "C"): -4, ("Z", "Q"): 4, ("X", "Z"): 0, ("F", "G"): -3,
    ("B", "E"): 0, ("X", "V"): 0, ("F", "K"): -1, ("B", "A"): -1,
    ("X", "R"): -1, ("D", "D"): 8, ("W", "G"): -1, ("Z", "F"): -3,
    ("S", "Q"): 0, ("W", "C"): -5, ("W", "K"): 0, ("H", "Q"): -1,
    ("L", "C"): -2, ("W", "N"): -2, ("S", "A"): 1, ("L", "G"): -3,
    ("W", "S"): -2, ("S", "E"): 0, ("H", "E"): -1, ("S", "I"): -2,
    ("H", "A"): -2, ("S", "M"): -1, ("Y", "L"): 0, ("Y", "H"): 0,
    ("Y", "D"): -2, ("E", "R"): -1, ("X", "P"): -1, ("G", "G"): 7,
    ("G", "C"): -3, ("E", "N"): -1, ("Y", "T"): -2, ("Y", "P"): -3,
    ("T", "K"): 0, ("A", "A"): 5, ("P", "Q"): 0, ("T", "C"): -1,
    ("V", "H"): -4, ("T", "G"): -2, ("I", "Q"): -2, ("Z", "T"): -1,
    ("C", "R"): -3, ("V", "P"): -3, ("P", "E"): 0, ("M", "C"): -4,
    ("K", "N"): 0, ("I", "I"): 5, ("P", "A"): -2, ("M", "G"): -1,
    ("T", "S"): 2, ("I", "E"): -3, ("P", "M"): -3, ("M", "K"): 0,
    ("I", "A"): -1, ("P", "I"): -1, ("R", "R"): 8, ("X", "M"): 0,
    ("L", "I"): 2, ("X", "I"): 0, ("Z", "B"): 0, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): 0, ("B", "R"): -1, ("B", "N"): 4,
    ("F", "D"): -3, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -3,
    ("B", "F"): -2, ("F", "L"): 2, ("X", "Q"): -1, ("B", "B"): 5
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum40.cmp
blosum40 = {
    ("W", "F"): 1, ("L", "R"): -2, ("S", "P"): -1, ("V", "T"): 1,
    ("Q", "Q"): 8, ("N", "A"): -1, ("Z", "Y"): -2, ("W", "R"): -2,
    ("Q", "A"): 0, ("S", "D"): 0, ("H", "H"): 13, ("S", "H"): -1,
    ("H", "D"): 0, ("L", "N"): -3, ("W", "A"): -3, ("Y", "M"): 1,
    ("G", "R"): -3, ("Y", "I"): 0, ("Y", "E"): -2, ("B", "Y"): -3,
    ("Y", "A"): -2, ("V", "D"): -3, ("B", "S"): 0, ("Y", "Y"): 9,
    ("G", "N"): 0, ("E", "C"): -2, ("Y", "Q"): -1, ("Z", "Z"): 5,
    ("V", "A"): 0, ("C", "C"): 16, ("M", "R"): -1, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 11, ("V", "I"): 4, ("V", "S"): -1,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -1, ("V", "Q"): -3,
    ("K", "K"): 6, ("P", "D"): -2, ("I", "H"): -3, ("I", "D"): -4,
    ("T", "R"): -2, ("P", "L"): -4, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): -2, ("F", "Q"): -4, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -3, ("X", "H"): -1, ("D", "R"): -1,
    ("B", "W"): -4, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -3,
    ("Z", "W"): -2, ("F", "E"): -3, ("D", "N"): 2, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 1, ("B", "G"): -1, ("X", "T"): 0,
    ("F", "M"): 0, ("B", "C"): -2, ("Z", "I"): -4, ("Z", "V"): -3,
    ("S", "S"): 5, ("L", "Q"): -2, ("W", "E"): -2, ("Q", "R"): 2,
    ("N", "N"): 8, ("W", "M"): -2, ("Q", "C"): -4, ("W", "I"): -3,
    ("S", "C"): -1, ("L", "A"): -2, ("S", "G"): 0, ("L", "E"): -2,
    ("W", "Q"): -1, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 1,
    ("N", "R"): 0, ("H", "C"): -4, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 4, ("C", "A"): -2, ("V", "L"): 2, ("G", "E"): -3,
    ("G", "A"): 1, ("K", "R"): 3, ("E", "D"): 2, ("Y", "R"): -1,
    ("M", "Q"): -1, ("T", "I"): -1, ("C", "D"): -2, ("V", "F"): 0,
    ("T", "A"): 0, ("T", "P"): 0, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -1, ("M", "A"): -1, ("K", "H"): -1,
    ("V", "R"): -2, ("P", "C"): -5, ("M", "E"): -2, ("K", "L"): -2,
    ("V", "V"): 5, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -4,
    ("P", "K"): -1, ("M", "M"): 7, ("K", "D"): 0, ("I", "C"): -4,
    ("Z", "D"): 1, ("F", "R"): -2, ("X", "K"): -1, ("Q", "D"): -1,
    ("X", "G"): -1, ("Z", "L"): -2, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -3, ("B", "H"): 0, ("F", "F"): 9, ("X", "W"): -2,
    ("B", "D"): 6, ("D", "A"): -1, ("S", "L"): -3, ("X", "S"): 0,
    ("F", "N"): -3, ("S", "R"): -1, ("W", "D"): -5, ("V", "Y"): -1,
    ("W", "L"): -1, ("H", "R"): 0, ("W", "H"): -5, ("H", "N"): 1,
    ("W", "T"): -4, ("T", "T"): 6, ("S", "F"): -2, ("W", "P"): -4,
    ("L", "D"): -3, ("B", "I"): -3, ("L", "H"): -2, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 6, ("Y", "K"): -1, ("E", "Q"): 2,
    ("Y", "G"): -3, ("Z", "S"): 0, ("Y", "C"): -4, ("G", "D"): -2,
    ("B", "V"): -3, ("E", "A"): -1, ("Y", "W"): 3, ("E", "E"): 7,
    ("Y", "S"): -2, ("C", "N"): -2, ("V", "C"): -2, ("T", "H"): -2,
    ("P", "R"): -3, ("V", "G"): -4, ("T", "L"): -1, ("V", "K"): -2,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -2, ("K", "I"): -3, ("M", "D"): -3,
    ("V", "W"): -3, ("W", "W"): 19, ("M", "H"): 1, ("P", "N"): -2,
    ("K", "A"): -1, ("M", "L"): 3, ("K", "E"): 1, ("Z", "E"): 5,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -1,
    ("K", "C"): -3, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -3,
    ("F", "C"): -2, ("Z", "Q"): 4, ("X", "Z"): -1, ("F", "G"): -3,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -3, ("B", "A"): -1,
    ("X", "R"): -1, ("D", "D"): 9, ("W", "G"): -2, ("Z", "F"): -4,
    ("S", "Q"): 1, ("W", "C"): -6, ("W", "K"): -2, ("H", "Q"): 0,
    ("L", "C"): -2, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -5, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -2,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): 0, ("Y", "H"): 2,
    ("Y", "D"): -3, ("E", "R"): -1, ("X", "P"): -2, ("G", "G"): 8,
    ("G", "C"): -3, ("E", "N"): -1, ("Y", "T"): -1, ("Y", "P"): -3,
    ("T", "K"): 0, ("A", "A"): 5, ("P", "Q"): -2, ("T", "C"): -1,
    ("V", "H"): -4, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -3, ("V", "P"): -3, ("P", "E"): 0, ("M", "C"): -3,
    ("K", "N"): 0, ("I", "I"): 6, ("P", "A"): -2, ("M", "G"): -2,
    ("T", "S"): 2, ("I", "E"): -4, ("P", "M"): -2, ("M", "K"): -1,
    ("I", "A"): -1, ("P", "I"): -2, ("R", "R"): 9, ("X", "M"): 0,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 2, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): 0, ("B", "R"): -1, ("B", "N"): 4,
    ("F", "D"): -4, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -2,
    ("B", "F"): -3, ("F", "L"): 2, ("X", "Q"): -1, ("B", "B"): 5
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum45.cmp
blosum45 = {
    ("W", "F"): 1, ("L", "R"): -2, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 6, ("N", "A"): -1, ("Z", "Y"): -2, ("W", "R"): -2,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 10, ("S", "H"): -1,
    ("H", "D"): 0, ("L", "N"): -3, ("W", "A"): -2, ("Y", "M"): 0,
    ("G", "R"): -2, ("Y", "I"): 0, ("Y", "E"): -2, ("B", "Y"): -2,
    ("Y", "A"): -2, ("V", "D"): -3, ("B", "S"): 0, ("Y", "Y"): 8,
    ("G", "N"): 0, ("E", "C"): -3, ("Y", "Q"): -1, ("Z", "Z"): 4,
    ("V", "A"): 0, ("C", "C"): 12, ("M", "R"): -1, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 9, ("V", "I"): 3, ("V", "S"): -1,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -1, ("V", "Q"): -3,
    ("K", "K"): 5, ("P", "D"): -1, ("I", "H"): -3, ("I", "D"): -4,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): -2, ("F", "Q"): -4, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -3, ("X", "H"): -1, ("D", "R"): -1,
    ("B", "W"): -4, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -2,
    ("Z", "W"): -2, ("F", "E"): -3, ("D", "N"): 2, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): -1, ("X", "T"): 0,
    ("F", "M"): 0, ("B", "C"): -2, ("Z", "I"): -3, ("Z", "V"): -3,
    ("S", "S"): 4, ("L", "Q"): -2, ("W", "E"): -3, ("Q", "R"): 1,
    ("N", "N"): 6, ("W", "M"): -2, ("Q", "C"): -3, ("W", "I"): -2,
    ("S", "C"): -1, ("L", "A"): -1, ("S", "G"): 0, ("L", "E"): -2,
    ("W", "Q"): -2, ("H", "G"): -2, ("S", "K"): -1, ("Q", "N"): 0,
    ("N", "R"): 0, ("H", "C"): -3, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): -1, ("V", "L"): 1, ("G", "E"): -2,
    ("G", "A"): 0, ("K", "R"): 3, ("E", "D"): 2, ("Y", "R"): -1,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -3, ("V", "F"): 0,
    ("T", "A"): 0, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -2, ("M", "A"): -1, ("K", "H"): -1,
    ("V", "R"): -2, ("P", "C"): -4, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 5, ("M", "I"): 2, ("T", "Q"): -1, ("I", "G"): -4,
    ("P", "K"): -1, ("M", "M"): 6, ("K", "D"): 0, ("I", "C"): -3,
    ("Z", "D"): 1, ("F", "R"): -2, ("X", "K"): -1, ("Q", "D"): 0,
    ("X", "G"): -1, ("Z", "L"): -2, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -3, ("B", "H"): 0, ("F", "F"): 8, ("X", "W"): -2,
    ("B", "D"): 5, ("D", "A"): -2, ("S", "L"): -3, ("X", "S"): 0,
    ("F", "N"): -2, ("S", "R"): -1, ("W", "D"): -4, ("V", "Y"): -1,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -3, ("H", "N"): 1,
    ("W", "T"): -3, ("T", "T"): 5, ("S", "F"): -2, ("W", "P"): -3,
    ("L", "D"): -3, ("B", "I"): -3, ("L", "H"): -2, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 5, ("Y", "K"): -1, ("E", "Q"): 2,
    ("Y", "G"): -3, ("Z", "S"): 0, ("Y", "C"): -3, ("G", "D"): -1,
    ("B", "V"): -3, ("E", "A"): -1, ("Y", "W"): 3, ("E", "E"): 6,
    ("Y", "S"): -2, ("C", "N"): -2, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -3, ("T", "L"): -1, ("V", "K"): -2,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -3, ("I", "N"): -2, ("K", "I"): -3, ("M", "D"): -3,
    ("V", "W"): -3, ("W", "W"): 15, ("M", "H"): 0, ("P", "N"): -2,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 1, ("Z", "E"): 4,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -1, ("X", "F"): -1,
    ("K", "C"): -3, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -2,
    ("F", "C"): -2, ("Z", "Q"): 4, ("X", "Z"): -1, ("F", "G"): -3,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -3, ("B", "A"): -1,
    ("X", "R"): -1, ("D", "D"): 7, ("W", "G"): -2, ("Z", "F"): -3,
    ("S", "Q"): 0, ("W", "C"): -5, ("W", "K"): -2, ("H", "Q"): 1,
    ("L", "C"): -2, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -3,
    ("W", "S"): -4, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -2,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): 0, ("Y", "H"): 2,
    ("Y", "D"): -2, ("E", "R"): 0, ("X", "P"): -1, ("G", "G"): 7,
    ("G", "C"): -3, ("E", "N"): 0, ("Y", "T"): -1, ("Y", "P"): -3,
    ("T", "K"): -1, ("A", "A"): 5, ("P", "Q"): -1, ("T", "C"): -1,
    ("V", "H"): -3, ("T", "G"): -2, ("I", "Q"): -2, ("Z", "T"): -1,
    ("C", "R"): -3, ("V", "P"): -3, ("P", "E"): 0, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 5, ("P", "A"): -1, ("M", "G"): -2,
    ("T", "S"): 2, ("I", "E"): -3, ("P", "M"): -2, ("M", "K"): -1,
    ("I", "A"): -1, ("P", "I"): -2, ("R", "R"): 7, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 2, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): 0, ("B", "R"): -1, ("B", "N"): 4,
    ("F", "D"): -4, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -2,
    ("B", "F"): -3, ("F", "L"): 1, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum50.cmp
blosum50 = {
    ("W", "F"): 1, ("L", "R"): -3, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 7, ("N", "A"): -1, ("Z", "Y"): -2, ("W", "R"): -3,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 10, ("S", "H"): -1,
    ("H", "D"): -1, ("L", "N"): -4, ("W", "A"): -3, ("Y", "M"): 0,
    ("G", "R"): -3, ("Y", "I"): -1, ("Y", "E"): -2, ("B", "Y"): -3,
    ("Y", "A"): -2, ("V", "D"): -4, ("B", "S"): 0, ("Y", "Y"): 8,
    ("G", "N"): 0, ("E", "C"): -3, ("Y", "Q"): -1, ("Z", "Z"): 5,
    ("V", "A"): 0, ("C", "C"): 13, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 10, ("V", "I"): 4, ("V", "S"): -2,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -2, ("V", "Q"): -3,
    ("K", "K"): 6, ("P", "D"): -1, ("I", "H"): -4, ("I", "D"): -4,
    ("T", "R"): -1, ("P", "L"): -4, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): -2, ("F", "Q"): -4, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -3, ("X", "H"): -1, ("D", "R"): -2,
    ("B", "W"): -5, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -3,
    ("Z", "W"): -2, ("F", "E"): -3, ("D", "N"): 2, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): -1, ("X", "T"): 0,
    ("F", "M"): 0, ("B", "C"): -3, ("Z", "I"): -3, ("Z", "V"): -3,
    ("S", "S"): 5, ("L", "Q"): -2, ("W", "E"): -3, ("Q", "R"): 1,
    ("N", "N"): 7, ("W", "M"): -1, ("Q", "C"): -3, ("W", "I"): -3,
    ("S", "C"): -1, ("L", "A"): -2, ("S", "G"): 0, ("L", "E"): -3,
    ("W", "Q"): -1, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -3, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 4, ("C", "A"): -1, ("V", "L"): 1, ("G", "E"): -3,
    ("G", "A"): 0, ("K", "R"): 3, ("E", "D"): 2, ("Y", "R"): -1,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -4, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -2, ("M", "A"): -1, ("K", "H"): 0,
    ("V", "R"): -3, ("P", "C"): -4, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 5, ("M", "I"): 2, ("T", "Q"): -1, ("I", "G"): -4,
    ("P", "K"): -1, ("M", "M"): 7, ("K", "D"): -1, ("I", "C"): -2,
    ("Z", "D"): 1, ("F", "R"): -3, ("X", "K"): -1, ("Q", "D"): 0,
    ("X", "G"): -2, ("Z", "L"): -3, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -4, ("B", "H"): 0, ("F", "F"): 8, ("X", "W"): -3,
    ("B", "D"): 5, ("D", "A"): -2, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -4, ("S", "R"): -1, ("W", "D"): -5, ("V", "Y"): -1,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -3, ("H", "N"): 1,
    ("W", "T"): -3, ("T", "T"): 5, ("S", "F"): -3, ("W", "P"): -4,
    ("L", "D"): -4, ("B", "I"): -4, ("L", "H"): -3, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 5, ("Y", "K"): -2, ("E", "Q"): 2,
    ("Y", "G"): -3, ("Z", "S"): 0, ("Y", "C"): -3, ("G", "D"): -1,
    ("B", "V"): -4, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 6,
    ("Y", "S"): -2, ("C", "N"): -2, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -3, ("V", "G"): -4, ("T", "L"): -1, ("V", "K"): -3,
    ("K", "Q"): 2, ("R", "A"): -2, ("I", "R"): -4, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -3, ("K", "I"): -3, ("M", "D"): -4,
    ("V", "W"): -3, ("W", "W"): 15, ("M", "H"): -1, ("P", "N"): -2,
    ("K", "A"): -1, ("M", "L"): 3, ("K", "E"): 1, ("Z", "E"): 5,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -1, ("X", "F"): -2,
    ("K", "C"): -3, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -3,
    ("F", "C"): -2, ("Z", "Q"): 4, ("X", "Z"): -1, ("F", "G"): -4,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -4, ("B", "A"): -2,
    ("X", "R"): -1, ("D", "D"): 8, ("W", "G"): -3, ("Z", "F"): -4,
    ("S", "Q"): 0, ("W", "C"): -5, ("W", "K"): -3, ("H", "Q"): 1,
    ("L", "C"): -2, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -4, ("S", "E"): -1, ("H", "E"): 0, ("S", "I"): -3,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -1, ("Y", "H"): 2,
    ("Y", "D"): -3, ("E", "R"): 0, ("X", "P"): -2, ("G", "G"): 8,
    ("G", "C"): -3, ("E", "N"): 0, ("Y", "T"): -2, ("Y", "P"): -3,
    ("T", "K"): -1, ("A", "A"): 5, ("P", "Q"): -1, ("T", "C"): -1,
    ("V", "H"): -4, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -4, ("V", "P"): -3, ("P", "E"): -1, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 5, ("P", "A"): -1, ("M", "G"): -3,
    ("T", "S"): 2, ("I", "E"): -4, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -1, ("P", "I"): -3, ("R", "R"): 7, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 2, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): -1, ("B", "R"): -1, ("B", "N"): 4,
    ("F", "D"): -5, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -1,
    ("B", "F"): -4, ("F", "L"): 1, ("X", "Q"): -1, ("B", "B"): 5
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum55.cmp
blosum55 = {
    ("W", "F"): 1, ("L", "R"): -3, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 7, ("N", "A"): -1, ("Z", "Y"): -2, ("W", "R"): -3,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 10, ("S", "H"): -1,
    ("H", "D"): -1, ("L", "N"): -4, ("W", "A"): -3, ("Y", "M"): 0,
    ("G", "R"): -3, ("Y", "I"): -1, ("Y", "E"): -2, ("B", "Y"): -3,
    ("Y", "A"): -2, ("V", "D"): -4, ("B", "S"): 0, ("Y", "Y"): 8,
    ("G", "N"): 0, ("E", "C"): -3, ("Y", "Q"): -1, ("Z", "Z"): 5,
    ("V", "A"): 0, ("C", "C"): 13, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 10, ("V", "I"): 4, ("V", "S"): -2,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -2, ("V", "Q"): -3,
    ("K", "K"): 6, ("P", "D"): -1, ("I", "H"): -4, ("I", "D"): -4,
    ("T", "R"): -1, ("P", "L"): -4, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): -2, ("F", "Q"): -4, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -3, ("X", "H"): -1, ("D", "R"): -2,
    ("B", "W"): -5, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -3,
    ("Z", "W"): -2, ("F", "E"): -3, ("D", "N"): 2, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): -1, ("X", "T"): 0,
    ("F", "M"): 0, ("B", "C"): -3, ("Z", "I"): -3, ("Z", "V"): -3,
    ("S", "S"): 5, ("L", "Q"): -2, ("W", "E"): -3, ("Q", "R"): 1,
    ("N", "N"): 7, ("W", "M"): -1, ("Q", "C"): -3, ("W", "I"): -3,
    ("S", "C"): -1, ("L", "A"): -2, ("S", "G"): 0, ("L", "E"): -3,
    ("W", "Q"): -1, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -3, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 4, ("C", "A"): -1, ("V", "L"): 1, ("G", "E"): -3,
    ("G", "A"): 0, ("K", "R"): 3, ("E", "D"): 2, ("Y", "R"): -1,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -4, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -2, ("M", "A"): -1, ("K", "H"): 0,
    ("V", "R"): -3, ("P", "C"): -4, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 5, ("M", "I"): 2, ("T", "Q"): -1, ("I", "G"): -4,
    ("P", "K"): -1, ("M", "M"): 7, ("K", "D"): -1, ("I", "C"): -2,
    ("Z", "D"): 1, ("F", "R"): -3, ("X", "K"): -1, ("Q", "D"): 0,
    ("X", "G"): -2, ("Z", "L"): -3, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -4, ("B", "H"): 0, ("F", "F"): 8, ("X", "W"): -3,
    ("B", "D"): 5, ("D", "A"): -2, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -4, ("S", "R"): -1, ("W", "D"): -5, ("V", "Y"): -1,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -3, ("H", "N"): 1,
    ("W", "T"): -3, ("T", "T"): 5, ("S", "F"): -3, ("W", "P"): -4,
    ("L", "D"): -4, ("B", "I"): -4, ("L", "H"): -3, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 5, ("Y", "K"): -2, ("E", "Q"): 2,
    ("Y", "G"): -3, ("Z", "S"): 0, ("Y", "C"): -3, ("G", "D"): -1,
    ("B", "V"): -4, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 6,
    ("Y", "S"): -2, ("C", "N"): -2, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -3, ("V", "G"): -4, ("T", "L"): -1, ("V", "K"): -3,
    ("K", "Q"): 2, ("R", "A"): -2, ("I", "R"): -4, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -3, ("K", "I"): -3, ("M", "D"): -4,
    ("V", "W"): -3, ("W", "W"): 15, ("M", "H"): -1, ("P", "N"): -2,
    ("K", "A"): -1, ("M", "L"): 3, ("K", "E"): 1, ("Z", "E"): 5,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -1, ("X", "F"): -2,
    ("K", "C"): -3, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -3,
    ("F", "C"): -2, ("Z", "Q"): 4, ("X", "Z"): -1, ("F", "G"): -4,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -4, ("B", "A"): -2,
    ("X", "R"): -1, ("D", "D"): 8, ("W", "G"): -3, ("Z", "F"): -4,
    ("S", "Q"): 0, ("W", "C"): -5, ("W", "K"): -3, ("H", "Q"): 1,
    ("L", "C"): -2, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -4, ("S", "E"): -1, ("H", "E"): 0, ("S", "I"): -3,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -1, ("Y", "H"): 2,
    ("Y", "D"): -3, ("E", "R"): 0, ("X", "P"): -2, ("G", "G"): 8,
    ("G", "C"): -3, ("E", "N"): 0, ("Y", "T"): -2, ("Y", "P"): -3,
    ("T", "K"): -1, ("A", "A"): 5, ("P", "Q"): -1, ("T", "C"): -1,
    ("V", "H"): -4, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -4, ("V", "P"): -3, ("P", "E"): -1, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 5, ("P", "A"): -1, ("M", "G"): -3,
    ("T", "S"): 2, ("I", "E"): -4, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -1, ("P", "I"): -3, ("R", "R"): 7, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 2, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): -1, ("B", "R"): -1, ("B", "N"): 4,
    ("F", "D"): -5, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -1,
    ("B", "F"): -4, ("F", "L"): 1, ("X", "Q"): -1, ("B", "B"): 5
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum60.cmp
blosum60 = {
    ("W", "F"): 1, ("L", "R"): -2, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 5, ("N", "A"): -1, ("Z", "Y"): -2, ("W", "R"): -3,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 7, ("S", "H"): -1,
    ("H", "D"): -1, ("L", "N"): -3, ("W", "A"): -3, ("Y", "M"): -1,
    ("G", "R"): -2, ("Y", "I"): -1, ("Y", "E"): -2, ("B", "Y"): -2,
    ("Y", "A"): -2, ("V", "D"): -3, ("B", "S"): 0, ("Y", "Y"): 6,
    ("G", "N"): 0, ("E", "C"): -3, ("Y", "Q"): -1, ("Z", "Z"): 3,
    ("V", "A"): 0, ("C", "C"): 9, ("M", "R"): -1, ("V", "E"): -2,
    ("T", "N"): 0, ("P", "P"): 7, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -2, ("V", "Q"): -2,
    ("K", "K"): 4, ("P", "D"): -1, ("I", "H"): -3, ("I", "D"): -3,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -1, ("M", "N"): -2,
    ("P", "H"): -2, ("F", "Q"): -3, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -3, ("X", "H"): -1, ("D", "R"): -1,
    ("B", "W"): -4, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -2,
    ("Z", "W"): -2, ("F", "E"): -3, ("D", "N"): 1, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): -1, ("X", "T"): 0,
    ("F", "M"): 0, ("B", "C"): -3, ("Z", "I"): -3, ("Z", "V"): -2,
    ("S", "S"): 4, ("L", "Q"): -2, ("W", "E"): -3, ("Q", "R"): 1,
    ("N", "N"): 6, ("W", "M"): -1, ("Q", "C"): -3, ("W", "I"): -2,
    ("S", "C"): -1, ("L", "A"): -1, ("S", "G"): 0, ("L", "E"): -3,
    ("W", "Q"): -2, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 0,
    ("N", "R"): 0, ("H", "C"): -3, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): 0, ("V", "L"): 1, ("G", "E"): -2,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 2, ("Y", "R"): -2,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -3, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -2, ("M", "A"): -1, ("K", "H"): -1,
    ("V", "R"): -2, ("P", "C"): -3, ("M", "E"): -2, ("K", "L"): -2,
    ("V", "V"): 4, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -3,
    ("P", "K"): -1, ("M", "M"): 5, ("K", "D"): -1, ("I", "C"): -1,
    ("Z", "D"): 1, ("F", "R"): -3, ("X", "K"): -1, ("Q", "D"): 0,
    ("X", "G"): -1, ("Z", "L"): -2, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -3, ("B", "H"): 0, ("F", "F"): 6, ("X", "W"): -2,
    ("B", "D"): 4, ("D", "A"): -2, ("S", "L"): -2, ("X", "S"): 0,
    ("F", "N"): -3, ("S", "R"): -1, ("W", "D"): -4, ("V", "Y"): -1,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -2, ("H", "N"): 1,
    ("W", "T"): -2, ("T", "T"): 4, ("S", "F"): -2, ("W", "P"): -4,
    ("L", "D"): -3, ("B", "I"): -3, ("L", "H"): -3, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 4, ("Y", "K"): -2, ("E", "Q"): 2,
    ("Y", "G"): -3, ("Z", "S"): 0, ("Y", "C"): -2, ("G", "D"): -1,
    ("B", "V"): -3, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 5,
    ("Y", "S"): -2, ("C", "N"): -2, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -3, ("T", "L"): -1, ("V", "K"): -2,
    ("K", "Q"): 1, ("R", "A"): -1, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -3, ("K", "I"): -3, ("M", "D"): -3,
    ("V", "W"): -3, ("W", "W"): 10, ("M", "H"): -1, ("P", "N"): -2,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 1, ("Z", "E"): 4,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -1, ("X", "F"): -1,
    ("K", "C"): -3, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -3,
    ("F", "C"): -2, ("Z", "Q"): 3, ("X", "Z"): -1, ("F", "G"): -3,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -3, ("B", "A"): -2,
    ("X", "R"): -1, ("D", "D"): 6, ("W", "G"): -2, ("Z", "F"): -3,
    ("S", "Q"): 0, ("W", "C"): -2, ("W", "K"): -3, ("H", "Q"): 1,
    ("L", "C"): -1, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -3, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -2,
    ("H", "A"): -2, ("S", "M"): -1, ("Y", "L"): -1, ("Y", "H"): 2,
    ("Y", "D"): -3, ("E", "R"): 0, ("X", "P"): -2, ("G", "G"): 6,
    ("G", "C"): -2, ("E", "N"): 0, ("Y", "T"): -2, ("Y", "P"): -3,
    ("T", "K"): -1, ("A", "A"): 4, ("P", "Q"): -1, ("T", "C"): -1,
    ("V", "H"): -3, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -3, ("V", "P"): -2, ("P", "E"): -1, ("M", "C"): -1,
    ("K", "N"): 0, ("I", "I"): 4, ("P", "A"): -1, ("M", "G"): -2,
    ("T", "S"): 1, ("I", "E"): -3, ("P", "M"): -2, ("M", "K"): -1,
    ("I", "A"): -1, ("P", "I"): -3, ("R", "R"): 5, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 1, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): 0, ("B", "R"): -1, ("B", "N"): 3,
    ("F", "D"): -3, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -1,
    ("B", "F"): -3, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum62.cmp
blosum62 = {
    ("W", "F"): 1, ("L", "R"): -2, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 5, ("N", "A"): -2, ("Z", "Y"): -2, ("W", "R"): -3,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 8, ("S", "H"): -1,
    ("H", "D"): -1, ("L", "N"): -3, ("W", "A"): -3, ("Y", "M"): -1,
    ("G", "R"): -2, ("Y", "I"): -1, ("Y", "E"): -2, ("B", "Y"): -3,
    ("Y", "A"): -2, ("V", "D"): -3, ("B", "S"): 0, ("Y", "Y"): 7,
    ("G", "N"): 0, ("E", "C"): -4, ("Y", "Q"): -1, ("Z", "Z"): 4,
    ("V", "A"): 0, ("C", "C"): 9, ("M", "R"): -1, ("V", "E"): -2,
    ("T", "N"): 0, ("P", "P"): 7, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -2, ("V", "Q"): -2,
    ("K", "K"): 5, ("P", "D"): -1, ("I", "H"): -3, ("I", "D"): -3,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): -2, ("F", "Q"): -3, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -3, ("X", "H"): -1, ("D", "R"): -2,
    ("B", "W"): -4, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -2,
    ("Z", "W"): -3, ("F", "E"): -3, ("D", "N"): 1, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): -1, ("X", "T"): 0,
    ("F", "M"): 0, ("B", "C"): -3, ("Z", "I"): -3, ("Z", "V"): -2,
    ("S", "S"): 4, ("L", "Q"): -2, ("W", "E"): -3, ("Q", "R"): 1,
    ("N", "N"): 6, ("W", "M"): -1, ("Q", "C"): -3, ("W", "I"): -3,
    ("S", "C"): -1, ("L", "A"): -1, ("S", "G"): 0, ("L", "E"): -3,
    ("W", "Q"): -2, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 0,
    ("N", "R"): 0, ("H", "C"): -3, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): 0, ("V", "L"): 1, ("G", "E"): -2,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 2, ("Y", "R"): -2,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -3, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -2, ("M", "A"): -1, ("K", "H"): -1,
    ("V", "R"): -3, ("P", "C"): -3, ("M", "E"): -2, ("K", "L"): -2,
    ("V", "V"): 4, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -4,
    ("P", "K"): -1, ("M", "M"): 5, ("K", "D"): -1, ("I", "C"): -1,
    ("Z", "D"): 1, ("F", "R"): -3, ("X", "K"): -1, ("Q", "D"): 0,
    ("X", "G"): -1, ("Z", "L"): -3, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -4, ("B", "H"): 0, ("F", "F"): 6, ("X", "W"): -2,
    ("B", "D"): 4, ("D", "A"): -2, ("S", "L"): -2, ("X", "S"): 0,
    ("F", "N"): -3, ("S", "R"): -1, ("W", "D"): -4, ("V", "Y"): -1,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -2, ("H", "N"): 1,
    ("W", "T"): -2, ("T", "T"): 5, ("S", "F"): -2, ("W", "P"): -4,
    ("L", "D"): -4, ("B", "I"): -3, ("L", "H"): -3, ("S", "N"): 1,
    ("B", "T"): -1, ("L", "L"): 4, ("Y", "K"): -2, ("E", "Q"): 2,
    ("Y", "G"): -3, ("Z", "S"): 0, ("Y", "C"): -2, ("G", "D"): -1,
    ("B", "V"): -3, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 5,
    ("Y", "S"): -2, ("C", "N"): -3, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -3, ("T", "L"): -1, ("V", "K"): -2,
    ("K", "Q"): 1, ("R", "A"): -1, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -3, ("K", "I"): -3, ("M", "D"): -3,
    ("V", "W"): -3, ("W", "W"): 11, ("M", "H"): -2, ("P", "N"): -2,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 1, ("Z", "E"): 4,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -1, ("X", "F"): -1,
    ("K", "C"): -3, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -3,
    ("F", "C"): -2, ("Z", "Q"): 3, ("X", "Z"): -1, ("F", "G"): -3,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -3, ("B", "A"): -2,
    ("X", "R"): -1, ("D", "D"): 6, ("W", "G"): -2, ("Z", "F"): -3,
    ("S", "Q"): 0, ("W", "C"): -2, ("W", "K"): -3, ("H", "Q"): 0,
    ("L", "C"): -1, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -3, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -2,
    ("H", "A"): -2, ("S", "M"): -1, ("Y", "L"): -1, ("Y", "H"): 2,
    ("Y", "D"): -3, ("E", "R"): 0, ("X", "P"): -2, ("G", "G"): 6,
    ("G", "C"): -3, ("E", "N"): 0, ("Y", "T"): -2, ("Y", "P"): -3,
    ("T", "K"): -1, ("A", "A"): 4, ("P", "Q"): -1, ("T", "C"): -1,
    ("V", "H"): -3, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -3, ("V", "P"): -2, ("P", "E"): -1, ("M", "C"): -1,
    ("K", "N"): 0, ("I", "I"): 4, ("P", "A"): -1, ("M", "G"): -3,
    ("T", "S"): 1, ("I", "E"): -3, ("P", "M"): -2, ("M", "K"): -1,
    ("I", "A"): -1, ("P", "I"): -3, ("R", "R"): 5, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 1, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): 0, ("B", "R"): -1, ("B", "N"): 3,
    ("F", "D"): -3, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -1,
    ("B", "F"): -3, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum65.cmp
blosum65 = {
    ("W", "F"): 1, ("L", "R"): -2, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 6, ("N", "A"): -2, ("Z", "Y"): -2, ("W", "R"): -3,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 8, ("S", "H"): -1,
    ("H", "D"): -1, ("L", "N"): -4, ("W", "A"): -3, ("Y", "M"): -1,
    ("G", "R"): -2, ("Y", "I"): -1, ("Y", "E"): -2, ("B", "Y"): -3,
    ("Y", "A"): -2, ("V", "D"): -3, ("B", "S"): 0, ("Y", "Y"): 7,
    ("G", "N"): -1, ("E", "C"): -4, ("Y", "Q"): -2, ("Z", "Z"): 4,
    ("V", "A"): 0, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 8, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -2, ("V", "Q"): -2,
    ("K", "K"): 5, ("P", "D"): -2, ("I", "H"): -3, ("I", "D"): -3,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): -2, ("F", "Q"): -3, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -4, ("X", "H"): -1, ("D", "R"): -2,
    ("B", "W"): -4, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -2,
    ("Z", "W"): -3, ("F", "E"): -3, ("D", "N"): 1, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): -1, ("X", "T"): -1,
    ("F", "M"): 0, ("B", "C"): -3, ("Z", "I"): -3, ("Z", "V"): -2,
    ("S", "S"): 4, ("L", "Q"): -2, ("W", "E"): -3, ("Q", "R"): 1,
    ("N", "N"): 6, ("W", "M"): -2, ("Q", "C"): -3, ("W", "I"): -2,
    ("S", "C"): -1, ("L", "A"): -2, ("S", "G"): 0, ("L", "E"): -3,
    ("W", "Q"): -2, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 0,
    ("N", "R"): 0, ("H", "C"): -3, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): 0, ("V", "L"): 1, ("G", "E"): -2,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 2, ("Y", "R"): -2,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -4, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -2, ("M", "A"): -1, ("K", "H"): -1,
    ("V", "R"): -3, ("P", "C"): -3, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 4, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -4,
    ("P", "K"): -1, ("M", "M"): 6, ("K", "D"): -1, ("I", "C"): -1,
    ("Z", "D"): 1, ("F", "R"): -3, ("X", "K"): -1, ("Q", "D"): 0,
    ("X", "G"): -2, ("Z", "L"): -3, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -4, ("B", "H"): 0, ("F", "F"): 6, ("X", "W"): -2,
    ("B", "D"): 4, ("D", "A"): -2, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -3, ("S", "R"): -1, ("W", "D"): -5, ("V", "Y"): -1,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -2, ("H", "N"): 1,
    ("W", "T"): -3, ("T", "T"): 5, ("S", "F"): -2, ("W", "P"): -4,
    ("L", "D"): -4, ("B", "I"): -3, ("L", "H"): -3, ("S", "N"): 1,
    ("B", "T"): -1, ("L", "L"): 4, ("Y", "K"): -2, ("E", "Q"): 2,
    ("Y", "G"): -3, ("Z", "S"): 0, ("Y", "C"): -2, ("G", "D"): -1,
    ("B", "V"): -3, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 5,
    ("Y", "S"): -2, ("C", "N"): -3, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -3, ("T", "L"): -1, ("V", "K"): -2,
    ("K", "Q"): 1, ("R", "A"): -1, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -3, ("K", "I"): -3, ("M", "D"): -3,
    ("V", "W"): -3, ("W", "W"): 10, ("M", "H"): -2, ("P", "N"): -2,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 1, ("Z", "E"): 4,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -3, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -3,
    ("F", "C"): -2, ("Z", "Q"): 3, ("X", "Z"): -1, ("F", "G"): -3,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -3, ("B", "A"): -2,
    ("X", "R"): -1, ("D", "D"): 6, ("W", "G"): -3, ("Z", "F"): -3,
    ("S", "Q"): 0, ("W", "C"): -2, ("W", "K"): -3, ("H", "Q"): 1,
    ("L", "C"): -1, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -3, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -2,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -1, ("Y", "H"): 2,
    ("Y", "D"): -3, ("E", "R"): 0, ("X", "P"): -2, ("G", "G"): 6,
    ("G", "C"): -3, ("E", "N"): 0, ("Y", "T"): -2, ("Y", "P"): -3,
    ("T", "K"): -1, ("A", "A"): 4, ("P", "Q"): -1, ("T", "C"): -1,
    ("V", "H"): -3, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -4, ("V", "P"): -2, ("P", "E"): -1, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 4, ("P", "A"): -1, ("M", "G"): -3,
    ("T", "S"): 1, ("I", "E"): -3, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -1, ("P", "I"): -3, ("R", "R"): 6, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 1, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): -1, ("B", "R"): -1, ("B", "N"): 3,
    ("F", "D"): -4, ("X", "Y"): -1, ("Z", "R"): 0, ("F", "H"): -1,
    ("B", "F"): -3, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum70.cmp
blosum70 = {
    ("W", "F"): 1, ("L", "R"): -3, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 6, ("N", "A"): -2, ("Z", "Y"): -2, ("W", "R"): -3,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 8, ("S", "H"): -1,
    ("H", "D"): -1, ("L", "N"): -4, ("W", "A"): -3, ("Y", "M"): -1,
    ("G", "R"): -3, ("Y", "I"): -1, ("Y", "E"): -3, ("B", "Y"): -3,
    ("Y", "A"): -2, ("V", "D"): -4, ("B", "S"): 0, ("Y", "Y"): 7,
    ("G", "N"): -1, ("E", "C"): -4, ("Y", "Q"): -2, ("Z", "Z"): 4,
    ("V", "A"): 0, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 8, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -2, ("V", "Q"): -2,
    ("K", "K"): 5, ("P", "D"): -2, ("I", "H"): -4, ("I", "D"): -4,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): -2, ("F", "Q"): -3, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -4, ("X", "H"): -1, ("D", "R"): -2,
    ("B", "W"): -4, ("X", "D"): -2, ("Z", "K"): 1, ("F", "A"): -2,
    ("Z", "W"): -3, ("F", "E"): -4, ("D", "N"): 1, ("B", "K"): -1,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): -1, ("X", "T"): -1,
    ("F", "M"): 0, ("B", "C"): -4, ("Z", "I"): -3, ("Z", "V"): -3,
    ("S", "S"): 4, ("L", "Q"): -2, ("W", "E"): -4, ("Q", "R"): 1,
    ("N", "N"): 6, ("W", "M"): -2, ("Q", "C"): -3, ("W", "I"): -3,
    ("S", "C"): -1, ("L", "A"): -2, ("S", "G"): -1, ("L", "E"): -3,
    ("W", "Q"): -2, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -4, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): -1, ("V", "L"): 1, ("G", "E"): -2,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 1, ("Y", "R"): -2,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -4, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -3, ("M", "A"): -1, ("K", "H"): -1,
    ("V", "R"): -3, ("P", "C"): -3, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 4, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -4,
    ("P", "K"): -1, ("M", "M"): 6, ("K", "D"): -1, ("I", "C"): -1,
    ("Z", "D"): 1, ("F", "R"): -3, ("X", "K"): -1, ("Q", "D"): -1,
    ("X", "G"): -2, ("Z", "L"): -3, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -4, ("B", "H"): -1, ("F", "F"): 6, ("X", "W"): -3,
    ("B", "D"): 4, ("D", "A"): -2, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -3, ("S", "R"): -1, ("W", "D"): -5, ("V", "Y"): -2,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -2, ("H", "N"): 0,
    ("W", "T"): -3, ("T", "T"): 5, ("S", "F"): -3, ("W", "P"): -4,
    ("L", "D"): -4, ("B", "I"): -4, ("L", "H"): -3, ("S", "N"): 0,
    ("B", "T"): -1, ("L", "L"): 4, ("Y", "K"): -2, ("E", "Q"): 2,
    ("Y", "G"): -4, ("Z", "S"): 0, ("Y", "C"): -3, ("G", "D"): -2,
    ("B", "V"): -3, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 5,
    ("Y", "S"): -2, ("C", "N"): -3, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -4, ("T", "L"): -2, ("V", "K"): -3,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -4, ("K", "I"): -3, ("M", "D"): -3,
    ("V", "W"): -3, ("W", "W"): 11, ("M", "H"): -2, ("P", "N"): -2,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 1, ("Z", "E"): 4,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -4, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -3,
    ("F", "C"): -2, ("Z", "Q"): 3, ("X", "Z"): -1, ("F", "G"): -4,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -3, ("B", "A"): -2,
    ("X", "R"): -1, ("D", "D"): 6, ("W", "G"): -3, ("Z", "F"): -4,
    ("S", "Q"): 0, ("W", "C"): -3, ("W", "K"): -3, ("H", "Q"): 1,
    ("L", "C"): -2, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -3, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -3,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -1, ("Y", "H"): 2,
    ("Y", "D"): -4, ("E", "R"): 0, ("X", "P"): -2, ("G", "G"): 6,
    ("G", "C"): -3, ("E", "N"): 0, ("Y", "T"): -2, ("Y", "P"): -3,
    ("T", "K"): -1, ("A", "A"): 4, ("P", "Q"): -2, ("T", "C"): -1,
    ("V", "H"): -3, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -4, ("V", "P"): -3, ("P", "E"): -1, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 4, ("P", "A"): -1, ("M", "G"): -3,
    ("T", "S"): 1, ("I", "E"): -4, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -2, ("P", "I"): -3, ("R", "R"): 6, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 0, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): -1, ("B", "R"): -1, ("B", "N"): 3,
    ("F", "D"): -4, ("X", "Y"): -2, ("Z", "R"): 0, ("F", "H"): -1,
    ("B", "F"): -4, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum75.cmp
blosum75 = {
    ("W", "F"): 1, ("L", "R"): -3, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 6, ("N", "A"): -2, ("Z", "Y"): -3, ("W", "R"): -3,
    ("Q", "A"): -1, ("S", "D"): -1, ("H", "H"): 8, ("S", "H"): -1,
    ("H", "D"): -1, ("L", "N"): -4, ("W", "A"): -3, ("Y", "M"): -2,
    ("G", "R"): -3, ("Y", "I"): -2, ("Y", "E"): -3, ("B", "Y"): -3,
    ("Y", "A"): -2, ("V", "D"): -4, ("B", "S"): 0, ("Y", "Y"): 7,
    ("G", "N"): -1, ("E", "C"): -5, ("Y", "Q"): -2, ("Z", "Z"): 4,
    ("V", "A"): 0, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 8, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -2, ("V", "M"): 1, ("T", "F"): -2, ("V", "Q"): -2,
    ("K", "K"): 5, ("P", "D"): -2, ("I", "H"): -4, ("I", "D"): -4,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -2, ("M", "N"): -3,
    ("P", "H"): -2, ("F", "Q"): -4, ("Z", "G"): -2, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -4, ("X", "H"): -1, ("D", "R"): -2,
    ("B", "W"): -5, ("X", "D"): -2, ("Z", "K"): 1, ("F", "A"): -3,
    ("Z", "W"): -3, ("F", "E"): -4, ("D", "N"): 1, ("B", "K"): -1,
    ("X", "X"): -1, ("F", "I"): 0, ("B", "G"): -1, ("X", "T"): -1,
    ("F", "M"): 0, ("B", "C"): -4, ("Z", "I"): -4, ("Z", "V"): -3,
    ("S", "S"): 5, ("L", "Q"): -3, ("W", "E"): -4, ("Q", "R"): 1,
    ("N", "N"): 6, ("W", "M"): -2, ("Q", "C"): -3, ("W", "I"): -3,
    ("S", "C"): -1, ("L", "A"): -2, ("S", "G"): -1, ("L", "E"): -4,
    ("W", "Q"): -2, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -4, ("Y", "N"): -3, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): -1, ("V", "L"): 1, ("G", "E"): -3,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 1, ("Y", "R"): -2,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -4, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -3, ("M", "A"): -1, ("K", "H"): -1,
    ("V", "R"): -3, ("P", "C"): -4, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 4, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -5,
    ("P", "K"): -1, ("M", "M"): 6, ("K", "D"): -1, ("I", "C"): -1,
    ("Z", "D"): 1, ("F", "R"): -3, ("X", "K"): -1, ("Q", "D"): -1,
    ("X", "G"): -2, ("Z", "L"): -3, ("X", "C"): -2, ("Z", "H"): 0,
    ("B", "L"): -4, ("B", "H"): -1, ("F", "F"): 6, ("X", "W"): -3,
    ("B", "D"): 4, ("D", "A"): -2, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -4, ("S", "R"): -1, ("W", "D"): -5, ("V", "Y"): -2,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -2, ("H", "N"): 0,
    ("W", "T"): -3, ("T", "T"): 5, ("S", "F"): -3, ("W", "P"): -5,
    ("L", "D"): -4, ("B", "I"): -4, ("L", "H"): -3, ("S", "N"): 0,
    ("B", "T"): -1, ("L", "L"): 4, ("Y", "K"): -2, ("E", "Q"): 2,
    ("Y", "G"): -4, ("Z", "S"): 0, ("Y", "C"): -3, ("G", "D"): -2,
    ("B", "V"): -4, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 5,
    ("Y", "S"): -2, ("C", "N"): -3, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -4, ("T", "L"): -2, ("V", "K"): -3,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -4, ("K", "I"): -3, ("M", "D"): -4,
    ("V", "W"): -3, ("W", "W"): 11, ("M", "H"): -2, ("P", "N"): -3,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 1, ("Z", "E"): 4,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -4, ("B", "Q"): 0, ("X", "B"): -2, ("B", "M"): -3,
    ("F", "C"): -2, ("Z", "Q"): 3, ("X", "Z"): -1, ("F", "G"): -4,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -4, ("B", "A"): -2,
    ("X", "R"): -1, ("D", "D"): 6, ("W", "G"): -3, ("Z", "F"): -4,
    ("S", "Q"): 0, ("W", "C"): -3, ("W", "K"): -4, ("H", "Q"): 1,
    ("L", "C"): -2, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -3, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -3,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -1, ("Y", "H"): 2,
    ("Y", "D"): -4, ("E", "R"): 0, ("X", "P"): -2, ("G", "G"): 6,
    ("G", "C"): -3, ("E", "N"): -1, ("Y", "T"): -2, ("Y", "P"): -4,
    ("T", "K"): -1, ("A", "A"): 4, ("P", "Q"): -2, ("T", "C"): -1,
    ("V", "H"): -4, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -4, ("V", "P"): -3, ("P", "E"): -1, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 4, ("P", "A"): -1, ("M", "G"): -3,
    ("T", "S"): 1, ("I", "E"): -4, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -2, ("P", "I"): -3, ("R", "R"): 6, ("X", "M"): -1,
    ("L", "I"): 1, ("X", "I"): -2, ("Z", "B"): 0, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): -1, ("B", "R"): -1, ("B", "N"): 3,
    ("F", "D"): -4, ("X", "Y"): -2, ("Z", "R"): 0, ("F", "H"): -2,
    ("B", "F"): -4, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum80.cmp
blosum80 = {
    ("W", "F"): 0, ("L", "R"): -3, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 6, ("N", "A"): -2, ("Z", "Y"): -3, ("W", "R"): -4,
    ("Q", "A"): -1, ("S", "D"): -1, ("H", "H"): 8, ("S", "H"): -1,
    ("H", "D"): -2, ("L", "N"): -4, ("W", "A"): -3, ("Y", "M"): -2,
    ("G", "R"): -3, ("Y", "I"): -2, ("Y", "E"): -3, ("B", "Y"): -3,
    ("Y", "A"): -2, ("V", "D"): -4, ("B", "S"): 0, ("Y", "Y"): 7,
    ("G", "N"): -1, ("E", "C"): -5, ("Y", "Q"): -2, ("Z", "Z"): 4,
    ("V", "A"): 0, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 8, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -2, ("V", "M"): 1, ("T", "F"): -2, ("V", "Q"): -3,
    ("K", "K"): 5, ("P", "D"): -2, ("I", "H"): -4, ("I", "D"): -4,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -2, ("M", "N"): -3,
    ("P", "H"): -3, ("F", "Q"): -4, ("Z", "G"): -3, ("X", "L"): -2,
    ("T", "M"): -1, ("Z", "C"): -4, ("X", "H"): -2, ("D", "R"): -2,
    ("B", "W"): -5, ("X", "D"): -2, ("Z", "K"): 1, ("F", "A"): -3,
    ("Z", "W"): -4, ("F", "E"): -4, ("D", "N"): 1, ("B", "K"): -1,
    ("X", "X"): -1, ("F", "I"): -1, ("B", "G"): -1, ("X", "T"): -1,
    ("F", "M"): 0, ("B", "C"): -4, ("Z", "I"): -4, ("Z", "V"): -3,
    ("S", "S"): 5, ("L", "Q"): -3, ("W", "E"): -4, ("Q", "R"): 1,
    ("N", "N"): 6, ("W", "M"): -2, ("Q", "C"): -4, ("W", "I"): -3,
    ("S", "C"): -2, ("L", "A"): -2, ("S", "G"): -1, ("L", "E"): -4,
    ("W", "Q"): -3, ("H", "G"): -3, ("S", "K"): -1, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -4, ("Y", "N"): -3, ("G", "Q"): -2,
    ("Y", "F"): 3, ("C", "A"): -1, ("V", "L"): 1, ("G", "E"): -3,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 1, ("Y", "R"): -3,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -4, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -2, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -4, ("P", "G"): -3, ("M", "A"): -1, ("K", "H"): -1,
    ("V", "R"): -3, ("P", "C"): -4, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 4, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -5,
    ("P", "K"): -1, ("M", "M"): 6, ("K", "D"): -1, ("I", "C"): -2,
    ("Z", "D"): 1, ("F", "R"): -4, ("X", "K"): -1, ("Q", "D"): -1,
    ("X", "G"): -2, ("Z", "L"): -3, ("X", "C"): -3, ("Z", "H"): 0,
    ("B", "L"): -4, ("B", "H"): -1, ("F", "F"): 6, ("X", "W"): -3,
    ("B", "D"): 4, ("D", "A"): -2, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -4, ("S", "R"): -1, ("W", "D"): -6, ("V", "Y"): -2,
    ("W", "L"): -2, ("H", "R"): 0, ("W", "H"): -3, ("H", "N"): 0,
    ("W", "T"): -4, ("T", "T"): 5, ("S", "F"): -3, ("W", "P"): -5,
    ("L", "D"): -5, ("B", "I"): -4, ("L", "H"): -3, ("S", "N"): 0,
    ("B", "T"): -1, ("L", "L"): 4, ("Y", "K"): -3, ("E", "Q"): 2,
    ("Y", "G"): -4, ("Z", "S"): 0, ("Y", "C"): -3, ("G", "D"): -2,
    ("B", "V"): -4, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 6,
    ("Y", "S"): -2, ("C", "N"): -3, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -4, ("T", "L"): -2, ("V", "K"): -3,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -4, ("I", "N"): -4, ("K", "I"): -3, ("M", "D"): -4,
    ("V", "W"): -3, ("W", "W"): 11, ("M", "H"): -2, ("P", "N"): -3,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 1, ("Z", "E"): 4,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -4, ("B", "Q"): 0, ("X", "B"): -2, ("B", "M"): -3,
    ("F", "C"): -3, ("Z", "Q"): 3, ("X", "Z"): -1, ("F", "G"): -4,
    ("B", "E"): 1, ("X", "V"): -1, ("F", "K"): -4, ("B", "A"): -2,
    ("X", "R"): -1, ("D", "D"): 6, ("W", "G"): -4, ("Z", "F"): -4,
    ("S", "Q"): 0, ("W", "C"): -3, ("W", "K"): -4, ("H", "Q"): 1,
    ("L", "C"): -2, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -4, ("S", "E"): 0, ("H", "E"): 0, ("S", "I"): -3,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -2, ("Y", "H"): 2,
    ("Y", "D"): -4, ("E", "R"): -1, ("X", "P"): -2, ("G", "G"): 6,
    ("G", "C"): -4, ("E", "N"): -1, ("Y", "T"): -2, ("Y", "P"): -4,
    ("T", "K"): -1, ("A", "A"): 5, ("P", "Q"): -2, ("T", "C"): -1,
    ("V", "H"): -4, ("T", "G"): -2, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -4, ("V", "P"): -3, ("P", "E"): -2, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 5, ("P", "A"): -1, ("M", "G"): -4,
    ("T", "S"): 1, ("I", "E"): -4, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -2, ("P", "I"): -4, ("R", "R"): 6, ("X", "M"): -1,
    ("L", "I"): 1, ("X", "I"): -2, ("Z", "B"): 0, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): -1, ("B", "R"): -2, ("B", "N"): 4,
    ("F", "D"): -4, ("X", "Y"): -2, ("Z", "R"): 0, ("F", "H"): -2,
    ("B", "F"): -4, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum85.cmp
blosum85 = {
    ("W", "F"): 0, ("L", "R"): -3, ("S", "P"): -1, ("V", "T"): 0,
    ("Q", "Q"): 6, ("N", "A"): -2, ("Z", "Y"): -3, ("W", "R"): -4,
    ("Q", "A"): -1, ("S", "D"): -1, ("H", "H"): 8, ("S", "H"): -1,
    ("H", "D"): -2, ("L", "N"): -4, ("W", "A"): -3, ("Y", "M"): -2,
    ("G", "R"): -3, ("Y", "I"): -2, ("Y", "E"): -4, ("B", "Y"): -4,
    ("Y", "A"): -3, ("V", "D"): -4, ("B", "S"): 0, ("Y", "Y"): 7,
    ("G", "N"): -1, ("E", "C"): -5, ("Y", "Q"): -2, ("Z", "Z"): 4,
    ("V", "A"): -1, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 8, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -2, ("V", "M"): 0, ("T", "F"): -3, ("V", "Q"): -3,
    ("K", "K"): 6, ("P", "D"): -2, ("I", "H"): -4, ("I", "D"): -5,
    ("T", "R"): -2, ("P", "L"): -4, ("K", "G"): -2, ("M", "N"): -3,
    ("P", "H"): -3, ("F", "Q"): -4, ("Z", "G"): -3, ("X", "L"): -2,
    ("T", "M"): -1, ("Z", "C"): -5, ("X", "H"): -2, ("D", "R"): -2,
    ("B", "W"): -5, ("X", "D"): -2, ("Z", "K"): 1, ("F", "A"): -3,
    ("Z", "W"): -4, ("F", "E"): -4, ("D", "N"): 1, ("B", "K"): -1,
    ("X", "X"): -2, ("F", "I"): -1, ("B", "G"): -1, ("X", "T"): -1,
    ("F", "M"): -1, ("B", "C"): -4, ("Z", "I"): -4, ("Z", "V"): -3,
    ("S", "S"): 5, ("L", "Q"): -3, ("W", "E"): -4, ("Q", "R"): 1,
    ("N", "N"): 7, ("W", "M"): -2, ("Q", "C"): -4, ("W", "I"): -3,
    ("S", "C"): -2, ("L", "A"): -2, ("S", "G"): -1, ("L", "E"): -4,
    ("W", "Q"): -3, ("H", "G"): -3, ("S", "K"): -1, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -5, ("Y", "N"): -3, ("G", "Q"): -3,
    ("Y", "F"): 3, ("C", "A"): -1, ("V", "L"): 0, ("G", "E"): -3,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 1, ("Y", "R"): -3,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -5, ("V", "F"): -1,
    ("T", "A"): 0, ("T", "P"): -2, ("B", "P"): -3, ("T", "E"): -1,
    ("V", "N"): -4, ("P", "G"): -3, ("M", "A"): -2, ("K", "H"): -1,
    ("V", "R"): -3, ("P", "C"): -4, ("M", "E"): -3, ("K", "L"): -3,
    ("V", "V"): 5, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -5,
    ("P", "K"): -2, ("M", "M"): 7, ("K", "D"): -1, ("I", "C"): -2,
    ("Z", "D"): 1, ("F", "R"): -4, ("X", "K"): -1, ("Q", "D"): -1,
    ("X", "G"): -2, ("Z", "L"): -4, ("X", "C"): -3, ("Z", "H"): 0,
    ("B", "L"): -5, ("B", "H"): -1, ("F", "F"): 7, ("X", "W"): -3,
    ("B", "D"): 4, ("D", "A"): -2, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -4, ("S", "R"): -1, ("W", "D"): -6, ("V", "Y"): -2,
    ("W", "L"): -3, ("H", "R"): 0, ("W", "H"): -3, ("H", "N"): 0,
    ("W", "T"): -4, ("T", "T"): 5, ("S", "F"): -3, ("W", "P"): -5,
    ("L", "D"): -5, ("B", "I"): -5, ("L", "H"): -3, ("S", "N"): 0,
    ("B", "T"): -1, ("L", "L"): 4, ("Y", "K"): -3, ("E", "Q"): 2,
    ("Y", "G"): -5, ("Z", "S"): -1, ("Y", "C"): -3, ("G", "D"): -2,
    ("B", "V"): -4, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 6,
    ("Y", "S"): -2, ("C", "N"): -4, ("V", "C"): -1, ("T", "H"): -2,
    ("P", "R"): -2, ("V", "G"): -4, ("T", "L"): -2, ("V", "K"): -3,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -4, ("T", "D"): -2,
    ("P", "F"): -4, ("I", "N"): -4, ("K", "I"): -3, ("M", "D"): -4,
    ("V", "W"): -3, ("W", "W"): 11, ("M", "H"): -3, ("P", "N"): -3,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 0, ("Z", "E"): 4,
    ("X", "N"): -2, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -4, ("B", "Q"): -1, ("X", "B"): -2, ("B", "M"): -4,
    ("F", "C"): -3, ("Z", "Q"): 4, ("X", "Z"): -1, ("F", "G"): -4,
    ("B", "E"): 0, ("X", "V"): -1, ("F", "K"): -4, ("B", "A"): -2,
    ("X", "R"): -2, ("D", "D"): 7, ("W", "G"): -4, ("Z", "F"): -4,
    ("S", "Q"): -1, ("W", "C"): -4, ("W", "K"): -5, ("H", "Q"): 1,
    ("L", "C"): -2, ("W", "N"): -5, ("S", "A"): 1, ("L", "G"): -5,
    ("W", "S"): -4, ("S", "E"): -1, ("H", "E"): -1, ("S", "I"): -3,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -2, ("Y", "H"): 2,
    ("Y", "D"): -4, ("E", "R"): -1, ("X", "P"): -2, ("G", "G"): 6,
    ("G", "C"): -4, ("E", "N"): -1, ("Y", "T"): -2, ("Y", "P"): -4,
    ("T", "K"): -1, ("A", "A"): 5, ("P", "Q"): -2, ("T", "C"): -2,
    ("V", "H"): -4, ("T", "G"): -2, ("I", "Q"): -4, ("Z", "T"): -1,
    ("C", "R"): -4, ("V", "P"): -3, ("P", "E"): -2, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 5, ("P", "A"): -1, ("M", "G"): -4,
    ("T", "S"): 1, ("I", "E"): -4, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -2, ("P", "I"): -4, ("R", "R"): 6, ("X", "M"): -1,
    ("L", "I"): 1, ("X", "I"): -2, ("Z", "B"): 0, ("X", "E"): -1,
    ("Z", "N"): -1, ("X", "A"): -1, ("B", "R"): -2, ("B", "N"): 4,
    ("F", "D"): -4, ("X", "Y"): -2, ("Z", "R"): 0, ("F", "H"): -2,
    ("B", "F"): -4, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum90.cmp
blosum90 = {
    ("W", "F"): 0, ("L", "R"): -3, ("S", "P"): -2, ("V", "T"): -1,
    ("Q", "Q"): 7, ("N", "A"): -2, ("Z", "Y"): -3, ("W", "R"): -4,
    ("Q", "A"): -1, ("S", "D"): -1, ("H", "H"): 8, ("S", "H"): -2,
    ("H", "D"): -2, ("L", "N"): -4, ("W", "A"): -4, ("Y", "M"): -2,
    ("G", "R"): -3, ("Y", "I"): -2, ("Y", "E"): -4, ("B", "Y"): -4,
    ("Y", "A"): -3, ("V", "D"): -5, ("B", "S"): 0, ("Y", "Y"): 8,
    ("G", "N"): -1, ("E", "C"): -6, ("Y", "Q"): -3, ("Z", "Z"): 4,
    ("V", "A"): -1, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 8, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -2, ("V", "M"): 0, ("T", "F"): -3, ("V", "Q"): -3,
    ("K", "K"): 6, ("P", "D"): -3, ("I", "H"): -4, ("I", "D"): -5,
    ("T", "R"): -2, ("P", "L"): -4, ("K", "G"): -2, ("M", "N"): -3,
    ("P", "H"): -3, ("F", "Q"): -4, ("Z", "G"): -3, ("X", "L"): -2,
    ("T", "M"): -1, ("Z", "C"): -5, ("X", "H"): -2, ("D", "R"): -3,
    ("B", "W"): -6, ("X", "D"): -2, ("Z", "K"): 1, ("F", "A"): -3,
    ("Z", "W"): -4, ("F", "E"): -5, ("D", "N"): 1, ("B", "K"): -1,
    ("X", "X"): -2, ("F", "I"): -1, ("B", "G"): -2, ("X", "T"): -1,
    ("F", "M"): -1, ("B", "C"): -4, ("Z", "I"): -4, ("Z", "V"): -3,
    ("S", "S"): 5, ("L", "Q"): -3, ("W", "E"): -5, ("Q", "R"): 1,
    ("N", "N"): 7, ("W", "M"): -2, ("Q", "C"): -4, ("W", "I"): -4,
    ("S", "C"): -2, ("L", "A"): -2, ("S", "G"): -1, ("L", "E"): -4,
    ("W", "Q"): -3, ("H", "G"): -3, ("S", "K"): -1, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -5, ("Y", "N"): -3, ("G", "Q"): -3,
    ("Y", "F"): 3, ("C", "A"): -1, ("V", "L"): 0, ("G", "E"): -3,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 1, ("Y", "R"): -3,
    ("M", "Q"): 0, ("T", "I"): -1, ("C", "D"): -5, ("V", "F"): -2,
    ("T", "A"): 0, ("T", "P"): -2, ("B", "P"): -3, ("T", "E"): -1,
    ("V", "N"): -4, ("P", "G"): -3, ("M", "A"): -2, ("K", "H"): -1,
    ("V", "R"): -3, ("P", "C"): -4, ("M", "E"): -3, ("K", "L"): -3,
    ("V", "V"): 5, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -5,
    ("P", "K"): -2, ("M", "M"): 7, ("K", "D"): -1, ("I", "C"): -2,
    ("Z", "D"): 0, ("F", "R"): -4, ("X", "K"): -1, ("Q", "D"): -1,
    ("X", "G"): -2, ("Z", "L"): -4, ("X", "C"): -3, ("Z", "H"): 0,
    ("B", "L"): -5, ("B", "H"): -1, ("F", "F"): 7, ("X", "W"): -3,
    ("B", "D"): 4, ("D", "A"): -3, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -4, ("S", "R"): -1, ("W", "D"): -6, ("V", "Y"): -3,
    ("W", "L"): -3, ("H", "R"): 0, ("W", "H"): -3, ("H", "N"): 0,
    ("W", "T"): -4, ("T", "T"): 6, ("S", "F"): -3, ("W", "P"): -5,
    ("L", "D"): -5, ("B", "I"): -5, ("L", "H"): -4, ("S", "N"): 0,
    ("B", "T"): -1, ("L", "L"): 5, ("Y", "K"): -3, ("E", "Q"): 2,
    ("Y", "G"): -5, ("Z", "S"): -1, ("Y", "C"): -4, ("G", "D"): -2,
    ("B", "V"): -4, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 6,
    ("Y", "S"): -3, ("C", "N"): -4, ("V", "C"): -2, ("T", "H"): -2,
    ("P", "R"): -3, ("V", "G"): -5, ("T", "L"): -2, ("V", "K"): -3,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -4, ("T", "D"): -2,
    ("P", "F"): -4, ("I", "N"): -4, ("K", "I"): -4, ("M", "D"): -4,
    ("V", "W"): -3, ("W", "W"): 11, ("M", "H"): -3, ("P", "N"): -3,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 0, ("Z", "E"): 4,
    ("X", "N"): -2, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -4, ("B", "Q"): -1, ("X", "B"): -2, ("B", "M"): -4,
    ("F", "C"): -3, ("Z", "Q"): 4, ("X", "Z"): -1, ("F", "G"): -5,
    ("B", "E"): 0, ("X", "V"): -2, ("F", "K"): -4, ("B", "A"): -2,
    ("X", "R"): -2, ("D", "D"): 7, ("W", "G"): -4, ("Z", "F"): -4,
    ("S", "Q"): -1, ("W", "C"): -4, ("W", "K"): -5, ("H", "Q"): 1,
    ("L", "C"): -2, ("W", "N"): -5, ("S", "A"): 1, ("L", "G"): -5,
    ("W", "S"): -4, ("S", "E"): -1, ("H", "E"): -1, ("S", "I"): -3,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -2, ("Y", "H"): 1,
    ("Y", "D"): -4, ("E", "R"): -1, ("X", "P"): -2, ("G", "G"): 6,
    ("G", "C"): -4, ("E", "N"): -1, ("Y", "T"): -2, ("Y", "P"): -4,
    ("T", "K"): -1, ("A", "A"): 5, ("P", "Q"): -2, ("T", "C"): -2,
    ("V", "H"): -4, ("T", "G"): -3, ("I", "Q"): -4, ("Z", "T"): -1,
    ("C", "R"): -5, ("V", "P"): -3, ("P", "E"): -2, ("M", "C"): -2,
    ("K", "N"): 0, ("I", "I"): 5, ("P", "A"): -1, ("M", "G"): -4,
    ("T", "S"): 1, ("I", "E"): -4, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -2, ("P", "I"): -4, ("R", "R"): 6, ("X", "M"): -1,
    ("L", "I"): 1, ("X", "I"): -2, ("Z", "B"): 0, ("X", "E"): -2,
    ("Z", "N"): -1, ("X", "A"): -1, ("B", "R"): -2, ("B", "N"): 4,
    ("F", "D"): -5, ("X", "Y"): -2, ("Z", "R"): 0, ("F", "H"): -2,
    ("B", "F"): -4, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/blosum95.cmp
blosum95 = {
    ("W", "F"): 0, ("L", "R"): -3, ("S", "P"): -2, ("V", "T"): -1,
    ("Q", "Q"): 7, ("N", "A"): -2, ("Z", "Y"): -4, ("W", "R"): -4,
    ("Q", "A"): -1, ("S", "D"): -1, ("H", "H"): 9, ("S", "H"): -2,
    ("H", "D"): -2, ("L", "N"): -5, ("W", "A"): -4, ("Y", "M"): -3,
    ("G", "R"): -4, ("Y", "I"): -2, ("Y", "E"): -4, ("B", "Y"): -4,
    ("Y", "A"): -3, ("V", "D"): -5, ("B", "S"): -1, ("Y", "Y"): 8,
    ("G", "N"): -1, ("E", "C"): -6, ("Y", "Q"): -3, ("Z", "Z"): 4,
    ("V", "A"): -1, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): -1, ("P", "P"): 8, ("V", "I"): 3, ("V", "S"): -3,
    ("Z", "P"): -2, ("V", "M"): 0, ("T", "F"): -3, ("V", "Q"): -3,
    ("K", "K"): 6, ("P", "D"): -3, ("I", "H"): -4, ("I", "D"): -5,
    ("T", "R"): -2, ("P", "L"): -4, ("K", "G"): -3, ("M", "N"): -3,
    ("P", "H"): -3, ("F", "Q"): -4, ("Z", "G"): -3, ("X", "L"): -2,
    ("T", "M"): -1, ("Z", "C"): -5, ("X", "H"): -2, ("D", "R"): -3,
    ("B", "W"): -6, ("X", "D"): -2, ("Z", "K"): 0, ("F", "A"): -3,
    ("Z", "W"): -4, ("F", "E"): -5, ("D", "N"): 1, ("B", "K"): -1,
    ("X", "X"): -2, ("F", "I"): -1, ("B", "G"): -2, ("X", "T"): -1,
    ("F", "M"): -1, ("B", "C"): -4, ("Z", "I"): -4, ("Z", "V"): -3,
    ("S", "S"): 5, ("L", "Q"): -3, ("W", "E"): -5, ("Q", "R"): 0,
    ("N", "N"): 7, ("W", "M"): -2, ("Q", "C"): -4, ("W", "I"): -4,
    ("S", "C"): -2, ("L", "A"): -2, ("S", "G"): -1, ("L", "E"): -4,
    ("W", "Q"): -3, ("H", "G"): -3, ("S", "K"): -1, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -5, ("Y", "N"): -3, ("G", "Q"): -3,
    ("Y", "F"): 3, ("C", "A"): -1, ("V", "L"): 0, ("G", "E"): -3,
    ("G", "A"): -1, ("K", "R"): 2, ("E", "D"): 1, ("Y", "R"): -3,
    ("M", "Q"): -1, ("T", "I"): -2, ("C", "D"): -5, ("V", "F"): -2,
    ("T", "A"): 0, ("T", "P"): -2, ("B", "P"): -3, ("T", "E"): -2,
    ("V", "N"): -4, ("P", "G"): -4, ("M", "A"): -2, ("K", "H"): -1,
    ("V", "R"): -4, ("P", "C"): -5, ("M", "E"): -3, ("K", "L"): -3,
    ("V", "V"): 5, ("M", "I"): 1, ("T", "Q"): -1, ("I", "G"): -6,
    ("P", "K"): -2, ("M", "M"): 7, ("K", "D"): -2, ("I", "C"): -2,
    ("Z", "D"): 0, ("F", "R"): -4, ("X", "K"): -1, ("Q", "D"): -1,
    ("X", "G"): -3, ("Z", "L"): -4, ("X", "C"): -3, ("Z", "H"): 0,
    ("B", "L"): -5, ("B", "H"): -1, ("F", "F"): 7, ("X", "W"): -4,
    ("B", "D"): 4, ("D", "A"): -3, ("S", "L"): -3, ("X", "S"): -1,
    ("F", "N"): -4, ("S", "R"): -2, ("W", "D"): -6, ("V", "Y"): -3,
    ("W", "L"): -3, ("H", "R"): -1, ("W", "H"): -3, ("H", "N"): 0,
    ("W", "T"): -4, ("T", "T"): 6, ("S", "F"): -3, ("W", "P"): -5,
    ("L", "D"): -5, ("B", "I"): -5, ("L", "H"): -4, ("S", "N"): 0,
    ("B", "T"): -1, ("L", "L"): 5, ("Y", "K"): -3, ("E", "Q"): 2,
    ("Y", "G"): -5, ("Z", "S"): -1, ("Y", "C"): -4, ("G", "D"): -2,
    ("B", "V"): -5, ("E", "A"): -1, ("Y", "W"): 2, ("E", "E"): 6,
    ("Y", "S"): -3, ("C", "N"): -4, ("V", "C"): -2, ("T", "H"): -2,
    ("P", "R"): -3, ("V", "G"): -5, ("T", "L"): -2, ("V", "K"): -3,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -4, ("T", "D"): -2,
    ("P", "F"): -5, ("I", "N"): -4, ("K", "I"): -4, ("M", "D"): -5,
    ("V", "W"): -3, ("W", "W"): 11, ("M", "H"): -3, ("P", "N"): -3,
    ("K", "A"): -1, ("M", "L"): 2, ("K", "E"): 0, ("Z", "E"): 4,
    ("X", "N"): -2, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -5, ("B", "Q"): -1, ("X", "B"): -2, ("B", "M"): -4,
    ("F", "C"): -3, ("Z", "Q"): 4, ("X", "Z"): -1, ("F", "G"): -5,
    ("B", "E"): 0, ("X", "V"): -2, ("F", "K"): -4, ("B", "A"): -3,
    ("X", "R"): -2, ("D", "D"): 7, ("W", "G"): -5, ("Z", "F"): -4,
    ("S", "Q"): -1, ("W", "C"): -4, ("W", "K"): -5, ("H", "Q"): 1,
    ("L", "C"): -3, ("W", "N"): -5, ("S", "A"): 1, ("L", "G"): -5,
    ("W", "S"): -4, ("S", "E"): -1, ("H", "E"): -1, ("S", "I"): -3,
    ("H", "A"): -3, ("S", "M"): -3, ("Y", "L"): -2, ("Y", "H"): 1,
    ("Y", "D"): -5, ("E", "R"): -1, ("X", "P"): -3, ("G", "G"): 6,
    ("G", "C"): -5, ("E", "N"): -1, ("Y", "T"): -3, ("Y", "P"): -5,
    ("T", "K"): -1, ("A", "A"): 5, ("P", "Q"): -2, ("T", "C"): -2,
    ("V", "H"): -4, ("T", "G"): -3, ("I", "Q"): -4, ("Z", "T"): -2,
    ("C", "R"): -5, ("V", "P"): -4, ("P", "E"): -2, ("M", "C"): -3,
    ("K", "N"): 0, ("I", "I"): 5, ("P", "A"): -1, ("M", "G"): -4,
    ("T", "S"): 1, ("I", "E"): -4, ("P", "M"): -3, ("M", "K"): -2,
    ("I", "A"): -2, ("P", "I"): -4, ("R", "R"): 7, ("X", "M"): -2,
    ("L", "I"): 1, ("X", "I"): -2, ("Z", "B"): 0, ("X", "E"): -2,
    ("Z", "N"): -1, ("X", "A"): -1, ("B", "R"): -2, ("B", "N"): 4,
    ("F", "D"): -5, ("X", "Y"): -2, ("Z", "R"): -1, ("F", "H"): -2,
    ("B", "F"): -5, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/feng.cmp
feng = {
    ("W", "F"): 3, ("L", "R"): 2, ("I", "I"): 6, ("Q", "Q"): 6,
    ("W", "N"): 0, ("V", "I"): 5, ("H", "T"): 2, ("H", "P"): 3,
    ("W", "V"): 3, ("Q", "E"): 4, ("W", "R"): 2, ("Q", "A"): 3,
    ("H", "H"): 6, ("H", "D"): 3, ("L", "N"): 1, ("Y", "M"): 2,
    ("Y", "I"): 3, ("Y", "E"): 1, ("E", "S"): 3, ("Y", "A"): 2,
    ("Y", "Y"): 6, ("T", "C"): 2, ("E", "C"): 0, ("Y", "Q"): 2,
    ("E", "G"): 4, ("V", "A"): 5, ("C", "C"): 6, ("M", "R"): 2,
    ("P", "T"): 4, ("V", "E"): 4, ("P", "P"): 6, ("I", "T"): 3,
    ("K", "S"): 3, ("R", "G"): 3, ("I", "P"): 2, ("R", "C"): 2,
    ("A", "T"): 5, ("K", "K"): 6, ("A", "P"): 5, ("V", "M"): 4,
    ("I", "D"): 1, ("K", "C"): 0, ("K", "G"): 2, ("R", "S"): 3,
    ("F", "Q"): 1, ("F", "A"): 2, ("V", "V"): 6, ("M", "N"): 1,
    ("F", "E"): 0, ("D", "N"): 5, ("F", "I"): 4, ("F", "M"): 2,
    ("M", "S"): 1, ("S", "S"): 6, ("L", "Q"): 2, ("W", "E"): 1,
    ("W", "A"): 2, ("W", "M"): 3, ("H", "S"): 3, ("W", "I"): 2,
    ("S", "C"): 4, ("L", "A"): 2, ("L", "E"): 1, ("W", "Q"): 1,
    ("H", "G"): 1, ("Q", "N"): 3, ("H", "C"): 2, ("L", "M"): 5,
    ("W", "Y"): 3, ("Y", "N"): 3, ("E", "P"): 3, ("Y", "F"): 5,
    ("E", "T"): 3, ("A", "A"): 6, ("I", "N"): 2, ("G", "A"): 5,
    ("Y", "V"): 3, ("E", "D"): 5, ("W", "H"): 1, ("Y", "R"): 1,
    ("M", "Q"): 2, ("P", "S"): 4, ("R", "H"): 4, ("A", "C"): 2,
    ("R", "D"): 2, ("K", "P"): 2, ("L", "D"): 1, ("K", "T"): 4,
    ("V", "N"): 2, ("M", "A"): 2, ("K", "H"): 3, ("V", "R"): 2,
    ("P", "C"): 2, ("M", "E"): 1, ("A", "S"): 5, ("T", "T"): 6,
    ("R", "T"): 3, ("I", "G"): 2, ("R", "P"): 3, ("K", "D"): 3,
    ("I", "C"): 2, ("F", "R"): 1, ("F", "V"): 4, ("L", "C"): 2,
    ("F", "F"): 6, ("D", "A"): 4, ("F", "N"): 1, ("W", "D"): 0,
    ("L", "P"): 3, ("Q", "S"): 3, ("N", "C"): 2, ("N", "G"): 3,
    ("H", "N"): 4, ("W", "T"): 1, ("Q", "G"): 2, ("W", "P"): 2,
    ("Q", "C"): 1, ("N", "S"): 5, ("L", "H"): 3, ("L", "L"): 6,
    ("G", "T"): 2, ("M", "M"): 6, ("G", "P"): 3, ("Y", "K"): 1,
    ("Y", "G"): 2, ("Y", "C"): 3, ("E", "A"): 4, ("E", "E"): 6,
    ("Y", "S"): 3, ("M", "P"): 2, ("V", "C"): 2, ("M", "T"): 3,
    ("V", "G"): 4, ("R", "E"): 2, ("V", "K"): 3, ("K", "Q"): 4,
    ("R", "A"): 2, ("I", "R"): 2, ("N", "A"): 3, ("V", "S"): 2,
    ("M", "D"): 0, ("M", "H"): 1, ("K", "A"): 3, ("R", "Q"): 3,
    ("K", "E"): 4, ("F", "S"): 3, ("I", "K"): 2, ("D", "P"): 2,
    ("D", "T"): 2, ("I", "M"): 4, ("F", "C"): 3, ("W", "L"): 4,
    ("F", "G"): 1, ("F", "K"): 0, ("F", "T"): 1, ("D", "D"): 6,
    ("Q", "T"): 3, ("W", "G"): 3, ("Q", "P"): 3, ("W", "C"): 3,
    ("W", "K"): 1, ("H", "Q"): 4, ("Q", "D"): 4, ("W", "W"): 6,
    ("V", "L"): 5, ("L", "G"): 2, ("W", "S"): 2, ("L", "K"): 2,
    ("N", "P"): 2, ("H", "E"): 2, ("N", "T"): 4, ("H", "A"): 2,
    ("Y", "L"): 3, ("Y", "H"): 3, ("G", "S"): 5, ("Y", "D"): 2,
    ("V", "Q"): 2, ("L", "T"): 2, ("G", "G"): 6, ("G", "C"): 3,
    ("E", "N"): 3, ("Y", "T"): 2, ("Y", "P"): 2, ("R", "N"): 2,
    ("V", "D"): 3, ("K", "R"): 5, ("V", "H"): 1, ("I", "Q"): 1,
    ("V", "P"): 3, ("M", "C"): 2, ("K", "N"): 4, ("V", "T"): 3,
    ("M", "G"): 1, ("T", "S"): 5, ("I", "E"): 1, ("M", "K"): 2,
    ("I", "A"): 2, ("N", "N"): 6, ("R", "R"): 6, ("F", "P"): 2,
    ("L", "I"): 5, ("I", "S"): 2, ("D", "S"): 3, ("L", "S"): 2,
    ("I", "H"): 1, ("F", "D"): 1, ("D", "C"): 1, ("F", "H"): 2,
    ("D", "G"): 4, ("F", "L"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/fitch.cmp
fitch = {
    ("W", "F"): 2, ("U", "I"): 1, ("W", "T"): 0, ("R", "R"): 3,
    ("N", "M"): 1, ("U", "M"): 2, ("U", "O"): 1, ("N", "A"): 2,
    ("U", "A"): 1, ("N", "C"): 1, ("U", "C"): 1, ("N", "E"): 2,
    ("U", "E"): 1, ("W", "H"): 1, ("W", "R"): 1, ("W", "V"): 2,
    ("Q", "E"): 1, ("S", "S"): 3, ("T", "T"): 3, ("S", "F"): 2,
    ("Q", "A"): 1, ("Q", "C"): 1, ("H", "F"): 1, ("S", "H"): 1,
    ("L", "H"): 1, ("S", "N"): 2, ("Q", "I"): 2, ("S", "L"): 2,
    ("L", "L"): 3, ("Y", "M"): 1, ("M", "M"): 3, ("Y", "I"): 1,
    ("W", "N"): 2, ("Y", "R"): 2, ("Y", "E"): 1, ("O", "O"): 3,
    ("Y", "A"): 2, ("Y", "C"): 2, ("U", "S"): 2, ("Y", "U"): 2,
    ("Q", "Q"): 3, ("E", "A"): 1, ("N", "L"): 1, ("E", "C"): 1,
    ("W", "W"): 3, ("E", "E"): 3, ("Y", "S"): 2, ("V", "A"): 2,
    ("C", "C"): 3, ("V", "C"): 2, ("T", "H"): 0, ("V", "E"): 1,
    ("T", "N"): 0, ("O", "I"): 1, ("R", "E"): 2, ("O", "E"): 1,
    ("R", "A"): 1, ("O", "C"): 2, ("R", "C"): 2, ("O", "A"): 2,
    ("V", "Q"): 1, ("V", "M"): 2, ("V", "S"): 1, ("Q", "O"): 2,
    ("I", "H"): 1, ("M", "F"): 1, ("R", "Q"): 2, ("M", "H"): 0,
    ("O", "F"): 2, ("T", "F"): 1, ("M", "I"): 2, ("M", "L"): 1,
    ("V", "T"): 1, ("Q", "H"): 2, ("Q", "N"): 1, ("T", "M"): 1,
    ("W", "M"): 2, ("Y", "Y"): 3, ("T", "S"): 2, ("R", "O"): 1,
    ("T", "R"): 2, ("H", "H"): 3, ("F", "A"): 1, ("F", "C"): 2,
    ("F", "E"): 1, ("W", "L"): 1, ("T", "L"): 1, ("U", "R"): 2,
    ("U", "H"): 2, ("N", "H"): 2, ("U", "Q"): 2, ("W", "E"): 2,
    ("U", "L"): 1, ("W", "C"): 1, ("U", "N"): 1, ("W", "A"): 1,
    ("W", "O"): 1, ("I", "F"): 0, ("U", "F"): 2, ("N", "F"): 1,
    ("L", "C"): 1, ("S", "C"): 2, ("L", "A"): 2, ("S", "A"): 1,
    ("W", "S"): 2, ("L", "E"): 2, ("S", "E"): 2, ("Q", "L"): 1,
    ("H", "E"): 1, ("S", "I"): 1, ("H", "C"): 1, ("S", "O"): 2,
    ("H", "A"): 2, ("S", "M"): 1, ("Y", "L"): 2, ("Y", "N"): 1,
    ("Y", "H"): 1, ("O", "M"): 0, ("Y", "Q"): 1, ("Y", "F"): 1,
    ("W", "I"): 1, ("C", "A"): 1, ("R", "I"): 2, ("Y", "O"): 1,
    ("Q", "M"): 1, ("S", "Q"): 2, ("U", "T"): 2, ("Y", "T"): 2,
    ("Y", "V"): 2, ("O", "L"): 1, ("R", "N"): 1, ("A", "A"): 3,
    ("N", "I"): 2, ("R", "L"): 1, ("T", "I"): 1, ("L", "F"): 1,
    ("T", "O"): 1, ("R", "H"): 2, ("O", "H"): 2, ("V", "F"): 2,
    ("T", "C"): 2, ("V", "H"): 1, ("T", "A"): 0, ("R", "F"): 1,
    ("V", "L"): 2, ("T", "E"): 1, ("V", "N"): 1, ("M", "A"): 0,
    ("Q", "F"): 1, ("M", "C"): 0, ("W", "U"): 2, ("I", "I"): 3,
    ("V", "V"): 3, ("O", "N"): 2, ("I", "E"): 2, ("T", "Q"): 2,
    ("I", "A"): 1, ("N", "N"): 3, ("I", "C"): 0, ("S", "R"): 2,
    ("V", "R"): 1, ("L", "I"): 1, ("V", "I"): 1, ("R", "M"): 2,
    ("Y", "W"): 1, ("M", "E"): 2, ("V", "U"): 2, ("W", "Q"): 0,
    ("U", "U"): 3, ("V", "O"): 1, ("F", "F"): 3
}


# http://www.embl-heidelberg.de/~vogt/matrices/genetic.cmp
genetic = {
    ("W", "F"): 0.0, ("L", "R"): -0.4, ("I", "I"): 4.1, ("Q", "Q"): 5.5,
    ("W", "N"): -3.0, ("V", "I"): 1.0, ("H", "T"): -1.8, ("H", "P"): 0.7,
    ("W", "V"): -2.1, ("Q", "E"): 2.0, ("W", "R"): 1.8, ("Q", "A"): -2.1,
    ("H", "H"): 4.7, ("H", "D"): 1.7, ("L", "N"): -2.2, ("Y", "M"): -2.9,
    ("Y", "I"): -1.6, ("Y", "E"): -0.9, ("E", "S"): -2.8, ("Y", "A"): -2.4,
    ("Y", "Y"): 6.5, ("T", "C"): -1.9, ("E", "C"): -3.0, ("Y", "Q"): -0.8,
    ("E", "G"): 1.4, ("V", "A"): 1.0, ("C", "C"): 5.5, ("M", "R"): -0.4,
    ("P", "T"): 1.1, ("V", "E"): 1.3, ("P", "P"): 3.8, ("I", "T"): 0.8,
    ("K", "S"): -1.5, ("R", "G"): 0.8, ("I", "P"): -1.6, ("R", "C"): 0.7,
    ("A", "T"): 0.9, ("K", "K"): 5.6, ("A", "P"): 0.8, ("V", "M"): 1.0,
    ("I", "D"): -2.1, ("K", "C"): -3.2, ("K", "G"): -2.2, ("R", "S"): 0.3,
    ("F", "Q"): -2.1, ("F", "A"): -2.4, ("V", "V"): 4.1, ("M", "N"): 0.1,
    ("F", "E"): -2.9, ("D", "N"): 1.7, ("F", "I"): 1.3, ("F", "M"): 0.5,
    ("M", "S"): -1.3, ("S", "S"): 2.6, ("L", "Q"): 0.1, ("W", "E"): -3.2,
    ("W", "A"): -2.2, ("W", "M"): -2.0, ("H", "S"): -1.6, ("W", "I"): -2.2,
    ("S", "C"): 1.5, ("L", "A"): -2.3, ("L", "E"): -2.5, ("W", "Q"): -2.3,
    ("H", "G"): -2.2, ("Q", "N"): 0.4, ("H", "C"): -1.6, ("L", "M"): 1.5,
    ("W", "Y"): -0.5, ("Y", "N"): 2.5, ("E", "P"): -2.1, ("Y", "F"): 2.0,
    ("E", "T"): -2.1, ("A", "A"): 4.0, ("I", "N"): 0.9, ("G", "A"): 1.2,
    ("Y", "V"): -2.2, ("E", "D"): 3.8, ("W", "H"): -2.1, ("Y", "R"): -1.9,
    ("M", "Q"): -1.2, ("P", "S"): 0.4, ("R", "H"): 3.6, ("A", "C"): -1.9,
    ("R", "D"): -2.3, ("K", "P"): -1.5, ("L", "D"): -2.4, ("K", "T"): 1.0,
    ("V", "N"): -2.2, ("M", "A"): -2.0, ("K", "H"): 0.6, ("V", "R"): -2.1,
    ("P", "C"): -1.9, ("M", "E"): -1.8, ("A", "S"): 0.1, ("T", "T"): 4.0,
    ("R", "T"): -0.6, ("I", "G"): -2.5, ("R", "P"): 0.3, ("K", "D"): 0.3,
    ("I", "C"): -1.9, ("F", "R"): -1.5, ("F", "V"): 1.0, ("L", "C"): -1.3,
    ("F", "F"): 4.5, ("D", "A"): 1.0, ("F", "N"): -1.3, ("W", "D"): -2.9,
    ("L", "P"): 0.0, ("Q", "S"): -2.3, ("N", "C"): -1.5, ("N", "G"): -2.6,
    ("H", "N"): 1.8, ("W", "T"): -2.2, ("Q", "G"): -2.1, ("W", "P"): -1.6,
    ("Q", "C"): -3.1, ("N", "S"): -0.3, ("L", "H"): -0.1, ("L", "L"): 3.4,
    ("G", "T"): -2.1, ("M", "M"): 5.4, ("G", "P"): -1.8, ("Y", "K"): -0.8,
    ("Y", "G"): -1.8, ("Y", "C"): 2.6, ("E", "A"): 1.3, ("E", "E"): 5.7,
    ("Y", "S"): 0.3, ("M", "P"): -1.4, ("V", "C"): -2.2, ("M", "T"): 0.7,
    ("V", "G"): 1.1, ("R", "E"): -2.0, ("V", "K"): -2.1, ("K", "Q"): 2.2,
    ("R", "A"): -1.6, ("I", "R"): -1.2, ("N", "A"): -1.7, ("V", "S"): -2.2,
    ("M", "D"): -2.5, ("M", "H"): -1.8, ("K", "A"): -1.9, ("R", "Q"): 0.3,
    ("K", "E"): 2.0, ("F", "S"): 0.0, ("I", "K"): 0.7, ("D", "P"): -2.2,
    ("D", "T"): -2.1, ("I", "M"): 3.3, ("F", "C"): 1.8, ("W", "L"): -0.3,
    ("F", "G"): -1.9, ("F", "K"): -2.8, ("F", "T"): -2.1, ("D", "D"): 4.8,
    ("Q", "T"): -1.7, ("W", "G"): 1.4, ("Q", "P"): 1.0, ("W", "C"): 4.1,
    ("W", "K"): -3.0, ("H", "Q"): 3.6, ("Q", "D"): 0.3, ("W", "W"): 7.5,
    ("V", "L"): 1.1, ("L", "G"): -2.2, ("W", "S"): 0.8, ("L", "K"): -2.0,
    ("N", "P"): -1.6, ("H", "E"): 0.3, ("N", "T"): 0.9, ("H", "A"): -2.1,
    ("Y", "L"): -1.6, ("Y", "H"): 2.3, ("G", "S"): -0.6, ("Y", "D"): 2.3,
    ("V", "Q"): -2.0, ("L", "T"): -1.9, ("G", "G"): 4.2, ("G", "C"): 1.0,
    ("E", "N"): 0.3, ("Y", "T"): -2.1, ("Y", "P"): -2.3, ("R", "N"): -1.5,
    ("V", "D"): 1.0, ("K", "R"): -0.2, ("V", "H"): -2.1, ("I", "Q"): -1.9,
    ("V", "P"): -2.1, ("M", "C"): -2.7, ("K", "N"): 3.5, ("V", "T"): -2.2,
    ("M", "G"): -2.3, ("T", "S"): 1.0, ("I", "E"): -2.3, ("M", "K"): 1.6,
    ("I", "A"): -1.8, ("N", "N"): 4.7, ("R", "R"): 2.9, ("F", "P"): -1.8,
    ("L", "I"): 1.2, ("I", "S"): -0.5, ("D", "S"): -2.1, ("L", "S"): -1.2,
    ("I", "H"): -1.8, ("F", "D"): -1.7, ("D", "C"): -1.6, ("F", "H"): -1.1,
    ("D", "G"): 1.1, ("F", "L"): 2.2
}


# http://www.embl-heidelberg.de/~vogt/matrices/gonnet.cmp
gonnet = {
    ("W", "F"): 3.6, ("L", "R"): -2.2, ("I", "I"): 4.0, ("Q", "Q"): 2.7,
    ("W", "N"): -3.6, ("V", "I"): 3.1, ("H", "T"): -0.3, ("H", "P"): -1.1,
    ("W", "V"): -2.6, ("Q", "E"): 1.7, ("W", "R"): -1.6, ("Q", "A"): -0.2,
    ("H", "H"): 6.0, ("H", "D"): 0.4, ("L", "N"): -3.0, ("Y", "M"): -0.2,
    ("Y", "I"): -0.7, ("Y", "E"): -2.7, ("E", "S"): 0.2, ("Y", "A"): -2.2,
    ("Y", "Y"): 7.8, ("T", "C"): -0.5, ("E", "C"): -3.0, ("Y", "Q"): -1.7,
    ("E", "G"): -0.8, ("V", "A"): 0.1, ("C", "C"): 11.5, ("M", "R"): -1.7,
    ("P", "T"): 0.1, ("V", "E"): -1.9, ("P", "P"): 7.6, ("I", "T"): -0.6,
    ("K", "S"): 0.1, ("R", "G"): -1.0, ("I", "P"): -2.6, ("R", "C"): -2.2,
    ("A", "T"): 0.6, ("K", "K"): 3.2, ("A", "P"): 0.3, ("V", "M"): 1.6,
    ("I", "D"): -3.8, ("K", "C"): -2.8, ("K", "G"): -1.1, ("R", "S"): -0.2,
    ("F", "Q"): -2.6, ("F", "A"): -2.3, ("V", "V"): 3.4, ("M", "N"): -2.2,
    ("F", "E"): -3.9, ("D", "N"): 2.2, ("F", "I"): 1.0, ("F", "M"): 1.6,
    ("M", "S"): -1.4, ("S", "S"): 2.2, ("L", "Q"): -1.6, ("W", "E"): -4.3,
    ("W", "A"): -3.6, ("W", "M"): -1.0, ("H", "S"): -0.2, ("W", "I"): -1.8,
    ("S", "C"): 0.1, ("L", "A"): -1.2, ("L", "E"): -2.8, ("W", "Q"): -2.7,
    ("H", "G"): -1.4, ("Q", "N"): 0.7, ("H", "C"): -1.3, ("L", "M"): 2.8,
    ("W", "Y"): 4.1, ("Y", "N"): -1.4, ("E", "P"): -0.5, ("Y", "F"): 5.1,
    ("E", "T"): -0.1, ("A", "A"): 2.4, ("I", "N"): -2.8, ("G", "A"): 0.5,
    ("Y", "V"): -1.1, ("E", "D"): 2.7, ("W", "H"): -0.8, ("Y", "R"): -1.8,
    ("M", "Q"): -1.0, ("P", "S"): 0.4, ("R", "H"): 0.6, ("A", "C"): 0.5,
    ("R", "D"): -0.3, ("K", "P"): -0.6, ("L", "D"): -4.0, ("K", "T"): 0.1,
    ("V", "N"): -2.2, ("M", "A"): -0.7, ("K", "H"): 0.6, ("V", "R"): -2.0,
    ("P", "C"): -3.1, ("M", "E"): -2.0, ("A", "S"): 1.1, ("T", "T"): 2.5,
    ("R", "T"): -0.2, ("I", "G"): -4.5, ("R", "P"): -0.9, ("K", "D"): 0.5,
    ("I", "C"): -1.1, ("F", "R"): -3.2, ("F", "V"): 0.1, ("L", "C"): -1.5,
    ("F", "F"): 7.0, ("D", "A"): -0.3, ("F", "N"): -3.1, ("W", "D"): -5.2,
    ("L", "P"): -2.3, ("Q", "S"): 0.2, ("N", "C"): -1.8, ("N", "G"): 0.4,
    ("H", "N"): 1.2, ("W", "T"): -3.5, ("Q", "G"): -1.0, ("W", "P"): -5.0,
    ("Q", "C"): -2.4, ("N", "S"): 0.9, ("L", "H"): -1.9, ("L", "L"): 4.0,
    ("G", "T"): -1.1, ("M", "M"): 4.3, ("G", "P"): -1.6, ("Y", "K"): -2.1,
    ("Y", "G"): -4.0, ("Y", "C"): -0.5, ("E", "A"): 0.0, ("E", "E"): 3.6,
    ("Y", "S"): -1.9, ("M", "P"): -2.4, ("V", "C"): 0.0, ("M", "T"): -0.6,
    ("V", "G"): -3.3, ("R", "E"): 0.4, ("V", "K"): -1.7, ("K", "Q"): 1.5,
    ("R", "A"): -0.6, ("I", "R"): -2.4, ("N", "A"): -0.3, ("V", "S"): -1.0,
    ("M", "D"): -3.0, ("M", "H"): -1.3, ("K", "A"): -0.4, ("R", "Q"): 1.5,
    ("K", "E"): 1.2, ("F", "S"): -2.8, ("I", "K"): -2.1, ("D", "P"): -0.7,
    ("D", "T"): 0.0, ("I", "M"): 2.5, ("F", "C"): -0.8, ("W", "L"): -0.7,
    ("F", "G"): -5.2, ("F", "K"): -3.3, ("F", "T"): -2.2, ("D", "D"): 4.7,
    ("Q", "T"): 0.0, ("W", "G"): -4.0, ("Q", "P"): -0.2, ("W", "C"): -1.0,
    ("W", "K"): -3.5, ("H", "Q"): 1.2, ("Q", "D"): 0.9, ("W", "W"): 14.2,
    ("V", "L"): 1.8, ("L", "G"): -4.4, ("W", "S"): -3.3, ("L", "K"): -2.1,
    ("N", "P"): -0.9, ("H", "E"): 0.4, ("N", "T"): 0.5, ("H", "A"): -0.8,
    ("Y", "L"): 0.0, ("Y", "H"): 2.2, ("G", "S"): 0.4, ("Y", "D"): -2.8,
    ("V", "Q"): -1.5, ("L", "T"): -1.3, ("G", "G"): 6.6, ("G", "C"): -2.0,
    ("E", "N"): 0.9, ("Y", "T"): -1.9, ("Y", "P"): -3.1, ("R", "N"): 0.3,
    ("V", "D"): -2.9, ("K", "R"): 2.7, ("V", "H"): -2.0, ("I", "Q"): -1.9,
    ("V", "P"): -1.8, ("M", "C"): -0.9, ("K", "N"): 0.8, ("V", "T"): 0.0,
    ("M", "G"): -3.5, ("T", "S"): 1.5, ("I", "E"): -2.7, ("M", "K"): -1.4,
    ("I", "A"): -0.8, ("N", "N"): 3.8, ("R", "R"): 4.7, ("F", "P"): -3.8,
    ("L", "I"): 2.8, ("I", "S"): -1.8, ("D", "S"): 0.5, ("L", "S"): -2.1,
    ("I", "H"): -2.2, ("F", "D"): -4.5, ("D", "C"): -3.2, ("F", "H"): -0.1,
    ("D", "G"): 0.1, ("F", "L"): 2.0
}


# from https://www.genome.jp/dbget-bin/www_bget?aaindex:GRAR740104,
# but subtracted from 215
# as per original reference now available at
# https://web.archive.org/web/19991111011852/http://www.embl-heidelberg.de:80/~vogt/matrices/grant.cmp
# which was found to have incorrect amino acid characters (issue 308)
grant = {
    ("A", "A"): 215, ("C", "A"): 20, ("C", "C"): 215, ("C", "D"): 61,
    ("C", "N"): 76, ("C", "R"): 35, ("D", "A"): 89, ("D", "D"): 215,
    ("D", "N"): 192, ("D", "R"): 119, ("E", "A"): 108, ("E", "C"): 45,
    ("E", "D"): 170, ("E", "E"): 215, ("E", "N"): 173, ("E", "Q"): 186,
    ("E", "R"): 161, ("F", "A"): 102, ("F", "C"): 10, ("F", "D"): 38,
    ("F", "E"): 75, ("F", "F"): 215, ("F", "G"): 62, ("F", "H"): 115,
    ("F", "I"): 194, ("F", "K"): 113, ("F", "L"): 193, ("F", "M"): 187,
    ("F", "N"): 57, ("F", "Q"): 99, ("F", "R"): 118, ("G", "A"): 155,
    ("G", "C"): 56, ("G", "D"): 121, ("G", "E"): 117, ("G", "G"): 215,
    ("G", "N"): 135, ("G", "Q"): 128, ("G", "R"): 90, ("H", "A"): 129,
    ("H", "C"): 41, ("H", "D"): 134, ("H", "E"): 175, ("H", "G"): 117,
    ("H", "H"): 215, ("H", "N"): 147, ("H", "Q"): 191, ("H", "R"): 186,
    ("I", "A"): 121, ("I", "C"): 17, ("I", "D"): 47, ("I", "E"): 81,
    ("I", "G"): 80, ("I", "H"): 121, ("I", "I"): 215, ("I", "N"): 66,
    ("I", "Q"): 106, ("I", "R"): 118, ("K", "A"): 109, ("K", "C"): 13,
    ("K", "D"): 114, ("K", "E"): 159, ("K", "G"): 88, ("K", "H"): 183,
    ("K", "I"): 113, ("K", "K"): 215, ("K", "L"): 108, ("K", "N"): 121,
    ("K", "Q"): 162, ("K", "R"): 189, ("L", "A"): 119, ("L", "C"): 17,
    ("L", "D"): 43, ("L", "E"): 77, ("L", "G"): 77, ("L", "H"): 116,
    ("L", "I"): 210, ("L", "L"): 215, ("L", "N"): 62, ("L", "Q"): 102,
    ("L", "R"): 113, ("M", "A"): 131, ("M", "C"): 19, ("M", "D"): 55,
    ("M", "E"): 89, ("M", "G"): 88, ("M", "H"): 128, ("M", "I"): 205,
    ("M", "K"): 120, ("M", "L"): 200, ("M", "M"): 215, ("M", "N"): 73,
    ("M", "Q"): 114, ("M", "R"): 124, ("N", "A"): 104, ("N", "N"): 215,
    ("N", "R"): 129, ("P", "A"): 188, ("P", "C"): 46, ("P", "D"): 107,
    ("P", "E"): 122, ("P", "F"): 101, ("P", "G"): 173, ("P", "H"): 138,
    ("P", "I"): 120, ("P", "K"): 112, ("P", "L"): 117, ("P", "M"): 128,
    ("P", "N"): 124, ("P", "P"): 215, ("P", "Q"): 139, ("P", "R"): 112,
    ("Q", "A"): 124, ("Q", "C"): 61, ("Q", "D"): 154, ("Q", "N"): 169,
    ("Q", "Q"): 215, ("Q", "R"): 172, ("R", "A"): 103, ("R", "R"): 215,
    ("S", "A"): 116, ("S", "C"): 103, ("S", "D"): 150, ("S", "E"): 135,
    ("S", "F"): 60, ("S", "G"): 159, ("S", "H"): 126, ("S", "I"): 73,
    ("S", "K"): 94, ("S", "L"): 70, ("S", "M"): 80, ("S", "N"): 169,
    ("S", "P"): 141, ("S", "Q"): 147, ("S", "R"): 105, ("S", "S"): 215,
    ("T", "A"): 157, ("T", "C"): 66, ("T", "D"): 130, ("T", "E"): 150,
    ("T", "F"): 112, ("T", "G"): 156, ("T", "H"): 168, ("T", "I"): 126,
    ("T", "K"): 137, ("T", "L"): 123, ("T", "M"): 134, ("T", "N"): 150,
    ("T", "P"): 177, ("T", "Q"): 173, ("T", "R"): 144, ("T", "S"): 157,
    ("T", "T"): 215, ("V", "A"): 151, ("V", "C"): 23, ("V", "D"): 63,
    ("V", "E"): 94, ("V", "F"): 165, ("V", "G"): 106, ("V", "H"): 131,
    ("V", "I"): 186, ("V", "K"): 118, ("V", "L"): 183, ("V", "M"): 194,
    ("V", "N"): 82, ("V", "P"): 147, ("V", "Q"): 119, ("V", "R"): 119,
    ("V", "S"): 91, ("V", "T"): 146, ("V", "V"): 215, ("V", "W"): 127,
    ("V", "Y"): 160, ("W", "A"): 67, ("W", "C"): 0, ("W", "D"): 34,
    ("W", "E"): 63, ("W", "F"): 175, ("W", "G"): 31, ("W", "H"): 100,
    ("W", "I"): 154, ("W", "K"): 105, ("W", "L"): 154, ("W", "M"): 148,
    ("W", "N"): 41, ("W", "P"): 68, ("W", "Q"): 85, ("W", "R"): 114,
    ("W", "S"): 38, ("W", "T"): 87, ("W", "W"): 215, ("Y", "A"): 103,
    ("Y", "C"): 21, ("Y", "D"): 55, ("Y", "E"): 93, ("Y", "F"): 193,
    ("Y", "G"): 68, ("Y", "H"): 132, ("Y", "I"): 182, ("Y", "K"): 130,
    ("Y", "L"): 179, ("Y", "M"): 179, ("Y", "N"): 72, ("Y", "P"): 105,
    ("Y", "Q"): 116, ("Y", "R"): 138, ("Y", "S"): 71, ("Y", "T"): 123,
    ("Y", "W"): 178, ("Y", "Y"): 215
}


# http://www.embl-heidelberg.de/~vogt/matrices/ident.cmp
ident = {
    ("W", "F"): -1, ("L", "R"): -1, ("I", "I"): 6, ("Q", "Q"): 6,
    ("W", "N"): -1, ("V", "I"): -1, ("H", "T"): -1, ("H", "P"): -1,
    ("W", "V"): -1, ("Q", "E"): -1, ("W", "R"): -1, ("Q", "A"): -1,
    ("H", "H"): 6, ("H", "D"): -1, ("L", "N"): -1, ("Y", "M"): -1,
    ("Y", "I"): -1, ("Y", "E"): -1, ("E", "S"): -1, ("Y", "A"): -1,
    ("Y", "Y"): 6, ("T", "C"): -1, ("E", "C"): -1, ("Y", "Q"): -1,
    ("E", "G"): -1, ("V", "A"): -1, ("C", "C"): 6, ("M", "R"): -1,
    ("P", "T"): -1, ("V", "E"): -1, ("P", "P"): 6, ("I", "T"): -1,
    ("K", "S"): -1, ("R", "G"): -1, ("I", "P"): -1, ("R", "C"): -1,
    ("A", "T"): -1, ("K", "K"): 6, ("A", "P"): -1, ("V", "M"): -1,
    ("I", "D"): -1, ("K", "C"): -1, ("K", "G"): -1, ("R", "S"): -1,
    ("F", "Q"): -1, ("F", "A"): -1, ("V", "V"): 6, ("M", "N"): -1,
    ("F", "E"): -1, ("D", "N"): -1, ("F", "I"): -1, ("F", "M"): -1,
    ("M", "S"): -1, ("S", "S"): 6, ("L", "Q"): -1, ("W", "E"): -1,
    ("W", "A"): -1, ("W", "M"): -1, ("H", "S"): -1, ("W", "I"): -1,
    ("S", "C"): -1, ("L", "A"): -1, ("L", "E"): -1, ("W", "Q"): -1,
    ("H", "G"): -1, ("Q", "N"): -1, ("H", "C"): -1, ("L", "M"): -1,
    ("W", "Y"): -1, ("Y", "N"): -1, ("E", "P"): -1, ("Y", "F"): -1,
    ("E", "T"): -1, ("A", "A"): 6, ("I", "N"): -1, ("G", "A"): -1,
    ("Y", "V"): -1, ("E", "D"): -1, ("W", "H"): -1, ("Y", "R"): -1,
    ("M", "Q"): -1, ("P", "S"): -1, ("R", "H"): -1, ("A", "C"): -1,
    ("R", "D"): -1, ("K", "P"): -1, ("L", "D"): -1, ("K", "T"): -1,
    ("V", "N"): -1, ("M", "A"): -1, ("K", "H"): -1, ("V", "R"): -1,
    ("P", "C"): -1, ("M", "E"): -1, ("A", "S"): -1, ("T", "T"): 6,
    ("R", "T"): -1, ("I", "G"): -1, ("R", "P"): -1, ("K", "D"): -1,
    ("I", "C"): -1, ("F", "R"): -1, ("F", "V"): -1, ("L", "C"): -1,
    ("F", "F"): 6, ("D", "A"): -1, ("F", "N"): -1, ("W", "D"): -1,
    ("L", "P"): -1, ("Q", "S"): -1, ("N", "C"): -1, ("N", "G"): -1,
    ("H", "N"): -1, ("W", "T"): -1, ("Q", "G"): -1, ("W", "P"): -1,
    ("Q", "C"): -1, ("N", "S"): -1, ("L", "H"): -1, ("L", "L"): 6,
    ("G", "T"): -1, ("M", "M"): 6, ("G", "P"): -1, ("Y", "K"): -1,
    ("Y", "G"): -1, ("Y", "C"): -1, ("E", "A"): -1, ("E", "E"): 6,
    ("Y", "S"): -1, ("M", "P"): -1, ("V", "C"): -1, ("M", "T"): -1,
    ("V", "G"): -1, ("R", "E"): -1, ("V", "K"): -1, ("K", "Q"): -1,
    ("R", "A"): -1, ("I", "R"): -1, ("N", "A"): -1, ("V", "S"): -1,
    ("M", "D"): -1, ("M", "H"): -1, ("K", "A"): -1, ("R", "Q"): -1,
    ("K", "E"): -1, ("F", "S"): -1, ("I", "K"): -1, ("D", "P"): -1,
    ("D", "T"): -1, ("I", "M"): -1, ("F", "C"): -1, ("W", "L"): -1,
    ("F", "G"): -1, ("F", "K"): -1, ("F", "T"): -1, ("D", "D"): 6,
    ("Q", "T"): -1, ("W", "G"): -1, ("Q", "P"): -1, ("W", "C"): -1,
    ("W", "K"): -1, ("H", "Q"): -1, ("Q", "D"): -1, ("W", "W"): 6,
    ("V", "L"): -1, ("L", "G"): -1, ("W", "S"): -1, ("L", "K"): -1,
    ("N", "P"): -1, ("H", "E"): -1, ("N", "T"): -1, ("H", "A"): -1,
    ("Y", "L"): -1, ("Y", "H"): -1, ("G", "S"): -1, ("Y", "D"): -1,
    ("V", "Q"): -1, ("L", "T"): -1, ("G", "G"): 6, ("G", "C"): -1,
    ("E", "N"): -1, ("Y", "T"): -1, ("Y", "P"): -1, ("R", "N"): -1,
    ("V", "D"): -1, ("K", "R"): -1, ("V", "H"): -1, ("I", "Q"): -1,
    ("V", "P"): -1, ("M", "C"): -1, ("K", "N"): -1, ("V", "T"): -1,
    ("M", "G"): -1, ("T", "S"): -1, ("I", "E"): -1, ("M", "K"): -1,
    ("I", "A"): -1, ("N", "N"): 6, ("R", "R"): 6, ("F", "P"): -1,
    ("L", "I"): -1, ("I", "S"): -1, ("D", "S"): -1, ("L", "S"): -1,
    ("I", "H"): -1, ("F", "D"): -1, ("D", "C"): -1, ("F", "H"): -1,
    ("D", "G"): -1, ("F", "L"): -1
}


# http://www.embl-heidelberg.de/~vogt/matrices/johnson.cmp
johnson = {
    ("W", "F"): 3.4, ("S", "P"): -1.0, ("N", "M"): -3.7, ("Q", "Q"): 9.0,
    ("N", "A"): -1.4, ("N", "E"): -0.7, ("W", "V"): -4.9, ("Q", "E"): 2.4,
    ("L", "H"): -4.2, ("W", "R"): -3.8, ("Q", "A"): -0.6, ("S", "D"): -0.2,
    ("H", "H"): 12.7, ("Q", "M"): -0.6, ("S", "H"): -2.6, ("H", "D"): -0.7,
    ("Q", "I"): -7.0, ("S", "L"): -5.2, ("Y", "M"): -1.3, ("Y", "I"): -2.5,
    ("Y", "E"): -3.7, ("Y", "A"): -4.0, ("G", "F"): -8.6, ("V", "T"): -1.9,
    ("Y", "Y"): 10.5, ("V", "H"): -3.9, ("E", "C"): -6.9, ("Y", "Q"): -5.1,
    ("V", "A"): -0.5, ("C", "C"): 16.1, ("V", "E"): -4.2, ("T", "N"): 0.1,
    ("R", "K"): 3.2, ("P", "P"): 10.3, ("V", "I"): 3.9, ("R", "G"): -2.8,
    ("V", "M"): 0.7, ("T", "F"): -5.0, ("R", "C"): -5.6, ("V", "Q"): -3.6,
    ("K", "K"): 7.6, ("P", "D"): -1.0, ("I", "H"): -5.1, ("M", "F"): -0.6,
    ("I", "D"): -4.8, ("K", "C"): -8.7, ("P", "L"): -2.8, ("K", "G"): -3.5,
    ("P", "H"): -4.3, ("T", "R"): -1.4, ("F", "A"): -3.2, ("F", "E"): -6.4,
    ("S", "S"): 5.8, ("W", "E"): -7.6, ("N", "N"): 8.0, ("W", "M"): -0.9,
    ("Q", "C"): -6.9, ("N", "F"): -3.8, ("S", "C"): -7.7, ("L", "A"): -3.3,
    ("S", "G"): -1.3, ("L", "E"): -5.6, ("W", "Q"): -8.2, ("H", "G"): -3.2,
    ("S", "K"): -1.5, ("Q", "N"): -0.8, ("V", "D"): -5.2, ("H", "C"): -8.2,
    ("Y", "N"): -1.3, ("Y", "F"): 3.4, ("W", "I"): -3.3, ("C", "A"): -3.4,
    ("G", "E"): -2.5, ("G", "A"): -0.5, ("Y", "V"): -1.8, ("E", "D"): 2.4,
    ("W", "H"): -4.0, ("Y", "R"): -2.1, ("N", "I"): -4.7, ("R", "L"): -3.7,
    ("T", "I"): -3.2, ("Q", "L"): -4.4, ("R", "H"): 0.1, ("T", "M"): -3.2,
    ("V", "F"): -1.3, ("R", "D"): -3.4, ("T", "A"): -0.8, ("T", "P"): -2.0,
    ("T", "E"): -0.5, ("V", "N"): -5.7, ("P", "G"): -2.5, ("M", "A"): -1.5,
    ("K", "H"): 0.1, ("V", "R"): -4.9, ("P", "C"): -8.9, ("M", "E"): -2.8,
    ("V", "V"): 7.0, ("T", "T"): 6.8, ("M", "I"): 2.6, ("T", "Q"): -0.4,
    ("I", "G"): -5.5, ("P", "K"): -0.6, ("M", "M"): 11.2, ("K", "D"): -1.5,
    ("I", "C"): -7.7, ("L", "C"): -8.7, ("F", "F"): 10.4, ("D", "A"): -1.6,
    ("S", "R"): -0.6, ("W", "D"): -6.0, ("N", "C"): -7.6, ("N", "G"): -1.4,
    ("W", "T"): -9.3, ("Q", "G"): -2.8, ("S", "F"): -4.8, ("W", "P"): -7.4,
    ("L", "D"): -8.0, ("H", "F"): -1.7, ("Q", "K"): 1.1, ("S", "N"): 1.0,
    ("L", "L"): 7.3, ("Q", "F"): -6.4, ("Y", "K"): -3.7, ("Y", "G"): -5.4,
    ("Y", "C"): -7.7, ("G", "D"): -2.1, ("E", "A"): -0.7, ("Y", "W"): 2.3,
    ("E", "E"): 8.6, ("Y", "S"): -3.4, ("R", "M"): -4.2, ("V", "C"): -4.8,
    ("T", "H"): -3.0, ("R", "I"): -5.4, ("V", "G"): -5.6, ("T", "L"): -4.6,
    ("R", "E"): -0.2, ("V", "K"): -3.7, ("R", "Q"): 2.1, ("R", "A"): -1.6,
    ("T", "D"): -1.8, ("P", "F"): -5.0, ("V", "S"): -4.3, ("K", "I"): -4.7,
    ("M", "D"): -5.9, ("W", "W"): 15.2, ("M", "H"): -2.3, ("P", "N"): -2.4,
    ("I", "F"): 0.5, ("K", "A"): -0.9, ("M", "L"): 4.4, ("K", "E"): 1.1,
    ("N", "K"): 0.1, ("R", "P"): -3.6, ("L", "F"): 1.8, ("F", "C"): -4.4,
    ("W", "G"): -6.3, ("W", "L"): -1.0, ("D", "D"): 8.5, ("N", "H"): 1.7,
    ("S", "Q"): -1.2, ("Q", "P"): -3.6, ("N", "L"): -4.8, ("W", "K"): -5.4,
    ("Q", "D"): -1.1, ("W", "N"): -6.1, ("S", "A"): 0.0, ("L", "G"): -7.2,
    ("W", "S"): -6.2, ("S", "E"): -2.2, ("L", "K"): -3.4, ("H", "E"): -2.3,
    ("S", "I"): -4.7, ("Q", "H"): 1.4, ("H", "A"): -3.1, ("S", "M"): -4.8,
    ("Y", "L"): -2.4, ("Y", "H"): -0.4, ("Y", "D"): -3.8, ("G", "G"): 8.0,
    ("G", "C"): -8.2, ("Y", "T"): -2.7, ("W", "C"): -9.1, ("Y", "P"): -7.0,
    ("T", "K"): -0.2, ("R", "N"): -1.5, ("A", "A"): 6.0, ("W", "A"): -5.8,
    ("T", "C"): -6.0, ("N", "D"): 2.6, ("R", "F"): -6.0, ("T", "G"): -3.8,
    ("V", "L"): 1.8, ("V", "P"): -5.2, ("P", "E"): -1.5, ("M", "C"): -4.4,
    ("I", "I"): 8.1, ("P", "A"): -1.0, ("M", "G"): -5.2, ("T", "S"): 2.0,
    ("I", "E"): -4.8, ("P", "M"): -9.8, ("M", "K"): -1.9, ("K", "F"): -5.6,
    ("I", "A"): -2.2, ("P", "I"): -5.7, ("R", "R"): 10.0, ("L", "I"): 2.6,
    ("F", "D"): -7.0, ("D", "C"): -9.7
}


# http://www.embl-heidelberg.de/~vogt/matrices/levin.cmp
levin = {
    ("W", "F"): 0, ("L", "R"): -1, ("S", "P"): 0, ("I", "I"): 2,
    ("Q", "Q"): 2, ("N", "A"): 0, ("H", "T"): 0, ("N", "E"): 0,
    ("H", "P"): 0, ("W", "V"): 0, ("Q", "E"): 1, ("W", "R"): 0,
    ("Q", "A"): 0, ("S", "D"): 0, ("H", "H"): 2, ("H", "D"): 0,
    ("L", "N"): -1, ("W", "A"): -1, ("Y", "M"): 0, ("Y", "I"): 0,
    ("Y", "E"): -1, ("Y", "A"): -1, ("Y", "Y"): 2, ("Y", "Q"): -1,
    ("E", "G"): 0, ("A", "D"): 0, ("C", "C"): 2, ("M", "R"): -1,
    ("V", "E"): -1, ("T", "N"): 0, ("R", "K"): 1, ("P", "P"): 3,
    ("I", "T"): 0, ("K", "S"): 0, ("R", "G"): 0, ("I", "P"): -1,
    ("C", "G"): 0, ("C", "S"): 0, ("A", "P"): -1, ("I", "D"): -1,
    ("M", "I"): 0, ("K", "G"): 0, ("M", "N"): -1, ("F", "Q"): -1,
    ("I", "V"): 1, ("F", "A"): -1, ("V", "V"): 2, ("F", "E"): -1,
    ("C", "M"): 0, ("F", "I"): 1, ("F", "M"): 0, ("S", "S"): 2,
    ("L", "Q"): -1, ("W", "E"): -1, ("N", "N"): 3, ("V", "A"): 0,
    ("C", "K"): 0, ("W", "M"): 0, ("H", "S"): 0, ("L", "V"): 1,
    ("L", "A"): 0, ("H", "K"): 0, ("S", "G"): 0, ("L", "E"): -1,
    ("W", "Q"): -1, ("H", "G"): 0, ("Q", "N"): 1, ("T", "A"): 0,
    ("L", "M"): 2, ("W", "Y"): 0, ("Y", "N"): -1, ("E", "P"): -1,
    ("Y", "F"): 1, ("W", "I"): 0, ("R", "S"): 0, ("Y", "V"): 0,
    ("E", "D"): 1, ("W", "H"): -1, ("Y", "R"): -1, ("M", "Q"): -1,
    ("A", "G"): 0, ("C", "D"): 0, ("R", "D"): 0, ("C", "H"): 0,
    ("T", "P"): 0, ("K", "T"): 0, ("V", "N"): -1, ("P", "G"): 0,
    ("M", "A"): 0, ("C", "P"): 0, ("V", "R"): -1, ("M", "V"): 0,
    ("M", "E"): -1, ("C", "T"): 0, ("I", "K"): -1, ("R", "T"): 0,
    ("T", "Q"): 0, ("I", "G"): -1, ("R", "P"): 0, ("K", "D"): 0,
    ("F", "R"): -1, ("F", "V"): 0, ("K", "P"): 0, ("L", "C"): 0,
    ("F", "F"): 2, ("F", "N"): -1, ("V", "D"): -1, ("L", "P"): -1,
    ("W", "K"): -1, ("L", "T"): 0, ("I", "N"): -1, ("I", "S"): -1,
    ("H", "R"): 0, ("N", "G"): 0, ("C", "I"): 0, ("H", "N"): 0,
    ("W", "T"): -1, ("Q", "G"): 0, ("W", "P"): -1, ("L", "D"): -1,
    ("L", "H"): -1, ("S", "N"): 0, ("L", "L"): 2, ("M", "M"): 2,
    ("Y", "K"): -1, ("Y", "G"): -1, ("Y", "C"): -1, ("E", "E"): 2,
    ("Y", "S"): -1, ("M", "P"): -1, ("C", "A"): 0, ("M", "T"): 0,
    ("V", "G"): -1, ("C", "E"): 0, ("R", "E"): 0, ("V", "K"): -1,
    ("K", "Q"): 0, ("R", "A"): 0, ("I", "R"): -1, ("T", "D"): 0,
    ("V", "S"): -1, ("C", "Q"): 0, ("M", "D"): -1, ("W", "W"): 2,
    ("M", "H"): -1, ("T", "G"): 0, ("K", "A"): 0, ("R", "Q"): 0,
    ("T", "T"): 2, ("F", "S"): -1, ("D", "P"): 0, ("F", "C"): -1,
    ("W", "L"): 0, ("F", "G"): -1, ("F", "K"): -1, ("D", "D"): 2,
    ("L", "S"): -1, ("W", "G"): -1, ("S", "Q"): 0, ("Q", "P"): 0,
    ("W", "C"): -1, ("N", "D"): 1, ("H", "Q"): 0, ("Q", "D"): 0,
    ("W", "N"): -1, ("S", "A"): 1, ("L", "G"): -1, ("W", "S"): -1,
    ("S", "E"): 0, ("L", "K"): -1, ("N", "P"): 0, ("H", "E"): 0,
    ("H", "A"): 0, ("Y", "L"): 0, ("Y", "H"): 0, ("Y", "D"): -1,
    ("V", "Q"): -1, ("G", "G"): 2, ("Y", "T"): -1, ("R", "N"): 0,
    ("Y", "P"): -1, ("A", "E"): 1, ("C", "V"): 0, ("M", "S"): -1,
    ("A", "A"): 2, ("V", "H"): -1, ("T", "E"): 0, ("C", "N"): 0,
    ("I", "Q"): -1, ("C", "R"): 0, ("V", "P"): -1, ("K", "E"): 0,
    ("K", "N"): 1, ("V", "T"): 0, ("M", "G"): -1, ("T", "S"): 0,
    ("I", "E"): -1, ("M", "K"): -1, ("I", "A"): 0, ("R", "R"): 2,
    ("F", "P"): -1, ("L", "I"): 0, ("W", "D"): -1, ("F", "T"): -1,
    ("K", "K"): 2, ("I", "H"): -1, ("F", "D"): -1, ("F", "H"): -1,
    ("D", "G"): 0, ("F", "L"): 0
}


# http://www.embl-heidelberg.de/~vogt/matrices/mclach.cmp
mclach = {
    ("N", "I"): 1, ("K", "V"): 2, ("S", "P"): 3, ("N", "M"): 2,
    ("L", "V"): 5, ("N", "A"): 3, ("H", "T"): 4, ("N", "E"): 4,
    ("Q", "Y"): 1, ("H", "P"): 3, ("W", "V"): 2, ("H", "L"): 2,
    ("Q", "A"): 3, ("H", "H"): 8, ("N", "Q"): 4, ("Q", "M"): 3,
    ("Q", "I"): 0, ("S", "L"): 2, ("G", "V"): 2, ("Y", "M"): 2,
    ("K", "L"): 2, ("Y", "I"): 3, ("E", "S"): 4, ("K", "A"): 3,
    ("E", "W"): 1, ("G", "F"): 0, ("E", "K"): 4, ("Y", "Y"): 9,
    ("K", "F"): 0, ("E", "G"): 3, ("C", "C"): 9, ("C", "G"): 1,
    ("M", "V"): 4, ("P", "P"): 8, ("A", "L"): 2, ("K", "S"): 3,
    ("R", "G"): 3, ("K", "W"): 1, ("R", "C"): 1, ("I", "L"): 5,
    ("C", "S"): 2, ("C", "W"): 2, ("K", "C"): 0, ("R", "W"): 3,
    ("P", "L"): 1, ("K", "G"): 3, ("R", "S"): 4, ("D", "R"): 1,
    ("D", "V"): 1, ("D", "N"): 5, ("F", "I"): 3, ("F", "M"): 5,
    ("D", "F"): 1, ("S", "S"): 8, ("Q", "V"): 2, ("S", "W"): 3,
    ("Q", "R"): 5, ("N", "N"): 8, ("H", "W"): 3, ("W", "M"): 1,
    ("H", "S"): 3, ("W", "I"): 3, ("T", "F"): 1, ("Q", "F"): 0,
    ("S", "G"): 3, ("H", "G"): 2, ("C", "P"): 0, ("N", "R"): 3,
    ("H", "C"): 3, ("N", "V"): 1, ("E", "P"): 4, ("Y", "F"): 6,
    ("E", "T"): 4, ("G", "Y"): 0, ("E", "H"): 2, ("E", "L"): 1,
    ("Y", "V"): 3, ("G", "M"): 1, ("G", "I"): 1, ("P", "W"): 0,
    ("R", "L"): 2, ("T", "I"): 3, ("A", "G"): 3, ("R", "H"): 5,
    ("T", "M"): 3, ("T", "A"): 3, ("K", "T"): 3, ("P", "G"): 3,
    ("T", "Y"): 1, ("A", "W"): 1, ("C", "T"): 2, ("V", "V"): 8,
    ("R", "T"): 3, ("R", "P"): 3, ("D", "Y"): 1, ("F", "V"): 3,
    ("D", "Q"): 4, ("K", "P"): 3, ("D", "I"): 1, ("D", "M"): 2,
    ("F", "F"): 9, ("D", "A"): 3, ("D", "E"): 5, ("N", "K"): 4,
    ("Q", "W"): 2, ("S", "V"): 2, ("Q", "S"): 4, ("H", "V"): 2,
    ("W", "L"): 3, ("N", "G"): 3, ("Q", "G"): 2, ("S", "F"): 2,
    ("Q", "C"): 0, ("H", "F"): 4, ("N", "S"): 5, ("Q", "K"): 4,
    ("N", "W"): 0, ("L", "L"): 8, ("E", "Y"): 2, ("M", "M"): 8,
    ("E", "Q"): 5, ("E", "I"): 1, ("E", "M"): 1, ("E", "A"): 4,
    ("G", "L"): 1, ("Y", "W"): 6, ("E", "E"): 8, ("R", "M"): 1,
    ("P", "V"): 2, ("A", "F"): 1, ("C", "A"): 1, ("R", "I"): 1,
    ("T", "L"): 3, ("I", "V"): 5, ("C", "I"): 1, ("R", "A"): 2,
    ("C", "Y"): 1, ("C", "M"): 3, ("P", "F"): 1, ("A", "V"): 3,
    ("K", "I"): 1, ("R", "Y"): 2, ("K", "M"): 1, ("K", "H"): 4,
    ("T", "P"): 3, ("M", "L"): 6, ("T", "T"): 8, ("C", "L"): 0,
    ("D", "P"): 3, ("N", "F"): 0, ("K", "Y"): 1, ("D", "T"): 3,
    ("D", "H"): 4, ("D", "L"): 1, ("K", "K"): 8, ("D", "D"): 8,
    ("Q", "T"): 3, ("N", "H"): 4, ("Q", "P"): 3, ("N", "L"): 1,
    ("H", "Y"): 4, ("S", "Y"): 3, ("W", "W"): 9, ("H", "M"): 3,
    ("S", "A"): 4, ("H", "I"): 2, ("Q", "L"): 3, ("N", "P"): 1,
    ("S", "I"): 2, ("Q", "H"): 4, ("N", "T"): 3, ("H", "A"): 3,
    ("S", "M"): 2, ("Y", "L"): 3, ("G", "W"): 1, ("E", "R"): 3,
    ("E", "V"): 2, ("G", "G"): 8, ("T", "V"): 3, ("E", "F"): 0,
    ("C", "F"): 0, ("A", "A"): 8, ("K", "R"): 5, ("A", "M"): 3,
    ("Q", "Q"): 8, ("R", "F"): 1, ("T", "G"): 2, ("A", "I"): 2,
    ("P", "Y"): 0, ("C", "V"): 1, ("I", "I"): 8, ("P", "A"): 4,
    ("T", "S"): 5, ("P", "M"): 1, ("R", "V"): 2, ("T", "W"): 2,
    ("A", "Y"): 1, ("P", "I"): 1, ("R", "R"): 8, ("N", "Y"): 2,
    ("D", "S"): 3, ("D", "W"): 0, ("M", "I"): 5, ("D", "K"): 3,
    ("N", "C"): 1, ("E", "C"): 0, ("D", "C"): 1, ("D", "G"): 3,
    ("F", "L"): 5, ("W", "F"): 6
}


# http://www.embl-heidelberg.de/~vogt/matrices/miyata.cmp
miyata = {
    ("W", "F"): 0.14, ("L", "R"): -1.37, ("S", "P"): 0.69, ("I", "I"): 1.25,
    ("Q", "Q"): 1.25, ("N", "A"): -0.53, ("H", "T"): -0.07, ("N", "E"): 0.4,
    ("H", "P"): -0.9, ("W", "V"): -1.26, ("W", "R"): -1.47, ("Q", "A"): -0.67,
    ("H", "H"): 1.25, ("N", "Q"): 0.26, ("H", "D"): -0.47, ("L", "N"): -2.24,
    ("Y", "M"): 0.32, ("Y", "I"): 0.39, ("Y", "E"): -1.97, ("E", "S"): -0.81,
    ("Y", "A"): -1.93, ("Y", "Y"): 1.25, ("E", "C"): -2.01, ("Y", "Q"): -1.23,
    ("E", "G"): -1.53, ("V", "A"): -0.6, ("C", "C"): 1.25, ("M", "R"): -1.04,
    ("V", "E"): -1.72, ("R", "K"): 0.85, ("P", "P"): 1.25, ("I", "T"): -0.89,
    ("K", "S"): -1.46, ("R", "G"): -2.33, ("I", "P"): -1.37, ("R", "C"): -1.81,
    ("V", "Q"): -0.88, ("K", "K"): 1.25, ("A", "P"): 1.19, ("I", "D"): -2.73,
    ("K", "C"): -2.02, ("M", "I"): 0.96, ("K", "G"): -2.29, ("R", "S"): -1.49,
    ("F", "Q"): -1.56, ("I", "V"): 0.4, ("M", "V"): 0.63, ("F", "A"): -1.98,
    ("V", "V"): 1.25, ("M", "N"): -1.83, ("F", "E"): -2.34, ("D", "N"): 0.6,
    ("F", "I"): 0.64, ("F", "M"): 0.43, ("M", "S"): -1.42, ("S", "S"): 1.25,
    ("L", "Q"): -1.45, ("W", "E"): -2.83, ("W", "A"): -2.98, ("W", "M"): -0.64,
    ("H", "S"): -0.69, ("L", "V"): 0.34, ("S", "C"): -0.59, ("L", "A"): -1.51,
    ("S", "G"): 0.4, ("L", "E"): -2.28, ("W", "Q"): -2.17, ("H", "G"): -1.53,
    ("H", "C"): -1.31, ("W", "Y"): 0.19, ("Y", "N"): -2.17, ("E", "P"): -1.23,
    ("I", "L"): 1.11, ("E", "T"): -0.58, ("W", "I"): -0.47, ("A", "A"): 1.25,
    ("I", "N"): -2.12, ("G", "A"): 0.34, ("Y", "V"): -0.27, ("W", "H"): -1.91,
    ("Y", "R"): -0.77, ("M", "Q"): -1.05, ("R", "H"): 0.43, ("A", "C"): -0.14,
    ("R", "D"): -1.09, ("T", "A"): 0.35, ("T", "P"): 0.38, ("L", "D"): -2.85,
    ("K", "T"): -0.85, ("V", "N"): -1.51, ("M", "A"): -1.17, ("K", "H"): 0.46,
    ("V", "R"): -1.18, ("P", "C"): -0.08, ("M", "E"): -1.88, ("I", "K"): -1.59,
    ("T", "T"): 1.25, ("R", "T"): -0.78, ("I", "G"): -2.35, ("R", "P"): -1.65,
    ("K", "D"): -0.8, ("I", "C"): -0.38, ("F", "R"): -1.22, ("F", "V"): -0.18,
    ("D", "Q"): -0.22, ("K", "P"): -1.69, ("F", "F"): 1.25, ("D", "A"): -1.12,
    ("D", "E"): 0.35, ("F", "N"): -2.45, ("W", "D"): -3.63, ("L", "P"): -1.45,
    ("Q", "S"): -0.4, ("N", "C"): -1.58, ("N", "G"): -0.71, ("H", "N"): -0.04,
    ("W", "T"): -2.25, ("Q", "G"): -1.23, ("W", "P"): -2.92, ("Q", "C"): -1.23,
    ("N", "S"): -0.06, ("L", "H"): -1.34, ("L", "L"): 1.25, ("M", "M"): 1.25,
    ("G", "P"): 0.28, ("Y", "K"): -1.17, ("E", "Q"): 0.41, ("Y", "G"): -2.83,
    ("Y", "C"): -1.13, ("E", "A"): -1.21, ("E", "E"): 1.25, ("Y", "S"): -2.08,
    ("M", "P"): -1.11, ("V", "C"): 0.39, ("M", "T"): -0.61, ("V", "G"): -1.51,
    ("R", "E"): -0.2, ("V", "K"): -1.45, ("K", "Q"): 0.19, ("R", "A"): -1.67,
    ("I", "R"): -1.24, ("V", "S"): -0.9, ("M", "L"): 0.84, ("M", "D"): -2.44,
    ("W", "W"): 1.25, ("M", "H"): -0.94, ("K", "A"): -1.71, ("R", "Q"): 0.12,
    ("K", "E"): 0.11, ("F", "S"): -2.2, ("D", "P"): -1.15, ("D", "T"): -0.8,
    ("F", "C"): -0.99, ("W", "L"): -0.48, ("F", "G"): -2.89, ("F", "K"): -1.6,
    ("F", "T"): -1.35, ("D", "D"): 1.25, ("Q", "T"): 0.13, ("W", "G"): -3.88,
    ("Q", "P"): -0.67, ("W", "C"): -2.09, ("W", "K"): -1.86, ("H", "Q"): 0.93,
    ("L", "C"): -0.4, ("W", "N"): -3.14, ("S", "A"): 0.74, ("L", "G"): -2.42,
    ("W", "S"): -3.13, ("L", "K"): -1.73, ("N", "P"): -0.55, ("H", "E"): 0.29,
    ("N", "T"): -0.15, ("H", "A"): -0.92, ("Y", "L"): 0.31, ("Y", "H"): -1.02,
    ("Y", "D"): -2.7, ("L", "T"): -1.0, ("G", "G"): 1.25, ("G", "C"): -0.97,
    ("Y", "T"): -1.2, ("Y", "P"): -1.87, ("R", "N"): -0.79, ("V", "D"): -2.15,
    ("T", "C"): -0.2, ("V", "H"): -0.86, ("T", "G"): -0.45, ("I", "Q"): -1.32,
    ("V", "P"): -0.54, ("M", "C"): -0.21, ("K", "N"): -0.59, ("V", "T"): -0.17,
    ("M", "G"): -2.09, ("T", "S"): 0.36, ("I", "E"): -2.14, ("M", "K"): -1.38,
    ("I", "A"): -1.44, ("N", "N"): 1.25, ("R", "R"): 1.25, ("F", "P"): -1.92,
    ("I", "S"): -1.7, ("D", "S"): -0.62, ("Y", "F"): 0.77, ("L", "S"): -1.79,
    ("I", "H"): -1.2, ("F", "D"): -3.02, ("D", "C"): -2.23, ("F", "H"): -1.38,
    ("D", "G"): -1.12, ("F", "L"): 0.62
}


# http://www.embl-heidelberg.de/~vogt/matrices/nwsgappep.cmp
nwsgappep = {
    ("W", "F"): 1.3, ("S", "P"): 0.4, ("W", "B"): -0.7, ("N", "N"): 1.5,
    ("N", "A"): 0.2, ("N", "E"): 0.5, ("Z", "Y"): -0.6, ("W", "V"): -0.8,
    ("L", "B"): -0.5, ("W", "R"): 1.4, ("Q", "A"): 0.2, ("S", "D"): 0.2,
    ("H", "H"): 1.5, ("Q", "M"): 0.0, ("S", "H"): -0.2, ("H", "D"): 0.4,
    ("Q", "I"): -0.3, ("S", "L"): -0.4, ("Y", "M"): -0.1, ("Q", "E"): 0.7,
    ("Y", "I"): 0.1, ("Y", "E"): -0.5, ("Y", "A"): -0.3, ("G", "F"): -0.6,
    ("V", "T"): 0.2, ("G", "B"): 0.6, ("Y", "Y"): 1.5, ("N", "L"): -0.4,
    ("E", "C"): -0.6, ("Y", "Q"): -0.6, ("Z", "Z"): 1.1, ("V", "A"): 0.2,
    ("C", "C"): 1.5, ("V", "E"): -0.2, ("T", "N"): 0.2, ("R", "K"): 0.8,
    ("P", "P"): 1.5, ("V", "I"): 1.1, ("T", "B"): 0.2, ("R", "G"): -0.3,
    ("V", "M"): 0.6, ("T", "F"): -0.3, ("R", "C"): -0.3, ("V", "Q"): -0.2,
    ("K", "K"): 1.5, ("M", "B"): -0.3, ("P", "D"): 0.1, ("I", "H"): -0.3,
    ("M", "F"): 0.5, ("I", "D"): -0.2, ("K", "C"): -0.6, ("L", "L"): 1.5,
    ("K", "G"): -0.1, ("P", "H"): 0.2, ("Z", "G"): 0.3, ("W", "M"): -0.3,
    ("Z", "C"): -0.6, ("T", "R"): -0.1, ("Z", "K"): 0.4, ("F", "A"): -0.5,
    ("Z", "W"): -0.8, ("F", "E"): -0.7, ("Z", "S"): 0.0, ("D", "B"): 1.1,
    ("S", "S"): 1.5, ("W", "E"): -1.1, ("W", "A"): -0.8, ("N", "B"): 1.1,
    ("Q", "C"): -0.6, ("Z", "Q"): 1.1, ("N", "F"): -0.5, ("S", "C"): 0.7,
    ("Q", "F"): -0.8, ("S", "G"): 0.6, ("Q", "B"): 0.5, ("W", "Q"): -0.5,
    ("H", "G"): -0.2, ("S", "K"): 0.2, ("L", "I"): 0.8, ("V", "D"): -0.2,
    ("H", "C"): -0.1, ("L", "E"): -0.3, ("Y", "N"): -0.1, ("Y", "F"): 1.4,
    ("W", "I"): -0.5, ("C", "A"): 0.3, ("G", "E"): 0.5, ("G", "A"): 0.7,
    ("Y", "V"): -0.1, ("E", "D"): 1.0, ("W", "H"): -0.1, ("Y", "R"): -0.6,
    ("N", "I"): -0.3, ("R", "L"): -0.4, ("T", "I"): 0.2, ("V", "B"): -0.2,
    ("R", "H"): 0.5, ("T", "M"): 0.0, ("V", "F"): 0.2, ("R", "D"): 0.0,
    ("T", "A"): 0.4, ("T", "P"): 0.3, ("T", "E"): 0.2, ("V", "N"): -0.3,
    ("P", "G"): 0.3, ("M", "A"): 0.0, ("K", "H"): 0.1, ("V", "R"): -0.3,
    ("P", "C"): 0.1, ("M", "E"): -0.2, ("V", "V"): 1.5, ("T", "T"): 1.5,
    ("M", "I"): 0.6, ("T", "Q"): -0.1, ("I", "G"): -0.3, ("P", "K"): 0.1,
    ("M", "M"): 1.5, ("K", "D"): 0.3, ("I", "C"): 0.2, ("Z", "D"): 0.9,
    ("Y", "W"): 1.1, ("Z", "L"): -0.2, ("P", "L"): -0.3, ("Z", "I"): -0.2,
    ("Z", "T"): 0.1, ("L", "C"): -0.8, ("F", "B"): -0.7, ("Z", "P"): 0.2,
    ("F", "F"): 1.5, ("D", "A"): 0.3, ("S", "R"): 0.1, ("W", "D"): -1.1,
    ("R", "R"): 1.5, ("W", "K"): 0.1, ("N", "M"): -0.3, ("N", "C"): -0.3,
    ("N", "G"): 0.4, ("S", "B"): 0.3, ("W", "T"): -0.6, ("Q", "G"): 0.2,
    ("S", "F"): -0.3, ("W", "P"): -0.8, ("L", "D"): -0.5, ("H", "F"): -0.1,
    ("L", "H"): -0.2, ("S", "N"): 0.3, ("H", "B"): 0.4, ("Q", "K"): 0.4,
    ("R", "P"): 0.3, ("Y", "K"): -0.6, ("Y", "B"): -0.3, ("Y", "G"): -0.7,
    ("Y", "C"): 1.0, ("G", "D"): 0.7, ("E", "A"): 0.3, ("T", "S"): 0.3,
    ("E", "E"): 1.5, ("Y", "S"): -0.4, ("R", "M"): 0.2, ("V", "C"): 0.2,
    ("T", "H"): -0.1, ("R", "I"): -0.3, ("V", "S"): -0.1, ("V", "G"): 0.2,
    ("T", "L"): -0.1, ("R", "E"): 0.0, ("V", "K"): -0.2, ("R", "Q"): 0.4,
    ("R", "A"): -0.3, ("Z", "H"): 0.5, ("T", "D"): 0.2, ("P", "F"): -0.7,
    ("L", "A"): -0.1, ("K", "I"): -0.2, ("M", "D"): -0.4, ("P", "B"): 0.1,
    ("W", "W"): 1.5, ("M", "H"): -0.3, ("P", "N"): 0.0, ("I", "F"): 0.7,
    ("K", "A"): 0.0, ("M", "L"): 1.3, ("I", "B"): -0.2, ("K", "E"): 0.3,
    ("Z", "E"): 1.1, ("Q", "N"): 0.4, ("Z", "A"): 0.2, ("Z", "M"): -0.1,
    ("L", "F"): 1.2, ("F", "C"): -0.1, ("W", "G"): -1.0, ("W", "L"): 0.5,
    ("C", "B"): -0.4, ("B", "A"): 0.2, ("D", "D"): 1.5, ("N", "H"): 0.5,
    ("S", "Q"): -0.1, ("Q", "P"): 0.3, ("W", "C"): -1.2, ("N", "D"): 0.7,
    ("Q", "D"): 0.7, ("W", "N"): -0.3, ("S", "A"): 0.4, ("L", "G"): -0.5,
    ("W", "S"): 0.3, ("S", "E"): 0.2, ("L", "K"): -0.3, ("H", "E"): 0.4,
    ("S", "I"): -0.1, ("Q", "H"): 0.7, ("H", "A"): -0.1, ("S", "M"): -0.3,
    ("Y", "L"): 0.3, ("Y", "H"): 0.3, ("Y", "D"): -0.5, ("G", "G"): 1.5,
    ("G", "C"): 0.2, ("Y", "T"): -0.3, ("E", "B"): 0.7, ("Y", "P"): -0.8,
    ("T", "K"): 0.2, ("R", "N"): 0.1, ("A", "A"): 1.5, ("N", "K"): 0.4,
    ("T", "C"): 0.2, ("V", "H"): -0.3, ("Q", "Q"): 1.5, ("R", "F"): -0.5,
    ("T", "G"): 0.4, ("V", "L"): 0.8, ("R", "B"): 0.1, ("V", "P"): 0.1,
    ("P", "E"): 0.1, ("M", "C"): -0.6, ("I", "I"): 1.5, ("P", "A"): 0.5,
    ("M", "G"): -0.3, ("K", "B"): 0.4, ("I", "E"): -0.2, ("P", "M"): -0.2,
    ("M", "K"): 0.2, ("K", "F"): -0.7, ("I", "A"): 0.0, ("P", "I"): -0.2,
    ("Q", "L"): -0.1, ("Z", "F"): -0.7, ("Z", "B"): 0.6, ("Z", "N"): 0.4,
    ("Z", "V"): -0.2, ("F", "D"): -1.0, ("Z", "R"): 0.2, ("D", "C"): -0.5,
    ("B", "B"): 1.1
}


# http://www.embl-heidelberg.de/~vogt/matrices/pam120.cmp
pam120 = {
    ("W", "F"): -1, ("L", "R"): -4, ("S", "P"): 1, ("V", "T"): 0,
    ("Q", "Q"): 6, ("N", "A"): -1, ("Z", "Y"): -5, ("W", "R"): 1,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 7, ("S", "H"): -2,
    ("H", "D"): 0, ("L", "N"): -4, ("W", "A"): -7, ("Y", "M"): -4,
    ("G", "R"): -4, ("Y", "I"): -2, ("Y", "E"): -5, ("B", "Y"): -3,
    ("Y", "A"): -4, ("V", "D"): -3, ("B", "S"): 0, ("Y", "Y"): 8,
    ("G", "N"): 0, ("E", "C"): -7, ("Y", "Q"): -5, ("Z", "Z"): 4,
    ("V", "A"): 0, ("C", "C"): 9, ("M", "R"): -1, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 6, ("V", "I"): 3, ("V", "S"): -2,
    ("Z", "P"): -1, ("V", "M"): 1, ("T", "F"): -4, ("V", "Q"): -3,
    ("K", "K"): 5, ("P", "D"): -3, ("I", "H"): -4, ("I", "D"): -3,
    ("T", "R"): -2, ("P", "L"): -3, ("K", "G"): -3, ("M", "N"): -3,
    ("P", "H"): -1, ("F", "Q"): -6, ("Z", "G"): -2, ("X", "L"): -2,
    ("T", "M"): -1, ("Z", "C"): -7, ("X", "H"): -2, ("D", "R"): -3,
    ("B", "W"): -6, ("X", "D"): -2, ("Z", "K"): -1, ("F", "A"): -4,
    ("Z", "W"): -7, ("F", "E"): -7, ("D", "N"): 2, ("B", "K"): 0,
    ("X", "X"): -2, ("F", "I"): 0, ("B", "G"): 0, ("X", "T"): -1,
    ("F", "M"): -1, ("B", "C"): -6, ("Z", "I"): -3, ("Z", "V"): -3,
    ("S", "S"): 3, ("L", "Q"): -2, ("W", "E"): -8, ("Q", "R"): 1,
    ("N", "N"): 4, ("W", "M"): -6, ("Q", "C"): -7, ("W", "I"): -6,
    ("S", "C"): 0, ("L", "A"): -3, ("S", "G"): 1, ("L", "E"): -4,
    ("W", "Q"): -6, ("H", "G"): -4, ("S", "K"): -1, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -4, ("Y", "N"): -2, ("G", "Q"): -3,
    ("Y", "F"): 4, ("C", "A"): -3, ("V", "L"): 1, ("G", "E"): -1,
    ("G", "A"): 1, ("K", "R"): 2, ("E", "D"): 3, ("Y", "R"): -5,
    ("M", "Q"): -1, ("T", "I"): 0, ("C", "D"): -7, ("V", "F"): -3,
    ("T", "A"): 1, ("T", "P"): -1, ("B", "P"): -2, ("T", "E"): -2,
    ("V", "N"): -3, ("P", "G"): -2, ("M", "A"): -2, ("K", "H"): -2,
    ("V", "R"): -3, ("P", "C"): -4, ("M", "E"): -3, ("K", "L"): -4,
    ("V", "V"): 5, ("M", "I"): 1, ("T", "Q"): -2, ("I", "G"): -4,
    ("P", "K"): -2, ("M", "M"): 8, ("K", "D"): -1, ("I", "C"): -3,
    ("Z", "D"): 3, ("F", "R"): -5, ("X", "K"): -2, ("Q", "D"): 1,
    ("X", "G"): -2, ("Z", "L"): -3, ("X", "C"): -4, ("Z", "H"): 1,
    ("B", "L"): -4, ("B", "H"): 1, ("F", "F"): 8, ("X", "W"): -5,
    ("B", "D"): 4, ("D", "A"): 0, ("S", "L"): -4, ("X", "S"): -1,
    ("F", "N"): -4, ("S", "R"): -1, ("W", "D"): -8, ("V", "Y"): -3,
    ("W", "L"): -3, ("H", "R"): 1, ("W", "H"): -3, ("H", "N"): 2,
    ("W", "T"): -6, ("T", "T"): 4, ("S", "F"): -3, ("W", "P"): -7,
    ("L", "D"): -5, ("B", "I"): -3, ("L", "H"): -3, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 5, ("Y", "K"): -5, ("E", "Q"): 2,
    ("Y", "G"): -6, ("Z", "S"): -1, ("Y", "C"): -1, ("G", "D"): 0,
    ("B", "V"): -3, ("E", "A"): 0, ("Y", "W"): -2, ("E", "E"): 5,
    ("Y", "S"): -3, ("C", "N"): -5, ("V", "C"): -3, ("T", "H"): -3,
    ("P", "R"): -1, ("V", "G"): -2, ("T", "L"): -3, ("V", "K"): -4,
    ("K", "Q"): 0, ("R", "A"): -3, ("I", "R"): -2, ("T", "D"): -1,
    ("P", "F"): -5, ("I", "N"): -2, ("K", "I"): -3, ("M", "D"): -4,
    ("V", "W"): -8, ("W", "W"): 12, ("M", "H"): -4, ("P", "N"): -2,
    ("K", "A"): -2, ("M", "L"): 3, ("K", "E"): -1, ("Z", "E"): 4,
    ("X", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -2, ("X", "F"): -3,
    ("K", "C"): -7, ("B", "Q"): 0, ("X", "B"): -1, ("B", "M"): -4,
    ("F", "C"): -6, ("Z", "Q"): 4, ("X", "Z"): -1, ("F", "G"): -5,
    ("B", "E"): 3, ("X", "V"): -1, ("F", "K"): -7, ("B", "A"): 0,
    ("X", "R"): -2, ("D", "D"): 5, ("W", "G"): -8, ("Z", "F"): -6,
    ("S", "Q"): -2, ("W", "C"): -8, ("W", "K"): -5, ("H", "Q"): 3,
    ("L", "C"): -7, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -5,
    ("W", "S"): -2, ("S", "E"): -1, ("H", "E"): -1, ("S", "I"): -2,
    ("H", "A"): -3, ("S", "M"): -2, ("Y", "L"): -2, ("Y", "H"): -1,
    ("Y", "D"): -5, ("E", "R"): -3, ("X", "P"): -2, ("G", "G"): 5,
    ("G", "C"): -4, ("E", "N"): 1, ("Y", "T"): -3, ("Y", "P"): -6,
    ("T", "K"): -1, ("A", "A"): 3, ("P", "Q"): 0, ("T", "C"): -3,
    ("V", "H"): -3, ("T", "G"): -1, ("I", "Q"): -3, ("Z", "T"): -2,
    ("C", "R"): -4, ("V", "P"): -2, ("P", "E"): -2, ("M", "C"): -6,
    ("K", "N"): 1, ("I", "I"): 6, ("P", "A"): 1, ("M", "G"): -4,
    ("T", "S"): 2, ("I", "E"): -3, ("P", "M"): -3, ("M", "K"): 0,
    ("I", "A"): -1, ("P", "I"): -3, ("R", "R"): 6, ("X", "M"): -2,
    ("L", "I"): 1, ("X", "I"): -1, ("Z", "B"): 2, ("X", "E"): -1,
    ("Z", "N"): 0, ("X", "A"): -1, ("B", "R"): -2, ("B", "N"): 3,
    ("F", "D"): -7, ("X", "Y"): -3, ("Z", "R"): -1, ("F", "H"): -3,
    ("B", "F"): -5, ("F", "L"): 0, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/pam180.cmp
pam180 = {
    ("W", "F"): 0, ("L", "R"): -4, ("S", "P"): 1, ("V", "T"): 0,
    ("Q", "Q"): 6, ("N", "A"): 0, ("Z", "Y"): -6, ("W", "R"): 2,
    ("Q", "A"): -1, ("S", "D"): 0, ("H", "H"): 8, ("S", "H"): -2,
    ("H", "D"): 0, ("L", "N"): -4, ("W", "A"): -8, ("Y", "M"): -4,
    ("G", "R"): -4, ("Y", "I"): -2, ("Y", "E"): -6, ("B", "Y"): -4,
    ("Y", "A"): -5, ("V", "D"): -3, ("B", "S"): 1, ("Y", "Y"): 11,
    ("G", "N"): 0, ("E", "C"): -7, ("Y", "Q"): -6, ("Z", "Z"): 5,
    ("V", "A"): 0, ("C", "C"): 13, ("M", "R"): -1, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 8, ("V", "I"): 5, ("V", "S"): -2,
    ("Z", "P"): -1, ("V", "M"): 2, ("T", "F"): -4, ("V", "Q"): -3,
    ("K", "K"): 6, ("P", "D"): -2, ("I", "H"): -4, ("I", "D"): -3,
    ("T", "R"): -2, ("P", "L"): -4, ("K", "G"): -3, ("M", "N"): -3,
    ("P", "H"): -1, ("F", "Q"): -6, ("Z", "G"): -1, ("X", "L"): -2,
    ("T", "M"): -1, ("Z", "C"): -7, ("X", "H"): -1, ("D", "R"): -3,
    ("B", "W"): -7, ("X", "D"): -1, ("Z", "K"): 0, ("F", "A"): -5,
    ("Z", "W"): -8, ("F", "E"): -7, ("D", "N"): 3, ("B", "K"): 0,
    ("X", "X"): -1, ("F", "I"): 1, ("B", "G"): 0, ("X", "T"): -1,
    ("F", "M"): 0, ("B", "C"): -6, ("Z", "I"): -3, ("Z", "V"): -3,
    ("S", "S"): 3, ("L", "Q"): -2, ("W", "E"): -9, ("Q", "R"): 1,
    ("N", "N"): 4, ("W", "M"): -6, ("Q", "C"): -7, ("W", "I"): -7,
    ("S", "C"): 0, ("L", "A"): -3, ("S", "G"): 1, ("L", "E"): -5,
    ("W", "Q"): -6, ("H", "G"): -3, ("S", "K"): -1, ("Q", "N"): 0,
    ("N", "R"): -1, ("H", "C"): -4, ("Y", "N"): -2, ("G", "Q"): -2,
    ("Y", "F"): 7, ("C", "A"): -3, ("V", "L"): 2, ("G", "E"): 0,
    ("G", "A"): 1, ("K", "R"): 4, ("E", "D"): 4, ("Y", "R"): -6,
    ("M", "Q"): -1, ("T", "I"): 0, ("C", "D"): -7, ("V", "F"): -2,
    ("T", "A"): 2, ("T", "P"): 0, ("B", "P"): -2, ("T", "E"): -1,
    ("V", "N"): -3, ("P", "G"): -1, ("M", "A"): -2, ("K", "H"): -1,
    ("V", "R"): -4, ("P", "C"): -4, ("M", "E"): -3, ("K", "L"): -4,
    ("V", "V"): 6, ("M", "I"): 2, ("T", "Q"): -2, ("I", "G"): -4,
    ("P", "K"): -2, ("M", "M"): 9, ("K", "D"): 0, ("I", "C"): -3,
    ("Z", "D"): 3, ("F", "R"): -6, ("X", "K"): -1, ("Q", "D"): 2,
    ("X", "G"): -2, ("Z", "L"): -3, ("X", "C"): -4, ("Z", "H"): 2,
    ("B", "L"): -5, ("B", "H"): 1, ("F", "F"): 10, ("X", "W"): -6,
    ("B", "D"): 4, ("D", "A"): 0, ("S", "L"): -4, ("X", "S"): 0,
    ("F", "N"): -5, ("S", "R"): -1, ("W", "D"): -9, ("V", "Y"): -4,
    ("W", "L"): -3, ("H", "R"): 2, ("W", "H"): -4, ("H", "N"): 2,
    ("W", "T"): -7, ("T", "T"): 4, ("S", "F"): -4, ("W", "P"): -7,
    ("L", "D"): -6, ("B", "I"): -3, ("L", "H"): -3, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 7, ("Y", "K"): -6, ("E", "Q"): 3,
    ("Y", "G"): -7, ("Z", "S"): -1, ("Y", "C"): 0, ("G", "D"): 0,
    ("B", "V"): -3, ("E", "A"): 0, ("Y", "W"): -1, ("E", "E"): 5,
    ("Y", "S"): -4, ("C", "N"): -5, ("V", "C"): -3, ("T", "H"): -2,
    ("P", "R"): -1, ("V", "G"): -2, ("T", "L"): -3, ("V", "K"): -4,
    ("K", "Q"): 0, ("R", "A"): -3, ("I", "R"): -3, ("T", "D"): -1,
    ("P", "F"): -6, ("I", "N"): -3, ("K", "I"): -3, ("M", "D"): -4,
    ("V", "W"): -8, ("W", "W"): 18, ("M", "H"): -3, ("P", "N"): -1,
    ("K", "A"): -2, ("M", "L"): 4, ("K", "E"): -1, ("Z", "E"): 5,
    ("X", "N"): -1, ("Z", "A"): 0, ("Z", "M"): -2, ("X", "F"): -3,
    ("K", "C"): -7, ("B", "Q"): 1, ("X", "B"): -1, ("B", "M"): -3,
    ("F", "C"): -6, ("Z", "Q"): 5, ("X", "Z"): -1, ("F", "G"): -6,
    ("B", "E"): 3, ("X", "V"): -1, ("F", "K"): -7, ("B", "A"): 0,
    ("X", "R"): -2, ("D", "D"): 5, ("W", "G"): -9, ("Z", "F"): -7,
    ("S", "Q"): -1, ("W", "C"): -10, ("W", "K"): -5, ("H", "Q"): 4,
    ("L", "C"): -8, ("W", "N"): -5, ("S", "A"): 1, ("L", "G"): -6,
    ("W", "S"): -3, ("S", "E"): -1, ("H", "E"): 0, ("S", "I"): -2,
    ("H", "A"): -2, ("S", "M"): -2, ("Y", "L"): -2, ("Y", "H"): 0,
    ("Y", "D"): -6, ("E", "R"): -2, ("X", "P"): -1, ("G", "G"): 6,
    ("G", "C"): -5, ("E", "N"): 2, ("Y", "T"): -4, ("Y", "P"): -7,
    ("T", "K"): 0, ("A", "A"): 3, ("P", "Q"): 0, ("T", "C"): -3,
    ("V", "H"): -3, ("T", "G"): -1, ("I", "Q"): -3, ("Z", "T"): -1,
    ("C", "R"): -5, ("V", "P"): -2, ("P", "E"): -1, ("M", "C"): -7,
    ("K", "N"): 1, ("I", "I"): 6, ("P", "A"): 1, ("M", "G"): -4,
    ("T", "S"): 2, ("I", "E"): -3, ("P", "M"): -3, ("M", "K"): 1,
    ("I", "A"): -1, ("P", "I"): -3, ("R", "R"): 8, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 3, ("X", "E"): -1,
    ("Z", "N"): 1, ("X", "A"): -1, ("B", "R"): -2, ("B", "N"): 3,
    ("F", "D"): -8, ("X", "Y"): -3, ("Z", "R"): 0, ("F", "H"): -3,
    ("B", "F"): -6, ("F", "L"): 1, ("X", "Q"): -1, ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/pam250.cmp
pam250 = {
    ("W", "F"): 0, ("L", "R"): -3, ("S", "P"): 1, ("V", "T"): 0,
    ("Q", "Q"): 4, ("N", "A"): 0, ("Z", "Y"): -4, ("W", "R"): 2,
    ("Q", "A"): 0, ("S", "D"): 0, ("H", "H"): 6, ("S", "H"): -1,
    ("H", "D"): 1, ("L", "N"): -3, ("W", "A"): -6, ("Y", "M"): -2,
    ("G", "R"): -3, ("Y", "I"): -1, ("Y", "E"): -4, ("B", "Y"): -3,
    ("Y", "A"): -3, ("V", "D"): -2, ("B", "S"): 0, ("Y", "Y"): 10,
    ("G", "N"): 0, ("E", "C"): -5, ("Y", "Q"): -4, ("Z", "Z"): 3,
    ("V", "A"): 0, ("C", "C"): 12, ("M", "R"): 0, ("V", "E"): -2,
    ("T", "N"): 0, ("P", "P"): 6, ("V", "I"): 4, ("V", "S"): -1,
    ("Z", "P"): 0, ("V", "M"): 2, ("T", "F"): -3, ("V", "Q"): -2,
    ("K", "K"): 5, ("P", "D"): -1, ("I", "H"): -2, ("I", "D"): -2,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): 0, ("F", "Q"): -5, ("Z", "G"): 0, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -5, ("X", "H"): -1, ("D", "R"): -1,
    ("B", "W"): -5, ("X", "D"): -1, ("Z", "K"): 0, ("F", "A"): -3,
    ("Z", "W"): -6, ("F", "E"): -5, ("D", "N"): 2, ("B", "K"): 1,
    ("X", "X"): -1, ("F", "I"): 1, ("B", "G"): 0, ("X", "T"): 0,
    ("F", "M"): 0, ("B", "C"): -4, ("Z", "I"): -2, ("Z", "V"): -2,
    ("S", "S"): 2, ("L", "Q"): -2, ("W", "E"): -7, ("Q", "R"): 1,
    ("N", "N"): 2, ("W", "M"): -4, ("Q", "C"): -5, ("W", "I"): -5,
    ("S", "C"): 0, ("L", "A"): -2, ("S", "G"): 1, ("L", "E"): -3,
    ("W", "Q"): -5, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 1,
    ("N", "R"): 0, ("H", "C"): -3, ("Y", "N"): -2, ("G", "Q"): -1,
    ("Y", "F"): 7, ("C", "A"): -2, ("V", "L"): 2, ("G", "E"): 0,
    ("G", "A"): 1, ("K", "R"): 3, ("E", "D"): 3, ("Y", "R"): -4,
    ("M", "Q"): -1, ("T", "I"): 0, ("C", "D"): -5, ("V", "F"): -1,
    ("T", "A"): 1, ("T", "P"): 0, ("B", "P"): -1, ("T", "E"): 0,
    ("V", "N"): -2, ("P", "G"): 0, ("M", "A"): -1, ("K", "H"): 0,
    ("V", "R"): -2, ("P", "C"): -3, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 4, ("M", "I"): 2, ("T", "Q"): -1, ("I", "G"): -3,
    ("P", "K"): -1, ("M", "M"): 6, ("K", "D"): 0, ("I", "C"): -2,
    ("Z", "D"): 3, ("F", "R"): -4, ("X", "K"): -1, ("Q", "D"): 2,
    ("X", "G"): -1, ("Z", "L"): -3, ("X", "C"): -3, ("Z", "H"): 2,
    ("B", "L"): -3, ("B", "H"): 1, ("F", "F"): 9, ("X", "W"): -4,
    ("B", "D"): 3, ("D", "A"): 0, ("S", "L"): -3, ("X", "S"): 0,
    ("F", "N"): -3, ("S", "R"): 0, ("W", "D"): -7, ("V", "Y"): -2,
    ("W", "L"): -2, ("H", "R"): 2, ("W", "H"): -3, ("H", "N"): 2,
    ("W", "T"): -5, ("T", "T"): 3, ("S", "F"): -3, ("W", "P"): -6,
    ("L", "D"): -4, ("B", "I"): -2, ("L", "H"): -2, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 6, ("Y", "K"): -4, ("E", "Q"): 2,
    ("Y", "G"): -5, ("Z", "S"): 0, ("Y", "C"): 0, ("G", "D"): 1,
    ("B", "V"): -2, ("E", "A"): 0, ("Y", "W"): 0, ("E", "E"): 4,
    ("Y", "S"): -3, ("C", "N"): -4, ("V", "C"): -2, ("T", "H"): -1,
    ("P", "R"): 0, ("V", "G"): -1, ("T", "L"): -2, ("V", "K"): -2,
    ("K", "Q"): 1, ("R", "A"): -2, ("I", "R"): -2, ("T", "D"): 0,
    ("P", "F"): -5, ("I", "N"): -2, ("K", "I"): -2, ("M", "D"): -3,
    ("V", "W"): -6, ("W", "W"): 17, ("M", "H"): -2, ("P", "N"): 0,
    ("K", "A"): -1, ("M", "L"): 4, ("K", "E"): 0, ("Z", "E"): 3,
    ("X", "N"): 0, ("Z", "A"): 0, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -5, ("B", "Q"): 1, ("X", "B"): -1, ("B", "M"): -2,
    ("F", "C"): -4, ("Z", "Q"): 3, ("X", "Z"): -1, ("F", "G"): -5,
    ("B", "E"): 3, ("X", "V"): -1, ("F", "K"): -5, ("B", "A"): 0,
    ("X", "R"): -1, ("D", "D"): 4, ("W", "G"): -7, ("Z", "F"): -5,
    ("S", "Q"): -1, ("W", "C"): -8, ("W", "K"): -3, ("H", "Q"): 3,
    ("L", "C"): -6, ("W", "N"): -4, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -2, ("S", "E"): 0, ("H", "E"): 1, ("S", "I"): -1,
    ("H", "A"): -1, ("S", "M"): -2, ("Y", "L"): -1, ("Y", "H"): 0,
    ("Y", "D"): -4, ("E", "R"): -1, ("X", "P"): -1, ("G", "G"): 5,
    ("G", "C"): -3, ("E", "N"): 1, ("Y", "T"): -3, ("Y", "P"): -5,
    ("T", "K"): 0, ("A", "A"): 2, ("P", "Q"): 0, ("T", "C"): -2,
    ("V", "H"): -2, ("T", "G"): 0, ("I", "Q"): -2, ("Z", "T"): -1,
    ("C", "R"): -4, ("V", "P"): -1, ("P", "E"): -1, ("M", "C"): -5,
    ("K", "N"): 1, ("I", "I"): 5, ("P", "A"): 1, ("M", "G"): -3,
    ("T", "S"): 1, ("I", "E"): -2, ("P", "M"): -2, ("M", "K"): 0,
    ("I", "A"): -1, ("P", "I"): -2, ("R", "R"): 6, ("X", "M"): -1,
    ("L", "I"): 2, ("X", "I"): -1, ("Z", "B"): 2, ("X", "E"): -1,
    ("Z", "N"): 1, ("X", "A"): 0, ("B", "R"): -1, ("B", "N"): 2,
    ("F", "D"): -6, ("X", "Y"): -2, ("Z", "R"): 0, ("F", "H"): -2,
    ("B", "F"): -4, ("F", "L"): 2, ("X", "Q"): -1, ("B", "B"): 3
}


# http://www.embl-heidelberg.de/~vogt/matrices/pam30.cmp
pam30 = {
    ("W", "F"): -4, ("L", "R"): -8, ("S", "P"): -2, ("V", "T"): -3,
    ("Q", "Q"): 8, ("N", "A"): -4, ("Z", "Y"): -9, ("W", "R"): -2,
    ("Q", "A"): -4, ("S", "D"): -4, ("H", "H"): 9, ("S", "H"): -6,
    ("H", "D"): -4, ("L", "N"): -7, ("W", "A"): -13, ("Y", "M"): -11,
    ("G", "R"): -9, ("Y", "I"): -6, ("Y", "E"): -8, ("B", "Y"): -6,
    ("Y", "A"): -8, ("V", "D"): -8, ("B", "S"): -1, ("Y", "Y"): 10,
    ("G", "N"): -3, ("E", "C"): -14, ("Y", "Q"): -12, ("Z", "Z"): 6,
    ("V", "A"): -2, ("C", "C"): 10, ("M", "R"): -4, ("V", "E"): -6,
    ("T", "N"): -2, ("P", "P"): 8, ("V", "I"): 2, ("V", "S"): -6,
    ("Z", "P"): -4, ("V", "M"): -1, ("T", "F"): -9, ("V", "Q"): -7,
    ("K", "K"): 7, ("P", "D"): -8, ("I", "H"): -9, ("I", "D"): -7,
    ("T", "R"): -6, ("P", "L"): -7, ("K", "G"): -7, ("M", "N"): -9,
    ("P", "H"): -4, ("F", "Q"): -13, ("Z", "G"): -5, ("X", "L"): -6,
    ("T", "M"): -4, ("Z", "C"): -14, ("X", "H"): -5, ("D", "R"): -10,
    ("B", "W"): -10, ("X", "D"): -5, ("Z", "K"): -4, ("F", "A"): -8,
    ("Z", "W"): -14, ("F", "E"): -14, ("D", "N"): 2, ("B", "K"): -2,
    ("X", "X"): -5, ("F", "I"): -2, ("B", "G"): -3, ("X", "T"): -4,
    ("F", "M"): -4, ("B", "C"): -12, ("Z", "I"): -6, ("Z", "V"): -6,
    ("S", "S"): 6, ("L", "Q"): -5, ("W", "E"): -17, ("Q", "R"): -2,
    ("N", "N"): 8, ("W", "M"): -13, ("Q", "C"): -14, ("W", "I"): -14,
    ("S", "C"): -3, ("L", "A"): -6, ("S", "G"): -2, ("L", "E"): -9,
    ("W", "Q"): -13, ("H", "G"): -9, ("S", "K"): -4, ("Q", "N"): -3,
    ("N", "R"): -6, ("H", "C"): -7, ("Y", "N"): -4, ("G", "Q"): -7,
    ("Y", "F"): 2, ("C", "A"): -6, ("V", "L"): -2, ("G", "E"): -4,
    ("G", "A"): -2, ("K", "R"): 0, ("E", "D"): 2, ("Y", "R"): -10,
    ("M", "Q"): -4, ("T", "I"): -2, ("C", "D"): -14, ("V", "F"): -8,
    ("T", "A"): -1, ("T", "P"): -4, ("B", "P"): -7, ("T", "E"): -6,
    ("V", "N"): -8, ("P", "G"): -6, ("M", "A"): -5, ("K", "H"): -6,
    ("V", "R"): -8, ("P", "C"): -8, ("M", "E"): -7, ("K", "L"): -8,
    ("V", "V"): 7, ("M", "I"): -1, ("T", "Q"): -5, ("I", "G"): -11,
    ("P", "K"): -6, ("M", "M"): 11, ("K", "D"): -4, ("I", "C"): -6,
    ("Z", "D"): 1, ("F", "R"): -9, ("X", "K"): -5, ("Q", "D"): -2,
    ("X", "G"): -5, ("Z", "L"): -7, ("X", "C"): -9, ("Z", "H"): -1,
    ("B", "L"): -9, ("B", "H"): -1, ("F", "F"): 9, ("X", "W"): -11,
    ("B", "D"): 6, ("D", "A"): -3, ("S", "L"): -8, ("X", "S"): -3,
    ("F", "N"): -9, ("S", "R"): -3, ("W", "D"): -15, ("V", "Y"): -7,
    ("W", "L"): -6, ("H", "R"): -2, ("W", "H"): -7, ("H", "N"): 0,
    ("W", "T"): -13, ("T", "T"): 7, ("S", "F"): -6, ("W", "P"): -14,
    ("L", "D"): -12, ("B", "I"): -6, ("L", "H"): -6, ("S", "N"): 0,
    ("B", "T"): -3, ("L", "L"): 7, ("Y", "K"): -9, ("E", "Q"): 1,
    ("Y", "G"): -14, ("Z", "S"): -5, ("Y", "C"): -4, ("G", "D"): -3,
    ("B", "V"): -8, ("E", "A"): -2, ("Y", "W"): -5, ("E", "E"): 8,
    ("Y", "S"): -7, ("C", "N"): -11, ("V", "C"): -6, ("T", "H"): -7,
    ("P", "R"): -4, ("V", "G"): -5, ("T", "L"): -7, ("V", "K"): -9,
    ("K", "Q"): -3, ("R", "A"): -7, ("I", "R"): -5, ("T", "D"): -5,
    ("P", "F"): -10, ("I", "N"): -5, ("K", "I"): -6, ("M", "D"): -11,
    ("V", "W"): -15, ("W", "W"): 13, ("M", "H"): -10, ("P", "N"): -6,
    ("K", "A"): -7, ("M", "L"): 1, ("K", "E"): -4, ("Z", "E"): 6,
    ("X", "N"): -3, ("Z", "A"): -3, ("Z", "M"): -5, ("X", "F"): -8,
    ("K", "C"): -14, ("B", "Q"): -3, ("X", "B"): -5, ("B", "M"): -10,
    ("F", "C"): -13, ("Z", "Q"): 6, ("X", "Z"): -5, ("F", "G"): -9,
    ("B", "E"): 1, ("X", "V"): -5, ("F", "K"): -14, ("B", "A"): -3,
    ("X", "R"): -6, ("D", "D"): 8, ("W", "G"): -15, ("Z", "F"): -13,
    ("S", "Q"): -5, ("W", "C"): -15, ("W", "K"): -12, ("H", "Q"): 1,
    ("L", "C"): -15, ("W", "N"): -8, ("S", "A"): 0, ("L", "G"): -10,
    ("W", "S"): -5, ("S", "E"): -4, ("H", "E"): -5, ("S", "I"): -7,
    ("H", "A"): -7, ("S", "M"): -5, ("Y", "L"): -7, ("Y", "H"): -3,
    ("Y", "D"): -11, ("E", "R"): -9, ("X", "P"): -5, ("G", "G"): 6,
    ("G", "C"): -9, ("E", "N"): -2, ("Y", "T"): -6, ("Y", "P"): -13,
    ("T", "K"): -3, ("A", "A"): 6, ("P", "Q"): -3, ("T", "C"): -8,
    ("V", "H"): -6, ("T", "G"): -6, ("I", "Q"): -8, ("Z", "T"): -6,
    ("C", "R"): -8, ("V", "P"): -6, ("P", "E"): -5, ("M", "C"): -13,
    ("K", "N"): -1, ("I", "I"): 8, ("P", "A"): -2, ("M", "G"): -8,
    ("T", "S"): 0, ("I", "E"): -5, ("P", "M"): -8, ("M", "K"): -2,
    ("I", "A"): -5, ("P", "I"): -8, ("R", "R"): 8, ("X", "M"): -5,
    ("L", "I"): -1, ("X", "I"): -5, ("Z", "B"): 0, ("X", "E"): -5,
    ("Z", "N"): -3, ("X", "A"): -3, ("B", "R"): -7, ("B", "N"): 6,
    ("F", "D"): -15, ("X", "Y"): -7, ("Z", "R"): -4, ("F", "H"): -6,
    ("B", "F"): -10, ("F", "L"): -3, ("X", "Q"): -5, ("B", "B"): 6
}


# http://www.embl-heidelberg.de/~vogt/matrices/pam300.cmp
pam300 = {
    ("W", "F"): 1, ("L", "R"): -3, ("S", "P"): 1, ("V", "T"): 0,
    ("Q", "Q"): 4, ("N", "A"): 0, ("Z", "Y"): -5, ("W", "R"): 3,
    ("Q", "A"): 0, ("S", "D"): 0, ("H", "H"): 7, ("S", "H"): -1,
    ("H", "D"): 1, ("L", "N"): -3, ("W", "A"): -6, ("Y", "M"): -2,
    ("G", "R"): -2, ("Y", "I"): -1, ("Y", "E"): -5, ("B", "Y"): -4,
    ("Y", "A"): -4, ("V", "D"): -2, ("B", "S"): 1, ("Y", "Y"): 12,
    ("G", "N"): 1, ("E", "C"): -6, ("Y", "Q"): -4, ("Z", "Z"): 3,
    ("V", "A"): 0, ("C", "C"): 15, ("M", "R"): 0, ("V", "E"): -2,
    ("T", "N"): 0, ("P", "P"): 6, ("V", "I"): 4, ("V", "S"): -1,
    ("Z", "P"): 0, ("V", "M"): 2, ("T", "F"): -3, ("V", "Q"): -2,
    ("K", "K"): 5, ("P", "D"): -1, ("I", "H"): -2, ("I", "D"): -2,
    ("T", "R"): -1, ("P", "L"): -3, ("K", "G"): -2, ("M", "N"): -2,
    ("P", "H"): 0, ("F", "Q"): -5, ("Z", "G"): 0, ("X", "L"): -1,
    ("T", "M"): -1, ("Z", "C"): -6, ("X", "H"): 0, ("D", "R"): -1,
    ("B", "W"): -6, ("X", "D"): -1, ("Z", "K"): 1, ("F", "A"): -4,
    ("Z", "W"): -6, ("F", "E"): -6, ("D", "N"): 2, ("B", "K"): 1,
    ("X", "X"): -1, ("F", "I"): 1, ("B", "G"): 1, ("X", "T"): 0,
    ("F", "M"): 1, ("B", "C"): -5, ("Z", "I"): -2, ("Z", "V"): -2,
    ("S", "S"): 1, ("L", "Q"): -2, ("W", "E"): -8, ("Q", "R"): 2,
    ("N", "N"): 2, ("W", "M"): -5, ("Q", "C"): -6, ("W", "I"): -6,
    ("S", "C"): 0, ("L", "A"): -2, ("S", "G"): 1, ("L", "E"): -4,
    ("W", "Q"): -5, ("H", "G"): -2, ("S", "K"): 0, ("Q", "N"): 1,
    ("N", "R"): 0, ("H", "C"): -4, ("Y", "N"): -2, ("G", "Q"): -1,
    ("Y", "F"): 9, ("C", "A"): -2, ("V", "L"): 2, ("G", "E"): 0,
    ("G", "A"): 2, ("K", "R"): 4, ("E", "D"): 4, ("Y", "R"): -5,
    ("M", "Q"): -1, ("T", "I"): 0, ("C", "D"): -6, ("V", "F"): -1,
    ("T", "A"): 1, ("T", "P"): 1, ("B", "P"): 0, ("T", "E"): 0,
    ("V", "N"): -2, ("P", "G"): 0, ("M", "A"): -1, ("K", "H"): 0,
    ("V", "R"): -3, ("P", "C"): -3, ("M", "E"): -2, ("K", "L"): -3,
    ("V", "V"): 5, ("M", "I"): 3, ("T", "Q"): -1, ("I", "G"): -3,
    ("P", "K"): -1, ("M", "M"): 6, ("K", "D"): 0, ("I", "C"): -3,
    ("Z", "D"): 3, ("F", "R"): -5, ("X", "K"): -1, ("Q", "D"): 2,
    ("X", "G"): -1, ("Z", "L"): -3, ("X", "C"): -3, ("Z", "H"): 2,
    ("B", "L"): -4, ("B", "H"): 1, ("F", "F"): 11, ("X", "W"): -4,
    ("B", "D"): 3, ("D", "A"): 0, ("S", "L"): -3, ("X", "S"): 0,
    ("F", "N"): -4, ("S", "R"): 0, ("W", "D"): -7, ("V", "Y"): -3,
    ("W", "L"): -2, ("H", "R"): 2, ("W", "H"): -3, ("H", "N"): 2,
    ("W", "T"): -6, ("T", "T"): 2, ("S", "F"): -4, ("W", "P"): -6,
    ("L", "D"): -4, ("B", "I"): -2, ("L", "H"): -2, ("S", "N"): 1,
    ("B", "T"): 0, ("L", "L"): 7, ("Y", "K"): -5, ("E", "Q"): 3,
    ("Y", "G"): -6, ("Z", "S"): 0, ("Y", "C"): 1, ("G", "D"): 1,
    ("B", "V"): -2, ("E", "A"): 0, ("Y", "W"): 0, ("E", "E"): 4,
    ("Y", "S"): -3, ("C", "N"): -4, ("V", "C"): -2, ("T", "H"): -1,
    ("P", "R"): 0, ("V", "G"): -1, ("T", "L"): -2, ("V", "K"): -2,
    ("K", "Q"): 1, ("R", "A"): -1, ("I", "R"): -2, ("T", "D"): 0,
    ("P", "F"): -5, ("I", "N"): -2, ("K", "I"): -2, ("M", "D"): -3,
    ("V", "W"): -7, ("W", "W"): 22, ("M", "H"): -2, ("P", "N"): 0,
    ("K", "A"): -1, ("M", "L"): 4, ("K", "E"): 0, ("Z", "E"): 3,
    ("X", "N"): 0, ("Z", "A"): 0, ("Z", "M"): -2, ("X", "F"): -2,
    ("K", "C"): -6, ("B", "Q"): 2, ("X", "B"): 0, ("B", "M"): -2,
    ("F", "C"): -5, ("Z", "Q"): 3, ("X", "Z"): -1, ("F", "G"): -5,
    ("B", "E"): 3, ("X", "V"): 0, ("F", "K"): -6, ("B", "A"): 0,
    ("X", "R"): -1, ("D", "D"): 4, ("W", "G"): -8, ("Z", "F"): -5,
    ("S", "Q"): 0, ("W", "C"): -9, ("W", "K"): -4, ("H", "Q"): 3,
    ("L", "C"): -7, ("W", "N"): -5, ("S", "A"): 1, ("L", "G"): -4,
    ("W", "S"): -3, ("S", "E"): 0, ("H", "E"): 1, ("S", "I"): -1,
    ("H", "A"): -1, ("S", "M"): -2, ("Y", "L"): 0, ("Y", "H"): 0,
    ("Y", "D"): -5, ("E", "R"): -1, ("X", "P"): -1, ("G", "G"): 5,
    ("G", "C"): -4, ("E", "N"): 2, ("Y", "T"): -3, ("Y", "P"): -5,
    ("T", "K"): 0, ("A", "A"): 2, ("P", "Q"): 0, ("T", "C"): -2,
    ("V", "H"): -2, ("T", "G"): 0, ("I", "Q"): -2, ("Z", "T"): 0,
    ("C", "R"): -4, ("V", "P"): -1, ("P", "E"): 0, ("M", "C"): -6,
    ("K", "N"): 1, ("I", "I"): 5, ("P", "A"): 1, ("M", "G"): -3,
    ("T", "S"): 1, ("I", "E"): -2, ("P", "M"): -2, ("M", "K"): 0,
    ("I", "A"): 0, ("P", "I"): -2, ("R", "R"): 7, ("X", "M"): -1,
    ("L", "I"): 3, ("X", "I"): -1, ("Z", "B"): 2, ("X", "E"): -1,
    ("Z", "N"): 1, ("X", "A"): 0, ("B", "R"): 0, ("B", "N"): 2,
    ("F", "D"): -6, ("X", "Y"): -2, ("Z", "R"): 0, ("F", "H"): -2,
    ("B", "F"): -5, ("F", "L"): 3, ("X", "Q"): 0, ("B", "B"): 3
}


# http://www.embl-heidelberg.de/~vogt/matrices/pam60.cmp
pam60 = {
    ("W", "F"): -3, ("L", "R"): -6, ("S", "P"): 0, ("V", "T"): -1,
    ("Q", "Q"): 7, ("N", "A"): -2, ("Z", "Y"): -7, ("W", "R"): 0,
    ("Q", "A"): -3, ("S", "D"): -2, ("H", "H"): 8, ("S", "H"): -4,
    ("H", "D"): -2, ("L", "N"): -5, ("W", "A"): -10, ("Y", "M"): -7,
    ("G", "R"): -7, ("Y", "I"): -4, ("Y", "E"): -7, ("B", "Y"): -5,
    ("Y", "A"): -6, ("V", "D"): -6, ("B", "S"): 0, ("Y", "Y"): 9,
    ("G", "N"): -1, ("E", "C"): -10, ("Y", "Q"): -8, ("Z", "Z"): 5,
    ("V", "A"): -1, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -4,
    ("T", "N"): -1, ("P", "P"): 7, ("V", "I"): 3, ("V", "S"): -4,
    ("Z", "P"): -2, ("V", "M"): 0, ("T", "F"): -6, ("V", "Q"): -5,
    ("K", "K"): 6, ("P", "D"): -5, ("I", "H"): -6, ("I", "D"): -5,
    ("T", "R"): -4, ("P", "L"): -5, ("K", "G"): -5, ("M", "N"): -6,
    ("P", "H"): -2, ("F", "Q"): -9, ("Z", "G"): -3, ("X", "L"): -4,
    ("T", "M"): -2, ("Z", "C"): -10, ("X", "H"): -3, ("D", "R"): -6,
    ("B", "W"): -8, ("X", "D"): -3, ("Z", "K"): -2, ("F", "A"): -6,
    ("Z", "W"): -11, ("F", "E"): -10, ("D", "N"): 2, ("B", "K"): -1,
    ("X", "X"): -3, ("F", "I"): -1, ("B", "G"): -2, ("X", "T"): -2,
    ("F", "M"): -2, ("B", "C"): -9, ("Z", "I"): -4, ("Z", "V"): -5,
    ("S", "S"): 5, ("L", "Q"): -3, ("W", "E"): -12, ("Q", "R"): 0,
    ("N", "N"): 6, ("W", "M"): -9, ("Q", "C"): -10, ("W", "I"): -10,
    ("S", "C"): -1, ("L", "A"): -4, ("S", "G"): 0, ("L", "E"): -7,
    ("W", "Q"): -9, ("H", "G"): -6, ("S", "K"): -2, ("Q", "N"): -2,
    ("N", "R"): -3, ("H", "C"): -6, ("Y", "N"): -3, ("G", "Q"): -5,
    ("Y", "F"): 3, ("C", "A"): -5, ("V", "L"): -1, ("G", "E"): -2,
    ("G", "A"): 0, ("K", "R"): 2, ("E", "D"): 3, ("Y", "R"): -8,
    ("M", "Q"): -2, ("T", "I"): -1, ("C", "D"): -10, ("V", "F"): -5,
    ("T", "A"): 1, ("T", "P"): -2, ("B", "P"): -4, ("T", "E"): -4,
    ("V", "N"): -5, ("P", "G"): -4, ("M", "A"): -3, ("K", "H"): -4,
    ("V", "R"): -5, ("P", "C"): -6, ("M", "E"): -5, ("K", "L"): -6,
    ("V", "V"): 6, ("M", "I"): 1, ("T", "Q"): -4, ("I", "G"): -7,
    ("P", "K"): -4, ("M", "M"): 10, ("K", "D"): -2, ("I", "C"): -4,
    ("Z", "D"): 2, ("F", "R"): -7, ("X", "K"): -3, ("Q", "D"): -1,
    ("X", "G"): -3, ("Z", "L"): -5, ("X", "C"): -6, ("Z", "H"): 0,
    ("B", "L"): -7, ("B", "H"): 0, ("F", "F"): 8, ("X", "W"): -8,
    ("B", "D"): 5, ("D", "A"): -2, ("S", "L"): -6, ("X", "S"): -2,
    ("F", "N"): -6, ("S", "R"): -2, ("W", "D"): -11, ("V", "Y"): -5,
    ("W", "L"): -4, ("H", "R"): 0, ("W", "H"): -5, ("H", "N"): 1,
    ("W", "T"): -9, ("T", "T"): 6, ("S", "F"): -5, ("W", "P"): -10,
    ("L", "D"): -9, ("B", "I"): -4, ("L", "H"): -4, ("S", "N"): 1,
    ("B", "T"): -2, ("L", "L"): 6, ("Y", "K"): -7, ("E", "Q"): 2,
    ("Y", "G"): -10, ("Z", "S"): -3, ("Y", "C"): -2, ("G", "D"): -2,
    ("B", "V"): -5, ("E", "A"): -1, ("Y", "W"): -3, ("E", "E"): 7,
    ("Y", "S"): -5, ("C", "N"): -7, ("V", "C"): -4, ("T", "H"): -5,
    ("P", "R"): -2, ("V", "G"): -4, ("T", "L"): -5, ("V", "K"): -6,
    ("K", "Q"): -1, ("R", "A"): -5, ("I", "R"): -4, ("T", "D"): -3,
    ("P", "F"): -7, ("I", "N"): -4, ("K", "I"): -4, ("M", "D"): -7,
    ("V", "W"): -11, ("W", "W"): 13, ("M", "H"): -7, ("P", "N"): -4,
    ("K", "A"): -5, ("M", "L"): 2, ("K", "E"): -3, ("Z", "E"): 5,
    ("X", "N"): -2, ("Z", "A"): -2, ("Z", "M"): -4, ("X", "F"): -5,
    ("K", "C"): -10, ("B", "Q"): -1, ("X", "B"): -3, ("B", "M"): -6,
    ("F", "C"): -9, ("Z", "Q"): 6, ("X", "Z"): -3, ("F", "G"): -7,
    ("B", "E"): 2, ("X", "V"): -3, ("F", "K"): -10, ("B", "A"): -2,
    ("X", "R"): -4, ("D", "D"): 7, ("W", "G"): -11, ("Z", "F"): -10,
    ("S", "Q"): -3, ("W", "C"): -12, ("W", "K"): -8, ("H", "Q"): 2,
    ("L", "C"): -11, ("W", "N"): -6, ("S", "A"): 1, ("L", "G"): -8,
    ("W", "S"): -4, ("S", "E"): -2, ("H", "E"): -3, ("S", "I"): -4,
    ("H", "A"): -5, ("S", "M"): -4, ("Y", "L"): -5, ("Y", "H"): -2,
    ("Y", "D"): -8, ("E", "R"): -6, ("X", "P"): -3, ("G", "G"): 6,
    ("G", "C"): -7, ("E", "N"): 0, ("Y", "T"): -5, ("Y", "P"): -10,
    ("T", "K"): -2, ("A", "A"): 5, ("P", "Q"): -1, ("T", "C"): -5,
    ("V", "H"): -5, ("T", "G"): -3, ("I", "Q"): -5, ("Z", "T"): -4,
    ("C", "R"): -6, ("V", "P"): -4, ("P", "E"): -3, ("M", "C"): -10,
    ("K", "N"): 0, ("I", "I"): 7, ("P", "A"): 0, ("M", "G"): -6,
    ("T", "S"): 1, ("I", "E"): -4, ("P", "M"): -6, ("M", "K"): 0,
    ("I", "A"): -3, ("P", "I"): -6, ("R", "R"): 8, ("X", "M"): -3,
    ("L", "I"): 0, ("X", "I"): -3, ("Z", "B"): 1, ("X", "E"): -3,
    ("Z", "N"): -1, ("X", "A"): -2, ("B", "R"): -5, ("B", "N"): 5,
    ("F", "D"): -11, ("X", "Y"): -5, ("Z", "R"): -2, ("F", "H"): -4,
    ("B", "F"): -8, ("F", "L"): -1, ("X", "Q"): -3, ("B", "B"): 5
}


# http://www.embl-heidelberg.de/~vogt/matrices/pam90.cmp
pam90 = {
    ("W", "F"): -2, ("L", "R"): -5, ("S", "P"): 0, ("V", "T"): -1,
    ("Q", "Q"): 6, ("N", "A"): -1, ("Z", "Y"): -6, ("W", "R"): 0,
    ("Q", "A"): -2, ("S", "D"): -1, ("H", "H"): 8, ("S", "H"): -3,
    ("H", "D"): -1, ("L", "N"): -4, ("W", "A"): -8, ("Y", "M"): -6,
    ("G", "R"): -5, ("Y", "I"): -3, ("Y", "E"): -6, ("B", "Y"): -4,
    ("Y", "A"): -5, ("V", "D"): -4, ("B", "S"): 0, ("Y", "Y"): 9,
    ("G", "N"): -1, ("E", "C"): -8, ("Y", "Q"): -6, ("Z", "Z"): 5,
    ("V", "A"): 0, ("C", "C"): 9, ("M", "R"): -2, ("V", "E"): -3,
    ("T", "N"): 0, ("P", "P"): 7, ("V", "I"): 3, ("V", "S"): -3,
    ("Z", "P"): -2, ("V", "M"): 1, ("T", "F"): -5, ("V", "Q"): -4,
    ("K", "K"): 5, ("P", "D"): -4, ("I", "H"): -5, ("I", "D"): -4,
    ("T", "R"): -3, ("P", "L"): -4, ("K", "G"): -4, ("M", "N"): -4,
    ("P", "H"): -2, ("F", "Q"): -7, ("Z", "G"): -2, ("T", "M"): -2,
    ("Z", "C"): -8, ("D", "R"): -5, ("B", "W"): -7, ("Z", "K"): -1,
    ("F", "A"): -5, ("Z", "W"): -8, ("F", "E"): -8, ("D", "N"): 3,
    ("B", "K"): 0, ("F", "I"): 0, ("B", "G"): -1, ("F", "M"): -1,
    ("B", "C"): -7, ("Z", "I"): -3, ("Z", "V"): -3, ("S", "S"): 4,
    ("L", "Q"): -3, ("W", "E"): -10, ("Q", "R"): 0, ("N", "N"): 5,
    ("W", "M"): -7, ("Q", "C"): -8, ("W", "I"): -8, ("S", "C"): -1,
    ("L", "A"): -3, ("S", "G"): 0, ("L", "E"): -5, ("W", "Q"): -7,
    ("H", "G"): -5, ("S", "K"): -1, ("L", "I"): 1, ("N", "R"): -2,
    ("H", "C"): -5, ("Y", "N"): -2, ("G", "Q"): -3, ("Y", "F"): 4,
    ("C", "A"): -3, ("V", "L"): 0, ("G", "E"): -1, ("G", "A"): 0,
    ("K", "R"): 2, ("E", "D"): 4, ("Y", "R"): -6, ("M", "Q"): -2,
    ("T", "I"): 0, ("C", "D"): -8, ("V", "F"): -4, ("T", "A"): 1,
    ("T", "P"): -1, ("B", "P"): -3, ("T", "E"): -2, ("V", "N"): -4,
    ("P", "G"): -3, ("M", "A"): -2, ("K", "H"): -2, ("V", "R"): -4,
    ("P", "C"): -5, ("M", "E"): -4, ("K", "L"): -5, ("V", "V"): 6,
    ("M", "I"): 1, ("T", "Q"): -3, ("I", "G"): -5, ("P", "K"): -3,
    ("M", "M"): 9, ("K", "D"): -2, ("I", "C"): -3, ("Z", "D"): 3,
    ("F", "R"): -6, ("Q", "D"): 0, ("Z", "L"): -4, ("Z", "H"): 1,
    ("B", "L"): -5, ("B", "H"): 1, ("F", "F"): 8, ("B", "D"): 5,
    ("D", "A"): -1, ("S", "L"): -5, ("F", "N"): -5, ("S", "R"): -1,
    ("W", "D"): -9, ("W", "L"): -3, ("H", "R"): 1, ("W", "H"): -4,
    ("H", "N"): 2, ("W", "T"): -7, ("T", "T"): 5, ("S", "F"): -4,
    ("W", "P"): -8, ("L", "D"): -7, ("B", "I"): -3, ("L", "H"): -3,
    ("S", "N"): 1, ("B", "T"): -1, ("L", "L"): 6, ("Y", "K"): -6,
    ("E", "Q"): 2, ("Y", "G"): -8, ("Z", "S"): -2, ("Y", "C"): -1,
    ("G", "D"): -1, ("B", "V"): -4, ("E", "A"): 0, ("Y", "W"): -2,
    ("E", "E"): 6, ("Y", "S"): -4, ("C", "N"): -6, ("V", "C"): -3,
    ("T", "H"): -3, ("P", "R"): -1, ("V", "G"): -3, ("T", "L"): -3,
    ("V", "K"): -5, ("K", "Q"): -1, ("R", "A"): -4, ("I", "R"): -3,
    ("T", "D"): -2, ("P", "F"): -6, ("I", "N"): -3, ("K", "I"): -3,
    ("M", "D"): -5, ("V", "W"): -9, ("W", "W"): 13, ("M", "H"): -5,
    ("P", "N"): -2, ("K", "A"): -3, ("M", "L"): 2, ("K", "E"): -2,
    ("Z", "E"): 5, ("Q", "N"): -1, ("Z", "A"): -1, ("Z", "M"): -3,
    ("K", "C"): -8, ("B", "Q"): 0, ("B", "M"): -5, ("F", "C"): -7,
    ("Z", "Q"): 5, ("F", "G"): -6, ("B", "E"): 2, ("F", "K"): -8,
    ("B", "A"): -1, ("D", "D"): 6, ("W", "G"): -9, ("S", "Q"): -2,
    ("W", "C"): -10, ("W", "K"): -6, ("H", "Q"): 2, ("L", "C"): -9,
    ("W", "N"): -5, ("S", "A"): 1, ("L", "G"): -6, ("W", "S"): -3,
    ("S", "E"): -2, ("H", "E"): -1, ("S", "I"): -3, ("H", "A"): -4,
    ("S", "M"): -3, ("Y", "L"): -3, ("Y", "H"): -1, ("Y", "D"): -6,
    ("E", "R"): -4, ("G", "G"): 5, ("G", "C"): -5, ("E", "N"): 0,
    ("Y", "T"): -4, ("Y", "P"): -8, ("T", "K"): -1, ("A", "A"): 4,
    ("P", "Q"): -1, ("T", "C"): -4, ("V", "H"): -4, ("T", "G"): -2,
    ("I", "Q"): -4, ("Z", "T"): -2, ("C", "R"): -5, ("V", "P"): -3,
    ("P", "E"): -2, ("M", "C"): -8, ("K", "N"): 1, ("I", "I"): 6,
    ("P", "A"): 0, ("M", "G"): -5, ("T", "S"): 2, ("I", "E"): -3,
    ("P", "M"): -4, ("M", "K"): 0, ("I", "A"): -2, ("P", "I"): -4,
    ("R", "R"): 7, ("Z", "F"): -8, ("Z", "B"): 2, ("Z", "N"): 0,
    ("B", "R"): -3, ("B", "N"): 4, ("F", "D"): -8, ("Z", "R"): -1,
    ("F", "H"): -3, ("B", "F"): -6, ("F", "L"): 0, ("V", "Y"): -4,
    ("B", "B"): 4
}


# http://www.embl-heidelberg.de/~vogt/matrices/rao.cmp
rao = {
    ("W", "F"): 11, ("S", "P"): 10, ("N", "M"): 6, ("Q", "Q"): 16,
    ("N", "A"): 9, ("N", "E"): 10, ("W", "V"): 11, ("Q", "E"): 11,
    ("L", "H"): 10, ("W", "R"): 7, ("Q", "A"): 11, ("S", "D"): 10,
    ("H", "H"): 16, ("Q", "M"): 9, ("S", "H"): 10, ("H", "D"): 9,
    ("Q", "I"): 6, ("S", "L"): 8, ("Y", "M"): 8, ("Y", "I"): 10,
    ("Y", "E"): 6, ("Y", "A"): 9, ("G", "F"): 7, ("V", "T"): 10,
    ("Y", "Y"): 16, ("V", "H"): 9, ("E", "C"): 9, ("Y", "Q"): 8,
    ("V", "A"): 9, ("C", "C"): 16, ("V", "E"): 4, ("T", "N"): 10,
    ("R", "K"): 11, ("P", "P"): 16, ("V", "I"): 12, ("R", "G"): 7,
    ("V", "M"): 9, ("T", "F"): 10, ("R", "C"): 8, ("V", "Q"): 6,
    ("K", "K"): 16, ("P", "D"): 8, ("I", "H"): 8, ("M", "F"): 10,
    ("I", "D"): 3, ("K", "C"): 9, ("P", "L"): 4, ("K", "G"): 7,
    ("P", "H"): 5, ("T", "R"): 9, ("F", "A"): 10, ("F", "E"): 6,
    ("S", "S"): 16, ("W", "E"): 7, ("N", "N"): 16, ("W", "M"): 10,
    ("Q", "C"): 10, ("N", "F"): 6, ("S", "C"): 10, ("L", "A"): 11,
    ("S", "G"): 11, ("L", "E"): 7, ("W", "Q"): 9, ("H", "G"): 7,
    ("S", "K"): 10, ("Q", "N"): 11, ("V", "D"): 3, ("H", "C"): 10,
    ("Y", "N"): 8, ("Y", "F"): 10, ("W", "I"): 11, ("C", "A"): 11,
    ("G", "E"): 6, ("G", "A"): 8, ("Y", "V"): 10, ("E", "D"): 11,
    ("W", "H"): 10, ("Y", "R"): 7, ("N", "I"): 5, ("R", "L"): 6,
    ("T", "I"): 10, ("Q", "L"): 9, ("R", "H"): 10, ("T", "M"): 8,
    ("V", "F"): 11, ("R", "D"): 10, ("T", "A"): 10, ("T", "P"): 8,
    ("T", "E"): 8, ("V", "N"): 5, ("P", "G"): 11, ("M", "A"): 11,
    ("K", "H"): 11, ("V", "R"): 5, ("P", "C"): 7, ("M", "E"): 8,
    ("V", "V"): 16, ("T", "T"): 16, ("M", "I"): 9, ("T", "Q"): 10,
    ("I", "G"): 6, ("P", "K"): 6, ("M", "M"): 16, ("K", "D"): 11,
    ("I", "C"): 8, ("L", "C"): 11, ("F", "F"): 16, ("D", "A"): 9,
    ("S", "R"): 9, ("W", "D"): 6, ("N", "C"): 9, ("N", "G"): 10,
    ("W", "T"): 11, ("Q", "G"): 8, ("S", "F"): 8, ("W", "P"): 6,
    ("L", "D"): 6, ("H", "F"): 9, ("Q", "K"): 12, ("S", "N"): 11,
    ("L", "L"): 16, ("Q", "F"): 7, ("Y", "K"): 7, ("Y", "G"): 10,
    ("Y", "C"): 10, ("G", "D"): 9, ("E", "A"): 10, ("Y", "W"): 11,
    ("E", "E"): 16, ("Y", "S"): 11, ("R", "M"): 6, ("V", "C"): 8,
    ("T", "H"): 10, ("R", "I"): 4, ("V", "G"): 6, ("T", "L"): 9,
    ("R", "E"): 9, ("V", "K"): 5, ("R", "Q"): 10, ("R", "A"): 8,
    ("T", "D"): 9, ("P", "F"): 4, ("V", "S"): 8, ("K", "I"): 4,
    ("M", "D"): 5, ("W", "W"): 16, ("M", "H"): 10, ("P", "N"): 9,
    ("I", "F"): 12, ("K", "A"): 10, ("M", "L"): 11, ("K", "E"): 11,
    ("N", "K"): 11, ("R", "P"): 6, ("L", "F"): 11, ("F", "C"): 10,
    ("W", "G"): 8, ("W", "L"): 11, ("D", "D"): 16, ("N", "H"): 10,
    ("S", "Q"): 10, ("Q", "P"): 7, ("N", "L"): 7, ("W", "K"): 7,
    ("Q", "D"): 11, ("W", "N"): 8, ("S", "A"): 10, ("L", "G"): 6,
    ("W", "S"): 10, ("S", "E"): 9, ("L", "K"): 7, ("H", "E"): 11,
    ("S", "I"): 8, ("Q", "H"): 11, ("H", "A"): 11, ("S", "M"): 7,
    ("Y", "L"): 9, ("Y", "H"): 9, ("Y", "D"): 7, ("G", "G"): 16,
    ("G", "C"): 8, ("Y", "T"): 11, ("W", "C"): 11, ("Y", "P"): 8,
    ("T", "K"): 9, ("R", "N"): 10, ("A", "A"): 16, ("W", "A"): 11,
    ("T", "C"): 10, ("N", "D"): 11, ("R", "F"): 5, ("T", "G"): 10,
    ("V", "L"): 10, ("V", "P"): 3, ("P", "E"): 5, ("M", "C"): 10,
    ("I", "I"): 16, ("P", "A"): 6, ("M", "G"): 4, ("T", "S"): 11,
    ("I", "E"): 4, ("P", "M"): 2, ("M", "K"): 8, ("K", "F"): 6,
    ("I", "A"): 9, ("P", "I"): 3, ("R", "R"): 16, ("L", "I"): 10,
    ("F", "D"): 4, ("D", "C"): 8
}


# http://www.embl-heidelberg.de/~vogt/matrices/risler.cmp
risler = {
    ("W", "F"): -0.9, ("S", "P"): -0.3, ("N", "M"): 0.0, ("Q", "Q"): 2.2,
    ("N", "A"): 1.3, ("N", "E"): 1.4, ("W", "V"): -0.7, ("Q", "E"): 2.1,
    ("L", "H"): -0.9, ("W", "R"): -0.8, ("Q", "A"): 1.8, ("S", "D"): 0.7,
    ("H", "H"): 2.2, ("Q", "M"): 1.2, ("S", "H"): -0.4, ("H", "D"): -1.3,
    ("Q", "I"): 1.4, ("S", "L"): 1.3, ("Y", "M"): -0.2, ("Y", "I"): 0.4,
    ("Y", "E"): 0.2, ("Y", "A"): 0.2, ("G", "F"): -0.4, ("V", "T"): 1.6,
    ("Y", "Y"): 2.2, ("V", "H"): -0.7, ("E", "C"): -1.5, ("Y", "Q"): 0.5,
    ("V", "A"): 2.0, ("C", "C"): 2.2, ("V", "E"): 1.6, ("T", "N"): 1.1,
    ("R", "K"): 2.1, ("P", "P"): 2.2, ("V", "I"): 2.2, ("R", "G"): 0.1,
    ("V", "M"): 0.8, ("T", "F"): 0.3, ("R", "C"): -1.5, ("V", "Q"): 1.5,
    ("K", "K"): 2.2, ("P", "D"): -1.2, ("I", "H"): -0.8, ("M", "F"): -0.2,
    ("I", "D"): 0.0, ("K", "C"): -1.6, ("P", "L"): -0.8, ("K", "G"): -0.1,
    ("P", "H"): -1.6, ("T", "R"): 1.9, ("F", "A"): 0.6, ("F", "E"): 0.6,
    ("S", "S"): 2.2, ("W", "E"): -1.0, ("N", "N"): 2.2, ("W", "M"): -1.3,
    ("Q", "C"): -1.4, ("N", "F"): 0.4, ("S", "C"): -1.3, ("L", "A"): 1.3,
    ("S", "G"): 0.7, ("L", "E"): 0.9, ("W", "Q"): -1.0, ("H", "G"): -1.2,
    ("S", "K"): 1.4, ("Q", "N"): 1.6, ("V", "D"): 0.0, ("H", "C"): -1.8,
    ("Y", "N"): -0.1, ("Y", "F"): 2.0, ("W", "I"): -0.7, ("C", "A"): -1.5,
    ("G", "E"): 0.3, ("G", "A"): 0.6, ("Y", "V"): 0.3, ("E", "D"): 1.0,
    ("W", "H"): -1.7, ("Y", "R"): 0.8, ("N", "I"): 0.9, ("R", "L"): 1.2,
    ("T", "I"): 1.6, ("Q", "L"): 1.1, ("R", "H"): -0.4, ("T", "M"): 0.8,
    ("V", "F"): 0.8, ("R", "D"): -0.1, ("T", "A"): 1.9, ("T", "P"): -0.5,
    ("T", "E"): 1.6, ("V", "N"): 1.1, ("P", "G"): -1.2, ("M", "A"): 1.0,
    ("K", "H"): -1.0, ("V", "R"): 1.5, ("P", "C"): -1.8, ("M", "E"): 0.6,
    ("V", "V"): 2.2, ("T", "T"): 2.2, ("M", "I"): 0.9, ("T", "Q"): 1.7,
    ("I", "G"): 0.0, ("P", "K"): -0.7, ("M", "M"): 2.2, ("K", "D"): 0.1,
    ("I", "C"): -1.6, ("L", "C"): -1.5, ("F", "F"): 2.2, ("D", "A"): 0.2,
    ("S", "R"): 2.0, ("W", "D"): -1.4, ("N", "C"): -1.6, ("N", "G"): 0.2,
    ("W", "T"): -1.0, ("Q", "G"): 0.2, ("S", "F"): 0.5, ("W", "P"): -1.6,
    ("L", "D"): -0.2, ("H", "F"): -1.1, ("Q", "K"): 1.7, ("S", "N"): 1.9,
    ("L", "L"): 2.2, ("Q", "F"): 0.7, ("Y", "K"): 0.5, ("Y", "G"): -0.2,
    ("Y", "C"): -1.1, ("G", "D"): -0.4, ("E", "A"): 1.7, ("Y", "W"): -0.6,
    ("E", "E"): 2.2, ("Y", "S"): 0.4, ("R", "M"): 1.1, ("V", "C"): -1.4,
    ("T", "H"): -0.9, ("R", "I"): 1.4, ("V", "G"): 0.1, ("T", "L"): 1.2,
    ("R", "E"): 1.9, ("V", "K"): 1.2, ("R", "Q"): 2.0, ("R", "A"): 1.5,
    ("T", "D"): 0.0, ("P", "F"): -1.1, ("V", "S"): 1.8, ("K", "I"): 1.0,
    ("M", "D"): -0.5, ("W", "W"): 2.2, ("M", "H"): -1.2, ("P", "N"): -1.0,
    ("I", "F"): 1.0, ("K", "A"): 1.4, ("M", "L"): 1.8, ("K", "E"): 1.4,
    ("N", "K"): 1.0, ("R", "P"): -0.3, ("L", "F"): 1.0, ("F", "C"): -1.6,
    ("W", "G"): -1.3, ("W", "L"): -0.8, ("D", "D"): 2.2, ("N", "H"): -0.3,
    ("S", "Q"): 1.8, ("Q", "P"): -0.6, ("N", "L"): 0.8, ("W", "K"): -1.1,
    ("Q", "D"): 0.6, ("W", "N"): -1.1, ("S", "A"): 2.0, ("L", "G"): -0.2,
    ("W", "S"): -0.8, ("S", "E"): 1.8, ("L", "K"): 0.7, ("H", "E"): -0.6,
    ("S", "I"): 1.6, ("Q", "H"): -0.5, ("H", "A"): -0.6, ("S", "M"): 0.6,
    ("Y", "L"): 0.5, ("Y", "H"): -0.8, ("Y", "D"): -0.4, ("G", "G"): 2.2,
    ("G", "C"): -1.7, ("Y", "T"): 0.3, ("W", "C"): -1.8, ("Y", "P"): -1.2,
    ("T", "K"): 1.2, ("R", "N"): 1.2, ("A", "A"): 2.2, ("W", "A"): -0.9,
    ("T", "C"): -1.4, ("N", "D"): 0.8, ("R", "F"): 0.4, ("T", "G"): 0.2,
    ("V", "L"): 2.0, ("V", "P"): -0.6, ("P", "E"): -0.1, ("M", "C"): -1.6,
    ("I", "I"): 2.2, ("P", "A"): -0.2, ("M", "G"): -0.4, ("T", "S"): 2.1,
    ("I", "E"): 1.5, ("P", "M"): -1.2, ("M", "K"): 0.4, ("K", "F"): 0.1,
    ("I", "A"): 1.7, ("P", "I"): -0.6, ("R", "R"): 2.2, ("L", "I"): 2.1,
    ("F", "D"): -0.3, ("D", "C"): -1.7
}


# http://www.embl-heidelberg.de/~vogt/matrices/str.cmp
structure = {
    ("W", "F"): 2, ("L", "R"): -3, ("I", "I"): 6, ("Q", "Q"): 6,
    ("W", "N"): -5, ("V", "I"): 2, ("H", "T"): -2, ("H", "P"): -3,
    ("W", "V"): -4, ("Q", "E"): 2, ("W", "R"): -2, ("Q", "A"): 0,
    ("H", "H"): 8, ("H", "D"): 0, ("L", "N"): -3, ("Y", "M"): -1,
    ("Y", "I"): -1, ("Y", "E"): -2, ("E", "S"): -1, ("Y", "A"): -3,
    ("Y", "Y"): 7, ("T", "C"): -5, ("E", "C"): -3, ("Y", "Q"): -3,
    ("E", "G"): -2, ("V", "A"): 0, ("C", "C"): 11, ("M", "R"): -4,
    ("P", "T"): -1, ("V", "E"): -2, ("P", "P"): 7, ("I", "T"): -2,
    ("K", "S"): -1, ("R", "G"): -2, ("I", "P"): -4, ("R", "C"): -2,
    ("A", "T"): -1, ("K", "K"): 5, ("A", "P"): -1, ("V", "M"): 0,
    ("I", "D"): -3, ("K", "C"): -4, ("K", "G"): -3, ("R", "S"): 0,
    ("F", "Q"): -4, ("F", "A"): -3, ("V", "V"): 5, ("M", "N"): -2,
    ("F", "E"): -4, ("D", "N"): 2, ("F", "I"): 1, ("F", "M"): 0,
    ("M", "S"): -4, ("S", "S"): 4, ("L", "Q"): -3, ("W", "E"): -6,
    ("W", "A"): -3, ("W", "M"): -2, ("H", "S"): -2, ("W", "I"): -2,
    ("S", "C"): -4, ("L", "A"): -2, ("L", "E"): -4, ("W", "Q"): -5,
    ("H", "G"): -3, ("Q", "N"): 0, ("H", "C"): -6, ("L", "M"): 3,
    ("W", "Y"): 2, ("Y", "N"): -1, ("E", "P"): -1, ("Y", "F"): 3,
    ("E", "T"): 0, ("A", "A"): 4, ("I", "N"): -3, ("G", "A"): 0,
    ("Y", "V"): -1, ("E", "D"): 2, ("W", "H"): -3, ("Y", "R"): -1,
    ("M", "Q"): 1, ("P", "S"): -1, ("R", "H"): 0, ("A", "C"): -2,
    ("R", "D"): -2, ("K", "P"): -1, ("L", "D"): -6, ("K", "T"): 0,
    ("V", "N"): -4, ("M", "A"): 0, ("K", "H"): 0, ("V", "R"): -3,
    ("P", "C"): -8, ("M", "E"): -2, ("A", "S"): 0, ("T", "T"): 5,
    ("R", "T"): -1, ("I", "G"): -5, ("R", "P"): -2, ("K", "D"): -1,
    ("I", "C"): -4, ("F", "R"): -4, ("F", "V"): -1, ("L", "C"): -6,
    ("F", "F"): 7, ("D", "A"): -1, ("F", "N"): -3, ("W", "D"): -6,
    ("L", "P"): -3, ("Q", "S"): -1, ("N", "C"): -6, ("N", "G"): -1,
    ("H", "N"): 2, ("W", "T"): -5, ("Q", "G"): -2, ("W", "P"): -4,
    ("Q", "C"): -3, ("N", "S"): 0, ("L", "H"): -3, ("L", "L"): 5,
    ("G", "T"): -3, ("M", "M"): 8, ("G", "P"): -2, ("Y", "K"): -2,
    ("Y", "G"): -3, ("Y", "C"): -6, ("E", "A"): 0, ("E", "E"): 5,
    ("Y", "S"): -2, ("M", "P"): -6, ("V", "C"): -4, ("M", "T"): -2,
    ("V", "G"): -4, ("R", "E"): 0, ("V", "K"): -3, ("K", "Q"): 1,
    ("R", "A"): -1, ("I", "R"): -3, ("N", "A"): -1, ("V", "S"): -3,
    ("M", "D"): -4, ("M", "H"): -2, ("K", "A"): -1, ("R", "Q"): 1,
    ("K", "E"): 1, ("F", "S"): -3, ("I", "K"): -3, ("D", "P"): -1,
    ("D", "T"): -1, ("I", "M"): 1, ("F", "C"): -2, ("W", "L"): -1,
    ("F", "G"): -6, ("F", "K"): -3, ("F", "T"): -3, ("D", "D"): 6,
    ("Q", "T"): 0, ("W", "G"): -4, ("Q", "P"): -2, ("W", "C"): -6,
    ("W", "K"): -3, ("H", "Q"): 0, ("Q", "D"): 0, ("W", "W"): 10,
    ("V", "L"): 1, ("L", "G"): -5, ("W", "S"): -5, ("L", "K"): -2,
    ("N", "P"): -2, ("H", "E"): -2, ("N", "T"): 0, ("H", "A"): -2,
    ("Y", "L"): -2, ("Y", "H"): 0, ("G", "S"): -1, ("Y", "D"): -3,
    ("V", "Q"): -2, ("L", "T"): -3, ("G", "G"): 5, ("G", "C"): -6,
    ("E", "N"): 0, ("Y", "T"): -2, ("Y", "P"): -6, ("R", "N"): -1,
    ("V", "D"): -4, ("K", "R"): 2, ("V", "H"): -2, ("I", "Q"): -5,
    ("V", "P"): -4, ("M", "C"): -5, ("K", "N"): 0, ("V", "T"): -1,
    ("M", "G"): -4, ("T", "S"): 1, ("I", "E"): -3, ("M", "K"): -1,
    ("I", "A"): -2, ("N", "N"): 5, ("R", "R"): 7, ("F", "P"): -5,
    ("L", "I"): 2, ("I", "S"): -3, ("D", "S"): 0, ("L", "S"): -4,
    ("I", "H"): -5, ("F", "D"): -5, ("D", "C"): -7, ("F", "H"): -2,
    ("D", "G"): -1, ("F", "L"): 2
}
