/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.framework.model;

import java.beans.PropertyChangeListener;
import java.beans.PropertyVetoException;

import org.jdom.Element;

import docking.DockingTool;
import docking.util.image.ToolIconURL;
import ghidra.framework.plugintool.PluginEvent;

/**
 * 
 * A Tool represents a collection of Plugins that are the basic building
 * blocks. The plugins cooperate with one another to achieve certain
 * functionality. The tool interface defines methods called by the 
 * Project that manages the tools.
 */
public interface Tool extends DockingTool, ToolListener {

	/**
	 * Name of the property for the tool name.
	 */
	public final static String TOOL_NAME_PROPERTY = "ToolName";
	/**
	 * Name of the property for the tool icon.
	 */
	public final static String ICON_PROPERTY_NAME = "Icon";
	/**
	 * Name of the property for the description of the tool.
	 */
	public final static String DESCRIPTION_PROPERTY_NAME = "Description";
	/**
	 * Name of the property for the number of plugins the tool has.
	 */
	public final static String PLUGIN_COUNT_PROPERTY_NAME = "PluginCount";

	/**
	 * Tells the tool to stop functioning and release its resources.
	 * The tool should dispose of all its windows and other resources.
	 */
	public void exit();

	/**
	 * Can this tool be closed?
	 * @param isExiting true if all of Ghidra is closing, false if just this tool is closing.
	 * @return true if the tool is in a state that it can be closed.
	 */
	public boolean canClose(boolean isExiting);

	/**
	 * Can the domain File be closed?
	 * <br>Note: This forces plugins to terminate any tasks they have running for the 
	 * indicated domain object and apply any unsaved data to the domain object. If they can't do 
	 * this or the user cancels then this returns false.
	 * @return false any of the plugins reports that the domain object 
	 * should not be closed
	 */
	public boolean canCloseDomainFile(DomainFile domainFile);

	/**
	 * Fire the plugin event by notifying the event manager which
	 * calls the listeners.
	 * @param event plugin event
	 */
	public void firePluginEvent(PluginEvent event);

	/**
	 * Returns the name associated with the tool
	 */
	public String getToolName();

	/**
	 * Sets the type name of the tool.
	 * @param toolName the basename to use when setting the tool's name
	 * @exception PropertyVetoException thrown if a VetoableChangeListener
	 * rejects the change
	 */
	public void setToolName(String toolName) throws PropertyVetoException;

	/**
	 * Associates a unique(within the active project) name to a tool instance.
	 * @param instanceName unique tool instance name
	 */
	public void putInstanceName(String instanceName);

	/**
	 * Returns the tool's unique name.
	 */
	public String getInstanceName();

	/**
	 * Returns the names of all the possible ToolEvents that this
	 * tool might generate.  Used by the ConnectionManager to connect
	 * tools together.
	 */
	public String[] getToolEventNames();

	/**
	 * Returns a list of eventNames that this Tool is interested in.
	 */
	public String[] getConsumedToolEventNames();

	/**
	 * When the user drags a data file onto a tool, an event will be fired
	 * that the tool will respond to by accepting the data.
	 *
	 * @param data the data to be used by the running tool
	 */
	public boolean acceptDomainFiles(DomainFile[] data);

	/**
	 * Get the domain files that this tool currently has open.
	 */
	public DomainFile[] getDomainFiles();

	/**
	 * Adds a ToolListener to be notified for any of a Tool's ToolEvents.
	 * The listener will be notified of any events that this tool generates.
	 *
	 * @param listener ToolListener to be added to receive all events
	 */
	public void addToolListener(ToolListener listener);

	/**
	 * Removes a ToolListener from receiving any event generated by this Tool.
	 * The tool will still recieve specific events that it has registered for.
	 *
	 * @param listener The ToolListener to be removed from receiving all events.
	 */
	public void removeToolListener(ToolListener listener);

	/**
	 * Add property change listener.
	 */
	public void addPropertyChangeListener(PropertyChangeListener l);

	/**
	 * Remove property change listener.
	 */
	public void removePropertyChangeListener(PropertyChangeListener l);

	/**
	 * Get the classes of the data types that this tool supports,
	 * i.e., what data types can be dropped onto this tool.
	 */
	public Class<?>[] getSupportedDataTypes();

	/**
	 * Tells tool to write its data state from the given output stream.
	 * @param isTransactionState true if saving the toolstate is for a potential undo/redo
	 * (database transaction)
	 */
	public Element saveDataStateToXml(boolean isTransactionState);

	/**
	 * Tells tool to read its data state from the given input stream.
	 * @param element XML data state
	 */
	public void restoreDataStateFromXml(Element element);

	/**
	 * Set the icon for this tool.
	 * @param iconURL icon location
	 */
	public void setIconURL(ToolIconURL iconURL);

	/**
	 * Get the url for the icon that this tool is using.
	 */
	public ToolIconURL getIconURL();

	/**
	 * Returns a ToolTemplate for this Tool that describes the state of the tool.
	 * @return a ToolTemplate for this Tool that describes the state of the tool.
	 */
	public ToolTemplate getToolTemplate(boolean includeConfigState);

	/**
	 * Save the tool and return its state as a ToolTemplate. Forces a complete 
	 * regeneration of the tool template.
	 * @return a toolTemplate for this tool.
	 */
	public ToolTemplate saveToolToToolTemplate();

	/**
	 * Saves the tool's Docking Window layout and positioning information to an XML element.
	 * @return the element containing the DockingWindow's layout information.
	 */
	public Element saveWindowingDataToXml();

	/**
	 * Restores the tool's Docking Window layout and positioning information from an XML element.
	 * @param windowData the element containing the information.
	 */
	public void restoreWindowingDataFromXml(Element windowData);

	/**
	 * Returns true if this tool should save its data, based upon its changed state and the state
	 * of Ghidra's saving method.
	 * @return true if this tool should save its data
	 */
	public boolean shouldSave();

	public Element saveToXml(boolean includeConfigState);
}
