/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.misc;

import ghidra.framework.main.FrontEndOnly;
import ghidra.framework.main.FrontEndTool;
import ghidra.framework.model.DomainFile;
import ghidra.framework.model.DomainFolderChangeListener;
import ghidra.framework.model.DomainFolderListenerAdapter;
import ghidra.framework.model.DomainObject;
import ghidra.framework.model.Project;
import ghidra.framework.model.ProjectListener;
import ghidra.framework.options.OptionsChangeListener;
import ghidra.framework.options.ToolOptions;
import ghidra.framework.plugintool.Plugin;
import ghidra.framework.plugintool.PluginInfo;
import ghidra.framework.plugintool.PluginTool;
import ghidra.framework.plugintool.util.PluginStatus;
import ghidra.util.Msg;
import ghidra.util.bean.opteditor.OptionsVetoException;
import ghidra.util.exception.AssertException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;
import javax.swing.Timer;

@PluginInfo(status=PluginStatus.RELEASED, packageName="Ghidra Core", category="Miscellaneous", shortDescription="Generates recovery snapshot files", description="Facilitates the periodic creation of recovery snapshot files.  In the event of a crash or application hang, these files may be used to recover any un-saved file changes at the time of failure.  The frequency in which these snapshots are generated is controlled via a Front-end Recovery Option.")
public class RecoverySnapshotMgrPlugin
extends Plugin
implements FrontEndOnly,
OptionsChangeListener,
ProjectListener {
    private static final String OPTIONS_TITLE = "Recovery";
    private static final String SNAPSHOT_PERIOD_OPTION = "Snapshot period (minutes, 0=disabled)";
    private static final int DELAYED_RETRY_PERIOD_MSEC = 10000;
    private int snapshotPeriodMin = 5;
    private Project currentProject;
    private DomainFolderChangeListener fileListener;
    private ActionListener snapshotAction;
    private Timer timer;
    private long timerStart = -1L;
    private int totalDelayTime;
    private SnapshotTask snapshotTask = new SnapshotTask();
    private Set<DomainFile> fileSet = new HashSet<DomainFile>();
    private TreeSet<DomainFile> pendingSnapshotSet = new TreeSet();

    public RecoverySnapshotMgrPlugin(PluginTool tool) {
        super(tool);
        if (tool instanceof FrontEndTool) {
            this.initSnapshotOptions();
        }
    }

    protected void init() {
        this.fileListener = new DomainFolderListenerAdapter(){

            public void domainFileObjectOpenedForUpdate(DomainFile file, DomainObject object) {
                RecoverySnapshotMgrPlugin.this.addFile(file);
            }

            public void domainFileObjectClosed(DomainFile file, DomainObject object) {
                RecoverySnapshotMgrPlugin.this.removeFile(file);
            }
        };
        this.snapshotAction = new ActionListener(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void actionPerformed(ActionEvent e) {
                2 var2_2 = this;
                synchronized (var2_2) {
                    if (RecoverySnapshotMgrPlugin.this.pendingSnapshotSet.isEmpty()) {
                        RecoverySnapshotMgrPlugin.this.pendingSnapshotSet.addAll(RecoverySnapshotMgrPlugin.this.fileSet);
                        RecoverySnapshotMgrPlugin.this.startSnapshotTimer(false);
                        return;
                    }
                }
                new Thread((Runnable)RecoverySnapshotMgrPlugin.this.snapshotTask, "recovery-snapshot-task").start();
            }
        };
        FrontEndTool feTool = (FrontEndTool)this.tool;
        feTool.addProjectListener((ProjectListener)this);
        Project prj = feTool.getProject();
        if (prj != null) {
            this.projectOpened(prj);
            this.findOpenFiles();
        }
        this.reportTimerSetting();
    }

    private void initSnapshotOptions() {
        ToolOptions opt = this.tool.getOptions(OPTIONS_TITLE);
        opt.registerOption(SNAPSHOT_PERIOD_OPTION, (Object)this.snapshotPeriodMin, null, "The time before creating an auto-save of a program");
        this.snapshotPeriodMin = opt.getInt(SNAPSHOT_PERIOD_OPTION, this.snapshotPeriodMin);
        if (this.snapshotPeriodMin < 0) {
            opt.setInt(SNAPSHOT_PERIOD_OPTION, 0);
            this.snapshotPeriodMin = 0;
        }
        opt.addOptionsChangeListener((OptionsChangeListener)this);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void optionsChanged(ToolOptions options, String optionName, Object oldValue, Object newValue) {
        if (optionName.equals(SNAPSHOT_PERIOD_OPTION)) {
            RecoverySnapshotMgrPlugin recoverySnapshotMgrPlugin = this;
            synchronized (recoverySnapshotMgrPlugin) {
                int oldSnapshotPeriod = this.snapshotPeriodMin;
                int newSnapshotPeriodMin = options.getInt(SNAPSHOT_PERIOD_OPTION, this.snapshotPeriodMin);
                if (newSnapshotPeriodMin < 0) {
                    throw new OptionsVetoException("The snapshot period must be >= 0");
                }
                this.snapshotPeriodMin = newSnapshotPeriodMin;
                if (oldSnapshotPeriod != this.snapshotPeriodMin) {
                    this.reportTimerSetting();
                    if (oldSnapshotPeriod == 0 || this.timerStart > 0L) {
                        if (this.snapshotPeriodMin > 0 && this.timerStart > 0L) {
                            this.totalDelayTime = (int)(new Date().getTime() - this.timerStart);
                        }
                        this.startSnapshotTimer(false);
                    }
                }
            }
        }
    }

    private void reportTimerSetting() {
        if (this.snapshotPeriodMin == 0) {
            Msg.debug((Object)((Object)this), (Object)"Recovery snapshot timer disabled!");
        } else {
            Msg.debug((Object)((Object)this), (Object)("Recovery snapshot timer set to " + this.snapshotPeriodMin + " minute(s)"));
        }
    }

    public void dispose() {
        ToolOptions opt = this.tool.getOptions(OPTIONS_TITLE);
        opt.removeOptionsChangeListener((OptionsChangeListener)this);
        super.dispose();
    }

    public synchronized void projectOpened(Project project) {
        if (project == this.currentProject) {
            return;
        }
        if (this.currentProject != null) {
            throw new AssertException("Unexpected - two or more projects active");
        }
        this.currentProject = project;
        this.findOpenFiles();
        this.currentProject.getProjectData().addDomainFolderChangeListener(this.fileListener);
        this.startSnapshotTimer(false);
    }

    public synchronized void projectClosed(Project project) {
        if (this.currentProject != null) {
            this.stopSnapshotTimer();
            this.currentProject.getProjectData().removeDomainFolderChangeListener(this.fileListener);
            this.pendingSnapshotSet.clear();
            this.fileSet.clear();
            this.currentProject = null;
        }
    }

    private void findOpenFiles() {
        ArrayList list = new ArrayList();
        this.currentProject.getProjectData().findOpenFiles(list);
        for (DomainFile df : list) {
            this.addFile(df);
        }
    }

    private synchronized void addFile(DomainFile df) {
        if (df.isInWritableProject()) {
            this.fileSet.add(df);
            if (this.totalDelayTime == 0) {
                this.pendingSnapshotSet.add(df);
            }
        }
    }

    private synchronized void removeFile(DomainFile df) {
        if (df.isInWritableProject()) {
            this.fileSet.remove(df);
            this.pendingSnapshotSet.remove(df);
        }
    }

    private synchronized void stopSnapshotTimer() {
        if (this.timer != null) {
            this.timer.stop();
        }
        this.timerStart = -1L;
    }

    private synchronized void startSnapshotTimer(boolean retryPeriod) {
        this.stopSnapshotTimer();
        if (this.snapshotPeriodMin == 0) {
            this.totalDelayTime = 0;
            return;
        }
        int msec = 10000;
        if (retryPeriod) {
            this.totalDelayTime += msec;
            if (this.totalDelayTime >= this.snapshotPeriodMin * 60000) {
                retryPeriod = false;
            }
        } else {
            this.totalDelayTime -= this.snapshotPeriodMin * 60000;
            if (this.totalDelayTime < -10000) {
                msec = -this.totalDelayTime;
            }
        }
        if (!retryPeriod) {
            this.pendingSnapshotSet.clear();
            this.pendingSnapshotSet.addAll(this.fileSet);
            this.totalDelayTime = 0;
            this.timerStart = new Date().getTime();
        }
        if (this.timer == null) {
            this.timer = new Timer(msec, this.snapshotAction);
            this.timer.setRepeats(false);
            this.timer.start();
        } else {
            this.timer.setInitialDelay(msec);
            this.timer.restart();
        }
    }

    private class SnapshotTask
    implements Runnable {
        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            RecoverySnapshotMgrPlugin recoverySnapshotMgrPlugin;
            ArrayList<DomainFile> unhandledList = new ArrayList<DomainFile>();
            DomainFile df = null;
            while (true) {
                recoverySnapshotMgrPlugin = RecoverySnapshotMgrPlugin.this;
                synchronized (recoverySnapshotMgrPlugin) {
                    if (RecoverySnapshotMgrPlugin.this.pendingSnapshotSet.isEmpty()) {
                        break;
                    }
                    df = RecoverySnapshotMgrPlugin.this.pendingSnapshotSet.first();
                    RecoverySnapshotMgrPlugin.this.pendingSnapshotSet.remove(df);
                }
                boolean completed = false;
                try {
                    completed = df.takeRecoverySnapshot();
                }
                catch (FileNotFoundException e) {
                    completed = true;
                }
                catch (Throwable t) {
                    completed = true;
                    Msg.showError((Object)this, null, (String)"Recovery Snapshot Error", (Object)("Failed to generate recovery snapshot for: " + df.getName()), (Throwable)t);
                }
                if (completed || !RecoverySnapshotMgrPlugin.this.fileSet.contains(df)) continue;
                unhandledList.add(df);
            }
            recoverySnapshotMgrPlugin = RecoverySnapshotMgrPlugin.this;
            synchronized (recoverySnapshotMgrPlugin) {
                if (!unhandledList.isEmpty()) {
                    RecoverySnapshotMgrPlugin.this.pendingSnapshotSet.clear();
                    RecoverySnapshotMgrPlugin.this.pendingSnapshotSet.addAll(unhandledList);
                    RecoverySnapshotMgrPlugin.this.startSnapshotTimer(true);
                } else {
                    RecoverySnapshotMgrPlugin.this.startSnapshotTimer(false);
                }
            }
        }
    }
}

