// Copyright 2005-2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the 'License');
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an 'AS IS' BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// See www.openfst.org for extensive documentation on this weighted
// finite-state transducer library.
//
// Getters for converting command-line arguments into the appropriate enums
// or bitmasks, with the simplest ones defined as inline.

#ifndef FST_SCRIPT_GETTERS_H_
#define FST_SCRIPT_GETTERS_H_

#include <string>

#include <fst/types.h>
#include <fst/log.h>
#include <fst/compose.h>         // For ComposeFilter.
#include <fst/determinize.h>     // For DeterminizeType.
#include <fst/encode.h>          // For kEncodeLabels (etc.).
#include <fst/epsnormalize.h>    // For EpsNormalizeType.
#include <fst/project.h>         // For ProjectType.
#include <fst/push.h>            // For kPushWeights (etc.).
#include <fst/queue.h>           // For QueueType.
#include <fst/rational.h>        // For ClosureType.
#include <fst/string.h>          // For TokenType.
#include <fst/script/arcsort.h>      // For ArcSortType.
#include <fst/script/map.h>          // For MapType.
#include <fst/script/script-impl.h>  // For RandArcSelection.

namespace fst {
namespace script {

bool GetArcSortType(const std::string &str, ArcSortType *sort_type);

inline ClosureType GetClosureType(bool closure_plus) {
  return closure_plus ? CLOSURE_PLUS : CLOSURE_STAR;
}

bool GetComposeFilter(const std::string &str, ComposeFilter *compose_filter);

bool GetDeterminizeType(const std::string &str, DeterminizeType *det_type);

inline uint8 GetEncodeFlags(bool encode_labels, bool encode_weights) {
  return (encode_labels ? kEncodeLabels : 0) |
         (encode_weights ? kEncodeWeights : 0);
}

inline EpsNormalizeType GetEpsNormalizeType(bool eps_norm_output) {
  return eps_norm_output ? EPS_NORM_OUTPUT : EPS_NORM_INPUT;
}

bool GetMapType(const std::string &str, MapType *map_type);

bool GetProjectType(const std::string &str, ProjectType *project_type);

inline uint8 GetPushFlags(bool push_weights, bool push_labels,
                          bool remove_total_weight, bool remove_common_affix) {
  return ((push_weights ? kPushWeights : 0) | (push_labels ? kPushLabels : 0) |
          (remove_total_weight ? kPushRemoveTotalWeight : 0) |
          (remove_common_affix ? kPushRemoveCommonAffix : 0));
}

bool GetQueueType(const std::string &str, QueueType *queue_type);

bool GetRandArcSelection(const std::string &str, RandArcSelection *ras);

bool GetReplaceLabelType(const std::string &str, bool epsilon_on_replace,
                         ReplaceLabelType *rlt);

inline ReweightType GetReweightType(bool to_final) {
  return to_final ? REWEIGHT_TO_FINAL : REWEIGHT_TO_INITIAL;
}

bool GetTokenType(const std::string &str, TokenType *token_type);

}  // namespace script
}  // namespace fst

#endif  // FST_SCRIPT_GETTERS_H_
