"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSampleNotebooks = void 0;

var _uuid = require("uuid");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const getDemoNotebook = (dateString, visId) => {
  const uuids = [(0, _uuid.v4)()];
  const oneWeekAgo = new Date();
  oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
  return {
    name: 'OpenSearch Notebooks Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'Default',
    paragraphs: [{
      output: [{
        result: `An OpenSearch Dashboards notebook is an interface that lets you easily combine code snippets, live visualizations, and narrative text in a single notebook interface.

Notebooks let you interactively explore data by running different visualizations that you can share with team members to collaborate on a project.

A notebook is a document composed of two elements: code blocks (Markdown/SQL/PPL) and visualizations.

Common use cases include creating postmortem reports, designing runbooks, building live infrastructure reports, and writing documentation.

You can also generate [reports](https://opensearch.org/docs/dashboards/reporting/) directly from your notebooks.

For more information, refer to the [documentation](https://opensearch.org/docs/dashboards/notebooks/).`,
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
An OpenSearch Dashboards notebook is an interface that lets you easily combine code snippets, live visualizations, and narrative text in a single notebook interface.

Notebooks let you interactively explore data by running different visualizations that you can share with team members to collaborate on a project.

A notebook is a document composed of two elements: code blocks (Markdown/SQL/PPL) and visualizations.

Common use cases include creating postmortem reports, designing runbooks, building live infrastructure reports, and writing documentation.

You can also generate [reports](https://opensearch.org/docs/dashboards/reporting/) directly from your notebooks.

For more information, refer to the [documentation](https://opensearch.org/docs/dashboards/notebooks/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Notebooks combine code blocks and visualizations for describing data. Code blocks support markdown, SQL, and PPL languages. Specify the input language on the first line using %\[language type\] syntax. For example, type %md for markdown, %sql for SQL, and %ppl for PPL. A sample visualization is shown below:',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
Notebooks combine code blocks and visualizations for describing data. Code blocks support markdown, SQL, and PPL languages. Specify the input language on the first line using %\[language type\] syntax. For example, type %md for markdown, %sql for SQL, and %ppl for PPL. A sample visualization is shown below:`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visId}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[0]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[0]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'You can add a new paragraph from the **Paragraph actions** menu on the top right of the page, the context menu on the top right of each paragraph, or the **Add paragraph** button on the bottom of the page.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
You can add a new paragraph from the **Paragraph actions** menu on the top right of the page, the context menu on the top right of each paragraph, or the **Add paragraph** button on the bottom of the page.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'You can also reorder, duplicate, or delete paragraphs from these menus.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
You can also reorder, duplicate, or delete paragraphs from these menus.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To execute a paragraph, choose **Run**. To make changes to the input block, choose the downward arrow that\'s next to the paragraph title.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
To execute a paragraph, choose **Run**. To make changes to the input block, choose the downward arrow that's next to the paragraph title.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `There are three view modes:
- **View both**: allows you to edit paragraphs and view outputs on the same page.
- **Input only**: only shows input of each paragraph, allowing easier editing.
- **Output only**: only shows output of each paragraph and hides panels.`,
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
There are three view modes:
- **View both**: allows you to edit paragraphs and view outputs on the same page.
- **Input only**: only shows input of each paragraph, allowing easier editing.
- **Output only**: only shows output of each paragraph and hides panels.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To go back, choose **Notebooks** on the top left of your screen.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
To go back, choose **Notebooks** on the top left of your screen.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};

const getRootCauseNotebook = (dateString, visIds) => {
  const uuids = [(0, _uuid.v4)(), (0, _uuid.v4)()];
  const oneWeekAgo = new Date();
  oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
  return {
    name: '[Logs] Sample Root Cause Event Analysis',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'Default',
    paragraphs: [{
      output: [{
        result: 'This notebook goes through a sample root cause event analysis using PPL and visualizations with the OpenSearch Dashboards sample web logs data.',
        outputType: 'MARKDOWN',
        execution_time: '0.016 ms'
      }],
      input: {
        inputText: `%md
This notebook goes through a sample root cause event analysis using PPL and visualizations with the OpenSearch Dashboards sample web logs data.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "First, let's bring up a few visualizations for a quick overview.",
        outputType: 'MARKDOWN',
        execution_time: '0.046 ms'
      }],
      input: {
        inputText: `%md
First, let's bring up a few visualizations for a quick overview.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visIds[0]}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[0]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[0]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visIds[1]}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[1]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[1]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Let's take a look at the source data by the selected fields (search and fields).`,
        outputType: 'MARKDOWN',
        execution_time: '0.013 ms'
      }],
      input: {
        inputText: `%md
Let's take a look at the source data by the selected fields (search and fields).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, 
  response, bytes | head 20

`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, 
  response, bytes | head 20

`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Check for any error log with response code 404 or 503 (filter).',
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Check for any error log with response code 404 or 503 (filter).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404' | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404' | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'We see too many events. Let\'s quickly check which host has the issue (dedup).',
        outputType: 'MARKDOWN',
        execution_time: '0.014 ms'
      }],
      input: {
        inputText: `%md
We see too many events. Let's quickly check which host has the issue (dedup).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | dedup host | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | dedup host | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'We get too few events. Let\'s dedup in consecutive mode (dedup).',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
We get too few events. Let's dedup in consecutive mode (dedup).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    dedup host consecutive=true | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    dedup host consecutive=true | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'How many IP addresses for each response (stats).',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
How many IP addresses for each response (stats).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count by response | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count by response | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To dive deep, let\'s group by host and response, count, and sum (stats).',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To dive deep, let's group by host and response, count, and sum (stats).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `We don't see a meaningful response. Let's change to resp_code (rename).`,
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
We don't see a meaningful response. Let's change to resp_code (rename).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response |
     rename response as resp_code | head 20

`,
        outputType: 'QUERY',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response |
     rename response as resp_code | head 20

`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'The data looks better now. Let\'s sort by `DESC count` and `ASC sum_bytes` (sort).',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
The data looks better now. Let's sort by \`DESC count\` and \`ASC sum_bytes\` (sort).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Let\'s check if we can perform aggregations after stats (eval).',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
Let's check if we can perform aggregations after stats (eval).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response | 
    rename response as resp_code | 
    sort - ip_count, + sum_bytes | 
    eval per_ip_bytes=sum_bytes/ip_count  | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response | 
    rename response as resp_code | 
    sort - ip_count, + sum_bytes | 
    eval per_ip_bytes=sum_bytes/ip_count  | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Wait, what's meant by an evaluation. Can we really perform an evaluation?`,
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
Wait, what's meant by an evaluation. Can we really perform an evaluation?`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | 
   eval per_ip_bytes=sum_bytes/ip_count, 
    double_per_ip_bytes = 2 * per_ip_bytes | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | 
   eval per_ip_bytes=sum_bytes/ip_count, 
    double_per_ip_bytes = 2 * per_ip_bytes | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};

const getSQLNotebook = dateString => {
  return {
    name: '[Flights] OpenSearch SQL Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'Default',
    paragraphs: [{
      output: [{
        result: `OpenSearch SQL lets you write queries in SQL rather than the [OpenSearch query domain-specific language (DSL)](https://opensearch.org/docs/opensearch/query-dsl/full-text/). If you’re already familiar with SQL and don’t want to learn the query DSL, this feature is a great option.

For more information, please refer to the [documentation](https://opensearch.org/docs/search-plugins/sql/index/).`,
        outputType: 'MARKDOWN',
        execution_time: '0.013 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL lets you write queries in SQL rather than the [OpenSearch query domain-specific language (DSL)](https://opensearch.org/docs/opensearch/query-dsl/full-text/). If you’re already familiar with SQL and don’t want to learn the query DSL, this feature is a great option.

For more information, please refer to the [documentation](https://opensearch.org/docs/search-plugins/sql/index/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'This notebook shows you how to use SQL with the [OpenSearch Dashboards sample flight data](/app/home#/tutorial_directory).',
        outputType: 'MARKDOWN',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%md
This notebook shows you how to use SQL with the [OpenSearch Dashboards sample flight data](/app/home#/tutorial_directory).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To use SQL, add a code paragraph, type %sql on the first line, and then add SQL queries on the next line.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
To use SQL, add a code paragraph, type %sql on the first line, and then add SQL queries on the next line.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Select * from opensearch_dashboards_sample_data_flights limit 20;',
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%sql
Select * from opensearch_dashboards_sample_data_flights limit 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'You can specify fields in the `SELECT` clause and use the `WHERE` clause to filter results. The following query finds flights heading to countries that start with \'A\' that are more than 5000 miles away.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
You can specify fields in the \`SELECT\` clause and use the \`WHERE\` clause to filter results. The following query finds flights heading to countries that start with 'A' that are more than 5000 miles away.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT FlightNum,OriginCountry,OriginCityName,DestCountry,DestCityName,DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry LIKE 'A%' LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT FlightNum,OriginCountry,OriginCityName,DestCountry,DestCityName,DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry LIKE 'A%' LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL also supports subqueries:',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL also supports subqueries:`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT opensearch_dashboards_sample_data_flights.FlightNum,opensearch_dashboards_sample_data_flights.OriginCountry,opensearch_dashboards_sample_data_flights.OriginCityName,opensearch_dashboards_sample_data_flights.DestCountry,opensearch_dashboards_sample_data_flights.DestCityName,opensearch_dashboards_sample_data_flights.DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE FlightNum IN (SELECT FlightNum FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry = 'AU') LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%sql
SELECT opensearch_dashboards_sample_data_flights.FlightNum,opensearch_dashboards_sample_data_flights.OriginCountry,opensearch_dashboards_sample_data_flights.OriginCityName,opensearch_dashboards_sample_data_flights.DestCountry,opensearch_dashboards_sample_data_flights.DestCityName,opensearch_dashboards_sample_data_flights.DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE FlightNum IN (SELECT FlightNum FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry = 'AU') LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL supports inner joins, cross joins, and left outer joins. The following query joins the flights index with itself to find flights departed from countries that are both origins and destinations.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL supports inner joins, cross joins, and left outer joins. The following query joins the flights index with itself to find flights departed from countries that are both origins and destinations.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
SELECT a.FlightNum,a.OriginCountry,a.OriginCityName,a.DestCountry,a.DestCityName,a.DistanceMiles FROM opensearch_dashboards_sample_data_flights a JOIN opensearch_dashboards_sample_data_flights b on a.OriginCountry = b.DestCountry LIMIT 20`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT a.FlightNum,a.OriginCountry,a.OriginCityName,a.DestCountry,a.DestCityName,a.DistanceMiles FROM opensearch_dashboards_sample_data_flights a JOIN opensearch_dashboards_sample_data_flights b on a.OriginCountry = b.DestCountry LIMIT 20`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'For aggregations, use the `GROUP BY` clause. The following query finds the countries with more than 500 flights departed.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
For aggregations, use the \`GROUP BY\` clause. The following query finds the countries with more than 500 flights departed.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT OriginCountry,COUNT(1) FROM opensearch_dashboards_sample_data_flights GROUP BY OriginCountry HAVING COUNT(1) > 500 LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT OriginCountry,COUNT(1) FROM opensearch_dashboards_sample_data_flights GROUP BY OriginCountry HAVING COUNT(1) > 500 LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL supports expressions.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL supports expressions.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT abs(-1.234), abs(-1 * abs(-5)), dayofmonth(DATE '2021-07-07');`,
        outputType: 'QUERY',
        execution_time: '0.005 ms'
      }],
      input: {
        inputText: `%sql
SELECT abs(-1.234), abs(-1 * abs(-5)), dayofmonth(DATE '2021-07-07');`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};

const getPPLNotebook = dateString => {
  return {
    name: '[Logs] OpenSearch Piped Processing Language (PPL) Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'Default',
    paragraphs: [{
      output: [{
        result: `Query DSL is powerful and fast, but it has a steep learning curve because it doesn’t have a human-readable interface. It's also difficult to create ad hoc queries and explore your data. SQL lets you extract and analyze data in a declarative manner.

OpenSearch makes its search and query engine robust by introducing Piped Processing Language (PPL).

PPL enables developers, DevOps engineers, support engineers, site reliability engineers (SREs), and IT managers to effectively discover and explore log data stored in OpenSearch.

With PPL, you can extract insights from your data with a sequence of commands delimited by pipes (|). PPL supports a comprehensive set of commands including search, where, fields, rename, dedup, sort, eval, head, top, and rare. PPL also supports functions, operators and expressions.

Even if you're new to OpenSearch and are only familiar with the pipe (|) syntax, you can still be productive from day one.`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Query DSL is powerful and fast, but it has a steep learning curve because it doesn’t have a human-readable interface. It's also difficult to create ad hoc queries and explore your data. SQL lets you extract and analyze data in a declarative manner.

OpenSearch makes its search and query engine robust by introducing Piped Processing Language (PPL).

PPL enables developers, DevOps engineers, support engineers, site reliability engineers (SREs), and IT managers to effectively discover and explore log data stored in OpenSearch.

With PPL, you can extract insights from your data with a sequence of commands delimited by pipes (|). PPL supports a comprehensive set of commands including search, where, fields, rename, dedup, sort, eval, head, top, and rare. PPL also supports functions, operators and expressions.

Even if you're new to OpenSearch and are only familiar with the pipe (|) syntax, you can still be productive from day one.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Piped Processing Language (PPL) is a query language that lets you use pipe (|) syntax to explore, discover, and query data stored in OpenSearch.

For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Piped Processing Language (PPL) is a query language that lets you use pipe (|) syntax to explore, discover, and query data stored in OpenSearch.

For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'In this notebook, we will demonstrate some simple PPL queries with the [OpenSearch Dashboards sample web logs](/app/home#/tutorial_directory). ',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
In this notebook, we will demonstrate some simple PPL queries with the [OpenSearch Dashboards sample web logs](/app/home#/tutorial_directory). `,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To use PPL, add a code paragraph, type `%ppl` on the first line, and add your PPL query on the next line. PPL uses `source` to specify indices and `head` to limit results.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To use PPL, add a code paragraph, type \`%ppl\` on the first line, and add your PPL query on the next line. PPL uses \`source\` to specify indices and \`head\` to limit results.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | head 20`,
        outputType: 'QUERY',
        execution_time: '0.026 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | head 20`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To specify fields to include and filter results, use the `field` and `where` commands. The next query returns hosts with failed responses.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
To specify fields to include and filter results, use the \`field\` and \`where\` commands. The next query returns hosts with failed responses.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404'`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404'`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To see most common hosts from the previous result, use the `top` command.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To see most common hosts from the previous result, use the \`top\` command.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | top host`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | top host`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To perform aggregations on search results, use the `stats` command.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To perform aggregations on search results, use the \`stats\` command.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | stats count(1) by host`,
        outputType: 'QUERY',
        execution_time: '0.011 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | stats count(1) by host`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).

To see how to perform sample root cause event analysis using PPL and visualizations, see the "\[Logs\] Sample Root Cause Event Analysis" notebook.`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).

To see how to perform sample root cause event analysis using PPL and visualizations, see the "\[Logs\] Sample Root Cause Event Analysis" notebook.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};

const getSampleNotebooks = visIds => {
  const dateString = new Date().toISOString();
  return [{
    notebook: getDemoNotebook(dateString, visIds[2]),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    notebook: getSQLNotebook(dateString),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    notebook: getPPLNotebook(dateString),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    notebook: getRootCauseNotebook(dateString, visIds),
    dateModified: dateString,
    dateCreated: dateString
  }];
};

exports.getSampleNotebooks = getSampleNotebooks;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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