"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("../utils/constants");

var _helpers = require("../utils/helpers");

var _DataStreamService = require("./DataStreamService");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class ManagedIndexService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "getManagedIndex", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          id,
          index: _constants.INDEX.OPENDISTRO_ISM_CONFIG
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const results = await callWithRequest("search", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error("Index Management - ManagedIndexService - getManagedIndex:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "getManagedIndices", async (context, request, response) => {
      try {
        const {
          from,
          size,
          sortDirection,
          sortField,
          terms,
          indices,
          dataStreams,
          showDataStreams
        } = request.query;
        const managedIndexSorts = {
          index: "managed_index.index",
          policyId: "managed_index.policy_id"
        };
        const explainParams = {
          sortField: sortField ? managedIndexSorts[sortField] : null,
          sortOrder: sortDirection,
          queryString: (0, _helpers.getSearchString)(terms, indices, dataStreams, showDataStreams),
          from: from,
          size: size
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const [explainAllResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("ism.explainAll", explainParams), (0, _DataStreamService.getIndexToDataStreamMapping)({
          callAsCurrentUser: callWithRequest
        })]);
        const managedIndices = [];

        for (const indexName in explainAllResponse) {
          if (indexName == "total_managed_indices") continue;
          const metadata = explainAllResponse[indexName]; // If showDataStreams is not true, then skip the managed index if it belongs to a data stream.

          const parentDataStream = indexToDataStreamMapping[metadata.index] || null;
          if (!showDataStreams && parentDataStream !== null) continue;
          let policy, seqNo, primaryTerm, getResponse;

          try {
            getResponse = await callWithRequest("ism.getPolicy", {
              policyId: metadata.policy_id
            });
          } catch (err) {
            if (err.statusCode === 404 && err.body.error.reason === "Policy not found") {
              console.log("managed index with not existing policy");
            } else {
              throw err;
            }
          }

          policy = _lodash.default.get(getResponse, "policy", null);
          seqNo = _lodash.default.get(getResponse, "_seq_no", null);
          primaryTerm = _lodash.default.get(getResponse, "_primary_term", null);
          managedIndices.push({
            index: metadata.index,
            indexUuid: metadata.index_uuid,
            dataStream: parentDataStream,
            policyId: metadata.policy_id,
            policySeqNo: seqNo,
            policyPrimaryTerm: primaryTerm,
            policy: policy,
            enabled: metadata.enabled,
            managedIndexMetaData: (0, _helpers.transformManagedIndexMetaData)(metadata)
          });
        }

        let totalManagedIndices = explainAllResponse.total_managed_indices;
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              managedIndices: managedIndices,
              totalManagedIndices: totalManagedIndices
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                managedIndices: [],
                totalManagedIndices: 0
              }
            }
          });
        }

        console.error("Index Management - ManagedIndexService - getManagedIndices", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "retryManagedIndexPolicy", async (context, request, response) => {
      try {
        const {
          index,
          state = null
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: index.join(",")
        };
        if (state) params.body = {
          state
        };
        const retryResponse = await callWithRequest("ism.retry", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: retryResponse.failures,
              updatedIndices: retryResponse.updated_indices,
              // TODO: remove ternary after fixing retry API to return empty array even if no failures
              failedIndices: retryResponse.failed_indices ? retryResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              })) : []
            }
          }
        });
      } catch (err) {
        console.error("Index Management - ManagedIndexService - retryManagedIndexPolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "changePolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId,
          include,
          state
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId,
            include,
            state
          }
        };
        const changeResponse = await callWithRequest("ism.change", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: changeResponse.failures,
              updatedIndices: changeResponse.updated_indices,
              failedIndices: changeResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - ManagedIndexService - changePolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "removePolicy", async (context, request, response) => {
      try {
        const {
          indices
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(",")
        };
        const addResponse = await callWithRequest("ism.remove", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - ManagedIndexService - removePolicy:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  } // TODO: Not finished, need UI page that uses this first


}

exports.default = ManagedIndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIk1hbmFnZWRJbmRleFNlcnZpY2UudHMiXSwibmFtZXMiOlsiTWFuYWdlZEluZGV4U2VydmljZSIsImNvbnN0cnVjdG9yIiwib3NEcml2ZXIiLCJjb250ZXh0IiwicmVxdWVzdCIsInJlc3BvbnNlIiwiaWQiLCJwYXJhbXMiLCJpbmRleCIsIklOREVYIiwiT1BFTkRJU1RST19JU01fQ09ORklHIiwiY2FsbEFzQ3VycmVudFVzZXIiLCJjYWxsV2l0aFJlcXVlc3QiLCJhc1Njb3BlZCIsInJlc3VsdHMiLCJjdXN0b20iLCJzdGF0dXNDb2RlIiwiYm9keSIsIm9rIiwiZXJyIiwiY29uc29sZSIsImVycm9yIiwibWVzc2FnZSIsImZyb20iLCJzaXplIiwic29ydERpcmVjdGlvbiIsInNvcnRGaWVsZCIsInRlcm1zIiwiaW5kaWNlcyIsImRhdGFTdHJlYW1zIiwic2hvd0RhdGFTdHJlYW1zIiwicXVlcnkiLCJtYW5hZ2VkSW5kZXhTb3J0cyIsInBvbGljeUlkIiwiZXhwbGFpblBhcmFtcyIsInNvcnRPcmRlciIsInF1ZXJ5U3RyaW5nIiwiZXhwbGFpbkFsbFJlc3BvbnNlIiwiaW5kZXhUb0RhdGFTdHJlYW1NYXBwaW5nIiwiUHJvbWlzZSIsImFsbCIsIm1hbmFnZWRJbmRpY2VzIiwiaW5kZXhOYW1lIiwibWV0YWRhdGEiLCJwYXJlbnREYXRhU3RyZWFtIiwicG9saWN5Iiwic2VxTm8iLCJwcmltYXJ5VGVybSIsImdldFJlc3BvbnNlIiwicG9saWN5X2lkIiwicmVhc29uIiwibG9nIiwiXyIsImdldCIsInB1c2giLCJpbmRleFV1aWQiLCJpbmRleF91dWlkIiwiZGF0YVN0cmVhbSIsInBvbGljeVNlcU5vIiwicG9saWN5UHJpbWFyeVRlcm0iLCJlbmFibGVkIiwibWFuYWdlZEluZGV4TWV0YURhdGEiLCJ0b3RhbE1hbmFnZWRJbmRpY2VzIiwidG90YWxfbWFuYWdlZF9pbmRpY2VzIiwidHlwZSIsInN0YXRlIiwiam9pbiIsInJldHJ5UmVzcG9uc2UiLCJmYWlsdXJlcyIsInVwZGF0ZWRJbmRpY2VzIiwidXBkYXRlZF9pbmRpY2VzIiwiZmFpbGVkSW5kaWNlcyIsImZhaWxlZF9pbmRpY2VzIiwibWFwIiwiZmFpbGVkSW5kZXgiLCJpbmRleF9uYW1lIiwiaW5jbHVkZSIsImNoYW5nZVJlc3BvbnNlIiwiYWRkUmVzcG9uc2UiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFLQTs7QUFTQTs7QUFDQTs7QUFlQTs7Ozs7O0FBRWUsTUFBTUEsbUJBQU4sQ0FBMEI7QUFHdkNDLEVBQUFBLFdBQVcsQ0FBQ0MsUUFBRCxFQUF1QztBQUFBOztBQUFBLDZDQUtoQyxPQUNoQkMsT0FEZ0IsRUFFaEJDLE9BRmdCLEVBR2hCQyxRQUhnQixLQUlnRDtBQUNoRSxVQUFJO0FBQ0YsY0FBTTtBQUFFQyxVQUFBQTtBQUFGLFlBQVNGLE9BQU8sQ0FBQ0csTUFBdkI7QUFDQSxjQUFNQSxNQUF5QixHQUFHO0FBQUVELFVBQUFBLEVBQUY7QUFBTUUsVUFBQUEsS0FBSyxFQUFFQyxpQkFBTUM7QUFBbkIsU0FBbEM7QUFDQSxjQUFNO0FBQUVDLFVBQUFBLGlCQUFpQixFQUFFQztBQUFyQixZQUF5QyxLQUFLVixRQUFMLENBQWNXLFFBQWQsQ0FBdUJULE9BQXZCLENBQS9DO0FBQ0EsY0FBTVUsT0FBNEIsR0FBRyxNQUFNRixlQUFlLENBQUMsUUFBRCxFQUFXTCxNQUFYLENBQTFEO0FBQ0EsZUFBT0YsUUFBUSxDQUFDVSxNQUFULENBQWdCO0FBQ3JCQyxVQUFBQSxVQUFVLEVBQUUsR0FEUztBQUVyQkMsVUFBQUEsSUFBSSxFQUFFO0FBQ0pDLFlBQUFBLEVBQUUsRUFBRSxJQURBO0FBRUpiLFlBQUFBLFFBQVEsRUFBRVM7QUFGTjtBQUZlLFNBQWhCLENBQVA7QUFPRCxPQVpELENBWUUsT0FBT0ssR0FBUCxFQUFZO0FBQ1pDLFFBQUFBLE9BQU8sQ0FBQ0MsS0FBUixDQUFjLDJEQUFkLEVBQTJFRixHQUEzRTtBQUNBLGVBQU9kLFFBQVEsQ0FBQ1UsTUFBVCxDQUFnQjtBQUNyQkMsVUFBQUEsVUFBVSxFQUFFLEdBRFM7QUFFckJDLFVBQUFBLElBQUksRUFBRTtBQUNKQyxZQUFBQSxFQUFFLEVBQUUsS0FEQTtBQUVKRyxZQUFBQSxLQUFLLEVBQUVGLEdBQUcsQ0FBQ0c7QUFGUDtBQUZlLFNBQWhCLENBQVA7QUFPRDtBQUNGLEtBaENpRDs7QUFBQSwrQ0FrQzlCLE9BQ2xCbkIsT0FEa0IsRUFFbEJDLE9BRmtCLEVBR2xCQyxRQUhrQixLQUlvRTtBQUN0RixVQUFJO0FBQ0YsY0FBTTtBQUFFa0IsVUFBQUEsSUFBRjtBQUFRQyxVQUFBQSxJQUFSO0FBQWNDLFVBQUFBLGFBQWQ7QUFBNkJDLFVBQUFBLFNBQTdCO0FBQXdDQyxVQUFBQSxLQUF4QztBQUErQ0MsVUFBQUEsT0FBL0M7QUFBd0RDLFVBQUFBLFdBQXhEO0FBQXFFQyxVQUFBQTtBQUFyRSxZQUF5RjFCLE9BQU8sQ0FBQzJCLEtBQXZHO0FBWUEsY0FBTUMsaUJBQXFDLEdBQUc7QUFBRXhCLFVBQUFBLEtBQUssRUFBRSxxQkFBVDtBQUFnQ3lCLFVBQUFBLFFBQVEsRUFBRTtBQUExQyxTQUE5QztBQUNBLGNBQU1DLGFBQWEsR0FBRztBQUNwQlIsVUFBQUEsU0FBUyxFQUFFQSxTQUFTLEdBQUdNLGlCQUFpQixDQUFDTixTQUFELENBQXBCLEdBQWtDLElBRGxDO0FBRXBCUyxVQUFBQSxTQUFTLEVBQUVWLGFBRlM7QUFHcEJXLFVBQUFBLFdBQVcsRUFBRSw4QkFBZ0JULEtBQWhCLEVBQXVCQyxPQUF2QixFQUFnQ0MsV0FBaEMsRUFBNkNDLGVBQTdDLENBSE87QUFJcEJQLFVBQUFBLElBQUksRUFBRUEsSUFKYztBQUtwQkMsVUFBQUEsSUFBSSxFQUFFQTtBQUxjLFNBQXRCO0FBUUEsY0FBTTtBQUFFYixVQUFBQSxpQkFBaUIsRUFBRUM7QUFBckIsWUFBeUMsS0FBS1YsUUFBTCxDQUFjVyxRQUFkLENBQXVCVCxPQUF2QixDQUEvQztBQUNBLGNBQU0sQ0FBQ2lDLGtCQUFELEVBQXFCQyx3QkFBckIsSUFBaUQsTUFBTUMsT0FBTyxDQUFDQyxHQUFSLENBQVksQ0FDdkU1QixlQUFlLENBQUMsZ0JBQUQsRUFBbUJzQixhQUFuQixDQUR3RCxFQUV2RSxvREFBNEI7QUFBRXZCLFVBQUFBLGlCQUFpQixFQUFFQztBQUFyQixTQUE1QixDQUZ1RSxDQUFaLENBQTdEO0FBSUEsY0FBTTZCLGNBQWtDLEdBQUcsRUFBM0M7O0FBQ0EsYUFBSyxNQUFNQyxTQUFYLElBQXdCTCxrQkFBeEIsRUFBNEM7QUFDMUMsY0FBSUssU0FBUyxJQUFJLHVCQUFqQixFQUEwQztBQUMxQyxnQkFBTUMsUUFBUSxHQUFHTixrQkFBa0IsQ0FBQ0ssU0FBRCxDQUFuQyxDQUYwQyxDQUkxQzs7QUFDQSxnQkFBTUUsZ0JBQWdCLEdBQUdOLHdCQUF3QixDQUFDSyxRQUFRLENBQUNuQyxLQUFWLENBQXhCLElBQTRDLElBQXJFO0FBQ0EsY0FBSSxDQUFDc0IsZUFBRCxJQUFvQmMsZ0JBQWdCLEtBQUssSUFBN0MsRUFBbUQ7QUFDbkQsY0FBSUMsTUFBSixFQUFZQyxLQUFaLEVBQW1CQyxXQUFuQixFQUFnQ0MsV0FBaEM7O0FBQ0EsY0FBSTtBQUNGQSxZQUFBQSxXQUFXLEdBQUcsTUFBTXBDLGVBQWUsQ0FBQyxlQUFELEVBQWtCO0FBQUVxQixjQUFBQSxRQUFRLEVBQUVVLFFBQVEsQ0FBQ007QUFBckIsYUFBbEIsQ0FBbkM7QUFDRCxXQUZELENBRUUsT0FBTzlCLEdBQVAsRUFBWTtBQUNaLGdCQUFJQSxHQUFHLENBQUNILFVBQUosS0FBbUIsR0FBbkIsSUFBMEJHLEdBQUcsQ0FBQ0YsSUFBSixDQUFTSSxLQUFULENBQWU2QixNQUFmLEtBQTBCLGtCQUF4RCxFQUE0RTtBQUMxRTlCLGNBQUFBLE9BQU8sQ0FBQytCLEdBQVIsQ0FBWSx3Q0FBWjtBQUNELGFBRkQsTUFFTztBQUNMLG9CQUFNaEMsR0FBTjtBQUNEO0FBQ0Y7O0FBQ0QwQixVQUFBQSxNQUFNLEdBQUdPLGdCQUFFQyxHQUFGLENBQU1MLFdBQU4sRUFBbUIsUUFBbkIsRUFBNkIsSUFBN0IsQ0FBVDtBQUNBRixVQUFBQSxLQUFLLEdBQUdNLGdCQUFFQyxHQUFGLENBQU1MLFdBQU4sRUFBbUIsU0FBbkIsRUFBOEIsSUFBOUIsQ0FBUjtBQUNBRCxVQUFBQSxXQUFXLEdBQUdLLGdCQUFFQyxHQUFGLENBQU1MLFdBQU4sRUFBbUIsZUFBbkIsRUFBb0MsSUFBcEMsQ0FBZDtBQUNBUCxVQUFBQSxjQUFjLENBQUNhLElBQWYsQ0FBb0I7QUFDbEI5QyxZQUFBQSxLQUFLLEVBQUVtQyxRQUFRLENBQUNuQyxLQURFO0FBRWxCK0MsWUFBQUEsU0FBUyxFQUFFWixRQUFRLENBQUNhLFVBRkY7QUFHbEJDLFlBQUFBLFVBQVUsRUFBRWIsZ0JBSE07QUFJbEJYLFlBQUFBLFFBQVEsRUFBRVUsUUFBUSxDQUFDTSxTQUpEO0FBS2xCUyxZQUFBQSxXQUFXLEVBQUVaLEtBTEs7QUFNbEJhLFlBQUFBLGlCQUFpQixFQUFFWixXQU5EO0FBT2xCRixZQUFBQSxNQUFNLEVBQUVBLE1BUFU7QUFRbEJlLFlBQUFBLE9BQU8sRUFBRWpCLFFBQVEsQ0FBQ2lCLE9BUkE7QUFTbEJDLFlBQUFBLG9CQUFvQixFQUFFLDRDQUE4QmxCLFFBQTlCO0FBVEosV0FBcEI7QUFXRDs7QUFFRCxZQUFJbUIsbUJBQW1CLEdBQUd6QixrQkFBa0IsQ0FBQzBCLHFCQUE3QztBQUNBLGVBQU8xRCxRQUFRLENBQUNVLE1BQVQsQ0FBZ0I7QUFDckJDLFVBQUFBLFVBQVUsRUFBRSxHQURTO0FBRXJCQyxVQUFBQSxJQUFJLEVBQUU7QUFDSkMsWUFBQUEsRUFBRSxFQUFFLElBREE7QUFFSmIsWUFBQUEsUUFBUSxFQUFFO0FBQUVvQyxjQUFBQSxjQUFjLEVBQUVBLGNBQWxCO0FBQWtDcUIsY0FBQUEsbUJBQW1CLEVBQUVBO0FBQXZEO0FBRk47QUFGZSxTQUFoQixDQUFQO0FBT0QsT0FyRUQsQ0FxRUUsT0FBTzNDLEdBQVAsRUFBWTtBQUNaLFlBQUlBLEdBQUcsQ0FBQ0gsVUFBSixLQUFtQixHQUFuQixJQUEwQkcsR0FBRyxDQUFDRixJQUFKLENBQVNJLEtBQVQsQ0FBZTJDLElBQWYsS0FBd0IsMkJBQXRELEVBQW1GO0FBQ2pGLGlCQUFPM0QsUUFBUSxDQUFDVSxNQUFULENBQWdCO0FBQ3JCQyxZQUFBQSxVQUFVLEVBQUUsR0FEUztBQUVyQkMsWUFBQUEsSUFBSSxFQUFFO0FBQ0pDLGNBQUFBLEVBQUUsRUFBRSxJQURBO0FBRUpiLGNBQUFBLFFBQVEsRUFBRTtBQUFFb0MsZ0JBQUFBLGNBQWMsRUFBRSxFQUFsQjtBQUFzQnFCLGdCQUFBQSxtQkFBbUIsRUFBRTtBQUEzQztBQUZOO0FBRmUsV0FBaEIsQ0FBUDtBQU9EOztBQUNEMUMsUUFBQUEsT0FBTyxDQUFDQyxLQUFSLENBQWMsNERBQWQsRUFBNEVGLEdBQTVFO0FBQ0EsZUFBT2QsUUFBUSxDQUFDVSxNQUFULENBQWdCO0FBQ3JCQyxVQUFBQSxVQUFVLEVBQUUsR0FEUztBQUVyQkMsVUFBQUEsSUFBSSxFQUFFO0FBQ0pDLFlBQUFBLEVBQUUsRUFBRSxLQURBO0FBRUpHLFlBQUFBLEtBQUssRUFBRUYsR0FBRyxDQUFDRztBQUZQO0FBRmUsU0FBaEIsQ0FBUDtBQU9EO0FBQ0YsS0EvSGlEOztBQUFBLHFEQWlJeEIsT0FDeEJuQixPQUR3QixFQUV4QkMsT0FGd0IsRUFHeEJDLFFBSHdCLEtBSThEO0FBQ3RGLFVBQUk7QUFDRixjQUFNO0FBQUVHLFVBQUFBLEtBQUY7QUFBU3lELFVBQUFBLEtBQUssR0FBRztBQUFqQixZQUEwQjdELE9BQU8sQ0FBQ2EsSUFBeEM7QUFDQSxjQUFNO0FBQUVOLFVBQUFBLGlCQUFpQixFQUFFQztBQUFyQixZQUF5QyxLQUFLVixRQUFMLENBQWNXLFFBQWQsQ0FBdUJULE9BQXZCLENBQS9DO0FBQ0EsY0FBTUcsTUFBbUIsR0FBRztBQUFFQyxVQUFBQSxLQUFLLEVBQUVBLEtBQUssQ0FBQzBELElBQU4sQ0FBVyxHQUFYO0FBQVQsU0FBNUI7QUFDQSxZQUFJRCxLQUFKLEVBQVcxRCxNQUFNLENBQUNVLElBQVAsR0FBYztBQUFFZ0QsVUFBQUE7QUFBRixTQUFkO0FBQ1gsY0FBTUUsYUFBNEIsR0FBRyxNQUFNdkQsZUFBZSxDQUFDLFdBQUQsRUFBY0wsTUFBZCxDQUExRDtBQUNBLGVBQU9GLFFBQVEsQ0FBQ1UsTUFBVCxDQUFnQjtBQUNyQkMsVUFBQUEsVUFBVSxFQUFFLEdBRFM7QUFFckJDLFVBQUFBLElBQUksRUFBRTtBQUNKQyxZQUFBQSxFQUFFLEVBQUUsSUFEQTtBQUVKYixZQUFBQSxRQUFRLEVBQUU7QUFDUitELGNBQUFBLFFBQVEsRUFBRUQsYUFBYSxDQUFDQyxRQURoQjtBQUVSQyxjQUFBQSxjQUFjLEVBQUVGLGFBQWEsQ0FBQ0csZUFGdEI7QUFHUjtBQUNBQyxjQUFBQSxhQUFhLEVBQUVKLGFBQWEsQ0FBQ0ssY0FBZCxHQUNYTCxhQUFhLENBQUNLLGNBQWQsQ0FBNkJDLEdBQTdCLENBQWtDQyxXQUFELEtBQWtCO0FBQ2pEaEMsZ0JBQUFBLFNBQVMsRUFBRWdDLFdBQVcsQ0FBQ0MsVUFEMEI7QUFFakRwQixnQkFBQUEsU0FBUyxFQUFFbUIsV0FBVyxDQUFDbEIsVUFGMEI7QUFHakROLGdCQUFBQSxNQUFNLEVBQUV3QixXQUFXLENBQUN4QjtBQUg2QixlQUFsQixDQUFqQyxDQURXLEdBTVg7QUFWSTtBQUZOO0FBRmUsU0FBaEIsQ0FBUDtBQWtCRCxPQXhCRCxDQXdCRSxPQUFPL0IsR0FBUCxFQUFZO0FBQ1pDLFFBQUFBLE9BQU8sQ0FBQ0MsS0FBUixDQUFjLG1FQUFkLEVBQW1GRixHQUFuRjtBQUNBLGVBQU9kLFFBQVEsQ0FBQ1UsTUFBVCxDQUFnQjtBQUNyQkMsVUFBQUEsVUFBVSxFQUFFLEdBRFM7QUFFckJDLFVBQUFBLElBQUksRUFBRTtBQUNKQyxZQUFBQSxFQUFFLEVBQUUsS0FEQTtBQUVKRyxZQUFBQSxLQUFLLEVBQUVGLEdBQUcsQ0FBQ0c7QUFGUDtBQUZlLFNBQWhCLENBQVA7QUFPRDtBQUNGLEtBeEtpRDs7QUFBQSwwQ0EwS25DLE9BQ2JuQixPQURhLEVBRWJDLE9BRmEsRUFHYkMsUUFIYSxLQUlvRTtBQUNqRixVQUFJO0FBQ0YsY0FBTTtBQUFFdUIsVUFBQUEsT0FBRjtBQUFXSyxVQUFBQSxRQUFYO0FBQXFCMkMsVUFBQUEsT0FBckI7QUFBOEJYLFVBQUFBO0FBQTlCLFlBQXdDN0QsT0FBTyxDQUFDYSxJQUF0RDtBQU1BLGNBQU07QUFBRU4sVUFBQUEsaUJBQWlCLEVBQUVDO0FBQXJCLFlBQXlDLEtBQUtWLFFBQUwsQ0FBY1csUUFBZCxDQUF1QlQsT0FBdkIsQ0FBL0M7QUFDQSxjQUFNRyxNQUFNLEdBQUc7QUFBRUMsVUFBQUEsS0FBSyxFQUFFb0IsT0FBTyxDQUFDc0MsSUFBUixDQUFhLEdBQWIsQ0FBVDtBQUE0QmpELFVBQUFBLElBQUksRUFBRTtBQUFFZ0MsWUFBQUEsU0FBUyxFQUFFaEIsUUFBYjtBQUF1QjJDLFlBQUFBLE9BQXZCO0FBQWdDWCxZQUFBQTtBQUFoQztBQUFsQyxTQUFmO0FBQ0EsY0FBTVksY0FBOEIsR0FBRyxNQUFNakUsZUFBZSxDQUFDLFlBQUQsRUFBZUwsTUFBZixDQUE1RDtBQUNBLGVBQU9GLFFBQVEsQ0FBQ1UsTUFBVCxDQUFnQjtBQUNyQkMsVUFBQUEsVUFBVSxFQUFFLEdBRFM7QUFFckJDLFVBQUFBLElBQUksRUFBRTtBQUNKQyxZQUFBQSxFQUFFLEVBQUUsSUFEQTtBQUVKYixZQUFBQSxRQUFRLEVBQUU7QUFDUitELGNBQUFBLFFBQVEsRUFBRVMsY0FBYyxDQUFDVCxRQURqQjtBQUVSQyxjQUFBQSxjQUFjLEVBQUVRLGNBQWMsQ0FBQ1AsZUFGdkI7QUFHUkMsY0FBQUEsYUFBYSxFQUFFTSxjQUFjLENBQUNMLGNBQWYsQ0FBOEJDLEdBQTlCLENBQW1DQyxXQUFELEtBQWtCO0FBQ2pFaEMsZ0JBQUFBLFNBQVMsRUFBRWdDLFdBQVcsQ0FBQ0MsVUFEMEM7QUFFakVwQixnQkFBQUEsU0FBUyxFQUFFbUIsV0FBVyxDQUFDbEIsVUFGMEM7QUFHakVOLGdCQUFBQSxNQUFNLEVBQUV3QixXQUFXLENBQUN4QjtBQUg2QyxlQUFsQixDQUFsQztBQUhQO0FBRk47QUFGZSxTQUFoQixDQUFQO0FBZUQsT0F6QkQsQ0F5QkUsT0FBTy9CLEdBQVAsRUFBWTtBQUNaQyxRQUFBQSxPQUFPLENBQUNDLEtBQVIsQ0FBYyx3REFBZCxFQUF3RUYsR0FBeEU7QUFDQSxlQUFPZCxRQUFRLENBQUNVLE1BQVQsQ0FBZ0I7QUFDckJDLFVBQUFBLFVBQVUsRUFBRSxHQURTO0FBRXJCQyxVQUFBQSxJQUFJLEVBQUU7QUFDSkMsWUFBQUEsRUFBRSxFQUFFLEtBREE7QUFFSkcsWUFBQUEsS0FBSyxFQUFFRixHQUFHLENBQUNHO0FBRlA7QUFGZSxTQUFoQixDQUFQO0FBT0Q7QUFDRixLQWxOaUQ7O0FBQUEsMENBb05uQyxPQUNibkIsT0FEYSxFQUViQyxPQUZhLEVBR2JDLFFBSGEsS0FJb0U7QUFDakYsVUFBSTtBQUNGLGNBQU07QUFBRXVCLFVBQUFBO0FBQUYsWUFBY3hCLE9BQU8sQ0FBQ2EsSUFBNUI7QUFDQSxjQUFNO0FBQUVOLFVBQUFBLGlCQUFpQixFQUFFQztBQUFyQixZQUF5QyxLQUFLVixRQUFMLENBQWNXLFFBQWQsQ0FBdUJULE9BQXZCLENBQS9DO0FBQ0EsY0FBTUcsTUFBTSxHQUFHO0FBQUVDLFVBQUFBLEtBQUssRUFBRW9CLE9BQU8sQ0FBQ3NDLElBQVIsQ0FBYSxHQUFiO0FBQVQsU0FBZjtBQUNBLGNBQU1ZLFdBQTJCLEdBQUcsTUFBTWxFLGVBQWUsQ0FBQyxZQUFELEVBQWVMLE1BQWYsQ0FBekQ7QUFDQSxlQUFPRixRQUFRLENBQUNVLE1BQVQsQ0FBZ0I7QUFDckJDLFVBQUFBLFVBQVUsRUFBRSxHQURTO0FBRXJCQyxVQUFBQSxJQUFJLEVBQUU7QUFDSkMsWUFBQUEsRUFBRSxFQUFFLElBREE7QUFFSmIsWUFBQUEsUUFBUSxFQUFFO0FBQ1IrRCxjQUFBQSxRQUFRLEVBQUVVLFdBQVcsQ0FBQ1YsUUFEZDtBQUVSQyxjQUFBQSxjQUFjLEVBQUVTLFdBQVcsQ0FBQ1IsZUFGcEI7QUFHUkMsY0FBQUEsYUFBYSxFQUFFTyxXQUFXLENBQUNOLGNBQVosQ0FBMkJDLEdBQTNCLENBQWdDQyxXQUFELEtBQWtCO0FBQzlEaEMsZ0JBQUFBLFNBQVMsRUFBRWdDLFdBQVcsQ0FBQ0MsVUFEdUM7QUFFOURwQixnQkFBQUEsU0FBUyxFQUFFbUIsV0FBVyxDQUFDbEIsVUFGdUM7QUFHOUROLGdCQUFBQSxNQUFNLEVBQUV3QixXQUFXLENBQUN4QjtBQUgwQyxlQUFsQixDQUEvQjtBQUhQO0FBRk47QUFGZSxTQUFoQixDQUFQO0FBZUQsT0FwQkQsQ0FvQkUsT0FBTy9CLEdBQVAsRUFBWTtBQUNaQyxRQUFBQSxPQUFPLENBQUNDLEtBQVIsQ0FBYyx3REFBZCxFQUF3RUYsR0FBeEU7QUFDQSxlQUFPZCxRQUFRLENBQUNVLE1BQVQsQ0FBZ0I7QUFDckJDLFVBQUFBLFVBQVUsRUFBRSxHQURTO0FBRXJCQyxVQUFBQSxJQUFJLEVBQUU7QUFDSkMsWUFBQUEsRUFBRSxFQUFFLEtBREE7QUFFSkcsWUFBQUEsS0FBSyxFQUFFRixHQUFHLENBQUNHO0FBRlA7QUFGZSxTQUFoQixDQUFQO0FBT0Q7QUFDRixLQXZQaUQ7O0FBQ2hELFNBQUtwQixRQUFMLEdBQWdCQSxRQUFoQjtBQUNELEdBTHNDLENBT3ZDOzs7QUFQdUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCBfIGZyb20gXCJsb2Rhc2hcIjtcbmltcG9ydCB7IFJlcXVlc3RQYXJhbXMgfSBmcm9tIFwiQGVsYXN0aWMvZWxhc3RpY3NlYXJjaFwiO1xuaW1wb3J0IHtcbiAgUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gIE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5LFxuICBJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZSxcbiAgSUxlZ2FjeUN1c3RvbUNsdXN0ZXJDbGllbnQsXG59IGZyb20gXCJvcGVuc2VhcmNoLWRhc2hib2FyZHMvc2VydmVyXCI7XG5pbXBvcnQgeyBJTkRFWCB9IGZyb20gXCIuLi91dGlscy9jb25zdGFudHNcIjtcbmltcG9ydCB7IGdldFNlYXJjaFN0cmluZywgdHJhbnNmb3JtTWFuYWdlZEluZGV4TWV0YURhdGEgfSBmcm9tIFwiLi4vdXRpbHMvaGVscGVyc1wiO1xuaW1wb3J0IHtcbiAgQ2hhbmdlUG9saWN5UmVzcG9uc2UsXG4gIEV4cGxhaW5BbGxSZXNwb25zZSxcbiAgRXhwbGFpbkFQSU1hbmFnZWRJbmRleE1ldGFEYXRhLFxuICBHZXRNYW5hZ2VkSW5kaWNlc1Jlc3BvbnNlLFxuICBSZW1vdmVQb2xpY3lSZXNwb25zZSxcbiAgUmVtb3ZlUmVzcG9uc2UsXG4gIFJldHJ5TWFuYWdlZEluZGV4UmVzcG9uc2UsXG4gIFJldHJ5UGFyYW1zLFxuICBSZXRyeVJlc3BvbnNlLFxuICBTZWFyY2hSZXNwb25zZSxcbn0gZnJvbSBcIi4uL21vZGVscy9pbnRlcmZhY2VzXCI7XG5pbXBvcnQgeyBNYW5hZ2VkSW5kaWNlc1NvcnQsIFNlcnZlclJlc3BvbnNlIH0gZnJvbSBcIi4uL21vZGVscy90eXBlc1wiO1xuaW1wb3J0IHsgTWFuYWdlZEluZGV4SXRlbSB9IGZyb20gXCIuLi8uLi9tb2RlbHMvaW50ZXJmYWNlc1wiO1xuaW1wb3J0IHsgZ2V0SW5kZXhUb0RhdGFTdHJlYW1NYXBwaW5nIH0gZnJvbSBcIi4vRGF0YVN0cmVhbVNlcnZpY2VcIjtcblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgTWFuYWdlZEluZGV4U2VydmljZSB7XG4gIG9zRHJpdmVyOiBJTGVnYWN5Q3VzdG9tQ2x1c3RlckNsaWVudDtcblxuICBjb25zdHJ1Y3Rvcihvc0RyaXZlcjogSUxlZ2FjeUN1c3RvbUNsdXN0ZXJDbGllbnQpIHtcbiAgICB0aGlzLm9zRHJpdmVyID0gb3NEcml2ZXI7XG4gIH1cblxuICAvLyBUT0RPOiBOb3QgZmluaXNoZWQsIG5lZWQgVUkgcGFnZSB0aGF0IHVzZXMgdGhpcyBmaXJzdFxuICBnZXRNYW5hZ2VkSW5kZXggPSBhc3luYyAoXG4gICAgY29udGV4dDogUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICByZXNwb25zZTogT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnlcbiAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxTZXJ2ZXJSZXNwb25zZTxhbnk+Pj4gPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGlkIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGlkOiBzdHJpbmcgfTtcbiAgICAgIGNvbnN0IHBhcmFtczogUmVxdWVzdFBhcmFtcy5HZXQgPSB7IGlkLCBpbmRleDogSU5ERVguT1BFTkRJU1RST19JU01fQ09ORklHIH07XG4gICAgICBjb25zdCB7IGNhbGxBc0N1cnJlbnRVc2VyOiBjYWxsV2l0aFJlcXVlc3QgfSA9IHRoaXMub3NEcml2ZXIuYXNTY29wZWQocmVxdWVzdCk7XG4gICAgICBjb25zdCByZXN1bHRzOiBTZWFyY2hSZXNwb25zZTxhbnk+ID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KFwic2VhcmNoXCIsIHBhcmFtcyk7XG4gICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgcmVzcG9uc2U6IHJlc3VsdHMsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUuZXJyb3IoXCJJbmRleCBNYW5hZ2VtZW50IC0gTWFuYWdlZEluZGV4U2VydmljZSAtIGdldE1hbmFnZWRJbmRleDpcIiwgZXJyKTtcbiAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgZXJyb3I6IGVyci5tZXNzYWdlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICB9O1xuXG4gIGdldE1hbmFnZWRJbmRpY2VzID0gYXN5bmMgKFxuICAgIGNvbnRleHQ6IFJlcXVlc3RIYW5kbGVyQ29udGV4dCxcbiAgICByZXF1ZXN0OiBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gICAgcmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8U2VydmVyUmVzcG9uc2U8R2V0TWFuYWdlZEluZGljZXNSZXNwb25zZT4+PiA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgZnJvbSwgc2l6ZSwgc29ydERpcmVjdGlvbiwgc29ydEZpZWxkLCB0ZXJtcywgaW5kaWNlcywgZGF0YVN0cmVhbXMsIHNob3dEYXRhU3RyZWFtcyB9ID0gcmVxdWVzdC5xdWVyeSBhcyB7XG4gICAgICAgIGZyb206IHN0cmluZztcbiAgICAgICAgc2l6ZTogc3RyaW5nO1xuICAgICAgICBzZWFyY2g6IHN0cmluZztcbiAgICAgICAgc29ydERpcmVjdGlvbjogc3RyaW5nO1xuICAgICAgICBzb3J0RmllbGQ6IHN0cmluZztcbiAgICAgICAgdGVybXM/OiBzdHJpbmdbXTtcbiAgICAgICAgaW5kaWNlcz86IHN0cmluZ1tdO1xuICAgICAgICBkYXRhU3RyZWFtcz86IHN0cmluZ1tdO1xuICAgICAgICBzaG93RGF0YVN0cmVhbXM6IGJvb2xlYW47XG4gICAgICB9O1xuXG4gICAgICBjb25zdCBtYW5hZ2VkSW5kZXhTb3J0czogTWFuYWdlZEluZGljZXNTb3J0ID0geyBpbmRleDogXCJtYW5hZ2VkX2luZGV4LmluZGV4XCIsIHBvbGljeUlkOiBcIm1hbmFnZWRfaW5kZXgucG9saWN5X2lkXCIgfTtcbiAgICAgIGNvbnN0IGV4cGxhaW5QYXJhbXMgPSB7XG4gICAgICAgIHNvcnRGaWVsZDogc29ydEZpZWxkID8gbWFuYWdlZEluZGV4U29ydHNbc29ydEZpZWxkXSA6IG51bGwsXG4gICAgICAgIHNvcnRPcmRlcjogc29ydERpcmVjdGlvbixcbiAgICAgICAgcXVlcnlTdHJpbmc6IGdldFNlYXJjaFN0cmluZyh0ZXJtcywgaW5kaWNlcywgZGF0YVN0cmVhbXMsIHNob3dEYXRhU3RyZWFtcyksXG4gICAgICAgIGZyb206IGZyb20sXG4gICAgICAgIHNpemU6IHNpemVcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IHsgY2FsbEFzQ3VycmVudFVzZXI6IGNhbGxXaXRoUmVxdWVzdCB9ID0gdGhpcy5vc0RyaXZlci5hc1Njb3BlZChyZXF1ZXN0KTtcbiAgICAgIGNvbnN0IFtleHBsYWluQWxsUmVzcG9uc2UsIGluZGV4VG9EYXRhU3RyZWFtTWFwcGluZ10gPSBhd2FpdCBQcm9taXNlLmFsbChbXG4gICAgICAgIGNhbGxXaXRoUmVxdWVzdChcImlzbS5leHBsYWluQWxsXCIsIGV4cGxhaW5QYXJhbXMpIGFzIFByb21pc2U8RXhwbGFpbkFsbFJlc3BvbnNlPixcbiAgICAgICAgZ2V0SW5kZXhUb0RhdGFTdHJlYW1NYXBwaW5nKHsgY2FsbEFzQ3VycmVudFVzZXI6IGNhbGxXaXRoUmVxdWVzdCB9KSxcbiAgICAgIF0pO1xuICAgICAgY29uc3QgbWFuYWdlZEluZGljZXM6IE1hbmFnZWRJbmRleEl0ZW1bXSA9IFtdO1xuICAgICAgZm9yIChjb25zdCBpbmRleE5hbWUgaW4gZXhwbGFpbkFsbFJlc3BvbnNlKSB7XG4gICAgICAgIGlmIChpbmRleE5hbWUgPT0gXCJ0b3RhbF9tYW5hZ2VkX2luZGljZXNcIikgY29udGludWU7XG4gICAgICAgIGNvbnN0IG1ldGFkYXRhID0gZXhwbGFpbkFsbFJlc3BvbnNlW2luZGV4TmFtZV0gYXMgRXhwbGFpbkFQSU1hbmFnZWRJbmRleE1ldGFEYXRhO1xuXG4gICAgICAgIC8vIElmIHNob3dEYXRhU3RyZWFtcyBpcyBub3QgdHJ1ZSwgdGhlbiBza2lwIHRoZSBtYW5hZ2VkIGluZGV4IGlmIGl0IGJlbG9uZ3MgdG8gYSBkYXRhIHN0cmVhbS5cbiAgICAgICAgY29uc3QgcGFyZW50RGF0YVN0cmVhbSA9IGluZGV4VG9EYXRhU3RyZWFtTWFwcGluZ1ttZXRhZGF0YS5pbmRleF0gfHwgbnVsbDtcbiAgICAgICAgaWYgKCFzaG93RGF0YVN0cmVhbXMgJiYgcGFyZW50RGF0YVN0cmVhbSAhPT0gbnVsbCkgY29udGludWU7XG4gICAgICAgIGxldCBwb2xpY3ksIHNlcU5vLCBwcmltYXJ5VGVybSwgZ2V0UmVzcG9uc2U7XG4gICAgICAgIHRyeSB7XG4gICAgICAgICAgZ2V0UmVzcG9uc2UgPSBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoXCJpc20uZ2V0UG9saWN5XCIsIHsgcG9saWN5SWQ6IG1ldGFkYXRhLnBvbGljeV9pZCB9KTtcbiAgICAgICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICAgICAgaWYgKGVyci5zdGF0dXNDb2RlID09PSA0MDQgJiYgZXJyLmJvZHkuZXJyb3IucmVhc29uID09PSBcIlBvbGljeSBub3QgZm91bmRcIikge1xuICAgICAgICAgICAgY29uc29sZS5sb2coXCJtYW5hZ2VkIGluZGV4IHdpdGggbm90IGV4aXN0aW5nIHBvbGljeVwiKTtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgdGhyb3cgZXJyO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBwb2xpY3kgPSBfLmdldChnZXRSZXNwb25zZSwgXCJwb2xpY3lcIiwgbnVsbCk7XG4gICAgICAgIHNlcU5vID0gXy5nZXQoZ2V0UmVzcG9uc2UsIFwiX3NlcV9ub1wiLCBudWxsKTtcbiAgICAgICAgcHJpbWFyeVRlcm0gPSBfLmdldChnZXRSZXNwb25zZSwgXCJfcHJpbWFyeV90ZXJtXCIsIG51bGwpO1xuICAgICAgICBtYW5hZ2VkSW5kaWNlcy5wdXNoKHtcbiAgICAgICAgICBpbmRleDogbWV0YWRhdGEuaW5kZXgsXG4gICAgICAgICAgaW5kZXhVdWlkOiBtZXRhZGF0YS5pbmRleF91dWlkLFxuICAgICAgICAgIGRhdGFTdHJlYW06IHBhcmVudERhdGFTdHJlYW0sXG4gICAgICAgICAgcG9saWN5SWQ6IG1ldGFkYXRhLnBvbGljeV9pZCxcbiAgICAgICAgICBwb2xpY3lTZXFObzogc2VxTm8sXG4gICAgICAgICAgcG9saWN5UHJpbWFyeVRlcm06IHByaW1hcnlUZXJtLFxuICAgICAgICAgIHBvbGljeTogcG9saWN5LFxuICAgICAgICAgIGVuYWJsZWQ6IG1ldGFkYXRhLmVuYWJsZWQsXG4gICAgICAgICAgbWFuYWdlZEluZGV4TWV0YURhdGE6IHRyYW5zZm9ybU1hbmFnZWRJbmRleE1ldGFEYXRhKG1ldGFkYXRhKSxcbiAgICAgICAgfSk7XG4gICAgICB9XG5cbiAgICAgIGxldCB0b3RhbE1hbmFnZWRJbmRpY2VzID0gZXhwbGFpbkFsbFJlc3BvbnNlLnRvdGFsX21hbmFnZWRfaW5kaWNlcztcbiAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZTogeyBtYW5hZ2VkSW5kaWNlczogbWFuYWdlZEluZGljZXMsIHRvdGFsTWFuYWdlZEluZGljZXM6IHRvdGFsTWFuYWdlZEluZGljZXN9LFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBpZiAoZXJyLnN0YXR1c0NvZGUgPT09IDQwNCAmJiBlcnIuYm9keS5lcnJvci50eXBlID09PSBcImluZGV4X25vdF9mb3VuZF9leGNlcHRpb25cIikge1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgICByZXNwb25zZTogeyBtYW5hZ2VkSW5kaWNlczogW10sIHRvdGFsTWFuYWdlZEluZGljZXM6IDAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICAgIGNvbnNvbGUuZXJyb3IoXCJJbmRleCBNYW5hZ2VtZW50IC0gTWFuYWdlZEluZGV4U2VydmljZSAtIGdldE1hbmFnZWRJbmRpY2VzXCIsIGVycik7XG4gICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICByZXRyeU1hbmFnZWRJbmRleFBvbGljeSA9IGFzeW5jIChcbiAgICBjb250ZXh0OiBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gICAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICAgIHJlc3BvbnNlOiBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeVxuICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPFNlcnZlclJlc3BvbnNlPFJldHJ5TWFuYWdlZEluZGV4UmVzcG9uc2U+Pj4gPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGluZGV4LCBzdGF0ZSA9IG51bGwgfSA9IHJlcXVlc3QuYm9keSBhcyB7IGluZGV4OiBzdHJpbmdbXTsgc3RhdGU/OiBzdHJpbmcgfTtcbiAgICAgIGNvbnN0IHsgY2FsbEFzQ3VycmVudFVzZXI6IGNhbGxXaXRoUmVxdWVzdCB9ID0gdGhpcy5vc0RyaXZlci5hc1Njb3BlZChyZXF1ZXN0KTtcbiAgICAgIGNvbnN0IHBhcmFtczogUmV0cnlQYXJhbXMgPSB7IGluZGV4OiBpbmRleC5qb2luKFwiLFwiKSB9O1xuICAgICAgaWYgKHN0YXRlKSBwYXJhbXMuYm9keSA9IHsgc3RhdGUgfTtcbiAgICAgIGNvbnN0IHJldHJ5UmVzcG9uc2U6IFJldHJ5UmVzcG9uc2UgPSBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoXCJpc20ucmV0cnlcIiwgcGFyYW1zKTtcbiAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZToge1xuICAgICAgICAgICAgZmFpbHVyZXM6IHJldHJ5UmVzcG9uc2UuZmFpbHVyZXMsXG4gICAgICAgICAgICB1cGRhdGVkSW5kaWNlczogcmV0cnlSZXNwb25zZS51cGRhdGVkX2luZGljZXMsXG4gICAgICAgICAgICAvLyBUT0RPOiByZW1vdmUgdGVybmFyeSBhZnRlciBmaXhpbmcgcmV0cnkgQVBJIHRvIHJldHVybiBlbXB0eSBhcnJheSBldmVuIGlmIG5vIGZhaWx1cmVzXG4gICAgICAgICAgICBmYWlsZWRJbmRpY2VzOiByZXRyeVJlc3BvbnNlLmZhaWxlZF9pbmRpY2VzXG4gICAgICAgICAgICAgID8gcmV0cnlSZXNwb25zZS5mYWlsZWRfaW5kaWNlcy5tYXAoKGZhaWxlZEluZGV4KSA9PiAoe1xuICAgICAgICAgICAgICAgICAgaW5kZXhOYW1lOiBmYWlsZWRJbmRleC5pbmRleF9uYW1lLFxuICAgICAgICAgICAgICAgICAgaW5kZXhVdWlkOiBmYWlsZWRJbmRleC5pbmRleF91dWlkLFxuICAgICAgICAgICAgICAgICAgcmVhc29uOiBmYWlsZWRJbmRleC5yZWFzb24sXG4gICAgICAgICAgICAgICAgfSkpXG4gICAgICAgICAgICAgIDogW10sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcihcIkluZGV4IE1hbmFnZW1lbnQgLSBNYW5hZ2VkSW5kZXhTZXJ2aWNlIC0gcmV0cnlNYW5hZ2VkSW5kZXhQb2xpY3k6XCIsIGVycik7XG4gICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBjaGFuZ2VQb2xpY3kgPSBhc3luYyAoXG4gICAgY29udGV4dDogUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICByZXNwb25zZTogT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnlcbiAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxTZXJ2ZXJSZXNwb25zZTxDaGFuZ2VQb2xpY3lSZXNwb25zZT4+PiA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgaW5kaWNlcywgcG9saWN5SWQsIGluY2x1ZGUsIHN0YXRlIH0gPSByZXF1ZXN0LmJvZHkgYXMge1xuICAgICAgICBpbmRpY2VzOiBzdHJpbmdbXTtcbiAgICAgICAgcG9saWN5SWQ6IHN0cmluZztcbiAgICAgICAgc3RhdGU6IHN0cmluZyB8IG51bGw7XG4gICAgICAgIGluY2x1ZGU6IHsgc3RhdGU6IHN0cmluZyB9W107XG4gICAgICB9O1xuICAgICAgY29uc3QgeyBjYWxsQXNDdXJyZW50VXNlcjogY2FsbFdpdGhSZXF1ZXN0IH0gPSB0aGlzLm9zRHJpdmVyLmFzU2NvcGVkKHJlcXVlc3QpO1xuICAgICAgY29uc3QgcGFyYW1zID0geyBpbmRleDogaW5kaWNlcy5qb2luKFwiLFwiKSwgYm9keTogeyBwb2xpY3lfaWQ6IHBvbGljeUlkLCBpbmNsdWRlLCBzdGF0ZSB9IH07XG4gICAgICBjb25zdCBjaGFuZ2VSZXNwb25zZTogUmVtb3ZlUmVzcG9uc2UgPSBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoXCJpc20uY2hhbmdlXCIsIHBhcmFtcyk7XG4gICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgcmVzcG9uc2U6IHtcbiAgICAgICAgICAgIGZhaWx1cmVzOiBjaGFuZ2VSZXNwb25zZS5mYWlsdXJlcyxcbiAgICAgICAgICAgIHVwZGF0ZWRJbmRpY2VzOiBjaGFuZ2VSZXNwb25zZS51cGRhdGVkX2luZGljZXMsXG4gICAgICAgICAgICBmYWlsZWRJbmRpY2VzOiBjaGFuZ2VSZXNwb25zZS5mYWlsZWRfaW5kaWNlcy5tYXAoKGZhaWxlZEluZGV4KSA9PiAoe1xuICAgICAgICAgICAgICBpbmRleE5hbWU6IGZhaWxlZEluZGV4LmluZGV4X25hbWUsXG4gICAgICAgICAgICAgIGluZGV4VXVpZDogZmFpbGVkSW5kZXguaW5kZXhfdXVpZCxcbiAgICAgICAgICAgICAgcmVhc29uOiBmYWlsZWRJbmRleC5yZWFzb24sXG4gICAgICAgICAgICB9KSksXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcihcIkluZGV4IE1hbmFnZW1lbnQgLSBNYW5hZ2VkSW5kZXhTZXJ2aWNlIC0gY2hhbmdlUG9saWN5OlwiLCBlcnIpO1xuICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgIHN0YXR1c0NvZGU6IDIwMCxcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICBlcnJvcjogZXJyLm1lc3NhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgcmVtb3ZlUG9saWN5ID0gYXN5bmMgKFxuICAgIGNvbnRleHQ6IFJlcXVlc3RIYW5kbGVyQ29udGV4dCxcbiAgICByZXF1ZXN0OiBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gICAgcmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8U2VydmVyUmVzcG9uc2U8UmVtb3ZlUG9saWN5UmVzcG9uc2U+Pj4gPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGluZGljZXMgfSA9IHJlcXVlc3QuYm9keSBhcyB7IGluZGljZXM6IHN0cmluZ1tdIH07XG4gICAgICBjb25zdCB7IGNhbGxBc0N1cnJlbnRVc2VyOiBjYWxsV2l0aFJlcXVlc3QgfSA9IHRoaXMub3NEcml2ZXIuYXNTY29wZWQocmVxdWVzdCk7XG4gICAgICBjb25zdCBwYXJhbXMgPSB7IGluZGV4OiBpbmRpY2VzLmpvaW4oXCIsXCIpIH07XG4gICAgICBjb25zdCBhZGRSZXNwb25zZTogUmVtb3ZlUmVzcG9uc2UgPSBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoXCJpc20ucmVtb3ZlXCIsIHBhcmFtcyk7XG4gICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgcmVzcG9uc2U6IHtcbiAgICAgICAgICAgIGZhaWx1cmVzOiBhZGRSZXNwb25zZS5mYWlsdXJlcyxcbiAgICAgICAgICAgIHVwZGF0ZWRJbmRpY2VzOiBhZGRSZXNwb25zZS51cGRhdGVkX2luZGljZXMsXG4gICAgICAgICAgICBmYWlsZWRJbmRpY2VzOiBhZGRSZXNwb25zZS5mYWlsZWRfaW5kaWNlcy5tYXAoKGZhaWxlZEluZGV4KSA9PiAoe1xuICAgICAgICAgICAgICBpbmRleE5hbWU6IGZhaWxlZEluZGV4LmluZGV4X25hbWUsXG4gICAgICAgICAgICAgIGluZGV4VXVpZDogZmFpbGVkSW5kZXguaW5kZXhfdXVpZCxcbiAgICAgICAgICAgICAgcmVhc29uOiBmYWlsZWRJbmRleC5yZWFzb24sXG4gICAgICAgICAgICB9KSksXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcihcIkluZGV4IE1hbmFnZW1lbnQgLSBNYW5hZ2VkSW5kZXhTZXJ2aWNlIC0gcmVtb3ZlUG9saWN5OlwiLCBlcnIpO1xuICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgIHN0YXR1c0NvZGU6IDIwMCxcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICBlcnJvcjogZXJyLm1lc3NhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG59XG4iXX0=