"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _cryptiles = require("@hapi/cryptiles");

var _querystring = require("querystring");

var _helper = require("./helper");

var _next_url = require("../../../utils/next_url");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class OpenIdAuthRoutes {
  constructor(router, config, sessionStorageFactory, openIdAuthConfig, securityClient, core, wreckClient) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.openIdAuthConfig = openIdAuthConfig;
    this.securityClient = securityClient;
    this.core = core;
    this.wreckClient = wreckClient;
  }

  redirectToLogin(request, response) {
    this.sessionStorageFactory.asScoped(request).clear();
    return response.redirected({
      headers: {
        location: `${this.core.http.basePath.serverBasePath}/auth/openid/login`
      }
    });
  }

  setupRoutes() {
    this.router.get({
      path: `/auth/openid/login`,
      validate: {
        query: _configSchema.schema.object({
          code: _configSchema.schema.maybe(_configSchema.schema.string()),
          nextUrl: _configSchema.schema.maybe(_configSchema.schema.string({
            validate: _next_url.validateNextUrl
          })),
          state: _configSchema.schema.maybe(_configSchema.schema.string()),
          refresh: _configSchema.schema.maybe(_configSchema.schema.string())
        }, {
          unknowns: 'allow'
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$openid2, _this$config$openid3;

      // implementation refers to https://github.com/hapijs/bell/blob/master/lib/oauth.js
      // Sign-in initialization
      if (!request.query.code) {
        var _this$config$openid;

        const nonce = (0, _cryptiles.randomString)(OpenIdAuthRoutes.NONCE_LENGTH);
        const query = {
          client_id: (_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.client_id,
          response_type: 'code',
          redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}/auth/openid/login`,
          state: nonce,
          scope: this.openIdAuthConfig.scope
        };
        const queryString = (0, _querystring.stringify)(query);
        const location = `${this.openIdAuthConfig.authorizationEndpoint}?${queryString}`;
        const cookie = {
          oidc: {
            state: nonce,
            nextUrl: request.query.nextUrl || '/'
          }
        };
        this.sessionStorageFactory.asScoped(request).set(cookie);
        return response.redirected({
          headers: {
            location
          }
        });
      } // Authentication callback
      // validate state first


      let cookie;

      try {
        var _cookie$oidc;

        cookie = await this.sessionStorageFactory.asScoped(request).get();

        if (!cookie || !((_cookie$oidc = cookie.oidc) !== null && _cookie$oidc !== void 0 && _cookie$oidc.state) || cookie.oidc.state !== request.query.state) {
          return this.redirectToLogin(request, response);
        }
      } catch (error) {
        return this.redirectToLogin(request, response);
      }

      const nextUrl = cookie.oidc.nextUrl;
      const clientId = (_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.client_id;
      const clientSecret = (_this$config$openid3 = this.config.openid) === null || _this$config$openid3 === void 0 ? void 0 : _this$config$openid3.client_secret;
      const query = {
        grant_type: 'authorization_code',
        code: request.query.code,
        redirect_uri: `${(0, _helper.getBaseRedirectUrl)(this.config, this.core, request)}/auth/openid/login`,
        client_id: clientId,
        client_secret: clientSecret
      };

      try {
        var _this$config$openid4;

        const tokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);
        const user = await this.securityClient.authenticateWithHeader(request, this.openIdAuthConfig.authHeaderName, `Bearer ${tokenResponse.idToken}`); // set to cookie

        const sessionStorage = {
          username: user.username,
          credentials: {
            authHeaderValue: `Bearer ${tokenResponse.idToken}`,
            expires_at: (0, _helper.getExpirationDate)(tokenResponse)
          },
          authType: 'openid',
          expiryTime: Date.now() + this.config.session.ttl
        };

        if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.refresh_tokens && tokenResponse.refreshToken) {
          Object.assign(sessionStorage.credentials, {
            refresh_token: tokenResponse.refreshToken
          });
        }

        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: nextUrl
          }
        });
      } catch (error) {
        context.security_plugin.logger.error(`OpenId authentication failed: ${error}`);

        if (error.toString().toLowerCase().includes('authentication exception')) {
          return response.unauthorized();
        } else {
          return this.redirectToLogin(request, response);
        }
      }
    });
    this.router.get({
      path: `/auth/logout`,
      validate: false
    }, async (context, request, response) => {
      var _this$config$openid5;

      const cookie = await this.sessionStorageFactory.asScoped(request).get();
      this.sessionStorageFactory.asScoped(request).clear(); // authHeaderValue is the bearer header, e.g. "Bearer <auth_token>"

      const token = cookie === null || cookie === void 0 ? void 0 : cookie.credentials.authHeaderValue.split(' ')[1]; // get auth token

      const logoutQueryParams = {
        post_logout_redirect_uri: (0, _helper.getBaseRedirectUrl)(this.config, this.core, request),
        id_token_hint: token
      };
      const endSessionUrl = (0, _helper.composeLogoutUrl)((_this$config$openid5 = this.config.openid) === null || _this$config$openid5 === void 0 ? void 0 : _this$config$openid5.logout_url, this.openIdAuthConfig.endSessionEndpoint, logoutQueryParams);
      return response.redirected({
        headers: {
          location: endSessionUrl
        }
      });
    });
  }

}

exports.OpenIdAuthRoutes = OpenIdAuthRoutes;

_defineProperty(OpenIdAuthRoutes, "NONCE_LENGTH", 22);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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