<?php

declare(strict_types=1);

/*
 * Copyright (c) 2017-2023 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\OAuth\Server\Http;

use fkooman\OAuth\Server\Exception\InvalidRequestException;
use fkooman\OAuth\Server\Exception\InvalidScopeException;

class Query
{
    private array $getData;

    public function __construct(array $getData)
    {
        $this->getData = $getData;
    }

    public function clientId(): string
    {
        $clientId = $this->requireParameter('client_id');

        // client-id  = *VSCHAR
        // VSCHAR     = %x20-7E
        if (1 !== preg_match('/^[\x20-\x7E]+$/', $clientId)) {
            throw new InvalidRequestException('invalid "client_id"');
        }

        return $clientId;
    }

    public function codeChallenge(): string
    {
        $codeChallenge = $this->requireParameter('code_challenge');

        // it seems the length of the codeChallenge is always 43 because it is
        // the output of the SHA256 hashing algorithm
        if (1 !== preg_match('/^[\x41-\x5A\x61-\x7A\x30-\x39-_]{43}$/', $codeChallenge)) {
            throw new InvalidRequestException('invalid "code_challenge"');
        }

        return $codeChallenge;
    }

    public function redirectUri(): string
    {
        // NOTE: no need to validate the redirect_uri, as we do strict string
        // comparison
        return $this->requireParameter('redirect_uri');
    }

    public function scope(): string
    {
        $scope = $this->requireParameter('scope');

        // scope       = scope-token *( SP scope-token )
        // scope-token = 1*NQCHAR
        // NQCHAR      = %x21 / %x23-5B / %x5D-7E
        foreach (explode(' ', $scope) as $scopeToken) {
            if (1 !== preg_match('/^[\x21\x23-\x5B\x5D-\x7E]+$/', $scopeToken)) {
                throw new InvalidScopeException('invalid "scope"');
            }
        }

        return $scope;
    }

    public function state(): string
    {
        $state = $this->requireParameter('state');

        // state      = 1*VSCHAR
        // VSCHAR     = %x20-7E
        if (1 !== preg_match('/^[\x20-\x7E]+$/', $state)) {
            throw new InvalidRequestException('invalid "state"');
        }

        return $state;
    }

    private function requireParameter(string $parameterName): string
    {
        if (!\array_key_exists($parameterName, $this->getData)) {
            throw new InvalidRequestException(sprintf('missing "%s" parameter', $parameterName));
        }

        if (!\is_string($this->getData[$parameterName])) {
            throw new InvalidRequestException(sprintf('"%s" parameter MUST be string', $parameterName));
        }

        return $this->getData[$parameterName];
    }
}
