/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.hnsw;

import java.io.IOException;
import java.util.Locale;
import java.util.Objects;
import java.util.SplittableRandom;
import org.apache.lucene.index.RandomAccessVectorValues;
import org.apache.lucene.index.RandomAccessVectorValuesProducer;
import org.apache.lucene.index.VectorSimilarityFunction;
import org.apache.lucene.util.FixedBitSet;
import org.apache.lucene.util.InfoStream;
import org.apache.lucene.util.hnsw.HnswGraphSearcher;
import org.apache.lucene.util.hnsw.NeighborArray;
import org.apache.lucene.util.hnsw.NeighborQueue;
import org.apache.lucene.util.hnsw.OnHeapHnswGraph;

public final class HnswGraphBuilder {
    private static final long DEFAULT_RAND_SEED = 42L;
    public static final String HNSW_COMPONENT = "HNSW";
    public static long randSeed = 42L;
    private final int M;
    private final int beamWidth;
    private final double ml;
    private final NeighborArray scratch;
    private final VectorSimilarityFunction similarityFunction;
    private final RandomAccessVectorValues vectorValues;
    private final SplittableRandom random;
    private final HnswGraphSearcher graphSearcher;
    final OnHeapHnswGraph hnsw;
    private InfoStream infoStream = InfoStream.getDefault();
    private RandomAccessVectorValues buildVectors;

    public HnswGraphBuilder(RandomAccessVectorValuesProducer vectors, VectorSimilarityFunction similarityFunction, int M, int beamWidth, long seed) throws IOException {
        this.vectorValues = vectors.randomAccess();
        this.buildVectors = vectors.randomAccess();
        this.similarityFunction = Objects.requireNonNull(similarityFunction);
        if (M <= 0) {
            throw new IllegalArgumentException("maxConn must be positive");
        }
        if (beamWidth <= 0) {
            throw new IllegalArgumentException("beamWidth must be positive");
        }
        this.M = M;
        this.beamWidth = beamWidth;
        this.ml = 1.0 / Math.log(1.0 * (double)M);
        this.random = new SplittableRandom(seed);
        int levelOfFirstNode = HnswGraphBuilder.getRandomGraphLevel(this.ml, this.random);
        this.hnsw = new OnHeapHnswGraph(M, levelOfFirstNode);
        this.graphSearcher = new HnswGraphSearcher(similarityFunction, new NeighborQueue(beamWidth, true), new FixedBitSet(this.vectorValues.size()));
        this.scratch = new NeighborArray(Math.max(beamWidth, M + 1), false);
    }

    public OnHeapHnswGraph build(RandomAccessVectorValues vectors) throws IOException {
        long start;
        if (vectors == this.vectorValues) {
            throw new IllegalArgumentException("Vectors to build must be independent of the source of vectors provided to HnswGraphBuilder()");
        }
        if (this.infoStream.isEnabled(HNSW_COMPONENT)) {
            this.infoStream.message(HNSW_COMPONENT, "build graph from " + vectors.size() + " vectors");
        }
        long t = start = System.nanoTime();
        for (int node = 1; node < vectors.size(); ++node) {
            this.addGraphNode(node, vectors.vectorValue(node));
            if (node % 10000 != 0 || !this.infoStream.isEnabled(HNSW_COMPONENT)) continue;
            t = this.printGraphBuildStatus(node, start, t);
        }
        return this.hnsw;
    }

    public void setInfoStream(InfoStream infoStream) {
        this.infoStream = infoStream;
    }

    void addGraphNode(int node, float[] value) throws IOException {
        NeighborQueue candidates;
        int level;
        int nodeLevel = HnswGraphBuilder.getRandomGraphLevel(this.ml, this.random);
        int curMaxLevel = this.hnsw.numLevels() - 1;
        int[] eps = new int[]{this.hnsw.entryNode()};
        for (level = nodeLevel; level > curMaxLevel; --level) {
            this.hnsw.addNode(level, node);
        }
        for (level = curMaxLevel; level > nodeLevel; --level) {
            candidates = this.graphSearcher.searchLevel(value, 1, level, eps, this.vectorValues, this.hnsw);
            eps = new int[]{candidates.pop()};
        }
        for (level = Math.min(nodeLevel, curMaxLevel); level >= 0; --level) {
            candidates = this.graphSearcher.searchLevel(value, this.beamWidth, level, eps, this.vectorValues, this.hnsw);
            eps = candidates.nodes();
            this.hnsw.addNode(level, node);
            this.addDiverseNeighbors(level, node, candidates);
        }
    }

    private long printGraphBuildStatus(int node, long start, long t) {
        long now = System.nanoTime();
        this.infoStream.message(HNSW_COMPONENT, String.format(Locale.ROOT, "built %d in %d/%d ms", node, (now - t) / 1000000L, (now - start) / 1000000L));
        return now;
    }

    private void addDiverseNeighbors(int level, int node, NeighborQueue candidates) throws IOException {
        NeighborArray neighbors = this.hnsw.getNeighbors(level, node);
        assert (neighbors.size() == 0);
        this.popToScratch(candidates);
        int maxConnOnLevel = level == 0 ? this.M * 2 : this.M;
        this.selectAndLinkDiverse(neighbors, this.scratch, maxConnOnLevel);
        int size = neighbors.size();
        for (int i = 0; i < size; ++i) {
            int nbr = neighbors.node[i];
            NeighborArray nbrNbr = this.hnsw.getNeighbors(level, nbr);
            nbrNbr.insertSorted(node, neighbors.score[i]);
            if (nbrNbr.size() <= maxConnOnLevel) continue;
            int indexToRemove = this.findWorstNonDiverse(nbrNbr);
            nbrNbr.removeIndex(indexToRemove);
        }
    }

    private void selectAndLinkDiverse(NeighborArray neighbors, NeighborArray candidates, int maxConnOnLevel) throws IOException {
        for (int i = candidates.size() - 1; neighbors.size() < maxConnOnLevel && i >= 0; --i) {
            int cNode = candidates.node[i];
            float cScore = candidates.score[i];
            assert (cNode < this.hnsw.size());
            if (!this.diversityCheck(this.vectorValues.vectorValue(cNode), cScore, neighbors, this.buildVectors)) continue;
            neighbors.add(cNode, cScore);
        }
    }

    private void popToScratch(NeighborQueue candidates) {
        this.scratch.clear();
        int candidateCount = candidates.size();
        for (int i = 0; i < candidateCount; ++i) {
            float maxSimilarity = candidates.topScore();
            this.scratch.add(candidates.pop(), maxSimilarity);
        }
    }

    private boolean diversityCheck(float[] candidate, float score, NeighborArray neighbors, RandomAccessVectorValues vectorValues) throws IOException {
        for (int i = 0; i < neighbors.size(); ++i) {
            float neighborSimilarity = this.similarityFunction.compare(candidate, vectorValues.vectorValue(neighbors.node[i]));
            if (!(neighborSimilarity >= score)) continue;
            return false;
        }
        return true;
    }

    private int findWorstNonDiverse(NeighborArray neighbors) throws IOException {
        for (int i = neighbors.size() - 1; i > 0; --i) {
            int cNode = neighbors.node[i];
            float[] cVector = this.vectorValues.vectorValue(cNode);
            float minAcceptedSimilarity = neighbors.score[i];
            for (int j = i - 1; j >= 0; --j) {
                float neighborSimilarity = this.similarityFunction.compare(cVector, this.buildVectors.vectorValue(neighbors.node[j]));
                if (!(neighborSimilarity >= minAcceptedSimilarity)) continue;
                return i;
            }
        }
        return neighbors.size() - 1;
    }

    private static int getRandomGraphLevel(double ml, SplittableRandom random) {
        double randDouble;
        while ((randDouble = random.nextDouble()) == 0.0) {
        }
        return (int)(-Math.log(randDouble) * ml);
    }
}

