"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthentication = void 0;

var _querystring = require("querystring");

var _security_cookie = require("../../../session/security_cookie");

var _routes = require("./routes");

var _authentication_type = require("../authentication_type");

var _common = require("../../../../common");

var _cookie_splitter = require("../../../session/cookie_splitter");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class SamlAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);

    _defineProperty(this, "type", 'saml');

    _defineProperty(this, "redirectSAMlCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
  }

  generateNextUrl(request) {
    const path = this.coreSetup.http.basePath.serverBasePath + (request.url.pathname || '/app/opensearch-dashboards');
    return (0, _querystring.escape)(path);
  } // Check if we can get the previous tenant information from the expired cookie.


  async init() {
    this.createExtraStorage();
    const samlAuthRoutes = new _routes.SamlAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.securityClient, this.coreSetup);
    samlAuthRoutes.setupRoutes();
  }

  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.saml.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie

    };

    for (let i = 1; i <= this.config.saml.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }

  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.saml.extra_storage.cookie_prefix,
      additionalCookies: this.config.saml.extra_storage.additional_cookies,
      logger
    };
  }

  requestIncludesAuthInfo(request) {
    return request.headers[SamlAuthentication.AUTH_HEADER_NAME] ? true : false;
  }

  async getAdditionalAuthHeader(request) {
    return {};
  }

  getCookie(request, authInfo) {
    const authorizationHeaderValue = request.headers[SamlAuthentication.AUTH_HEADER_NAME];
    (0, _cookie_splitter.setExtraAuthStorage)(request, authorizationHeaderValue, this.getExtraAuthStorageOptions(this.logger));
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: _common.AuthType.SAML,
      expiryTime: Date.now() + this.config.session.ttl
    };
  } // Can be improved to check if the token is expiring.


  async isValidCookie(cookie, request) {
    var _cookie$credentials;

    return cookie.authType === _common.AuthType.SAML && cookie.username && cookie.expiryTime && (((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue) || this.getExtraAuthStorageValue(request, cookie));
  }

  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectSAMlCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }

  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;

    let extraValue = '';

    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }

    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions(this.logger));
    } catch (error) {
      this.logger.info(error);
    }

    return extraValue;
  }

  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;

    const headers = {};

    if ((_cookie$credentials3 = cookie.credentials) !== null && _cookie$credentials3 !== void 0 && _cookie$credentials3.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        headers[SamlAuthentication.AUTH_HEADER_NAME] = extraAuthStorageValue;
      } catch (error) {
        this.logger.error(error); // @todo Re-throw?
        // throw error;
      }
    } else {
      var _cookie$credentials4;

      headers[SamlAuthentication.AUTH_HEADER_NAME] = (_cookie$credentials4 = cookie.credentials) === null || _cookie$credentials4 === void 0 ? void 0 : _cookie$credentials4.authHeaderValue;
    }

    return headers;
  }

}

exports.SamlAuthentication = SamlAuthentication;

_defineProperty(SamlAuthentication, "AUTH_HEADER_NAME", 'authorization');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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