/*
 * Decompiled with CFR 0.152.
 */
package org.traccar.session;

import io.netty.channel.Channel;
import io.netty.util.Timeout;
import io.netty.util.Timer;
import jakarta.inject.Inject;
import jakarta.inject.Singleton;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.traccar.Protocol;
import org.traccar.broadcast.BroadcastInterface;
import org.traccar.broadcast.BroadcastService;
import org.traccar.config.Config;
import org.traccar.config.Keys;
import org.traccar.database.DeviceLookupService;
import org.traccar.database.NotificationManager;
import org.traccar.model.BaseModel;
import org.traccar.model.Device;
import org.traccar.model.Event;
import org.traccar.model.LogRecord;
import org.traccar.model.Position;
import org.traccar.model.User;
import org.traccar.session.DeviceSession;
import org.traccar.session.cache.CacheManager;
import org.traccar.storage.Storage;
import org.traccar.storage.StorageException;
import org.traccar.storage.query.Columns;
import org.traccar.storage.query.Condition;
import org.traccar.storage.query.Request;

@Singleton
public class ConnectionManager
implements BroadcastInterface {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionManager.class);
    private final long deviceTimeout;
    private final boolean showUnknownDevices;
    private final Map<Long, DeviceSession> sessionsByDeviceId = new ConcurrentHashMap<Long, DeviceSession>();
    private final Map<SocketAddress, Map<String, DeviceSession>> sessionsByEndpoint = new ConcurrentHashMap<SocketAddress, Map<String, DeviceSession>>();
    private final Map<SocketAddress, String> unknownByEndpoint = new ConcurrentHashMap<SocketAddress, String>();
    private final Config config;
    private final CacheManager cacheManager;
    private final Storage storage;
    private final NotificationManager notificationManager;
    private final Timer timer;
    private final BroadcastService broadcastService;
    private final DeviceLookupService deviceLookupService;
    private final Map<Long, Set<UpdateListener>> listeners = new HashMap<Long, Set<UpdateListener>>();
    private final Map<Long, Set<Long>> userDevices = new HashMap<Long, Set<Long>>();
    private final Map<Long, Set<Long>> deviceUsers = new HashMap<Long, Set<Long>>();
    private final Map<Long, Timeout> timeouts = new ConcurrentHashMap<Long, Timeout>();

    @Inject
    public ConnectionManager(Config config, CacheManager cacheManager, Storage storage, NotificationManager notificationManager, Timer timer, BroadcastService broadcastService, DeviceLookupService deviceLookupService) {
        this.config = config;
        this.cacheManager = cacheManager;
        this.storage = storage;
        this.notificationManager = notificationManager;
        this.timer = timer;
        this.broadcastService = broadcastService;
        this.deviceLookupService = deviceLookupService;
        this.deviceTimeout = config.getLong(Keys.STATUS_TIMEOUT);
        this.showUnknownDevices = config.getBoolean(Keys.WEB_SHOW_UNKNOWN_DEVICES);
        broadcastService.registerListener(this);
    }

    public DeviceSession getDeviceSession(long deviceId) {
        return this.sessionsByDeviceId.get(deviceId);
    }

    public DeviceSession getDeviceSession(Protocol protocol, Channel channel, SocketAddress remoteAddress, String ... uniqueIds) throws Exception {
        Map endpointSessions = this.sessionsByEndpoint.getOrDefault(remoteAddress, new ConcurrentHashMap());
        if ((uniqueIds = (String[])Arrays.stream(uniqueIds).filter(Objects::nonNull).toArray(String[]::new)).length > 0) {
            for (String uniqueId : uniqueIds) {
                DeviceSession deviceSession = (DeviceSession)endpointSessions.get(uniqueId);
                if (deviceSession == null) continue;
                return deviceSession;
            }
        } else {
            return endpointSessions.values().stream().findAny().orElse(null);
        }
        Device device = this.deviceLookupService.lookup(uniqueIds);
        String firstUniqueId = uniqueIds[0];
        if (device == null && this.config.getBoolean(Keys.DATABASE_REGISTER_UNKNOWN) && firstUniqueId.matches(this.config.getString(Keys.DATABASE_REGISTER_UNKNOWN_REGEX))) {
            device = this.addUnknownDevice(firstUniqueId);
        }
        if (device != null) {
            this.unknownByEndpoint.remove(remoteAddress);
            device.checkDisabled();
            DeviceSession oldSession = this.sessionsByDeviceId.remove(device.getId());
            if (oldSession != null) {
                Map<String, DeviceSession> oldEndpointSessions = this.sessionsByEndpoint.get(oldSession.getRemoteAddress());
                if (oldEndpointSessions != null && oldEndpointSessions.size() > 1) {
                    oldEndpointSessions.remove(device.getUniqueId());
                } else {
                    this.sessionsByEndpoint.remove(oldSession.getRemoteAddress());
                }
            }
            DeviceSession deviceSession = new DeviceSession(device.getId(), device.getUniqueId(), protocol, channel, remoteAddress);
            endpointSessions.put(device.getUniqueId(), deviceSession);
            this.sessionsByEndpoint.put(remoteAddress, endpointSessions);
            this.sessionsByDeviceId.put(device.getId(), deviceSession);
            if (oldSession == null) {
                this.cacheManager.addDevice(device.getId());
            }
            return deviceSession;
        }
        this.unknownByEndpoint.put(remoteAddress, firstUniqueId);
        LOGGER.warn("Unknown device - " + String.join((CharSequence)" ", uniqueIds) + " (" + ((InetSocketAddress)remoteAddress).getHostString() + ")");
        return null;
    }

    private Device addUnknownDevice(String uniqueId) {
        Device device = new Device();
        device.setName(uniqueId);
        device.setUniqueId(uniqueId);
        device.setCategory(this.config.getString(Keys.DATABASE_REGISTER_UNKNOWN_DEFAULT_CATEGORY));
        long defaultGroupId = this.config.getLong(Keys.DATABASE_REGISTER_UNKNOWN_DEFAULT_GROUP_ID);
        if (defaultGroupId != 0L) {
            device.setGroupId(defaultGroupId);
        }
        try {
            device.setId(this.storage.addObject(device, new Request(new Columns.Exclude("id"))));
            LOGGER.info("Automatically registered " + uniqueId);
            return device;
        }
        catch (StorageException e) {
            LOGGER.warn("Automatic registration failed", (Throwable)e);
            return null;
        }
    }

    public void deviceDisconnected(Channel channel, boolean supportsOffline) {
        SocketAddress remoteAddress = channel.remoteAddress();
        Map<String, DeviceSession> endpointSessions = this.sessionsByEndpoint.remove(remoteAddress);
        if (endpointSessions != null) {
            for (DeviceSession deviceSession : endpointSessions.values()) {
                if (supportsOffline) {
                    this.updateDevice(deviceSession.getDeviceId(), "offline", null);
                }
                this.sessionsByDeviceId.remove(deviceSession.getDeviceId());
                this.cacheManager.removeDevice(deviceSession.getDeviceId());
            }
        }
        this.unknownByEndpoint.remove(remoteAddress);
    }

    public void deviceUnknown(long deviceId) {
        this.updateDevice(deviceId, "unknown", null);
        this.removeDeviceSession(deviceId);
    }

    private void removeDeviceSession(long deviceId) {
        DeviceSession deviceSession = this.sessionsByDeviceId.remove(deviceId);
        if (deviceSession != null) {
            this.cacheManager.removeDevice(deviceId);
            this.sessionsByEndpoint.computeIfPresent(deviceSession.getRemoteAddress(), (e, sessions) -> {
                sessions.remove(deviceSession.getUniqueId());
                return sessions.isEmpty() ? null : sessions;
            });
        }
    }

    public void updateDevice(long deviceId, String status, Date time) {
        Timeout timeout;
        Device device = this.cacheManager.getObject(Device.class, deviceId);
        if (device == null) {
            try {
                device = this.storage.getObject(Device.class, new Request((Columns)new Columns.All(), new Condition.Equals("id", deviceId)));
            }
            catch (StorageException e) {
                LOGGER.warn("Failed to get device", (Throwable)e);
            }
            if (device == null) {
                return;
            }
        }
        String oldStatus = device.getStatus();
        device.setStatus(status);
        if (!status.equals(oldStatus)) {
            String eventType;
            HashMap<Event, Position> events = new HashMap<Event, Position>();
            switch (status) {
                case "online": {
                    eventType = "deviceOnline";
                    break;
                }
                case "unknown": {
                    eventType = "deviceUnknown";
                    break;
                }
                default: {
                    eventType = "deviceOffline";
                }
            }
            events.put(new Event(eventType, deviceId), null);
            this.notificationManager.updateEvents(events);
        }
        if (time != null) {
            device.setLastUpdate(time);
        }
        if ((timeout = this.timeouts.remove(deviceId)) != null) {
            timeout.cancel();
        }
        if (status.equals("online")) {
            this.timeouts.put(deviceId, this.timer.newTimeout(timeout1 -> {
                if (!timeout1.isCancelled()) {
                    this.deviceUnknown(deviceId);
                }
            }, this.deviceTimeout, TimeUnit.SECONDS));
        }
        try {
            this.storage.updateObject(device, new Request((Columns)new Columns.Include("status", "lastUpdate"), new Condition.Equals("id", deviceId)));
        }
        catch (StorageException e) {
            LOGGER.warn("Update device status error", (Throwable)e);
        }
        this.updateDevice(true, device);
    }

    public synchronized void sendKeepalive() {
        for (Set<UpdateListener> userListeners : this.listeners.values()) {
            for (UpdateListener listener : userListeners) {
                listener.onKeepalive();
            }
        }
    }

    @Override
    public synchronized void updateDevice(boolean local, Device device) {
        if (local) {
            this.broadcastService.updateDevice(true, device);
        } else if ("online".equals(device.getStatus())) {
            this.timeouts.remove(device.getId());
            this.removeDeviceSession(device.getId());
        }
        Iterator iterator = this.deviceUsers.getOrDefault(device.getId(), Collections.emptySet()).iterator();
        while (iterator.hasNext()) {
            long userId = (Long)iterator.next();
            if (!this.listeners.containsKey(userId)) continue;
            for (UpdateListener listener : this.listeners.get(userId)) {
                listener.onUpdateDevice(device);
            }
        }
    }

    @Override
    public synchronized void updatePosition(boolean local, Position position) {
        if (local) {
            this.broadcastService.updatePosition(true, position);
        }
        Iterator iterator = this.deviceUsers.getOrDefault(position.getDeviceId(), Collections.emptySet()).iterator();
        while (iterator.hasNext()) {
            long userId = (Long)iterator.next();
            if (!this.listeners.containsKey(userId)) continue;
            for (UpdateListener listener : this.listeners.get(userId)) {
                listener.onUpdatePosition(position);
            }
        }
    }

    @Override
    public synchronized void updateEvent(boolean local, long userId, Event event) {
        if (local) {
            this.broadcastService.updateEvent(true, userId, event);
        }
        if (this.listeners.containsKey(userId)) {
            for (UpdateListener listener : this.listeners.get(userId)) {
                listener.onUpdateEvent(event);
            }
        }
    }

    @Override
    public synchronized <T1 extends BaseModel, T2 extends BaseModel> void invalidatePermission(boolean local, Class<T1> clazz1, long id1, Class<T2> clazz2, long id2, boolean link) {
        if (link && clazz1.equals(User.class) && clazz2.equals(Device.class) && this.listeners.containsKey(id1)) {
            this.userDevices.get(id1).add(id2);
            this.deviceUsers.put(id2, new HashSet<Long>(List.of(Long.valueOf(id1))));
        }
    }

    public synchronized void updateLog(LogRecord record) {
        Map sessions = this.sessionsByEndpoint.getOrDefault(record.getAddress(), Map.of());
        if (sessions.isEmpty()) {
            String unknownUniqueId = this.unknownByEndpoint.get(record.getAddress());
            if (unknownUniqueId != null && this.showUnknownDevices) {
                record.setUniqueId(unknownUniqueId);
                this.listeners.values().stream().flatMap(Collection::stream).forEach(listener -> listener.onUpdateLog(record));
            }
        } else {
            Map.Entry firstEntry = sessions.entrySet().iterator().next();
            record.setUniqueId((String)firstEntry.getKey());
            record.setDeviceId(((DeviceSession)firstEntry.getValue()).getDeviceId());
            Iterator iterator = this.deviceUsers.getOrDefault(record.getDeviceId(), Set.of()).iterator();
            while (iterator.hasNext()) {
                long userId = (Long)iterator.next();
                for (UpdateListener listener2 : this.listeners.getOrDefault(userId, Set.of())) {
                    listener2.onUpdateLog(record);
                }
            }
        }
    }

    public synchronized void addListener(long userId, UpdateListener listener) throws StorageException {
        Set<UpdateListener> set = this.listeners.get(userId);
        if (set == null) {
            set = new HashSet<UpdateListener>();
            this.listeners.put(userId, set);
            List<Device> devices = this.storage.getObjects(Device.class, new Request((Columns)new Columns.Include("id"), new Condition.Permission(User.class, userId, Device.class)));
            this.userDevices.put(userId, devices.stream().map(BaseModel::getId).collect(Collectors.toSet()));
            devices.forEach(device -> this.deviceUsers.computeIfAbsent(device.getId(), id -> new HashSet()).add(userId));
        }
        set.add(listener);
    }

    public synchronized void removeListener(long userId, UpdateListener listener) {
        Set<UpdateListener> set = this.listeners.get(userId);
        set.remove(listener);
        if (set.isEmpty()) {
            this.listeners.remove(userId);
            this.userDevices.remove(userId).forEach(deviceId -> this.deviceUsers.computeIfPresent((Long)deviceId, (x, userIds) -> {
                userIds.remove(userId);
                return userIds.isEmpty() ? null : userIds;
            }));
        }
    }

    public static interface UpdateListener {
        public void onKeepalive();

        public void onUpdateDevice(Device var1);

        public void onUpdatePosition(Position var1);

        public void onUpdateEvent(Event var1);

        public void onUpdateLog(LogRecord var1);
    }
}

