"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class RollupService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "putRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          seqNo,
          primaryTerm
        } = request.query;
        let method = "ism.putRollup";
        let params = {
          rollupId: id,
          if_seq_no: seqNo,
          if_primary_term: primaryTerm,
          body: JSON.stringify(request.body)
        };

        if (seqNo === undefined || primaryTerm === undefined) {
          method = "ism.createRollup";
          params = {
            rollupId: id,
            body: JSON.stringify(request.body)
          };
        }

        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const putRollupResponse = await callWithRequest(method, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: putRollupResponse
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - putRollup", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "deleteRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const deleteRollupResponse = await callWithRequest("ism.deleteRollup", params);

        if (deleteRollupResponse.result !== "deleted") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: deleteRollupResponse.result
            }
          });
        }

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: true
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - deleteRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "startRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const startResponse = await callWithRequest("ism.startRollup", params);

        const acknowledged = _lodash.default.get(startResponse, "acknowledged");

        if (acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to start rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - startRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "stopRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const stopResponse = await callWithRequest("ism.stopRollup", params);

        const acknowledged = _lodash.default.get(stopResponse, "acknowledged");

        if (acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to stop rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - stopRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "getRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getResponse = await callWithRequest("ism.getRollup", params);
        const metadata = await callWithRequest("ism.explainRollup", params);

        const rollup = _lodash.default.get(getResponse, "rollup", null);

        const seqNo = _lodash.default.get(getResponse, "_seq_no");

        const primaryTerm = _lodash.default.get(getResponse, "_primary_term"); //Form response


        if (rollup) {
          if (metadata) {
            return response.custom({
              statusCode: 200,
              body: {
                ok: true,
                response: {
                  _id: id,
                  _seqNo: seqNo,
                  _primaryTerm: primaryTerm,
                  rollup: rollup,
                  metadata: metadata
                }
              }
            });
          } else return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to load metadata"
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to load rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - getRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "getMappings", async (context, request, response) => {
      try {
        const {
          index
        } = request.body;
        const params = {
          index: index
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const mappings = await callWithRequest("indices.getMapping", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: mappings
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - getMapping:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "getRollups", async (context, request, response) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField
        } = request.query;
        const rollupSortMap = {
          _id: "rollup.rollup_id.keyword",
          "rollup.source_index": "rollup.source_index.keyword",
          "rollup.target_index": "rollup.target_index.keyword",
          "rollup.rollup.enabled": "rollup.enabled"
        };
        const params = {
          from: parseInt(from, 10),
          size: parseInt(size, 10),
          search,
          sortField: rollupSortMap[sortField] || rollupSortMap._id,
          sortDirection
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getRollupResponse = await callWithRequest("ism.getRollups", params);
        const totalRollups = getRollupResponse.total_rollups;
        const rollups = getRollupResponse.rollups.map(rollup => ({
          _seqNo: rollup._seqNo,
          _primaryTerm: rollup._primaryTerm,
          _id: rollup._id,
          rollup: rollup.rollup,
          metadata: null
        })); // Call getExplain if any rollup job exists

        if (totalRollups) {
          // Concat rollup job ids
          const ids = rollups.map(rollup => rollup._id).join(",");
          const explainResponse = await callWithRequest("ism.explainRollup", {
            rollupId: ids
          });

          if (!explainResponse.error) {
            rollups.map(rollup => {
              rollup.metadata = explainResponse[rollup._id];
            });
            return response.custom({
              statusCode: 200,
              body: {
                ok: true,
                response: {
                  rollups: rollups,
                  totalRollups: totalRollups,
                  metadata: explainResponse
                }
              }
            });
          } else return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: explainResponse ? explainResponse.error : "An error occurred when calling getExplain API."
            }
          });
        }

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              rollups: rollups,
              totalRollups: totalRollups,
              metadata: {}
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                rollups: [],
                totalRollups: 0,
                metadata: null
              }
            }
          });
        }

        console.error("Index Management - RollupService - getRollups", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: "Error in getRollups " + err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }
  /**
   * Calls backend Put Rollup API
   */

  /**
   * Calls backend Delete Rollup API
   */

  /**
   * Calls backend Get Rollup API
   */

  /**
   * Performs a fuzzy search request on rollup id
   */


}

exports.default = RollupService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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