"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _constants = require("../utils/constants");

var _helpers = require("../utils/helpers");

var _DataStreamService = require("./DataStreamService");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class IndexService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "getIndices", async (context, request, response) => {
      try {
        // @ts-ignore
        const {
          from,
          size,
          sortField,
          sortDirection,
          terms,
          indices,
          dataStreams,
          showDataStreams,
          expandWildcards,
          exactSearch
        } = request.query;
        const params = {
          index: (0, _helpers.getSearchString)(terms, indices, dataStreams),
          format: "json",
          s: `${sortField}:${sortDirection}`
        };

        if (expandWildcards) {
          params.expand_wildcards = expandWildcards;
        }

        if (exactSearch) {
          params.index = exactSearch;
        }

        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const [recoverys, tasks, indicesResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("cat.recovery", {
          format: "json",
          detailed: true
        }).catch(() => []), callWithRequest("cat.tasks", {
          format: "json",
          detailed: true,
          actions: "indices:data/write/reindex"
        }).catch(() => []), callWithRequest("cat.indices", params), (0, _DataStreamService.getIndexToDataStreamMapping)({
          callAsCurrentUser: callWithRequest
        })]);
        const formattedTasks = tasks.map(item => {
          const {
            description
          } = item;
          const regexp = /reindex from \[([^\]]+)\] to \[([^\]]+)\]/i;
          const matchResult = description.match(regexp);

          if (matchResult) {
            const [, fromIndex, toIndex] = matchResult;
            return { ...item,
              fromIndex,
              toIndex
            };
          } else {
            return { ...item,
              fromIndex: "",
              toIndex: ""
            };
          }
        });
        const onGoingRecovery = recoverys.filter(item => item.stage !== "done"); // Augment the indices with their parent data stream name.

        indicesResponse.forEach(index => {
          index.data_stream = indexToDataStreamMapping[index.index] || null;
          let extraStatus = index.status;

          if (index.health === "green") {
            if (formattedTasks.find(item => item.toIndex === index.index)) {
              extraStatus = "reindex";
            }
          } else {
            if (onGoingRecovery.find(item => item.index === index.index)) {
              extraStatus = "recovery";
            }
          }

          if (extraStatus) {
            index.extraStatus = extraStatus;
          }
        });

        if (sortField === "status") {
          // add new more status to status field so we need to sort
          indicesResponse.sort((a, b) => {
            let flag;
            const aStatus = a.extraStatus;
            const bStatus = b.extraStatus;

            if (sortDirection === "asc") {
              flag = aStatus < bStatus;
            } else {
              flag = aStatus > bStatus;
            }

            return flag ? -1 : 1;
          });
        } // Filtering out indices that belong to a data stream. This must be done before pagination.


        const filteredIndices = showDataStreams ? indicesResponse : indicesResponse.filter(index => index.data_stream === null); // _cat doesn't support pagination, do our own in server pagination to at least reduce network bandwidth

        const fromNumber = parseInt(from, 10);
        const sizeNumber = parseInt(size, 10);
        const paginatedIndices = filteredIndices.slice(fromNumber, fromNumber + sizeNumber);
        const indexNames = paginatedIndices.map(value => value.index);
        const managedStatus = await this._getManagedStatus(request, indexNames); // NOTE: Cannot use response.ok due to typescript type checking

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              indices: paginatedIndices.map(catIndex => ({ ...catIndex,
                managed: managedStatus[catIndex.index] ? "Yes" : "No",
                managedPolicy: managedStatus[catIndex.index]
              })),
              totalIndices: filteredIndices.length
            }
          }
        });
      } catch (err) {
        // Throws an error if there is no index matching pattern
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                indices: [],
                totalIndices: 0
              }
            }
          });
        }

        console.error("Index Management - IndexService - getIndices:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "_getManagedStatus", async (request, indexNames) => {
      try {
        const explainParamas = {
          index: indexNames.toString()
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const explainResponse = await callWithRequest("ism.explain", explainParamas);
        const managed = {};

        for (const indexName in explainResponse) {
          if (indexName === "total_managed_indices") continue;
          const explain = explainResponse[indexName];
          managed[indexName] = explain["index.plugins.index_state_management.policy_id"] === null ? "" : explain["index.plugins.index_state_management.policy_id"];
        }

        return managed;
      } catch (err) {
        // otherwise it could be an unauthorized access error to config index or some other error
        // in which case we will return managed status N/A
        console.error("Index Management - IndexService - _getManagedStatus:", err);
        return indexNames.reduce((accu, value) => ({ ...accu,
          [value]: "N/A"
        }), {});
      }
    });

    _defineProperty(this, "applyPolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId
          }
        };
        const addResponse = await callWithRequest("ism.add", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - applyPolicy:", err); // return { ok: false, error: err.message };

        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "editRolloverAlias", async (context, request, response) => {
      try {
        const {
          alias,
          index
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index,
          body: {
            [_constants.Setting.RolloverAlias]: alias
          }
        };
        const rollOverResponse = await callWithRequest("indices.putSettings", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: rollOverResponse
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - editRolloverAlias", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }

}

exports.default = IndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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