"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configRoutes = configRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helper = require("../utils/helper");
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../common/constants");
var _MDSEnabledClientService = require("../../common/MDSEnabledClientService");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function configRoutes(router, dataSourceEnabled) {
  const getConfigsQuerySchema = {
    from_index: _configSchema.schema.number(),
    max_items: _configSchema.schema.number(),
    query: _configSchema.schema.maybe(_configSchema.schema.string()),
    config_type: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]),
    is_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
    sort_field: _configSchema.schema.string(),
    sort_order: _configSchema.schema.string(),
    config_id_list: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])),
    'smtp_account.method': _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]))
  };
  if (dataSourceEnabled) {
    getConfigsQuerySchema.dataSourceId = _configSchema.schema.string();
  }
  const genericBodyAndDataSourceIdQuery = {
    body: _configSchema.schema.any()
  };
  if (dataSourceEnabled) {
    genericBodyAndDataSourceIdQuery.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const genericParamsAndDataSourceIdQuery = {
    params: _configSchema.schema.any()
  };
  if (dataSourceEnabled) {
    genericParamsAndDataSourceIdQuery.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const updateQuerySchema = {
    body: _configSchema.schema.any(),
    params: _configSchema.schema.object({
      configId: _configSchema.schema.string()
    })
  };
  if (dataSourceEnabled) {
    updateQuerySchema.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const deleteQuerySchema = {
    config_id_list: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])
  };
  if (dataSourceEnabled) {
    deleteQuerySchema.dataSourceId = _configSchema.schema.string();
  }
  router.get({
    path: _common.NODE_API.GET_CONFIGS,
    validate: {
      query: _configSchema.schema.object(getConfigsQuerySchema)
    }
  }, async (context, request, response) => {
    const config_type = (0, _helper.joinRequestParams)(request.query.config_type);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    const encryption_method = (0, _helper.joinRequestParams)(request.query['smtp_account.method']);
    const query = request.query.query;
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getConfigs', {
        from_index: request.query.from_index,
        max_items: request.query.max_items,
        is_enabled: request.query.is_enabled,
        sort_field: request.query.sort_field,
        sort_order: request.query.sort_order,
        config_type,
        ...(query && {
          text_query: query
        }),
        // text_query will exclude keyword fields
        ...(config_id_list && {
          config_id_list
        }),
        ...(encryption_method && {
          'smtp_account.method': encryption_method
        })
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_common.NODE_API.GET_CONFIG}/{configId}`,
    validate: genericParamsAndDataSourceIdQuery
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getConfigById', {
        configId: request.params.configId
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _common.NODE_API.CREATE_CONFIG,
    validate: genericBodyAndDataSourceIdQuery
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.createConfig', {
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_common.NODE_API.UPDATE_CONFIG}/{configId}`,
    validate: updateQuerySchema
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.updateConfigById', {
        configId: request.params.configId,
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: _common.NODE_API.DELETE_CONFIGS,
    validate: {
      query: _configSchema.schema.object(deleteQuerySchema)
    }
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    try {
      const resp = await client('notifications.deleteConfigs', {
        config_id_list
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: _common.NODE_API.GET_AVAILABLE_FEATURES,
    validate: false
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getServerFeatures');
      const config_type_list = resp.allowed_config_type_list;
      const channelTypes = {};
      for (let channel of config_type_list) {
        if (_constants.CHANNEL_TYPE[channel]) {
          channelTypes[channel] = _constants.CHANNEL_TYPE[channel];
        }
      }
      const availableFeature = {
        availableChannels: channelTypes,
        availableConfigTypes: config_type_list,
        tooltipSupport: _lodash.default.get(response, ['plugin_features', 'tooltip_support']) === 'true'
      };
      return response.ok({
        body: availableFeature
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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