"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityPlugin = void 0;
var _migrations = require("./adaptors/integrations/migrations");
var _opensearch_observability_plugin = require("./adaptors/opensearch_observability_plugin");
var _ppl_plugin = require("./adaptors/ppl_plugin");
var _ppl_parser = require("./parsers/ppl_parser");
var _index = require("./routes/index");
var _observability_saved_object = require("./saved_objects/observability_saved_object");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class ObservabilityPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    this.logger = initializerContext.logger.get();
  }
  async setup(core, deps) {
    const {
      assistantDashboards,
      dataSource
    } = deps;
    this.logger.debug('Observability: Setup');
    const router = core.http.createRouter();
    const dataSourceEnabled = !!dataSource;
    const openSearchObservabilityClient = core.opensearch.legacy.createClient('opensearch_observability', {
      plugins: [_ppl_plugin.PPLPlugin, _opensearch_observability_plugin.OpenSearchObservabilityPlugin]
    });
    if (dataSourceEnabled) {
      dataSource.registerCustomApiSchema(_ppl_plugin.PPLPlugin);
      dataSource.registerCustomApiSchema(_opensearch_observability_plugin.OpenSearchObservabilityPlugin);
    }
    // @ts-ignore
    core.http.registerRouteHandlerContext('observability_plugin', (_context, _request) => {
      return {
        logger: this.logger,
        observabilityClient: openSearchObservabilityClient
      };
    });
    const obsPanelType = {
      name: 'observability-panel',
      hidden: false,
      namespaceType: 'single',
      mappings: {
        dynamic: false,
        properties: {
          title: {
            type: 'text'
          },
          description: {
            type: 'text'
          }
        }
      },
      management: {
        importableAndExportable: true,
        getInAppUrl() {
          return {
            path: `/app/management/observability/settings`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return `Observability Settings [${obj.id}]`;
        }
      },
      migrations: {
        '3.0.0': doc => ({
          ...doc,
          description: ''
        }),
        '3.0.1': doc => ({
          ...doc,
          description: 'Some Description Text'
        }),
        '3.0.2': doc => ({
          ...doc,
          dateCreated: parseInt(doc.dateCreated || '0', 10)
        })
      }
    };
    const integrationInstanceType = {
      name: 'integration-instance',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/installed/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          templateName: {
            type: 'text'
          },
          dataSource: {
            type: 'text'
          },
          creationDate: {
            type: 'date'
          },
          assets: {
            type: 'nested'
          }
        }
      }
    };
    const integrationTemplateType = {
      name: 'integration-template',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/available/${obj.attributes.name}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          var _obj$attributes$displ;
          return (_obj$attributes$displ = obj.attributes.displayName) !== null && _obj$attributes$displ !== void 0 ? _obj$attributes$displ : obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          version: {
            type: 'text'
          },
          displayName: {
            type: 'text'
          },
          license: {
            type: 'text'
          },
          type: {
            type: 'text'
          },
          labels: {
            type: 'text'
          },
          author: {
            type: 'text'
          },
          description: {
            type: 'text'
          },
          sourceUrl: {
            type: 'text'
          },
          statics: {
            type: 'nested'
          },
          components: {
            type: 'nested'
          },
          assets: {
            type: 'nested'
          },
          sampleData: {
            type: 'nested'
          }
        }
      },
      migrations: {
        '3.0.0': _migrations.migrateV1IntegrationToV2Integration
      }
    };
    core.savedObjects.registerType(obsPanelType);
    core.savedObjects.registerType(integrationInstanceType);
    core.savedObjects.registerType(integrationTemplateType);

    // Register server side APIs
    (0, _index.setupRoutes)({
      router,
      client: openSearchObservabilityClient,
      dataSourceEnabled
    });
    core.savedObjects.registerType(_observability_saved_object.visualizationSavedObject);
    core.savedObjects.registerType(_observability_saved_object.searchSavedObject);
    core.savedObjects.registerType(_observability_saved_object.notebookSavedObject);
    core.capabilities.registerProvider(() => ({
      observability: {
        show: true
      }
    }));
    assistantDashboards === null || assistantDashboards === void 0 || assistantDashboards.registerMessageParser(_ppl_parser.PPLParsers);
    return {};
  }
  start(_core) {
    this.logger.debug('Observability: Started');
    return {};
  }
  stop() {}
}
exports.ObservabilityPlugin = ObservabilityPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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