"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthentication = void 0;
var _querystring = require("querystring");
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SamlAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "type", _common.AuthType.SAML);
    // Check if we can get the previous tenant information from the expired cookie.
    _defineProperty(this, "redirectSAMlCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
  }
  generateNextUrl(request) {
    let path = this.coreSetup.http.basePath.serverBasePath + (request.url.pathname || '/app/opensearch-dashboards');
    if (request.url.search) {
      path += request.url.search;
    }
    return (0, _querystring.escape)(path);
  }
  async init() {
    this.createExtraStorage();
    const samlAuthRoutes = new _routes.SamlAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.securityClient, this.coreSetup);
    samlAuthRoutes.setupRoutes();
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.saml.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.saml.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.saml.extra_storage.cookie_prefix,
      additionalCookies: this.config.saml.extra_storage.additional_cookies,
      logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers[SamlAuthentication.AUTH_HEADER_NAME] ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }

  // SAML expiry time is set by the IDP and returned via the security backend. Keep alive should not modify this value.
  getKeepAliveExpiry(cookie, request) {
    return cookie.expiryTime;
  }
  getCookie(request, authInfo) {
    const authorizationHeaderValue = request.headers[SamlAuthentication.AUTH_HEADER_NAME];
    (0, _cookie_splitter.setExtraAuthStorage)(request, authorizationHeaderValue, this.getExtraAuthStorageOptions(this.logger));
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: _common.AuthType.SAML,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }

  // Can be improved to check if the token is expiring.
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    return cookie.authType === _common.AuthType.SAML && cookie.username && cookie.expiryTime && (((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue) || this.getExtraAuthStorageValue(request, cookie));
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectSAMlCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions(this.logger));
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const headers = {};
    if ((_cookie$credentials3 = cookie.credentials) !== null && _cookie$credentials3 !== void 0 && _cookie$credentials3.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        headers[SamlAuthentication.AUTH_HEADER_NAME] = extraAuthStorageValue;
      } catch (error) {
        this.logger.error(error);
        // @todo Re-throw?
        // throw error;
      }
    } else {
      var _cookie$credentials4;
      headers[SamlAuthentication.AUTH_HEADER_NAME] = (_cookie$credentials4 = cookie.credentials) === null || _cookie$credentials4 === void 0 ? void 0 : _cookie$credentials4.authHeaderValue;
    }
    return headers;
  }
}
exports.SamlAuthentication = SamlAuthentication;
_defineProperty(SamlAuthentication, "AUTH_HEADER_NAME", 'authorization');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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